/*
 * Copyright (C) 2001-2003 Clemens Fuchslocher <clfuit00@fht-esslingen.de>
 *
 * bk_edit_tree.c - 10.07.2003 - v0.9 - converting capability
 *                  17.04.2003 - v0.8 - sort order
 *                  05.04.2003 - v0.7 - edit with middle mouse click
 *                  11.02.2003 - v0.6 - pos1, end
 *                  23.12.2002 - v0.5 - plugin api clean-up
 *                  26.02.2002 - v0.4 - plugin functionality
 *                  05.09.2001 - v0.3 - undo functionality
 *                  08.08.2001 - v0.2 - mozilla support
 *                  24.07.2001 - v0.1
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */

#include <time.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h>

#include "bk_edit.h"
#include "bk_edit_tree.h"
#include "bk_edit_undo.h"
#include "bk_edit_icon.h"
#include "bk_edit_edit.h"
#include "bk_edit_sort.h"
#include "bk_edit_dialog_main.h"

#include "config.h"
#include "plugin.h"
#include "convert.h"

extern config *conf;
extern convert conv;

extern bk_edit_tree tree;
extern bk_edit_dialog_main dialog_main;


int bk_edit_tree_add_root_node (bk_edit_tree_data *node_data)
{
	int i;

	if (conv.mode == CONVERT)
	{
		bk_edit_tree_data *data = (bk_edit_tree_data *) malloc (sizeof (bk_edit_tree_data));

		data->id = 0;
		data->type = TOP;
		data->order = 0;

		for (i = 0; i < ELEMENTS; i++)
		{
			data->elements[i] = strdup (node_data->elements[i]  == NULL ? "" : node_data->elements[i]);
		}

		conv.tree = conv.parent = g_node_new ((gpointer) data);
		if (g_stack_push (conv.parent_stack, (void *) conv.parent) == G_STACK_FULL)
		{
			fprintf (stderr, "%s[%d]: bk_edit_tree_add_root_node: parent stack overrun (fatal error).\n", __FILE__, __LINE__);
			return BK_EDIT_TREE_KO;
		}
	}
	else
	{
		bk_edit_tree_data *root_data;

		tree.id = 0;
		tree.order = 0;

		root_data = (bk_edit_tree_data *) malloc (sizeof (bk_edit_tree_data));
		root_data->type = TOP;
		root_data->id = tree.id++;
		root_data->order = tree.order++;

		if (g_stack_push (tree.order_stack, (void *) tree.order) == G_STACK_FULL)
		{
			fprintf (stderr, "%s[%d]: bk_edit_tree_add_root_node: order stack overrun (fatal error).\n", __FILE__, __LINE__);
			return BK_EDIT_TREE_KO;
		}

		for (i = NAME; i < ELEMENTS; i++)
		{
			root_data->elements[i] = strdup (node_data->elements[i]  == NULL ? "" : node_data->elements[i]);
		}

		tree.tree_node = gtk_ctree_insert_node (GTK_CTREE (tree.tree), NULL, NULL, root_data->elements, 4, bookmark_folder_closed_pix, bookmark_folder_closed_mask, bookmark_folder_open_pix, bookmark_folder_open_mask, FALSE, FALSE);
		gtk_ctree_node_set_row_data_full (GTK_CTREE (tree.tree), tree.tree_node, root_data, bk_edit_tree_destroy);

		if (g_stack_push (tree.tree_stack, (void *) tree.tree_node) == G_STACK_FULL)
		{
			fprintf (stderr, "%s[%d]: bk_edit_tree_add_root_node: folder stack overrun (fatal error).\n", __FILE__, __LINE__);
			return BK_EDIT_TREE_KO;
		}
	}

	return BK_EDIT_TREE_OK;
}


int bk_edit_tree_add_bookmark_node (bk_edit_tree_data *node_data)
{
	int i;

	if (conv.mode == CONVERT)
	{
		bk_edit_tree_data *data = (bk_edit_tree_data *) malloc (sizeof (bk_edit_tree_data));

		data->id = 0;
		data->type = BOOKMARK;
		data->order = 0;

		for (i = 0; i < ELEMENTS; i++)
		{
			data->elements[i] = strdup (node_data->elements[i]  == NULL ? "" : node_data->elements[i]);
		}

		g_node_append_data (conv.parent, (gpointer) data);
	}
	else
	{
		bk_edit_tree_data *bookmark_data;
		GtkCTreeNode *node;

		bookmark_data = (bk_edit_tree_data *) malloc (sizeof (bk_edit_tree_data));
		bookmark_data->type = BOOKMARK;
		bookmark_data->id = tree.id++;
		bookmark_data->order = tree.order++;

		for (i = NAME; i < ELEMENTS; i++)
		{
			bookmark_data->elements[i] = strdup (node_data->elements[i]  == NULL ? "" : node_data->elements[i]);
		}

		node = gtk_ctree_insert_node (GTK_CTREE (tree.tree), tree.tree_node, NULL, bookmark_data->elements, 4, bookmark_item_pix, bookmark_item_mask, bookmark_item_pix, bookmark_item_mask, TRUE, FALSE);
		gtk_ctree_node_set_row_data_full (GTK_CTREE (tree.tree), node, bookmark_data, bk_edit_tree_destroy);
		bk_edit_tree_date_update (GTK_CTREE (tree.tree), node, bookmark_data);
	}

	return BK_EDIT_TREE_OK;
}


int bk_edit_tree_add_folder_node (bk_edit_tree_data *node_data)
{
	int i;

	if (conv.mode == CONVERT)
	{
		bk_edit_tree_data *data = (bk_edit_tree_data *) malloc (sizeof (bk_edit_tree_data));

		if (g_stack_push (conv.parent_stack, (void *) conv.parent) == G_STACK_FULL)
		{
			fprintf (stderr, "%s[%d]: bk_edit_tree_add_folder_node: parent stack overrun (fatal error).\n", __FILE__, __LINE__);
			return BK_EDIT_TREE_KO;
		}

		data->id = 0;
		data->type = FOLDER;
		data->order = 0;

		for (i = 0; i < ELEMENTS; i++)
		{
			data->elements[i] = strdup (node_data->elements[i]  == NULL ? "" : node_data->elements[i]);
		}

		conv.parent = g_node_append_data (conv.parent, (gpointer) data);
	}
	else
	{
		bk_edit_tree_data *folder_data;

		folder_data = (bk_edit_tree_data *) malloc (sizeof (bk_edit_tree_data));
		folder_data->type = FOLDER;
		folder_data->id = tree.id++;
		folder_data->order = tree.order++;

		if (g_stack_push (tree.order_stack, (void *) tree.order) == G_STACK_FULL)
		{
			fprintf (stderr, "%s[%d]: bk_edit_tree_add_folder_node: order stack overrun (fatal error).\n", __FILE__, __LINE__);
			return BK_EDIT_TREE_KO;
		}
		tree.order = 0;

		for (i = NAME; i < ELEMENTS; i++)
		{
			folder_data->elements[i] = strdup (node_data->elements[i]  == NULL ? "" : node_data->elements[i]);
		}

		if (g_stack_push (tree.tree_stack, (void *) tree.tree_node) == G_STACK_FULL)
		{
			fprintf (stderr, "%s[%d]: bk_edit_tree_add_folder_node: folder stack overrun (fatal error).\n", __FILE__, __LINE__);
			return BK_EDIT_TREE_KO;
		}

		tree.tree_node = (GtkCTreeNode *) gtk_ctree_insert_node (GTK_CTREE (tree.tree), tree.tree_node, NULL, folder_data->elements, 4, bookmark_folder_closed_pix, bookmark_folder_closed_mask, bookmark_folder_open_pix, bookmark_folder_open_mask, FALSE, FALSE);
		gtk_ctree_node_set_row_data_full (GTK_CTREE (tree.tree), tree.tree_node, folder_data, bk_edit_tree_destroy);
		bk_edit_tree_date_update (GTK_CTREE (tree.tree), tree.tree_node, folder_data);
	}

	return BK_EDIT_TREE_OK;
}


int bk_edit_tree_add_separator (bk_edit_tree_data *node_data)
{
	int i;

	if (conv.mode == CONVERT)
	{
		bk_edit_tree_data *data = (bk_edit_tree_data *) malloc (sizeof (bk_edit_tree_data));

		data->id = 0;
		data->type = SEPARATOR;
		data->order = 0;

		for (i = 0; i < ELEMENTS; i++)
		{
			data->elements[i] = strdup (node_data->elements[i]  == NULL ? "" : node_data->elements[i]);
		}

		g_node_append_data (conv.parent, (gpointer) data);
	}
	else
	{
		bk_edit_tree_data *separator_data;
		GtkCTreeNode *node;

		separator_data = (bk_edit_tree_data *) malloc (sizeof (bk_edit_tree_data));
		separator_data->type = SEPARATOR;
		separator_data->id = tree.id++;
		separator_data->order = tree.order++;

		for (i = NAME; i < ELEMENTS; i++)
		{
			separator_data->elements[i] = strdup (node_data->elements[i]  == NULL ? "" : node_data->elements[i]);
		}

		node = gtk_ctree_insert_node (GTK_CTREE (tree.tree), tree.tree_node, NULL, separator_data->elements, 4, NULL, NULL, NULL, NULL, TRUE, FALSE);
		gtk_ctree_node_set_row_data_full (GTK_CTREE (tree.tree), node, separator_data, bk_edit_tree_destroy);
	}

	return BK_EDIT_TREE_OK;
}


int bk_edit_tree_leave_folder (void)
{
	if (conv.mode == CONVERT)
	{
		if ((conv.parent = (GNode *) g_stack_pop (conv.parent_stack)) == NULL)
		{
			fprintf (stderr, "%s[%d]: bk_edit_tree_leave_folder: parent stack underrun (fatal error).\n", __FILE__, __LINE__);
			return BK_EDIT_TREE_KO;
		}
	}
	else
	{
		void *order = g_stack_pop (tree.order_stack);
		if (order == NULL)
		{
			fprintf (stderr, "%s[%d]: bk_edit_tree_leave_folder: order stack underrun (fatal error).\n", __FILE__, __LINE__);
			return BK_EDIT_TREE_KO;
		}
		tree.order = (int) order;

		if ((tree.tree_node = (GtkCTreeNode *) g_stack_pop (tree.tree_stack)) == NULL)

		{
			fprintf (stderr, "%s[%d]: bk_edit_tree_leave_folder: folder stack underrun (fatal error).\n", __FILE__, __LINE__);
			return BK_EDIT_TREE_KO;
		}
	}

	return BK_EDIT_TREE_OK;
}


void bk_edit_tree_save (plugin *plug, char *filename)
{
	bk_edit_tree_data *node_data = (bk_edit_tree_data *) gtk_ctree_node_get_row_data (GTK_CTREE (tree.tree), GTK_CTREE_NODE (GTK_CLIST (tree.tree)->row_list));

	if (plug->save_begin (filename, node_data) != PLUGIN_OK)
	{
		return;
	}

	bk_edit_tree_save_traverse (plug,  GTK_CTREE_ROW (GTK_CTREE_NODE (GTK_CLIST (tree.tree)->row_list))->children);
	plug->save_end ();
}


void bk_edit_tree_save_traverse (plugin *plug, GtkCTreeNode *node)
{
	while (node != NULL)
	{
		bk_edit_tree_data *node_data = (bk_edit_tree_data *) gtk_ctree_node_get_row_data (GTK_CTREE (tree.tree), node);
		if (node_data != NULL)
		{
			if (node_data->type == FOLDER)
			{
				plug->save_folder_in (node_data);
				bk_edit_tree_save_traverse (plug, GTK_CTREE_ROW (node)->children);
				plug->save_folder_out ();
			}
			else if (node_data->type == BOOKMARK)
			{
				plug->save_bookmark (node_data);
			}
			else if (node_data->type == SEPARATOR)
			{
				plug->save_separator (node_data);
			}
		}
		node = GTK_CTREE_ROW (node)->sibling;
	}
}


void bk_edit_tree_destroy (gpointer data)
{
	gint i;

	for (i = NAME; i < ELEMENTS; i++)
	{
		free (((bk_edit_tree_data *) data)->elements[i]);
	}
	free ((bk_edit_tree_data *) data);
}


void bk_edit_tree_select_row (GtkCTree *ctree, GtkCTreeNode *node, gint column, gpointer null)
{
	tree.selected_node = node;
}


void bk_edit_tree_unselect_row (GtkCTree *ctree, GtkCTreeNode *node, gint column, gpointer null)
{
	tree.selected_node = NULL;
}


void bk_edit_tree_button_press_event (GtkWidget *ctree, GdkEventButton *event, gpointer data)
{
	gint row = -1, col = -1;

	if (event->type == GDK_KEY_PRESS)
	{
	}
	else if (event->type == GDK_BUTTON_PRESS)
	{
		gtk_clist_get_selection_info (GTK_CLIST (ctree), event->x, event->y, &row, &col);

		if ((row == -1) && (col == -1))
		{
			return;
		}

		if (event->button == 1) /* dnd */
		{
			tree.selected_node = gtk_ctree_node_nth (GTK_CTREE (tree.tree), row);

			if (gtk_ctree_is_hot_spot (GTK_CTREE (ctree), event->x, event->y))
			{
				gtk_clist_select_row (GTK_CLIST (ctree), row, col);
			}
		}
		else if (event->button == 2)
		{
			if (config_middle_mouse_get (conf))
			{
				gtk_clist_select_row (GTK_CLIST (ctree), row, col);
				bk_edit_edit ();
			}
		}
		else if (event->button == 3) /* popup */
		{
			gtk_clist_select_row (GTK_CLIST (ctree), row, col);
			gtk_item_factory_popup (GTK_ITEM_FACTORY (data), event->x_root, event->y_root, event->button, event->time);
		}
	}
}


void bk_edit_tree_move (GtkCTree *ctree, GtkCTreeNode *node, GtkCTreeNode *new_parent, GtkCTreeNode *new_sibling, gpointer data)
{
	if (gtk_signal_n_emissions_by_name (GTK_OBJECT (tree.tree), "tree_move") > 1) /* avoid recursion */
	{
		gtk_signal_emit_stop_by_name (GTK_OBJECT (ctree), "tree_move");
		return;
	}

	if (new_parent == NULL) /* you can't move something out of the first folder */
	{
		gtk_signal_emit_stop_by_name (GTK_OBJECT (ctree), "tree_move");
		return;
	}

	gtk_signal_emit_stop_by_name (GTK_OBJECT (ctree), "tree_move");

	/* move */
	gtk_ctree_move (GTK_CTREE (tree.tree), node, new_parent, new_sibling);

	/*
	 * When the tree is sorted in some way, you can only move
	 * a bookmark or a folder if the target is also a folder.
	 */
	if (bk_edit_sort_menu_state_is_unsorted () == KO)
	{
		int x, y, row, column;

		gtk_widget_get_pointer (tree.tree, &x, &y);

		if (gtk_clist_get_selection_info (GTK_CLIST (tree.tree), x, y, &row, &column) == 1)
		{
			GtkCTreeNode *folder = gtk_ctree_node_nth (GTK_CTREE (tree.tree), row - 1);
			if (folder != NULL)
			{
				bk_edit_tree_data *node_data = (bk_edit_tree_data *) gtk_ctree_node_get_row_data (GTK_CTREE (tree.tree), node);
				bk_edit_tree_data *folder_data = (bk_edit_tree_data *) gtk_ctree_node_get_row_data (GTK_CTREE (tree.tree), folder);
				bk_edit_tree_data *new_parent_data = (bk_edit_tree_data *) gtk_ctree_node_get_row_data (GTK_CTREE (tree.tree), new_parent);

				if (folder_data->type != FOLDER)
				{
					gtk_signal_emit_stop_by_name (GTK_OBJECT (ctree), "tree_move");
					return;
				}

				if (strcmp (new_parent_data->elements[NAME], folder_data->elements[NAME]) != 0)
				{
					gtk_signal_emit_stop_by_name (GTK_OBJECT (ctree), "tree_move");
					return;
				}

				/* undo */
				bk_edit_undo_add_move_item (node, new_parent, new_sibling);

				/*
				 * The node will get the highest order. So it apears at the
				 * end of the tree when we are activating the unsorted view.
				 */
				node_data->order = bk_edit_sort_get_highest_order (new_parent, node);
			}
		}
		else
		{
			gtk_signal_emit_stop_by_name (GTK_OBJECT (ctree), "tree_move");
			return;
		}
	}
	else
	{
		bk_edit_tree_data *parent_data = (bk_edit_tree_data *) gtk_ctree_node_get_row_data (GTK_CTREE (tree.tree), GTK_CTREE_ROW (node)->parent);
		bk_edit_tree_data *new_parent_data = (bk_edit_tree_data *) gtk_ctree_node_get_row_data (GTK_CTREE (tree.tree), new_parent);

		/* undo */
		bk_edit_undo_add_move_item (node, new_parent, new_sibling);

		if (parent_data->id == new_parent_data->id)
		{
			bk_edit_sort_reorder_folder (GTK_CTREE_ROW (node)->parent, NULL);
		}
		else
		{
			bk_edit_sort_reorder_folder (GTK_CTREE_ROW (node)->parent, node);
		}
	}

	gtk_ctree_select (GTK_CTREE (tree.tree), node);
	gtk_ctree_unselect (GTK_CTREE (tree.tree), node);

	tree.selected_node = NULL;
}


void bk_edit_tree_move_after (GtkCTree *ctree, GtkCTreeNode *node, GtkCTreeNode *new_parent, GtkCTreeNode *new_sibling, gpointer data)
{
	if (bk_edit_sort_menu_state_is_unsorted () != OK)
	{
		bk_edit_sort_sort (bk_edit_sort_menu_get_menu_state (), gtk_ctree_sort_node, new_parent);
		return;
	}

	bk_edit_sort_reorder_folder (GTK_CTREE_ROW (node)->parent, NULL);
}


void bk_edit_tree_date_update (GtkCTree *tree, GtkCTreeNode *node, bk_edit_tree_data *node_data)
{
	gint i;

	for (i = ADD_DATE; i <= LAST_MODIFIED; i++)
	{
		time_t timestamp = atoi (node_data->elements[i]);
		char *timestr = ctime (&timestamp);

		if (timestamp == 0)
		{
			timestr = "0";
		}
		else
		{
			timestr[strlen (timestr) - 1] = '\0';
		}

		gtk_ctree_node_set_text (GTK_CTREE (tree), node, i, timestr);
	}
}


void bk_edit_tree_expand (void)
{
	gtk_ctree_expand_recursive (GTK_CTREE (tree.tree), NULL);
}


void bk_edit_tree_collapse (void)
{
	gtk_ctree_collapse_recursive (GTK_CTREE (tree.tree), NULL);
}


gboolean bk_edit_tree_to_gnode (GtkCTree *ctree, guint depth, GNode *gnode, GtkCTreeNode *cnode, gpointer data)
{
	gint i;

	bk_edit_tree_data *node_data = (bk_edit_tree_data *) malloc (sizeof (bk_edit_tree_data));
	if (node_data == NULL)
	{
		fprintf (stderr, "%s[%d]: bk_edit_undo_tree2gnode: out of memory (fatal error).\n", __FILE__, __LINE__);
		return FALSE;
	}

	node_data->type = ((bk_edit_tree_data *) GTK_CTREE_ROW (cnode)->row.data)->type;
	if (node_data->type == TOP)
	{
		node_data->type = FOLDER;
	}
	node_data->id = ((bk_edit_tree_data *) GTK_CTREE_ROW (cnode)->row.data)->id;
	for (i = 0; i < ELEMENTS; i++)
	{
		node_data->elements[i] = strdup (((bk_edit_tree_data *) GTK_CTREE_ROW (cnode)->row.data)->elements[i]);
	}

	node_data->order = ((bk_edit_tree_data *) GTK_CTREE_ROW (cnode)->row.data)->order;

	gnode->data = node_data;

	return TRUE;
}


gboolean bk_edit_gnode_to_tree (GtkCTree *ctree, guint depth, GNode *gnode, GtkCTreeNode *cnode, gpointer data)
{
	gint i;

	bk_edit_tree_data *node_data = (bk_edit_tree_data *) malloc (sizeof (bk_edit_tree_data));
	if (node_data == NULL)
	{
		fprintf (stderr, "%s[%d]: bk_edit_undo_gnode2tree: out of memory (fatal error).\n", __FILE__, __LINE__);
		return FALSE;
	}

	node_data->type = ((bk_edit_tree_data *) gnode->data)->type;
	node_data->id = ((bk_edit_tree_data *) gnode->data)->id;
	node_data->order = ((bk_edit_tree_data *) gnode->data)->order;

	for (i = 0; i < ELEMENTS; i++)
	{
		node_data->elements[i] = strdup (((bk_edit_tree_data *) gnode->data)->elements[i]);
	}

	gtk_ctree_node_set_row_data_full (GTK_CTREE (tree.tree), cnode, node_data, bk_edit_tree_destroy);

	switch (node_data->type)
	{
		case BOOKMARK:
		{
			gtk_ctree_set_node_info (GTK_CTREE (tree.tree), cnode, node_data->elements[NAME], 4, bookmark_item_pix, bookmark_item_mask, bookmark_item_pix, bookmark_item_mask, TRUE, FALSE);
			for (i = URI; i < ALIASOF; i++)
			{
				gtk_ctree_node_set_text (GTK_CTREE (tree.tree), cnode, i, node_data->elements[i]);
			}
			bk_edit_tree_date_update (GTK_CTREE (tree.tree), cnode, node_data);

			break;
		}

		case TOP:
		{
			gtk_ctree_set_node_info (GTK_CTREE (tree.tree), cnode, node_data->elements[NAME], 4,  bookmark_folder_closed_pix, bookmark_folder_closed_mask, bookmark_folder_open_pix, bookmark_folder_open_mask, FALSE, TRUE);

			break;
		}

		case FOLDER:
		{
			gtk_ctree_set_node_info (GTK_CTREE (tree.tree), cnode, node_data->elements[NAME], 4,  bookmark_folder_closed_pix, bookmark_folder_closed_mask, bookmark_folder_open_pix, bookmark_folder_open_mask, FALSE, TRUE);
			for (i = URI; i < ALIASOF; i++)
			{
				gtk_ctree_node_set_text (GTK_CTREE (tree.tree), cnode, i, node_data->elements[i]);
			}
			bk_edit_tree_date_update (GTK_CTREE (tree.tree), cnode, node_data);

			break;
		}

		case SEPARATOR:
		{
			gtk_ctree_set_node_info (GTK_CTREE (tree.tree), cnode, node_data->elements[NAME], 4, NULL, NULL, NULL, NULL, TRUE, FALSE);
			for (i = URI; i < ALIASOF; i++)
			{
				gtk_ctree_node_set_text (GTK_CTREE (tree.tree), cnode, i, node_data->elements[i]);
			}

			break;
		}
	}

	return TRUE;
}


void bk_edit_tree_destroy_gnode_tree (GNode *gnode_tree)
{
	g_node_traverse (gnode_tree, G_IN_ORDER, G_TRAVERSE_ALL, -1, bk_edit_destroy_gnode, NULL);
	g_node_destroy (gnode_tree);
}


gboolean bk_edit_destroy_gnode (GNode *node, gpointer data)
{
	int i;

	for (i = NAME; i < ELEMENTS; i++)
	{
		free (((bk_edit_tree_data *) node->data)->elements[i]);
	}
	free ((bk_edit_tree_data *) node->data);

	return FALSE;
}


void bk_edit_tree_key_pressed (GtkCTree *ctree, GdkEventKey *event, gpointer *data)
{
	if (event == NULL)
	{
		return;
	}

	switch (event->keyval)
	{
		case GDK_End:
		{
			if (GTK_CLIST (tree.tree)->focus_row != (GTK_CLIST (tree.tree)->rows - 1))
			{
				GTK_CLIST (tree.tree)->focus_row = GTK_CLIST (tree.tree)->rows - 1;
				gtk_clist_moveto (GTK_CLIST (tree.tree), GTK_CLIST (tree.tree)->rows - 1, 0, 0, 0);
				if (GTK_CLIST (tree.tree)->freeze_count == 0)
				{
					gtk_widget_draw (GTK_WIDGET (tree.tree), NULL);
				}
			}
			break;
		}

		case GDK_Home:
		{
			if (GTK_CLIST (tree.tree)->focus_row != 0)
			{
				GTK_CLIST (tree.tree)->focus_row = 0;
				gtk_clist_moveto (GTK_CLIST (tree.tree), 0, 0, 0, 0);
				if (GTK_CLIST (tree.tree)->freeze_count == 0)
				{
					gtk_widget_draw (GTK_WIDGET (tree.tree), NULL);
				}
			}
			break;
		}

		case GDK_Delete:
		{
			bk_edit_delete ();
			break;
		}
	}
}

