/*
 * Copyright (C) 2002-2003 Clemens Fuchslocher <clfuit00@fht-esslingen.de>
 *
 * bk_edit_misc.c - 10.07.2003 - v0.8 - bk_edit_misc_create_backup_file
 *                  18.05.2003 - v0.7 - bk_edit_misc_html_entities_encode
 *                  15.05.2003 - v0.6 - bugfix - HTML character entity references
 *                    (Reported by Michael S. Moulton <mmoulton@vt.edu>)
 *                  05.04.2003 - v0.5 - bk_edit_misc_mixed_utf8_to_latin1
 *                                      bk_edit_misc_utf8_to_latin1_in_place
 *                                      bk_edit_misc_utf8_to_latin1
 *                                      bk_edit_misc_latin1_to_utf8
 *                                      bk_edit_misc_remove_char_from_str
 *                                      bk_edit_misc_xml_entities_encode
 *                                      bk_edit_misc_xml_entities_decode
 *                                      bk_edit_misc_html_entities_encode_without_utf8
 *                                      bk_edit_misc_html_entities_decode
 *                  15.02.2003 - v0.4 - bk_edit_misc_clist_get_selection
 *                                      bk_edit_misc_strescape
 *                  12.02.2003 - v0.3 - bk_edit_misc_strcasestr
 *                                      bk_edit_misc_input_create
 *                                      bk_edit_misc_get_menu_option
 *                  11.02.2003 - v0.2 - bk_edit_misc_get_user_home_dir
 *                  23.12.2002 - v0.1 - bk_edit_misc_set_menu_option
 *                                      bk_edit_misc_toggle_widget
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */

#include <time.h>
#include <stdio.h>
#include <errno.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include <sys/stat.h>

#include <pwd.h>

#include <gtk/gtk.h>

#include <libxml/parser.h>
#include <libxml/parserInternals.h>

#include "bk_edit.h"
#include "bk_edit_icon.h"
#include "bk_edit_misc.h"
#include "bk_edit_dialog_info.h"

#include "convert.h"

extern convert conv;


int bk_edit_misc_set_menu_option (GtkOptionMenu *menu_option, char *set_to_this)
{
	GList *iter = GTK_MENU_SHELL (menu_option->menu)->children;

	int index = 0;

	if (set_to_this == NULL)
	{
		return -1;
	}

	for (; iter != NULL; iter = g_list_next (iter))
	{
		GtkMenuItem *item = (GtkMenuItem *) iter->data;
		if (item != NULL)
		{
			if (item->item.bin.child != NULL)
			{
				gchar *name;

				gtk_label_get (GTK_LABEL (item->item.bin.child), &name);
				if (!strcmp (name, set_to_this))
				{
					gtk_menu_shell_select_item (GTK_MENU_SHELL (menu_option->menu), GTK_WIDGET (item));
					break;
				}
			}
		}
		index++;
	}
	gtk_option_menu_set_history (menu_option, index);

	return index;
}


char *bk_edit_misc_get_menu_option (GtkOptionMenu *menu_option)
{
	char *option;

	if (GTK_BIN (menu_option)->child)
	{
		GtkWidget *child = GTK_BIN (menu_option)->child;
		if (GTK_IS_LABEL (child))
		{
			gtk_label_get (GTK_LABEL (child), &option);
		}
	}

	return option;
}


void bk_edit_misc_toggle_widget (GtkWidget *widget, int flag)
{
	if (flag == SHOW)
	{
		gtk_widget_show (widget);
	}
	else
	{
		gtk_widget_hide (widget);
	}
}


char *bk_edit_misc_get_user_home_dir (void)
{
	char *home = getenv ("HOME");
	if (home == NULL)
	{
		struct passwd *user = getpwuid (getuid ());
		if (user == NULL)
		{
			fprintf (stderr, "%s[%d]: WARNING: /tmp/ will be used as home directory.\n", __FILE__, __LINE__);
			return "/tmp/";
		}

		home = user->pw_dir;
	}

	return home;
}


void bk_edit_misc_input_create (bk_edit_input *input, gchar *title)
{
	input->entry = gtk_entry_new ();
	input->label = gtk_label_new (title);
	gtk_misc_set_alignment (GTK_MISC (input->label), 0.0, 0.5);
}


int bk_edit_misc_get_selection (GtkCList *list)
{
	if (list->selection == NULL)
	{
		return -1;
	}

	return (int) list->selection->data;
}


int bk_edit_misc_mixed_utf8_to_latin1 (unsigned char *out, int *out_len, unsigned char *in, int *in_len)
{
	unsigned char *in_index = in;
	int in_size = *in_len;

	unsigned char *out_index = out;
	int out_size = *out_len;

	while (UTF8Toisolat1 (out_index, &out_size, in_index, &in_size) < 0)
	{
		if (((out_index + out_size) - out) >= *out_len)
		{
			fprintf (stderr, "%s[%d]: convert_mixed_utf8_to_latin1 (): no free space in out buffer.\n", __FILE__, __LINE__);
			return KO;
		}

		out_index += out_size;
		*out_index = *(in_index + in_size);
		out_index++;
		out_size = *out_len - out_size;

		in_index += ++in_size;
		in_size = strlen (in_index);
	}
	out_index += out_size;

	*out_len = (out_index - out);

	return OK;
}


int bk_edit_misc_utf8_to_latin1_in_place (char **text)
{
	if (*text != NULL)
	{
		int in_len = strlen (*text);
		char *in = *text;

		int out_len = in_len;
		char *out = (char *) malloc (out_len + 1);
		if (out == NULL)
		{
			fprintf (stderr, "%s[%d]: malloc (): %s\n", __FILE__, __LINE__, strerror (errno));
			return KO;
		}

		if (UTF8Toisolat1 (out, &out_len, in, &in_len) < 0)
		{
			fprintf (stderr, "%s[%d]: UTF8Toisolat1 ()\n", __FILE__, __LINE__);
		}
		out[out_len] = '\0';

		free (*text);
		*text = out;
	}

	return OK;
}


char *bk_edit_misc_utf8_to_latin1 (char *in)
{
	if (in != NULL)
	{
		int in_len = strlen (in);

		int out_len = in_len;
		unsigned char *out = (unsigned char *) malloc (in_len + 1);

		if (out == NULL)
		{
			fprintf (stderr, "%s[%d]: malloc (): %s\n", __FILE__, __LINE__, strerror (errno));
			return NULL;
		}

		if (bk_edit_misc_mixed_utf8_to_latin1 (out, &out_len, in, &in_len) != OK)
		{
			fprintf (stderr, "%s[%d]: convert_mixed_utf8_to_latin1 ()\n", __FILE__, __LINE__);
			free (out);
			return NULL;
		}
		out[out_len] = '\0';

		return out;
	}

	return NULL;
}


char *bk_edit_misc_latin1_to_utf8 (char *in)
{
	if (in != NULL)
	{
		int in_len = strlen (in) * 3;

		int out_len = in_len;
		unsigned char *out = (unsigned char *) malloc (in_len + 1);
		if (out == NULL)
		{
			fprintf (stderr, "%s[%d]: malloc (): %s\n", __FILE__, __LINE__, strerror (errno));
			return NULL;
		}

		if (isolat1ToUTF8 (out, &out_len, in, &in_len) < 0)
		{
			fprintf (stderr, "%s[%d]: isolat1ToUTF8 ()\n", __FILE__, __LINE__);
		}
		out[out_len] = '\0';

		return out;
	}

	return NULL;
}


char *bk_edit_misc_html_entities_encode_without_utf8 (char *in, char *charset)
{
	GString *out = g_string_sized_new (42);
	char *o;

	if (in == NULL)
	{
		return NULL;
	}

	while (*in != '\0')
	{
		if ((unsigned char) *in > 0x7F)
		{
			g_string_append_c (out, *in);
		}
		else if ((charset == NULL) || strchr (charset, *in))
		{
			if (!isalnum (*in) && !isspace (*in))
			{
				const htmlEntityDesc *entity = htmlEntityValueLookup ((unsigned char) *in);
				if (entity == NULL)
				{
					g_string_append_c (out, *in);
				}
				else
				{
					g_string_append_c (out, '&');
					g_string_append (out, entity->name);
					g_string_append_c (out, ';');
				}
			}
			else
			{
				g_string_append_c (out, *in);
			}
		}
		else
		{
			g_string_append_c (out, *in);
		}

		in++;
	}

	o = out->str;
	g_string_free (out, FALSE);

	return o;
}


char *bk_edit_misc_html_entities_encode (char *in, char *charset)
{
	GString *out = g_string_sized_new (42);
	char *o;

	if (in == NULL)
	{
		return NULL;
	}

	while (*in != '\0')
	{
		if ((charset == NULL) || strchr (charset, *in))
		{
			if (!isalnum (*in) && !isspace (*in))
			{
				const htmlEntityDesc *entity = htmlEntityValueLookup ((unsigned char) *in);
				if (entity == NULL)
				{
					g_string_append_c (out, *in);
				}
				else
				{
					g_string_append_c (out, '&');
					g_string_append (out, entity->name);
					g_string_append_c (out, ';');
				}
			}
			else
			{
				g_string_append_c (out, *in);
			}
		}
		else
		{
			g_string_append_c (out, *in);
		}

		in++;
	}

	o = out->str;
	g_string_free (out, FALSE);

	return o;
}


char *bk_edit_misc_html_entities_decode (char *in)
{
	char *out, *o;
	char *i = in;

	if (in == NULL)
	{
		return NULL;
	}

	out = (char *) malloc (strlen (in) + 1);
	if (out == NULL)
	{
		return NULL;
	}
	o = out;

	while (*i != '\0')
	{
		if (*i == '&')
		{
			char *end = strchr (i, ';');
			if (end != NULL)
			{
				char *ii = i + 1;

				char *e = (char *) alloca (end - ii + 1);
				char *ee = e;

				const htmlEntityDesc *entity;

				while (ii != end)
				{
					*ee++ = *ii++;
				}
				*ee = '\0';

				entity = htmlEntityLookup ((const xmlChar *) e);
				if (entity != NULL)
				{
					*o = entity->value;
					i = end;
				}
				else
				{
					fprintf (stderr, "%s[%d]: htmlEntityLookup (%s)\n", __FILE__, __LINE__, e);
					*o = *i;
				}
			}
			else
			{
				*o = *i;
			}
		}
		else
		{
			*o = *i;
		}

		i++;
		o++;
	}
	*o = '\0';

	return out;
}


char *bk_edit_misc_xml_entities_encode (char *in)
{
	char *inn = bk_edit_misc_latin1_to_utf8 (in);
	char *out = xmlEncodeEntitiesReentrant (NULL, inn);

	free (inn);

	return out;
}


char *bk_edit_misc_xml_entities_decode (char *in)
{
	xmlParserCtxtPtr ctx = xmlNewParserCtxt ();

	char *out = xmlStringDecodeEntities (ctx, in, XML_SUBSTITUTE_REF, 0, 0, 0);
	char *outt = bk_edit_misc_utf8_to_latin1 (in);

	free (out);

	return outt;
}


void bk_edit_misc_remove_char_from_str (char *string, char character)
{
	char *c = string;

	while (*c != '\0')
	{
		if (*c == character)
		{
			char *cc = c;
			while (*cc != '\0')
			{
				*cc = *(cc + 1);

				cc++;
			}
		}
		else
		{
			c++;
		}
	}
}


int bk_edit_misc_create_backup_file (char *name)
{
	struct stat st;

	char *backup;
	int revision = 0;

	time_t date = time (NULL);
	char date_string[9] = "YYYYMMDD";
	strftime (date_string, sizeof (date_string), "%Y%m%d", localtime (&date));

	for (;;)
	{
		backup = g_strdup_printf ("%s-%s-%03d", name, date_string, revision++);

		stat (backup, &st);
		if (errno == ENOENT)
		{
			break;
		}

		if (revision > 999)
		{
			char *error = g_strdup_printf ("Save failed. Can't create backup file %s.\n\nYou may want to save the file under another name or into another directory.", backup);

			if (conv.mode == CONVERT)
			{
				fprintf (stderr, "%s[%d]: Saved failed. Can't create backup file %s.\n", __FILE__, __LINE__, backup);
			}
			else
			{
				bk_edit_dialog_info ("bk edit - error", error, icon_warning);
			}

			free (error);
			free (backup);

			return KO;
		}

		free (backup);
	}

	if (rename (name, backup) == -1)
	{
		char *error = g_strdup_printf ("Save failed. Can't create backup file %s.\n\nYou may want to save the file under another name or into another directory.", backup);

		if (conv.mode == CONVERT)
		{
			fprintf (stderr, "%s[%d]: Save failed. Can't create backup file %s.\n", __FILE__, __LINE__, backup);
		}
		else
		{
			bk_edit_dialog_info ("bk edit - error", error, icon_warning);
		}

		free (error);
		free (backup);

		return KO;
	}

	free (backup);

	return OK;
}


/* This is taken from glib-1.2.8/gstrfuncs.c */
char *bk_edit_misc_strescape (char *string, char escape)
{
	gchar *q;
	gchar *escaped;
	guint backslashes = 0;
	gchar *p = string;

	g_return_val_if_fail (string != NULL, NULL);

	while (*p != '\000')
		backslashes += (*p++ == escape);

	if (!backslashes)
		return g_strdup (string);

	escaped = g_new (gchar, strlen (string) + backslashes + 1);

	p = string;
	q = escaped;

	while (*p != '\000')
	{
		if (*p == escape)
			*q++ = escape;
		*q++ = *p++;
	}
	*q = '\000';

	return escaped;
}


/* This is taken from gnome-libs-1.4.2/gtk-xmhtml/parse.c */
char *bk_edit_misc_strcasestr (const char *s1, const char *s2)
{
	register int i;
	register const char *p1, *p2, *s = s1;

	for (p2 = s2, i = 0; *s; p2 = s2, i++, s++)
	{
		for (p1 = s; *p1 && *p2 && tolower(*p1) == tolower(*p2);
				p1++, p2++)
			;
		if (!*p2)
			break;
	}
	if (!*p2)
		return((char*)s1 + i);
	return(NULL);
}

