/*
 * Copyright (C) 2001-2003 Clemens Fuchslocher <clfuit00@fht-esslingen.de>
 *
 * bk_edit_file.c - 23.05.2003 - v0.10 - modify indicator rewrite
 *                  22.05.2003 - v0.9 - hidden files
 *                  17.04.2003 - v0.8 - sort order
 *                  05.04.2003 - v0.7 - default browser type
 *                  01.03.2003 - v0.6 - bk_edit_file_direct_open
 *                  12.02.2003 - v0.5 - backup files are now created
 *                  12.02.2003 - v0.5 - modify indicator added
 *                  23.12.2002 - v0.4 - plugin api clean-up
 *                  26.02.2002 - v0.3 - plugin functionality
 *                  08.08.2001 - v0.2 - mozilla support
 *                  24.07.2001 - v0.1
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */

#include <time.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>

#include <sys/stat.h>

#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h>

#include "bk_edit.h"
#include "bk_edit_tree.h"

#include "bk_edit_undo.h"
#include "bk_edit_file.h"
#include "bk_edit_icon.h"
#include "bk_edit_misc.h"
#include "bk_edit_sort.h"
#include "bk_edit_dialog_main.h"
#include "bk_edit_dialog_info.h"
#include "bk_edit_dialog_edit.h"
#include "bk_edit_dialog_undo.h"
#include "bk_edit_recent_document.h"
#include "bk_edit_modify_indicator.h"

#include "config.h"
#include "plugin.h"

extern bk_edit_tree tree;
extern bk_edit_undo *undo;
extern bk_edit_modify_indicator *modify_indicator;

extern bk_edit_dialog_main dialog_main;
extern bk_edit_dialog_edit dialog_edit;

extern plugins *plugs;

extern config *conf;

static GtkWidget *open_menu_option;
static GtkWidget *save_menu_option;

static currently_opened_file opened_file = { NULL, NULL };

char *selected_plugin = NULL;


static void file_save_as_ok (GtkWidget *w, GtkFileSelection *fs);
static void file_open_ok (GtkWidget *w, GtkFileSelection *fs);
static void show_hidden_files (GtkWidget *w, GtkFileSelection *fs);
static void key_pressed (GtkWidget *widget, GdkEventKey *event, gpointer *dialog);


void bk_edit_file_open (void)
{
	GList *iter;

	GtkWidget *fs = gtk_file_selection_new ("bk edit - open");

	GtkWidget *box, *frame, *label;
	GtkWidget *menu, *menu_option;
	GtkWidget *hidden_files;

	/*
	 * you can't edit an old bookmark corresponding to an old bookmarkfile
	 * if you have opened a new one. so we will hide the edit dialog.
	 */
	if (dialog_edit.dialog != NULL)
	{
		gtk_widget_hide (GTK_WIDGET (dialog_edit.dialog));
	}

	tree.selected_node = NULL;

	gtk_window_position (GTK_WINDOW (fs), GTK_WIN_POS_MOUSE);
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (fs)->ok_button), "clicked",
					GTK_SIGNAL_FUNC (file_open_ok), GTK_OBJECT (fs));
	gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (fs)->cancel_button), "clicked",
					GTK_SIGNAL_FUNC (gtk_widget_destroy),
					GTK_OBJECT (fs));
	gtk_signal_connect (GTK_OBJECT (fs), "key_press_event",
				GTK_SIGNAL_FUNC (key_pressed), (gpointer) fs);

	/* plugin */
	box = gtk_hbox_new (FALSE, 1);
	gtk_container_set_border_width (GTK_CONTAINER (box), 4);
	frame = gtk_frame_new ("I/O Plugin");
	label = gtk_label_new ("File Type: ");
	menu = gtk_menu_new ();
	menu_option = open_menu_option = gtk_option_menu_new ();

	for (iter = plugs->list; iter != NULL; iter = g_list_next (iter))
	{
		if (((plugin *) iter->data)->type & PLUGIN_LOAD)
		{
			GtkWidget *menu_item = gtk_menu_item_new_with_label (((plugin *) iter->data)->name);
			gtk_menu_append (GTK_MENU (menu), menu_item);
			gtk_widget_show (menu_item);
		}
	}
	gtk_option_menu_set_menu (GTK_OPTION_MENU (menu_option), menu);

	if (selected_plugin == NULL)
	{
		selected_plugin = strdup (config_browser_type_get (conf));
	}
	bk_edit_misc_set_menu_option (GTK_OPTION_MENU (menu_option), selected_plugin);

	/* hidden files */
	hidden_files = gtk_button_new_with_label ("Hidden Files");
	gtk_signal_connect (GTK_OBJECT (hidden_files), "clicked", GTK_SIGNAL_FUNC (show_hidden_files), (gpointer) fs);

	gtk_box_pack_start (GTK_BOX (box), label, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (box), menu_option, TRUE, TRUE, 0);
	gtk_container_add (GTK_CONTAINER (frame), box);
	gtk_box_pack_end (GTK_BOX (GTK_FILE_SELECTION (fs)->action_area), frame, TRUE, TRUE, 4);
	gtk_box_pack_end (GTK_BOX (GTK_FILE_SELECTION (fs)->button_area), hidden_files, TRUE, TRUE, 0);

	gtk_widget_show_all (fs);
}


void bk_edit_file_save (void)
{
	plugin *plug;

	if (opened_file.name != NULL)
	{
		if (bk_edit_misc_create_backup_file (opened_file.name) != OK)
		{
			return;
		}

		/* plugin */
		plug = plugin_get_by_name (plugs, opened_file.used_plugin);
		if (plug == NULL)
		{
			bk_edit_dialog_info ("bk edit - error", "Unknown plugin.", icon_warning);
		}
		else
		{
			gtk_clist_freeze (GTK_CLIST (tree.tree));
			bk_edit_sort (NULL, UNSORTED, NULL);
			bk_edit_tree_save (plug, opened_file.name);
			bk_edit_sort_reset_to_menu_value ();
			gtk_clist_thaw (GTK_CLIST (tree.tree));
		}

		bk_edit_modify_indicator_set (modify_indicator, CURRENT_UNDO_STACK_ENTRY);
		bk_edit_modify_indicator_update (modify_indicator, CURRENT_UNDO_STACK_ENTRY);
	}
	else
	{
		bk_edit_file_save_as ();
	}
}


void bk_edit_file_save_as (void)
{
	GList *iter;

	GtkWidget *fs = gtk_file_selection_new ("bk edit - save as");

	GtkWidget *box, *frame, *label;
	GtkWidget *menu, *menu_option;
	GtkWidget *button;

	gtk_window_position (GTK_WINDOW (fs), GTK_WIN_POS_MOUSE);
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (fs)->ok_button), "clicked",
					GTK_SIGNAL_FUNC (file_save_as_ok), GTK_OBJECT (fs));
	gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (fs)->cancel_button), "clicked",
					GTK_SIGNAL_FUNC (gtk_widget_destroy),
					GTK_OBJECT (fs));
	gtk_signal_connect (GTK_OBJECT (fs), "key_press_event",
				GTK_SIGNAL_FUNC (key_pressed), (gpointer) fs);

	/* plugin */
	box = gtk_hbox_new (FALSE, 1);
	gtk_container_set_border_width (GTK_CONTAINER (box), 4);
	frame = gtk_frame_new ("I/O Plugin");
	label = gtk_label_new ("File Type: ");
	menu = gtk_menu_new ();
	menu_option = save_menu_option = gtk_option_menu_new ();

	for (iter = plugs->list; iter != NULL; iter = g_list_next (iter))
	{
		if (((plugin *) iter->data)->type & PLUGIN_SAVE)
		{
			GtkWidget *menu_item = gtk_menu_item_new_with_label (((plugin *) iter->data)->name);
			gtk_menu_append (GTK_MENU (menu), menu_item);
			gtk_widget_show (menu_item);
		}
	}
	gtk_option_menu_set_menu (GTK_OPTION_MENU (menu_option), menu);

	if (selected_plugin == NULL)
	{
		selected_plugin = strdup (config_browser_type_get (conf));
	}
	bk_edit_misc_set_menu_option (GTK_OPTION_MENU (menu_option), selected_plugin);

	gtk_box_pack_end (GTK_BOX (GTK_FILE_SELECTION (fs)->action_area), frame, TRUE, TRUE, 4);

	/* hidden files */
	button = gtk_button_new_with_label ("Hidden Files");
	gtk_signal_connect (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (show_hidden_files), (gpointer) fs);
	gtk_box_pack_end (GTK_BOX (GTK_FILE_SELECTION (fs)->button_area), button, TRUE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX (box), label, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (box), menu_option, TRUE, TRUE, 0);
	gtk_container_add (GTK_CONTAINER (frame), box);

	gtk_widget_show_all (fs);
}


void bk_edit_file_direct_open (char *filename, char *plugin_name)
{
	struct stat st;
	plugin *plug;

	if (stat (filename, &st) == -1)
	{
		bk_edit_dialog_info ("bk edit - error", "The selected file does not exist.", icon_warning);
		return;
	}

	if (!S_ISREG (st.st_mode))
	{
		bk_edit_dialog_info ("bk edit - error", "Can't open file.", icon_warning);
		return;
	}

	plug = plugin_get_by_name (plugs, plugin_name);
	if (plug == NULL)
	{
		plug = plugin_get_by_short_name (plugs, plugin_name);
		if (plug == NULL)
		{
			bk_edit_dialog_info ("bk edit - error", "Unknown plugin.", icon_warning);
			return;
		}
	}
	plugin_name = plug->name;

	gtk_clist_freeze (GTK_CLIST (tree.tree));
	gtk_clist_clear (GTK_CLIST (tree.tree));

	tree.tree_stack = g_stack_new (tree.tree_stack);
	tree.order_stack = g_stack_new (tree.order_stack);

	if (plug->load (filename) != PLUGIN_OK)
	{
		bk_edit_dialog_info ("bk edit - open failed", "The selected file couldn't be loaded.\nDid you select the correct file type?", icon_warning);
	}
	else
	{
		if (opened_file.name != NULL)
		{
			free (opened_file.name);
		}
		opened_file.name = strdup (filename);

		if (opened_file.used_plugin != NULL)
		{
			free (opened_file.used_plugin);
		}
		opened_file.used_plugin = strdup (plugin_name);

		if (selected_plugin != NULL)
		{
			free (selected_plugin);
		}
		selected_plugin = strdup (plugin_name);

		bk_edit_modify_indicator_set (modify_indicator, NULL);
		bk_edit_modify_indicator_update (modify_indicator, NULL);
	}

	g_stack_delete (tree.order_stack);
	g_stack_delete (tree.tree_stack);

	bk_edit_sort_menu_update (config_sort_order_get (conf));

	gtk_clist_thaw (GTK_CLIST (tree.tree));

	/* undo */
	bk_edit_undo_delete (undo);
	undo = bk_edit_undo_new (undo);
	bk_edit_dialog_undo_update ();
	gtk_widget_grab_focus (tree.tree);
}


currently_opened_file *bk_edit_file_get_currently_opened_file (void)
{
	return &opened_file;
}


void bk_edit_file_unset_currently_opened_file (void)
{
	if (opened_file.name != NULL)
	{
		free (opened_file.name);
	}
	opened_file.name = NULL;

	if (opened_file.used_plugin != NULL)
	{
		free (opened_file.used_plugin);
	}
	opened_file.used_plugin = NULL;
}


static void file_open_ok (GtkWidget *w, GtkFileSelection *fs)
{
	struct stat st;
	plugin *plug;
	gchar *plug_name;

	if (strlen (gtk_entry_get_text (GTK_ENTRY(fs->selection_entry))) == 0)
	{
		return;
	}

	if (stat (gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs)), &st) == -1)
	{
		bk_edit_dialog_info ("bk edit - error", "The selected file does not exist.", icon_warning);
		return;
	}

	if (S_ISDIR (st.st_mode))
	{
		gtk_file_selection_set_filename (GTK_FILE_SELECTION (fs), gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs)));
		return;
	}

	if (!S_ISREG (st.st_mode))
	{
		bk_edit_dialog_info ("bk edit - error", "Can't open file.", icon_warning);
		return;
	}

	/* plugin */
	plug_name = bk_edit_misc_get_menu_option (GTK_OPTION_MENU (open_menu_option));

	if (selected_plugin != NULL)
	{
		free (selected_plugin);
	}
	selected_plugin = strdup (plug_name);

	plug = plugin_get_by_name (plugs, plug_name);
	if (plug == NULL)
	{
		bk_edit_dialog_info ("bk edit - error", "Unknown plugin.", icon_warning);
	}
	else
	{
		gtk_clist_freeze (GTK_CLIST (tree.tree));
		gtk_clist_clear (GTK_CLIST (tree.tree));

		tree.tree_stack = g_stack_new (tree.tree_stack);
		tree.order_stack = g_stack_new (tree.order_stack);

		if (plug->load (gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs))) != PLUGIN_OK)
		{
			bk_edit_dialog_info ("bk edit - open failed", "The selected file couldn't be loaded.\nDid you select the correct file type?", icon_warning);
		}
		else
		{
			if (opened_file.name != NULL)
			{
				free (opened_file.name);
			}
			opened_file.name = strdup (gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs)));

			if (opened_file.used_plugin != NULL)
			{
				free (opened_file.used_plugin);
			}
			opened_file.used_plugin = strdup (plug_name);

			bk_edit_modify_indicator_set (modify_indicator, NULL);
			bk_edit_modify_indicator_update (modify_indicator, NULL);

			bk_edit_recent_document_add (opened_file.name, opened_file.used_plugin);
		}

		g_stack_delete (tree.order_stack);
		g_stack_delete (tree.tree_stack);

		bk_edit_sort_menu_update (config_sort_order_get (conf));

		gtk_clist_thaw (GTK_CLIST (tree.tree));
	}

	gtk_widget_destroy (GTK_WIDGET (fs));

	/* undo */
	bk_edit_undo_delete (undo);
	undo = bk_edit_undo_new (undo);
	bk_edit_dialog_undo_update ();
	gtk_widget_grab_focus (tree.tree);
}


static void file_save_as_ok (GtkWidget *w, GtkFileSelection *fs)
{
	struct stat st;
	plugin *plug;
	gchar *plug_name;

	GtkCTreeNode *node = GTK_CTREE_NODE (GTK_CLIST (tree.tree)->row_list);
	if (node == NULL)
	{
		bk_edit_dialog_info ("bk edit - error", "There is no data to save.", icon_warning);
		gtk_widget_destroy (GTK_WIDGET (fs));
		return;
	}

	if (strlen (gtk_entry_get_text (GTK_ENTRY(fs->selection_entry))) == 0)
	{
		return;
	}

	stat (gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs)), &st);
	if (S_ISDIR (st.st_mode))
	{
		gtk_file_selection_set_filename (GTK_FILE_SELECTION (fs), gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs)));
		return;
	}

	if (S_ISREG (st.st_mode))
	{
		if (bk_edit_misc_create_backup_file (gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs))) != OK)
		{
			return;
		}
	}

	/* plugin */
	plug_name = bk_edit_misc_get_menu_option (GTK_OPTION_MENU (save_menu_option));

	plug = plugin_get_by_name (plugs, plug_name);
	if (plug == NULL)
	{
		bk_edit_dialog_info ("bk edit - error", "Unknown plugin.", icon_warning);
	}
	else
	{
		gtk_clist_freeze (GTK_CLIST (tree.tree));
		bk_edit_sort (NULL, UNSORTED, NULL);
		bk_edit_tree_save (plug, gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs)));
		bk_edit_sort_reset_to_menu_value ();
		gtk_clist_thaw (GTK_CLIST (tree.tree));

		if (opened_file.name != NULL)
		{
			free (opened_file.name);
		}
		opened_file.name = strdup (gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs)));

		if (opened_file.used_plugin != NULL)
		{
			free (opened_file.used_plugin);
		}
		opened_file.used_plugin = strdup (plug_name);

		bk_edit_modify_indicator_set (modify_indicator, CURRENT_UNDO_STACK_ENTRY);
		bk_edit_modify_indicator_update (modify_indicator, CURRENT_UNDO_STACK_ENTRY);
	}

	gtk_widget_destroy (GTK_WIDGET (fs));
}


static void key_pressed (GtkWidget *widget, GdkEventKey *event, gpointer *dialog)
{
	if (event == NULL)
	{
		return;
	}

	switch (event->keyval)
	{
		case GDK_Escape:
		{
			gtk_widget_destroy (GTK_WIDGET (dialog));
			break;
		}
	}
}


static void show_hidden_files (GtkWidget *button, GtkFileSelection *fs)
{
	gtk_file_selection_complete (GTK_FILE_SELECTION (fs), ".");
}

