/*
 * Copyright (c) 2001 Tommy Bohlin <tommy@gatespace.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/* irobex.c
 */

#include <irda.h>

#include <stdio.h>
#include <stdlib.h>

static int verbosity;
static u_char* objPtr;
static int objLength;

/**********************************************************************
 * OBEX control
 **********************************************************************/

static void obexStatus(OBEXClient* oc, int event)
{
  switch(event) {
  case OBEX_SUCCESS:
    if(verbosity>0) birda_log("put succeeded\n");
    break;
  case OBEX_FAILURE:
    birda_log("put failed\n");
    break;
  }
  obexCltClose(oc);
  lapDisconnect(optLap);
}

/**********************************************************************
 * LAP control
 **********************************************************************/

static void lapConnected(LAP* lap)
{
  OBEXClient* obex;

  if(verbosity>0) birda_log("LAP connected\n");
  obex=createOBEXClient(lap);
  if(!obex) {
    birda_log("OBEX connect failed\n");
    lapDisconnect(lap);
  }
  if(verbosity>0) obex->debug|=OBEX_DEBUG_INFO;
  obex->status=obexStatus;
  obexCltPut(obex,objPtr,objLength);
}

static void lapDisconnected(LAP* lap)
{
  exit(0);
}

/**********************************************************************
 * Main function
 **********************************************************************/

static int collectInput(FILE* f, u_char** bufp)
{
  int length=0;
  int maxLength=1024;
  u_char* buf=malloc(maxLength);
  int n;

  while((n=fread(buf+length,1,maxLength-length,f))>0) {
    length+=n;
    if(maxLength-length<128) {
      maxLength*=2;
      buf=realloc(buf,maxLength);
    }
  }
  if(ferror(f)) {
    birda_log("Error reading input\n");
    length=0;
  }
  if(length>0) *bufp=buf; else free(buf);

  return length;
}

static const char* description="Send OBEX object from stdin or pty";

int main(int argc, char** argv)
{
  doOptions(argc,argv,0,0,&verbosity,description);

  optLap->flags|=HINT_IROBEX;
  optLapConnected=lapConnected;
  optLapDisconnected=lapDisconnected;

  objLength=collectInput(stdin,&objPtr);
  if(!objLength) return 0;

  doConnect();
  evtLoop();

  return 0;
}
