 /* BenAri Concurrent PCODE Interpreter Utilities */

#include <stdio.h>
#include <string.h>
#include <assert.h>

#include "../include/globdata.h"
#include "../include/readtab.h"
#include "../include/genutil.h"
#include "random.h"
#include "bainterp.h"

extern int fprintf(FILE*,const char*,...);
extern int fclose(FILE*);
extern int printf(const char*,...);
extern int atoi(char*);
extern int build_cbtab();
extern int sscanf(const char*, const char*,...);

FNAME_STRING interp_pname;

char interp_pnroot[]  = 
   {"BenAri Concurrent PCODE Interpreter, " };
   
extern char date_string[];

char xpcode_suffix[] = { ".xpc" };

FNAME_STRING xpcode_fname;
FILE  *xpcode;

   /* debug flags */
int   end_db=0, initAR_db=0, finalAR_db=0, pcexec_db=0,
      debugging = 0;

FNAME_STRING   buf;  /* input buffer for reading tables */

extern int charl, charh;   /* lowest char ord = ord' ', highest= ord '~'*/
extern ORDER ir;           /* holds current instruction */
extern enum pstat ps;      /* current interpreter state */
extern int s[];            /* the stack */
extern proctab ptab[];     /* the process table */
extern CODEBLOCK cbtab[];  /* code block table */
extern int last_cbtab;     /* index of last element cbtab */

extern int npr;            /* number of concurrent processes */
extern int curpr;          /* current process executing */
extern int stepcount;      /* number of steps before switch */
extern int pflag;          /* 1 ==> in COBEGIN, 0 at COEND */
extern unsigned iclock;    /* total no. of pcode instructions executed */

extern int singlestep;

#define MAXBREAK 100          /* This many breaks active at once */
/* array of breakpoint information */
struct {
   int adr;          /* lc of the break */
   int tix;          /* tab index of the code block owning the break*/ 
   int uix;          /* index of the break that user refers to */
} breakpoint[MAXBREAK];
int break_cnt;       /* # active entries in breakpoint array */
int next_uix = 0;    /* next index of user break index to be assigned */


void  show_help(char **argv)
{
   fprintf(stderr,"%s\n",interp_pname);
   fprintf(stderr,"Usage:   %s [optional_flags]  pcode_filename\n",argv[0]);
   fprintf(stderr,"Optional flags:\n");
   fprintf(stderr,
      "   -d  enter the debugger, single step, set breakpoints\n");
   fprintf(stderr,
      "   -e  show the activation record (AR) on entry to each process\n");
   fprintf(stderr, "   -x  show the AR on exit from each process\n");
   fprintf(stderr, "   -t  announce process termination\n");
   fprintf(stderr, "   -h  show this help\n");
   fprintf(stderr, "   -p  show PCODE instructions as they are executed\n");
   fprintf(stderr, "The name of the PCODE file is required.  If missing, you\n");
   fprintf(stderr, "will be prompted for it.  The file suffix \"%s\" will\n",
         pcode_suffix);
   fprintf(stderr, "be appended to the filename you give.\n");
}  /* show_help */

void  get_command_options(int argc, char** argv)
{
   int   aix;
   char  c; 
   
   /* open the object file */
   if (argc == 1) 
      pcode_fname[0] = '\0';
   else {
      /*  process options on command line */
      for (aix= 1;aix < argc;aix++)
      {
         if (argv[aix][0] != '-')
            break;      /* all done with options */
         /* otherwise argv[aix] is '-'*/
         c = argv[aix][1];
         switch (c)
         {
            case 'd': debugging = 1; break;
            case 'h': show_help(argv); break;
            case 'x': finalAR_db = 1; break;
            case 'e': initAR_db = 1; break;
            case 't': end_db = 1; break;
            case 'p': pcexec_db = 1; break;
            default:
               fprintf(stderr,"Invalid option '%s'\n",argv[aix]);
         }  /* switch */
      } /* for processing options */
      if (aix < argc) 
         strcpy(pcode_fname,argv[aix]);
   }  /* else process options */
}  /* get_command_options */

FILE  *pcode;

void get_filenames()
{
   int tlen;

   strcpy(interp_pname,comp_proot);
   strcat(interp_pname,interp_pnroot);
   strcat(interp_pname,date_string);
   if (pcode_fname[0] == '\0'){
      fprintf(stderr,"Enter name of PCODE file (%s suffix assumed): ",
         pcode_suffix);
      fgets(pcode_fname,MAX_FNAME,stdin);
      tlen = strlen(pcode_fname);
      pcode_fname[tlen-1]='\0'; /* get rid of \n */
      if (tlen == 1) {
         fprintf(stderr,"Name of the PCODE file is required!\n");
         fprintf(stderr,"%s\n",interp_pname);
         exit(0);
      }
   }
   strcpy(xpcode_fname,pcode_fname);
   strcat(xpcode_fname,xpcode_suffix);
   strcat(pcode_fname,pcode_suffix);
   if ((pcode = fopen(pcode_fname,"r")) == NULL) {
      fprintf(stderr,"Cannot open %s\n",pcode_fname);
      fprintf(stderr,"%s\n",interp_pname);
      exit(1);
   }
}  /* get_filenames */

void init_interpreter();

void global_init(int argc, char **argv)
{
   get_command_options(argc,argv);
   get_filenames();
   read_pcode_file(pcode);
   init_interpreter();
   last_cbtab = build_cbtab();
   break_cnt = 0;
   singlestep = 1;
   fprintf(stderr,"Executing PCODE ...\n");
}


void chooseproc()
/* from a random starting point, look through each entry of ptab
   for a process that is active and not suspended.  If no such
   process is found, declare deadlock and blame it on the most
   recently seen suspended process, if there is one, otherwise,
   blame it on the most recently seen active process */
{
   int   active, suspended, look, add, d;

   d = PMAXP1;
   suspended = active = 0;
   look = random(PMAXP1);
   add = 1 + random(PMAX); /* 1 <= add <= PMAXP1 */
   while ( ((!ptab[look].active) || (ptab[look].suspend>=0) ||
            (ptab[look].tabix < 0)) && (d >= 0) ){
/* keep looking until a called process that is active and not
   suspended is found */
      d--;
      if (ptab[look].active > 0) active = look;
      if (ptab[look].suspend >= 0) suspended = look;
      /* VERY IMPORTANT that equation below has full period of PMAXP1 */
      look = (look + add) % PMAXP1;
   } /* while */
   if (d < 0) {  /* no runnable process was found */
      /* choose most recently seen suspended proc, else m.r.seen active */
      curpr = (suspended ? suspended : active );
      ps = deadlock;
   }
   else {
      stepcount = random(STEPMAXP1);
      if (ptab[curpr].atomic){
         if (ptab[curpr].suspend >= 0) {
            ps = deadlock;
         }
         /* if curpr is atomic & not suspended, then let it run */
      }
      else  /* curpr is not atomic, let someone else run */
         curpr = look; 
   }
}  /* chooseproc */

void show_display(int pr)
{
int   ix;
   if ((pr >= 0) && (pr <= PMAX)) {
      printf("Display for process: ");
      if (ptab[pr].tabix > 0)
         printf("%s\n",tab[ptab[pr].tabix].name);
      else
         printf("???\n");
      for (ix = LMAX; ix >= 0; ix--)
         printf("%3d %4d\n",ix,ptab[pr].display[ix]);
   } /* if */
}  /* show_display */
         
void show_vars(int prno,int tix)
   /* show the variables in the block with tab index tix */
{
   int ix;
   int bix = tab[tix].ref;
   int tmp, tmp2;
      /* if psize == vsize, then this block has no variables */
   if (btab[bix].psize == btab[bix].vsize) return;
   ix = btab[bix].last; 
   while (ix) {
      if ((tab[ix].obj == variable) && (tab[ix].lev <= 2)) {
         if ((tab[ix].typ == notyp) || (tab[ix].typ == ints) ||
             (tab[ix].typ == bools) || (tab[ix].typ == chars) ||
             (tab[ix].typ == sems) || (tab[ix].typ == bsems) ||
             (tab[ix].typ == conds)) {
            printf("%-6s %-12s %3d %3d",
               typenames[tab[ix].typ],tab[ix].name,
               tab[ix].lev,tab[ix].adr);
            if ((tab[tix].obj == outerblock)||(tab[tix].obj == mainproc))
               tmp = ptab[0].display[tab[ix].lev]+tab[ix].adr;
            else if (tab[tix].obj == monitor)
               tmp = tab[tix].mon + tab[ix].adr;
            else  /* use current process */
               /* global variables start at s[0] */
               if (tab[ix].lev == 0)
                  tmp = tab[ix].adr;
               else { /* for a process we have to calculate it's frame base */
                  assert(prno > 0);
                  tmp = (ptab[prno-1].stacksize + 1) + tab[ix].adr;
               }
            switch (tab[ix].typ) {
            case  bsems:
            case  sems:
            case  bools:
            case  ints: printf("%6d\n",s[tmp]); break;
            case  chars:   
               tmp2 = s[tmp] % 256;
               if ((tmp2 < charl) || (tmp2 > charh)) 
                  printf("  ord(char) = %d\n",tmp2);
               else
                  printf("  %c\n",tmp2);
               break;
            case  strings:
               printf("%s\n",(char *) s[tmp]);
            default:
               printf("\n");
            }  /* switch */
         } /* inner if */
      }  /* outer if */
      ix = tab[ix].link;
   }  /* while */
}  /* show_vars */

void write_process_table()
   /* write the process table to stdout */
{
   int   x;
   extern int last_tab;

   printf("\nProcess Table\n");   
   printf(" Process        Active  Suspend  PC    xpc   atomic\n");
   for (x = 0; x <= PMAX; x++)
      if (ptab[x].active >= 0) {
         if (ptab[x].tabix > 0)
            printf("%2d %-12s %5d  %5d  %5d  %5d  %5d\n",
               x,tab[ptab[x].tabix].name,ptab[x].active,
               ptab[x].suspend,ptab[x].pc,ptab[x].xpc,ptab[x].atomic);
         else
            printf("%2d %-12s %5d  %5d  %5d  %5d  %5d\n", x, " ",
               ptab[x].active,ptab[x].suspend,ptab[x].pc,
               ptab[x].xpc,ptab[x].atomic);
      }
   printf("Global Variables \n");
   printf("type  name         level adr value\n");
   show_vars(0,0);
   printf("Mainproc Variables\n");
   show_vars(0,ptab[0].tabix);
   printf("Monitor Variables\n");
   for (x = 0; x <= last_tab; x++)
      if (tab[x].obj == monitor) {
         printf("%s:\n",tab[x].name);
         show_vars(curpr,x);
      }
   printf("Process Variables\n");
   for (x = 1; x <= PMAX; x++)
      if (ptab[x].active >= 0) {
         printf("Process #%d  %s\n",x,tab[ptab[x].tabix].name);
         show_vars(x,ptab[x].tabix);
      }
}  /* write_process_table */

#define ITEMSPERLINE 10

void dump_stack(int b, int t)
   /* write the stack from s[t] down to s[b] */
{
   int ix, count;
   for (ix = t, count = 0; ix >= b; ix--){
      if (s[ix] == UNDEF_VALUE)
         printf("----  ");
      else
         printf("%4d  ",s[ix]);
      if (++count == ITEMSPERLINE){
         printf("\n");
         count = 0;
      }
   }  /* for */
   printf("\n");
}

void show_AR(int b,int t)
   /* show the AR s[t] down to s[b] for the process that owns it */
{
int localb;
   localb = ((b < 0) ? 0 : b);
   printf("Activation Record  t = %3d down to b = %3d\n",t,b);
   printf("  for program unit: %s\n",tab[s[b+4]].name);
   dump_stack(localb,t);
   if ((--localb >= 0)&&(curpr == 0)) {
      printf("Remainder of stack from %d down to 0\n",localb);
      dump_stack(0,localb);
   }
}  /* show_AR */

void stop_interp(enum pstat ps)
   /* stop the interpreter and dump debug information */
{
   int i;

   printf("\n");
   if (ps != fin) { 
      printf("*** Halt at %5d in process %4d because of ",
         ptab[curpr].pc,curpr);
      switch (ps){
         case deadlock: printf("deadlock\n"); break;
         case divchk  : printf("division by 0\n"); break;
         case inxchk  : printf("invalid array index\n"); break;
         case stkchk  : printf("invalid stack address\n"); break;
         case padrchk : printf("invalid code address\n"); break;
         case charchk : printf("invalid character\n"); break;
         case inpchk  : printf("invalid input\n"); break;
         case redchk  : printf("reading past end of file\n"); break;
         case semchk  : printf("a negative semaphore initial value\n"); 
                        break;
         case bsemchk : printf("a binary semaphore not 0 or 1\n");
                        break;
         case uisemchk: printf("an un-initialized semaphore\n"); 
                        break;
         case xmonchk : printf("cross monitor call\n"); break;
         case pcodchk : printf("invalid PCODE\n"); break;
         case rdchk   : printf("cannot read this variable type"); break;
         case wrtchk  : printf("cannot write this variable type"); break;
         case procchk : printf("invalid process number"); break;
         default: ;
      }  /* switch */
      write_process_table();
      printf("Stack for main proc from %d down to 0\n", ptab[0].t);
      dump_stack(0,ptab[0].t);
      printf("Stacks of concurrent processes\n");
      for (i = 1; i <= PMAX; i++)
         if (ptab[i].active != -1) {
            printf("Process # %d:  %s stack from %d down to %d\n",i,
            tab[ptab[i].tabix].name,
            ptab[i].t, ptab[i].b);
            dump_stack(ptab[i].b,ptab[i].t);
         }
   }
   else { /* ps == fin */
      if (finalAR_db) {
         printf("PROGRAM EXIT\n");
         show_display(curpr);
         printf("Entire Stack for main proc from %d down to 0\n",
            ptab[curpr].t);
         dump_stack(0,ptab[curpr].t);
      }
   } /* else ps == fin  */
   if (pcexec_db) {
      fclose(xpcode);
      printf("PCODE execution trace stored in %s\n",xpcode_fname);
   }
}  /* stop_interp */

int saddr_check(int addr)
   /* check a stack address */
{
   if ((addr >= ptab[curpr].stacksize)||(addr < 0)){
      printf("\nBad stack address:  %d  at loc %d in process %d\n",
         addr,ptab[curpr].pc,curpr);
      stop_interp(stkchk);
      exit(1);
   }
   return addr;
}

int paddr_check(int addr)
   /* check a pcode address */
{
   if ((addr >= CMAX)||(addr < 0)) {
      printf("\nBad pcode address:  %d    at loc  %d of current process:  %d\n",
         addr,ptab[curpr].pc,curpr);
      stop_interp(padrchk);
      exit(1);
   }
   return addr;
}

void local_strcpy(char* d, char* s)
   /* When the stack words were initialized to 0x00003f3f, the C-Lib */
   /* strcpy issued a SEGV when it hit the 3f's.  The bare-bones     */
   /* local version below is too dumb to care.                       */
{
   while (*s) {
      *d = *s;
      d++;s++;
   }
   *d = '\0';
}

void clock_tick()
   /* increment the number of PCODE instructions executed */
{
   int ix;

   iclock++;
   ptab[curpr].xpc++; /* one more instr for this proc */
   if (pcexec_db){
      fprintf(xpcode,"%5d   %2d %5d  %3d %4d %4d",
         iclock, curpr,ptab[curpr].pc-1,ir.f,ir.x,ir.y);
      for (ix = 0; ix <= PMAX; ix++)
         if (ptab[ix].active >= 0)
            fprintf(xpcode," %4d",ptab[ix].xpc);
      fprintf(xpcode,"\n");
   }
}

void init_proc_displays()
   /* initialize the displays for the concurrent threads to point to    */
   /* the stack frames of the outer, global scope and the scope of the  */
   /* main block                                                        */
{
   int ix, jx;
   int main_b = btab[0].vsize;   /* size of the global frame, base of main */
   for (ix = 1; ix <= PMAX; ix++) {
      ptab[ix].display[0] = 0;      /* base of the global frame */
      ptab[ix].display[1] = main_b; /* base of the main frame */
      for (jx = 2; jx <= LMAX; jx++) ptab[ix].display[jx] = 0;
   }
}  /* init_proc_displays */

void init_interpreter()
   /* initialize the interpreter's data structures */
{

   int ix;           /* loop counter */
   int main_b;       /* bottom of stackframe of main proc */
   int main_tix;     /* tab index of main proc */
   unsigned block0_vsize; /* size of outer block */

   if (btab[1].vsize > STMAX - STKINCR * PMAX) {
      fprintf(stderr,"Stack required is too great to run. Abort!\n");
      exit(1);
   }

   charl = ' ';    /* lowest char ord */ 
   charh = '~';    /* highest character ordinal */

   /* initialize stack */
   for (ix = 0; ix < STMAX; ix++) s[ix] = 0;

   /* initialize space for variables in the outer block */
   block0_vsize = btab[0].vsize;
   if (block0_vsize >= STMAX) {
      fprintf(stderr,
         "init_interpreter: btab[0].vsize (%u) larger than stacksize %d\n",
            btab[0].vsize,STMAX);
      exit(1);
   }
   /* establish stack frame for main pgm */ 
   ptab[0].b = main_b = block0_vsize;
   s[main_b] = 0;                 /* main proc result */
   s[main_b+1] = 0;               /* main proc return address */
   s[main_b+2] = 0;               /* main proc static link */
   s[main_b+3] = -1;              /* main proc dynamic link */
   s[main_b+4] = btab[0].lastpar; /* tab index of main proc temp stored here */
   ptab[0].tabix = s[main_b+4];   /* need to store in ptab, too */
   if (pcexec_db) {
      if ((xpcode = fopen(xpcode_fname,"w+")) == NULL){
         fprintf(stderr,"Can't open file for executed PCODE!\n");
         exit(1);
      }
      fprintf(xpcode,"Executing PCODE: %s\n",filename_line);
      fprintf(xpcode,"clock  proc  pc    f    x    y proc 0    1    2    3    4    5    6    7\n");
   } /* if pcexec_db */
   ptab[0].suspend = -1;
   /* set tab index of main proc */
   main_tix = ptab[0].tabix = s[main_b+4]; 
   /* initialize display of main proc */
   for (ix = 0; ix <= LMAX; ix++) ptab[0].display[ix] = 0;
   ptab[0].display[0] = 0;                   /* outer block */
   ptab[0].display[1] = main_b;              /* main proc */
   /* set top of stack to sum of sizes of stackframes of outer block */
   /* and main proc                                                  */
   ptab[0].t = block0_vsize + btab[tab[main_tix].ref].vsize - 1;
   if (ptab[0].t >= STMAX) {
      fprintf(stderr,
         "init_interpeter: ptab[0].t (%u) greater than stack size %d\n",
         ptab[0].t,STMAX);
      exit(1);
   }
   /* zero out the local variable storage of the main proc */
   for (ix = main_b+5; ix <= ptab[0].t; ix++) s[ix] = 0;
   ptab[0].pc = tab[main_tix].adr;  /* set entry point */
   ptab[0].active = 1;              /* main proc is active */
   ptab[0].monitor = 0;             /* not in a monitor call */
   ptab[0].priority = 0;            /* at this priority level */ 
   ptab[0].xpc = 0;                 /* # PCODE instructions executed */
   /* main proc gets what's left over after allocating the stack to  */
   /* the other procs                                                */
   ptab[0].stacksize = STMAX - PMAX * STKINCR;

   /* show main proc startup information, if necessary */
   if (initAR_db) {
      printf("STARTUP ");
      show_display(1); 
      show_AR(ptab[1].b,ptab[1].t);
   }

   /* initialize proc table entries for all other processes */
   for (ix = 1; ix <= PMAX; ix++) {
      ptab[ix].active = -1; /* for process in use, active = 0 or 1 */
      ptab[ix].pc = 0;
      ptab[ix].monitor = 0;
      ptab[ix].priority = ptab[ix].xpc = 0;
      ptab[ix].tabix = -1;
      ptab[ix].suspend = -1;
      ptab[ix].atomic = 0;
      ptab[ix].b = ptab[ix- 1].stacksize + 1;
      /* make sure that return addr stored in AR of each proc is zero, */
      /* because EXIT_PROC keys on it                                  */
      s[ptab[ix].b+1] = 0;
      ptab[ix].stacksize = ptab[ix].b + STKINCR - 1;
      ptab[ix].t = ptab[ix].b - 1;
   }
   init_proc_displays();
   /* initialize the biggies */
   npr = curpr = 0;
   pflag = 0;
   randomize();   /* wake up the random number generator */
   stepcount = 0; 
   ps = run;
   iclock = 0;
}  /* init_interpreter */

int find_words(char* s,char* word[])
   /* splits the string 's' into words on the tabs & blanks */
   /* returns the number of words found                     */
{
   char* tmp;
   int i = 0;
   tmp = strtok(s,"\t ");
   while (tmp) {
      word[i] =  tmp;
      i++;
      tmp = strtok(NULL,"\t ");
   } 
   return i;
}

int current_codeblock(int pc)
   /* return the cbtab index of the codeblock corresponding to 'pc' */
{
   extern int last_code;
   int ix;
   for(ix = 0; ix <= last_code; ix++)
      if (pc >= cbtab[ix].adr)
         return ix;
   return 0;
}


void set_break(int pc)
{
   int ix;

   if (break_cnt == MAXBREAK - 1) {
      printf("Can't set another breakpoint.\n");  
      printf("At most %d breakpoints can be active simultaneously\n",
         MAXBREAK);
      return;
   }
   for(ix = 0; ix < break_cnt; ix++) 
      if (breakpoint[ix].adr == pc) {
         printf("Breakpoint # %d already set at location %d\n",ix,pc);
         return;
      }
   breakpoint[break_cnt].adr = pc;
   breakpoint[break_cnt].tix = cbtab[current_codeblock(pc)].tix;
   breakpoint[break_cnt].uix = next_uix;
   next_uix++;
   break_cnt++;
}

void unset_break(int break_uix)
{
   int ix, jx;

   if (break_uix < 0) {
      printf("Invalid breakpoint index %d\n",break_uix);
      return;
   }
   jx = -1;
   for (ix = 0; ix < break_cnt; ix++)
      if (breakpoint[ix].uix == break_uix) {
         jx = ix;
         break;
   }
   if (jx < 0) {
      printf("Breakpoint %d is not set\n",break_uix);
      return;
   }
   for (ix = jx; ix < break_cnt; ix++){
      breakpoint[ix].adr = breakpoint[ix+1].adr;
      breakpoint[ix].tix = breakpoint[ix+1].tix;
      breakpoint[ix].uix = breakpoint[ix+1].uix;
   }
   breakpoint[break_cnt].adr = breakpoint[break_cnt].tix =
      breakpoint[break_cnt].uix = 0;
   break_cnt--;
}

void show_breakpoints()
{
   int ix;

   printf("Current breakpoints\n");
   printf("index location codeblock\n");
   for (ix = 0; ix < break_cnt; ix++)
      printf("%3d     %3d    %s\n",breakpoint[ix].uix,breakpoint[ix].adr,
         tab[breakpoint[ix].tix].name);
}

int at_breakpoint(int pc)
   /* if 'pc' is at a breakpoint, then return its index in breakpoint[], */
   /* else return -1                                                     */
{
   int ix;
   for (ix = 0; ix < break_cnt; ix++)
      if (pc == breakpoint[ix].adr)
         return breakpoint[ix].uix;
   return -1;
}

void debugger_help()
{
   printf("Debugger Commands:\n");
   printf("   b  lc     -- set a break at location 'lc'\n");
   printf("   c         -- continue to the next breakpoint\n");
   printf("   d         -- dump the stack of the current process\n");
   printf("   d  t      -- dump 10 stack words from s[t] down to s[t-10]\n");
   printf("   d  t b    -- dump stack words from s[t] down to s[b]\n");
   printf("   h         -- show this help\n");
   printf("   i         -- show current breakpoints\n");
   printf("   p         -- show process table\n");
   printf("   q         -- terminate execution\n");
   printf("   s         -- execute one PCODE instruction\n");
   printf("   RETURN    -- repeat singlestep or continue\n");
   printf("   u  i      -- unset breakpoint[i]\n");
   printf("   w         -- show where current execution is\n"); 
   printf("   x         -- disassemble the next 10 instructions\n");
   printf("   x  loc    -- dissassemble 10 instructions starting at 'loc'\n");
}
   
extern void disassemble(FILE*,int,int);

void debugger(int curpr, int* singlestep,int *continuing)
{
   int reading_cmds = 1;
   char* word[50];
   int word_cnt;
   char* prompt = "(h = help)>";
   int b, t, pc, oldpc;
   int count;

   while (reading_cmds && *singlestep) {
      printf("%s ", prompt);
      fgets(buf,MAX_FNAME,stdin);
      word_cnt = find_words(buf,word);
      if (word_cnt == 0) {
         *singlestep = 1;
         break;
      }
      switch(word[0][0]) {
         case 'b':
            pc = atoi(word[1]);
            set_break(pc);
            break;
         case 'c':
            *singlestep = 0;
            *continuing = 1;
            reading_cmds = 0;
            break;
         case 'd':
            if (word_cnt == 1) {
               t = ptab[curpr].t;
               b = ptab[curpr].b;
            } else if (word_cnt == 2) {
               t = atoi(word[1]);
               if (t >= STMAX) t = STMAX - 1;
               b = t - 10;
               if (b < 0) b = 0;
            } 
            else {
               t = atoi(word[1]);
               if (t >= STMAX) t = STMAX - 1;
               b = atoi(word[2]);
               if (b < 0) b = 0;
            }
            if (t >= b ) {
               printf("Stack for Process #%d: %s from %d down to %d\n",
                  curpr,tab[ptab[curpr].tabix].name,t,b);
               dump_stack(b,t);
            }
            break;
         case 'h':
            debugger_help();
            break;
         case 'i':
            printf("Current process # %d: %s  pc = %d  b = %d  t = %d\n",
               curpr,tab[ptab[curpr].tabix].name, ptab[curpr].pc,
               ptab[curpr].b, ptab[curpr].t);
            show_display(curpr);
            show_breakpoints();
            break;
         case 'p':
            write_process_table();
            break;
         case 'q':
            exit(0);
            break;
         case 's':
            reading_cmds = 0;
            *singlestep = 1;
            *continuing = 0;
            break;
         case 'u':
            pc = atoi(word[1]);
            unset_break(pc);
            break;
         case 'w':
            printf("Current process # %d: %s  pc = %d  b = %d  t = %d\n",
               curpr,tab[ptab[curpr].tabix].name, ptab[curpr].pc,
               ptab[curpr].b, ptab[curpr].t);
            break;
         case 'x':
            count = 10;
            if (word_cnt == 1) 
               pc = ptab[curpr].pc;
            else {
               pc = atoi(word[1]);
               if (pc < 0) pc = 0;
            }
            t = ptab[curpr].tabix;
            for (oldpc = pc; pc - oldpc < count; pc++)
               disassemble(stdout,t,pc);
            break;
         default:
            printf("Unknown command: '%c'\n",word[0][0]);
            break;
      }  /* switch */
   }  /* while */
}

void check_unresolved_externs()
{
   extern int last_tab;
   int ix;
   int count = 0;
   char name[30];
   char what[30];
   name[0] = '\0';
   for(ix = 0; ix < last_tab; ix++) {
      switch (tab[ix].obj) {
         case ext_procedure:
            strcpy(name,tab[ix].name);
            strcpy(what,"void function");
            break;
         case ext_function:
            strcpy(name,tab[ix].name);
            strcpy(what,"function");
            break;
         case ext_variable:
            strcpy(name,tab[ix].name);
            strcpy(what,"variable");
            break;
         case ext_monitor:
            strcpy(name,tab[ix].name);
            strcpy(what,"monitor");
            break;
         default:  /* nothing */ ;
      }
      if (name[0]) {
         fprintf(stderr,"Source file contains unresolved external %s: %s\n",
            what,name);
         name[0] = '\0';
         count++;
      }
      if (count) 
         fatal("Cannot execute source file because of %d unresolved externals",
               count);
   }     
}  /* check_unresolved_externs */
            

int next_word(char* buf, char** scanstr)
   /* copy into 'buf' the next whitespace substring in the  */
   /* string pointed to by *scanstr and update *scanstr     */
{
   char* p = *scanstr;
   while ((*p)&&((*p == ' ')||(*p == '\t'))) p++;
   if (*p == '\0') return -1;
   while ((*p)&&(*p != ' ')&&(*p != '\t')) {
      *buf = *p;
      buf++; p++;
   }
   *buf = '\0';
   *scanstr = p;
   return 0;
}

int next_string(char* buf, char** scanstr)
   /* copy into 'buf' the next "-delimited substring in the  */
   /* string pointed to by *scanstr and update *scanstr      */
   /* returns -1 on error, 0 if OK                           */
{
   char* p = *scanstr;

   while ((*p)&&((*p == ' ')||(*p == '\t'))) p++;
   if (*p != '"') return -1;
   p++; 
   while ((*p)&&(*p != '"')){
      *buf = *p;
      buf++; p++;
   }
   *buf = '\0';
   if (*p == '"') p++;
   *scanstr = p;
   return 0;
}

static char tempbuf[1024];

int xsscanf(int scan_strix, int six, int parm_cnt)
   /* s[scan_strix] has the address of the string to be scanned */
   /* stab[six] is the beginning of the format string */
   /* parm_cnt is the number of parms (not counting addr of scan string) */
{
   char *sp;
   char *fp;
   int i;
   int parm_offset;
   int op_addr;

   sp = (char* ) &s[saddr_check(s[scan_strix])]; 
   fp = (char* ) &stab[six];
   i = 0;
   parm_offset = scan_strix + 1;
   while (*fp) {
      if ((*fp == ' ')||(*fp == '\t')) {
         fp++; continue;
      }
      if (*fp == '%') {
         fp++;
         if (i == parm_cnt) return i;
         op_addr = saddr_check(s[parm_offset]);
         parm_offset++;
         switch (*fp) {
            case 'd': 
               if (next_word(tempbuf,&sp)< 0) return i;
               sscanf(tempbuf,"%d",(int *) &s[op_addr]);
               i++;
               break;
            case 'x': 
               if (next_word(tempbuf,&sp)< 0) return i;
               sscanf(tempbuf,"%x",(int *) &s[op_addr]);
               i++;
               break;
            case 's': 
               if (next_word(tempbuf,&sp)< 0) return i;
               strcpy((char*) &s[op_addr], tempbuf);
               i++;
               break;
            case 'q':
               if (next_string(tempbuf,&sp) < 0) return i;
               strcpy((char*) &s[op_addr], tempbuf);
               i++;
               break;
            default:  
               return i;
         }
      }
      fp++;
   }
   return i;
}
      
static char tfmt[1024];

char cpfmt(char* tfmt, char** fp)
   /* copies the format string pointed to by *fp into the tfmt buffer */
   /* returns what is probably the format specifier character         */
{
   char* p = tfmt;
   char* q = *fp;
   char t = *q;
   while ((*q != ' ')&&(*q != '\t')&&(*q)) {
      *p = t = *q; 
      p++; q++;
      if ((t == 'd')||(t == 'i')||(t == 'u')||(t == 'o')||
         (t == 'x')||(t == 'X')||(t == 'c')||(t == 's')) break;
   }
   *p = '\0';
   *fp = q;
   return t;
}

void xsprintf(int scan_strix, int six, int parm_cnt)
   /* s[scan_strix] has the address of the string to be built */
   /* stab[six] is the beginning of the format string */
   /* parm_cnt is the number of parms (not counting addr of scan string) */
{
   int i;
   char* fp;
   char* bp;
   char* buf;
   char  fspec;
   int parm_offset;
   int op_addr;

   bp =  tempbuf;
   buf = (char* ) &s[saddr_check(s[scan_strix])];
   fp = (char* ) &stab[six];
   i = 0;
   parm_offset = scan_strix + 1;
   while(*fp) {
      while((*fp)&&(*fp != '%')) {
         /* copy non-format stuff directly into output buffer */
         *bp = *fp;
         fp++; bp++;
      }
      /* if there's format left, use it */
      if (*fp == '%') {
         if (i == parm_cnt) return;
         /* must check for %q -- it's a BACI special */
         if (*(fp+1) == 'q') {
            if (i <= parm_cnt)  {
               op_addr = saddr_check(s[parm_offset]);
               parm_offset++;
               sprintf(bp,"\"%s\"",(char *) &s[op_addr]);
            }
            else
               sprintf(bp,"\"\"");
            fp += 2;
         }
         else {
            fspec = cpfmt(tfmt,&fp);
            if (fspec == 's') {
               if (i <= parm_cnt) {
                  op_addr = saddr_check(s[saddr_check(parm_offset)]);
                  sprintf(bp,tfmt,&s[op_addr]);
               }
               else
                  sprintf(bp," (null) ");
            }
            else
               sprintf(bp,tfmt,(i <= parm_cnt ? s[saddr_check(parm_offset)] : 0));
            parm_offset++;
         }
         /* move up the buffer pointer to the inserted NULL */
         while (*bp) bp++;
         i++;
      }
   }
   *bp = '\0';
   strcpy(buf,tempbuf);
}

/*
 *
 *  $Log: baiutils.c,v $
 *  Revision 2.9  1999/07/19 15:51:09  bynum
 *  switch gets call to fgets call
 *
 *  Revision 2.8  1998/08/09 14:27:35  bynum
 *  add init_proc_displays() and call it in init_interpreter()
 *
 *  Revision 2.7  1997/10/24 09:34:35  bynum
 *  add semicolon to empty default case, fix off-by-one in clock_tick(xpc)
 *
 * Revision 2.6  1997/10/23  13:26:47  bynum
 * change breakpoint machinery to display a "logical" breakpoint index
 *
 * Revision 2.5  1997/10/19  07:06:06  bynum
 * make sscanf declaration acceptable to LINUX
 *
 *  Revision 2.4  1997/09/05 16:39:02  bynum
 *  fix cpfmt() proc
 *
 * Revision 2.3  1997/09/04  10:57:43  bynum
 * add sprintf, sscanf
 *
 * Revision 2.2  1997/07/10  17:20:14  bynum
 * improve write_process_table output
 *
 * Revision 2.1  1997/07/02  13:41:15  bynum
 * move interpreter utils from bainterp.c here, add debugger utils
 *
 * Revision 2.0  1997/06/30  06:59:21  bynum
 * first version with level 0 globals
 *
 * Revision 1.8  1997/03/25  14:55:59  bynum
 * ncorporate name changes from the include directory, add fprintf prototype
 *
 * Revision 1.7  1996/03/07  09:32:41  bynum
 * remove 'filename_line' declaration
 *
 * Revision 1.6  1995/11/20  16:07:26  bynum
 * change "magic phrase" that intepreter looks for, change interpreter
 * flag
 *
 * Revision 1.5  1995/09/07  14:53:40  bynum
 * remove baiglobs.h include, tweak pgm name flag slightly
 *
 * Revision 1.4  1995/09/07  14:42:00  bynum
 * move code to read the global tables to the ../lib/read_tables.c file,
 * change includes to conform to new structure of ../include directory
 * change from K&R proc headers to ANSI C headers
 *
 * Revision 1.3  1995/08/29  15:57:20  bynum
 * add -p option message in 'show_help', add code to create the interp
 * program name string automatically from the date string
 *
 * Revision 1.2  1995/07/06  14:33:15  bynum
 * add 'atomic' field
 *
 * Revision 1.1  1995/06/22  06:51:31  bynum
 * Initial revision
 *
 *
 */
