#include "reflate.h"

#include "block.h"
#include "par.h"
#include "parse.h"
#include "ref.h"

static int ndim;
static int *block_size;
static int *npoints;

static int nstore;
static int dir_size;
static float *store;
static int *directory;

static Ref_info *ref;

static char *input_file;
static Line output_file;
static FILE *file_in;
static FILE *file_out;
static Bool swapped;
static float level;

static Bool input_found;
static Bool output_found;
static Bool par_found;
static Line output_par_file;
static char *reflate_file;

static int nblocks;

static int parse_string[] = { PARSE_STRING };

#define  FOUND_TWICE(string) \
	 {   sprintf(error_msg, "in \"%s\" '%s' found twice", \
				reflate_file, string);  return  ERROR;   }

#define  FOUND_BEFORE(string1, string2) \
	 {   sprintf(error_msg, "in \"%s\" '%s' found before '%s'", \
			reflate_file, string1, string2);  return  ERROR;   }

#define  NOT_FOUND(string) \
	 {   sprintf(error_msg, "in \"%s\" no '%s' found", \
				reflate_file, string);  return  ERROR;   }

static Status allocate_memory(String error_msg)
{
    sprintf(error_msg, "allocating memory for store");
    MALLOC(store, float, nstore);

    sprintf(error_msg, "allocating memory for directory");
    MALLOC(directory, int, dir_size);

    return  OK;
}

static void determine_params()
{
    int i;

    VECTOR_PRODUCT(nstore, block_size, ndim);

    nblocks = 1;
    for (i = 0; i < ndim; i++)
        nblocks *= BLOCK(npoints[i], block_size[i]);
}

static Status input_parse(Generic_ptr *var, String error_msg)
{
    String input_par_file = (String) (*var);
    Par_info par_info;

    if (input_found)
	FOUND_TWICE("input");

    CHECK_STATUS(read_par_file(input_par_file, &par_info, error_msg));

    input_file = par_info.file;
    ndim = par_info.ndim;
    npoints = par_info.npoints;
    block_size = par_info.block_size;
    ref = par_info.ref;
    swapped = par_info.swapped;
    level = par_info.level;
    dir_size = par_info.dir_size;

    if (!(par_info.deflated))
    {
	sprintf(error_msg, "input file \"%s\" must be deflated", input_file);
	return  ERROR;
    }

    input_found = TRUE;

    return  OK;
}

static Status output_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (!input_found)
	FOUND_BEFORE("output", "input");

    if (output_found)
	FOUND_TWICE("output");

    strcpy(output_file, name);

    output_found = TRUE;

    return  OK;
}

static Status par_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (!input_found)
	FOUND_BEFORE("par", "input");

    if (par_found)
	FOUND_TWICE("par");

    strcpy(output_par_file, name);

    par_found = TRUE;

    return  OK;
}

static Parse_line reflate_table[] =
{
    { "input",		1,	parse_string,		input_parse },
    { "output",		1,	parse_string,		output_parse },
    { "par",		1,	parse_string,		par_parse },
    { (String) NULL,	0,	(int *) NULL,		no_parse_func }
};

static Status read_reflate_file(String error_msg)
{
    input_found = FALSE;
    output_found = FALSE;
    par_found = FALSE;

    CHECK_STATUS(parse_file(reflate_file, reflate_table, TRUE, error_msg));

    if (!input_found)
	NOT_FOUND("input");

    if (!output_found)
	NOT_FOUND("output");

    return  OK;
}

static void print_reflate_info()
{
    printf("number of dimensions of data = %d\n", ndim);
    printf("original deflation level = %f\n", level);
}

void main(int argc, char **argv)
{
    Line error_msg;
    Store_info store_info;
    File_info file_info;
    Reflate_info reflate_info;
    Par_info par_info;
    String par_file;

    printf(product);

    if (help_request(argc, argv, help_table))
	exit (0);

    if (argc != 2)
    {
        sprintf(error_msg, "correct usage: %s <reflate file>", argv[0]);
        ERROR_AND_EXIT(error_msg);
    }

    reflate_file = argv[1];

    if (read_reflate_file(error_msg) == ERROR)
        ERROR_AND_EXIT(error_msg);

    determine_params();

    if (allocate_memory(error_msg) == ERROR)
        ERROR_AND_EXIT(error_msg);

    if (OPEN_FOR_READING(file_in, input_file))
    {
	sprintf(error_msg, "opening \"%s\" for reading", input_file);
	ERROR_AND_EXIT(error_msg);
    }

    if (OPEN_FOR_WRITING(file_out, output_file))
    {
	sprintf(error_msg, "opening \"%s\" for writing", output_file);
	ERROR_AND_EXIT(error_msg);
    }

    store_info.nstore = nstore;
    store_info.dir_size = dir_size;
    store_info.store = store;
    store_info.directory = directory;

    file_info.input_file = input_file;
    file_info.output_file = output_file;
    file_info.file_in = file_in;
    file_info.file_out = file_out;
    file_info.swapped = swapped;

    reflate_info.nblocks = nblocks;

    print_reflate_info();
    FLUSH;

    if (block_process(&store_info, &file_info, &reflate_info, error_msg)
								== ERROR)
	ERROR_AND_EXIT(error_msg);

    par_info.file = output_file;
    par_info.ndim = ndim;
    par_info.npoints = npoints;
    par_info.block_size = block_size;
    par_info.ref = ref;
    par_info.blocked = TRUE;
    par_info.deflated = FALSE;
    par_info.level = level;
    par_info.param_dim = -1;

    if (par_found)
	par_file = output_par_file;
    else
	par_file = NULL;

    if (write_par_file(par_file, &par_info, error_msg) == ERROR)
	ERROR_AND_EXIT(error_msg);
}
