#include "block_io.h"

#include "utility.h"

#define  MAX_FSEEK_SIZE         (16*1024*1024)
	/* for negative argument more than 256*1024*1024 bytes,
	   i.e. 64*1024*1024 words, seems that fseek does not work;
	   play extra safe and use 16 instead of 32 (cannot use 64
	   because of code below) */

/*  normally expect ntimes to be a power of 2  */
static Status safe_fseek_relative(FILE *fp, int size, int ntimes,
					int byte_size, String error_msg)
{
    int i;
    long int offset, n, seek_size;
    static long int max_fseek_size = MAX_FSEEK_SIZE;

    sprintf(error_msg, "seek: size = %d, ntimes = %d", size, ntimes);

    if (size == 0)
    /* needed for files opened for read and write, see fopen man page */
    {
	offset = 0;

	if (FSEEK_RELATIVE(fp, offset))
	    return  ERROR;
	else
	    return  OK;
    }

    for (offset = size;
		((ntimes % 2) == 0) && (ABS(offset) < max_fseek_size);
						offset *= 2, ntimes /= 2)
	;

    for (i = 0; i < ntimes; i++)
    {
	for (n = ABS(offset); n > 0; n -= ABS(seek_size))
	{
            seek_size = MIN(n, max_fseek_size);

            if (offset < 0)
        	seek_size = - seek_size;

	    if (FSEEK_RELATIVE_SIZE(fp, seek_size, byte_size))
		return  ERROR;
	}
    }

    return  OK;
}

static String short_file_name(String name)
{
    char *ptr;

    if (ptr = strrchr(name, DIRECTORY_SYMBOL))
	ptr++;
    else
	ptr = name;

    return  ptr;
}

static Status read_directory(Block_IO *block_io, String error_msg)
{
    if (FREAD(block_io->directory, BYTES_PER_WORD, block_io->dir_size, block_io->file))
	RETURN_ERROR_MSG("reading directory");

    if (block_io->swapped)
	swap_bytes((char *) (block_io->directory),
					BYTES_PER_WORD*block_io->dir_size);
    return  OK;
}

Status init_block_read(Block_IO *block_io, String error_msg)
{
    sprintf(error_msg, "\"%s\": ", short_file_name(block_io->name));
    error_msg += strlen(error_msg);

    if (FSEEK_ABSOLUTE(block_io->file, block_io->header))  /* skip header */
	RETURN_ERROR_MSG("skipping over header");

    if (block_io->deflated)
	CHECK_STATUS(read_directory(block_io, error_msg));

    block_io->last_done = 0;

    return  OK;
}

static Status read_normal_blocks(Block_IO *block_io, int block, int nblocks,
					float *data, String error_msg)
{
    int delta_block, n, i, *iptr;
    char *ptr;
    short *sptr, x;

    delta_block = block - block_io->last_done;

    CHECK_STATUS(safe_fseek_relative(block_io->file, delta_block, block_io->block_size, block_io->byte_size, error_msg));

    n = nblocks * block_io->block_size;

    if (FREAD(data, block_io->byte_size, n, block_io->file))
	RETURN_ERROR_MSG("reading data");

    if (block_io->byte_size == BYTES_PER_WORD)
    {
	if (block_io->swapped)
	    swap_bytes((char *) data, BYTES_PER_WORD*n);
    }
    else /* block_io->byte_size == 2 bytes */
    {
	if (block_io->swapped)
	{
	    ptr = (char *) data;

	    for (i = 0; i < n; i++)
		SWAP(ptr[2*i], ptr[2*i+1], char);
	}

	if (block_io->integer)
	{
	    sptr = (short *) data;
	    iptr = (int *) data;

	    for (i = n-1; i >= 0; i--)
	    {
		x = sptr[i];
		iptr[i] = (int) x;
	    }
	}
	/* 2 byte data should be floating point */
    }

    if (block_io->integer)
	float_words(data, n);

    block_io->last_done = block + nblocks;

    return  OK;
}

static Status decode_block(Block_IO *block_io, float *data, int *nwords,
							String error_msg)
{
    int n, nz, nv;

    *nwords = 0;
    n = 0;

    while (n < block_io->block_size)
    {
    	if (FREAD(&nz, BYTES_PER_WORD, 1, block_io->file))
	    RETURN_ERROR_MSG("reading number of zeroes");

    	if (block_io->swapped)
	    swap_bytes((char *) &nz, BYTES_PER_WORD);

	(*nwords)++;

	if (nz == -1)
	    return  OK;

	n += nz;

    	if (FREAD(&nv, BYTES_PER_WORD, 1, block_io->file))
	    RETURN_ERROR_MSG("reading number of values");

    	if (block_io->swapped)
	    swap_bytes((char *) &nv, BYTES_PER_WORD);

	(*nwords)++;

	if (nv == -1)
	    return  OK;

	if (nv > 0) /* always happens? */
	{
    	    if (FREAD(data+n, BYTES_PER_WORD, nv, block_io->file))
	    	RETURN_ERROR_MSG("reading values");

    	    if (block_io->swapped)
	    	swap_bytes((char *) (data+n), BYTES_PER_WORD*nv);

	    (*nwords) += nv;
	}

	n += nv;
    }

    if (FREAD(&nz, BYTES_PER_WORD, 1, block_io->file))
	RETURN_ERROR_MSG("reading end flag");

    if (block_io->swapped)
	swap_bytes((char *) &nz, BYTES_PER_WORD);

    (*nwords)++;

    return  OK;
}

static Status read_deflated_block(Block_IO *block_io, int block, float *data,
							String error_msg)
{
    int delta_word, word, nwords;

    ZERO_VECTOR(data, block_io->block_size);

    if ((word = block_io->directory[block]) == -1)
	return  OK;

    delta_word = word - block_io->last_done;

    CHECK_STATUS(safe_fseek_relative(block_io->file, delta_word, 1, BYTES_PER_WORD, error_msg));

    CHECK_STATUS(decode_block(block_io, data, &nwords, error_msg));

    block_io->last_done = word + nwords;

    return  OK;
}

Status read_file_block(Block_IO *block_io, int block, float *data,
							String error_msg)
{
    sprintf(error_msg, "\"%s\": ", short_file_name(block_io->name));
    error_msg += strlen(error_msg);

    if (!(block_io->deflated))
    {
	CHECK_STATUS(read_normal_blocks(block_io, block, 1, data, error_msg));
    }
    else
    {
	CHECK_STATUS(read_deflated_block(block_io, block, data, error_msg));
    }

    return  OK;
}

Status read_file_blocks(Block_IO *block_io, int block, int nblocks,
					float *data, String error_msg)
{
    int i;
    float *d;

    sprintf(error_msg, "\"%s\": ", short_file_name(block_io->name));
    error_msg += strlen(error_msg);

    if (!(block_io->deflated))
    {
	CHECK_STATUS(read_normal_blocks(block_io, block, nblocks, data, error_msg));
    }
    else
    {
	for (i = 0; i < nblocks; i++)
	{
	    d = data + i * block_io->block_size;
	    CHECK_STATUS(read_deflated_block(block_io, block+i, d, error_msg));
	}
    }

    return  OK;
}

static Status write_directory(Block_IO *block_io, String error_msg)
{
/*
    if (FWRITE(block_io->directory, BYTES_PER_WORD, block_io->dir_size, block_io->file))
*/
    if (endian_fwrite((char *) block_io->directory, block_io->dir_size,
						block_io->file) == ERROR)
	RETURN_ERROR_MSG("writing directory");

    return  OK;
}

Status init_block_write(Block_IO *block_io, String error_msg)
{
    sprintf(error_msg, "\"%s\": ", short_file_name(block_io->name));
    error_msg += strlen(error_msg);

    if (block_io->deflated)
    {
	ZERO_VECTOR(block_io->directory, block_io->dir_size);

	CHECK_STATUS(write_directory(block_io, error_msg));
    }

    block_io->last_done = 0;

    return  OK;
}

static Status write_normal_blocks(Block_IO *block_io, int block, int nblocks,
					float *data, String error_msg)
{
    int delta_block, n;

    delta_block = block - block_io->last_done;

    CHECK_STATUS(safe_fseek_relative(block_io->file, delta_block, block_io->block_size, BYTES_PER_WORD, error_msg));

    n = nblocks * block_io->block_size;

/*
    if (FWRITE(data, BYTES_PER_WORD, n, block_io->file))
*/
    if (endian_fwrite((char *) data, n, block_io->file) == ERROR)
	RETURN_ERROR_MSG("writing data");

    block_io->last_done = block + nblocks;

    return  OK;
}

#define  VALUE_IS_ZERO(d)  (ABS(d) < level)

static int number_zeroes(float level, float *s, int n)
{
    int m;

    for (m = 0; m < n; m++)
    {
	if (!VALUE_IS_ZERO(s[m]))
	    return  m;
    }

    return  n;
}

static int number_values(float level, float *s, int n)
{
    int m;

    for (m = 0; m < n; m++)
    {
	if (VALUE_IS_ZERO(s[m]))
	    return  m;
    }

    return  n;
}

static Status encode_block(Block_IO *block_io, float *data, int *nwords,
							String error_msg)
{
    int n, nz, nv;

    n = 0;

    *nwords = 0;

    while (n < block_io->block_size)
    {
	nz = number_zeroes(block_io->level, data+n, block_io->block_size-n);

	if (nz == block_io->block_size)
	    return  OK;

	n += nz;
	if (n < block_io->block_size)
	    nv = number_values(block_io->level, data+n, block_io->block_size-n);
	else
	    nz = -1;

/*
    	if (FWRITE(&nz, BYTES_PER_WORD, 1, block_io->file))
*/
	if (endian_fwrite((char *) &nz, 1, block_io->file) == ERROR)
	    RETURN_ERROR_MSG("writing number zeroes");

	(*nwords)++;

	if (nz == -1)
	{
	    return  OK;
	}
	else
	{
/*
    	    if (FWRITE(&nv, BYTES_PER_WORD, 1, block_io->file))
*/
	    if (endian_fwrite((char *) &nv, 1, block_io->file) == ERROR)
		RETURN_ERROR_MSG("writing number values");

	    (*nwords)++;

	    if (nv > 0)
	    {
/*
    	    	if (FWRITE(data+n, BYTES_PER_WORD, nv, block_io->file))
*/
		if (endian_fwrite((char *) (data+n), nv, block_io->file)
								== ERROR)
		    RETURN_ERROR_MSG("writing values");

	    	(*nwords) += nv;
	    }
	}

	n += nv;
    }

    nz = -1;

/*
    if (FWRITE(&nz, BYTES_PER_WORD, 1, block_io->file))
*/
    if (endian_fwrite((char *) &nz, 1, block_io->file) == ERROR)
	RETURN_ERROR_MSG("writing end flag");

    (*nwords)++;

    return  OK;
}

static Status write_deflated_block(Block_IO *block_io, int block, float *data,
							String error_msg)
{
    int nwords;

    CHECK_STATUS(encode_block(block_io, data, &nwords, error_msg));

    if (nwords == 0)
	block_io->directory[block] = -1;
    else
	block_io->directory[block] = block_io->last_done;

    block_io->last_done += nwords;

    return  OK;
}

Status write_file_block(Block_IO *block_io, int block, float *data,
							String error_msg)
{
    sprintf(error_msg, "\"%s\": ", short_file_name(block_io->name));
    error_msg += strlen(error_msg);

    if (!(block_io->deflated))
    {
	CHECK_STATUS(write_normal_blocks(block_io, block, 1, data, error_msg));
    }
    else
    {
	CHECK_STATUS(write_deflated_block(block_io, block, data, error_msg));
    }

    return  OK;
}

Status write_file_blocks(Block_IO *block_io, int block, int nblocks,
					float *data, String error_msg)
{
    int i;
    float *d;

    sprintf(error_msg, "\"%s\": ", short_file_name(block_io->name));
    error_msg += strlen(error_msg);

    if (!(block_io->deflated))
    {
	CHECK_STATUS(write_normal_blocks(block_io, block, nblocks, data, error_msg));
    }
    else
    {
	for (i = 0; i < nblocks; i++)
	{
	    d = data + i * block_io->block_size;
	    CHECK_STATUS(write_deflated_block(block_io, block+i, d, error_msg));
	}
    }

    return  OK;
}

static Status pad_file(FILE *fp, int n, String error_msg) /* needed for Ansig */
{
    int i, pad_size = 4096;
    float zero[1];

    n %= pad_size;

    if (n > 0)
    {
	zero[0] = 0;

	for (i = n; i < pad_size; i++)
	{
	    if (FWRITE(zero, BYTES_PER_WORD, 1, fp))
		RETURN_ERROR_MSG("padding file");
	}
    }

    return  OK;
}

Status end_block_write(Block_IO *block_io, String error_msg)
{
    sprintf(error_msg, "\"%s\": ", short_file_name(block_io->name));
    error_msg += strlen(error_msg);

    if (block_io->deflated)
    {
	CHECK_STATUS(pad_file(block_io->file, block_io->last_done, error_msg));

	rewind(block_io->file);
	CHECK_STATUS(write_directory(block_io, error_msg));
    }

    return  OK;
}

Status skip_file_blocks(Block_IO *block_io, int nblocks, String error_msg)
{
    CHECK_STATUS(safe_fseek_relative(block_io->file, nblocks, block_io->block_size, BYTES_PER_WORD, error_msg));

/*  should not do below, this is a skip, so pretend it has not happened
    block_io->last_done += nblocks;
*/

    return  OK;
}
