/*  Note, program below assumes that block_size_out is
    multiple of block_size_in in the projected dimensions  */

#include "block.h"

#include "block_io.h"

static int ndim;
static int ndim_proj;
static int *dim_proj;
static int *npoints_in;
static int *npoints_out;
static int *block_size_in;
static int *block_size_out;

static float *store_in;
static float *store_out;

static int ndim_rest;
static int total_nblocks_in;
static int total_nblocks_out;
static int size_of_block_in;
static int size_of_block_out;
static int size_of_block_proj;
static int size_of_block_rest;
static int nsub_blocks;
static int nproj_blocks;
static int point;

static float threshold;

static int dim_rest[MAX_NDIM];
static int nblocks_in[MAX_NDIM];
static int nblocks_out[MAX_NDIM];
static int block_in[MAX_NDIM];
static int block_out[MAX_NDIM];
static int sub_block[MAX_NDIM];
static int array[MAX_NDIM];
static int ratio_block_size[MAX_NDIM];
static int cum_nblocks_in[MAX_NDIM];
static int cum_nblocks_out[MAX_NDIM];
static int cum_nblocks_rest[MAX_NDIM];
static int cum_block_size_in[MAX_NDIM];
static int cum_block_size_out[MAX_NDIM];
static int cum_block_size_proj1[MAX_NDIM];
static int cum_block_size_proj2[MAX_NDIM];
static int cum_block_size_rest1[MAX_NDIM];
static int cum_block_size_rest2[MAX_NDIM];
static int cum_ratio[MAX_NDIM];

static Block_IO block_io_in;
static Block_IO block_io_out;

static void init_arrays()
{
    int i, j;

    ndim_rest = 0;
    for (i = 0; i < ndim; i++)
    {
	for (j = 0; j < ndim_proj; j++)
	    if (i == dim_proj[j])
		break;

	if (j == ndim_proj)
	    dim_rest[ndim_rest++] = i;
    }

    BLOCKS(nblocks_in, npoints_in, block_size_in, ndim);
    BLOCKS(nblocks_out, npoints_out, block_size_out, ndim_proj);

    CUMULATIVE(cum_nblocks_in, nblocks_in, total_nblocks_in, ndim);
    CUMULATIVE(cum_nblocks_out, nblocks_out, total_nblocks_out, ndim_proj);

    CUMULATIVE(cum_block_size_in, block_size_in, size_of_block_in, ndim);
   CUMULATIVE(cum_block_size_out, block_size_out, size_of_block_out, ndim_proj);

    for (i = 0; i < ndim_proj; i++)
    {
	j = dim_proj[i];
	ratio_block_size[i] = block_size_out[i] / block_size_in[j];
    }

    CUMULATIVE(cum_ratio, ratio_block_size, nsub_blocks, ndim_proj);

    for (i = 0; i < ndim_proj; i++)
    {
	j = dim_proj[i];
	cum_block_size_proj2[i] = cum_block_size_in[j];
	array[i] = block_size_in[j];
    }

    CUMULATIVE(cum_block_size_proj1, array, size_of_block_proj, ndim_proj);

    for (i = 0; i < ndim_rest; i++)
    {
	j = dim_rest[i];
	array[i] = nblocks_in[j];
	cum_block_size_rest2[i] = cum_block_size_in[j];
    }

    CUMULATIVE(cum_nblocks_rest, array, nproj_blocks, ndim_rest);

    for (i = 0; i < ndim_rest; i++)
    {
	j = dim_rest[i];
	cum_block_size_rest2[i] = cum_block_size_in[j];
	array[i] = block_size_in[j];
    }

    CUMULATIVE(cum_block_size_rest1, array, size_of_block_rest, ndim_rest);
}

static Status project_blocks(String error_msg)
{
    int i, j, k, pt_in, pt_out, block;
    float *s_in, *s_out, t, s;

    s_out = store_out + point;

    for (i = 0; i < nproj_blocks; i++)
    {
	ARRAY_OF_INDEX(array, i, cum_nblocks_rest, ndim_rest);

	for (j = 0; j < ndim_rest; j++)
	{
	    k = dim_rest[j];
	    block_in[k] = array[j];
	}

	INDEX_OF_ARRAY(block, block_in, cum_nblocks_in, ndim);

	CHECK_STATUS(read_file_block(&block_io_in, block, store_in, error_msg));

	for (j = 0; j < size_of_block_proj; j++)
	{
	    ARRAY_OF_INDEX(array, j, cum_block_size_proj1, ndim_proj);
	    INDEX_OF_ARRAY(pt_out, array, cum_block_size_out, ndim_proj);

	    INDEX_OF_ARRAY(pt_in, array, cum_block_size_proj2, ndim_proj);
	    s_in = store_in + pt_in;

	    t = 0;
	    for (k = 0; k < size_of_block_rest; k++)
	    {
		ARRAY_OF_INDEX(array, k, cum_block_size_rest1, ndim_rest);
		INDEX_OF_ARRAY(pt_in, array, cum_block_size_rest2, ndim_rest);

		s = s_in[pt_in];
		if (ABS(s) >= threshold)
		    t += s;
	    }

	    s_out[pt_out] += t;
	}
    }

    return  OK;
}

Status block_process(Size_info *size_in, Size_info *size_out,
			Store_info *store_info, File_info *file_info,
			Project_info *project_info, String error_msg)
{
    int i, j, k, l, p;
    Bool do_block;

    ndim = size_in->ndim;
    npoints_in = size_in->npoints;
    block_size_in = size_in->block_size;

    ndim_proj = size_out->ndim;
    npoints_out = size_out->npoints;
    block_size_out = size_out->block_size;

    store_in = store_info->store_in;
    store_out = store_info->store_out;

    dim_proj = project_info->dim_proj;
    threshold = project_info->threshold;

    if (!(file_info->blocked))
	RETURN_ERROR_MSG("input file must be blocked");

    init_arrays();

    block_io_in.name = file_info->input_file;
    block_io_in.file = file_info->file_in;
    block_io_in.swapped = file_info->swapped;
    block_io_in.integer = file_info->integer;
    block_io_in.deflated = file_info->deflated;
    block_io_in.header = file_info->header;
    block_io_in.dir_size = file_info->dir_size;
    block_io_in.directory = store_info->directory;
    block_io_in.block_size = size_of_block_in;
    block_io_in.byte_size = BYTES_PER_WORD;

    block_io_out.name = file_info->output_file;
    block_io_out.file = file_info->file_out;
    block_io_out.block_size = size_of_block_out;
    block_io_out.deflated = FALSE;

    CHECK_STATUS(init_block_read(&block_io_in, error_msg));
    CHECK_STATUS(init_block_write(&block_io_out, error_msg));

    p = MAX(total_nblocks_out/32, 1);
    for (i = 0; i < total_nblocks_out; i++)
    {
	if (!(i % p))
        {
	    printf("\t... working on output block %d of %d\n",
						i+1, total_nblocks_out);
	    FLUSH;
	}

	ARRAY_OF_INDEX(block_out, i, cum_nblocks_out, ndim_proj);

	ZERO_VECTOR(store_out, size_of_block_out);

	for (j = 0; j < nsub_blocks; j++)
	{
	    ARRAY_OF_INDEX(sub_block, j, cum_ratio, ndim_proj);

	    do_block = TRUE;
	    for (k = 0; k < ndim_proj; k++)
	    {
		l = dim_proj[k];

		block_in[l] = block_out[k]*ratio_block_size[k] + sub_block[k];

		if (block_in[l] >= nblocks_in[l])
		{
		    do_block = FALSE;
		    break;
		}

		array[k] = sub_block[k] * block_size_in[l];
	    }


	    if (do_block)
	    {
		INDEX_OF_ARRAY(point, array, cum_block_size_out, ndim_proj);

		CHECK_STATUS(project_blocks(error_msg));
	    }
	}

	CHECK_STATUS(write_file_block(&block_io_out, i, store_out, error_msg));
    }

    return  OK;
}
