#include "window.h"

#include "list.h"

static XtAppContext application;
static Widget topshell = (Widget) NULL;
static Display *display;
static List shell_list = (List) NULL;
static Cursor cursor;

Widget initialize_application(String name, int *argc, char **argv)
{
    topshell = XtVaAppInitialize(&application, name,
                NULL, 0, argc, argv, NULL, NULL);

    if (topshell)
    {
	display = XtDisplay(topshell);

	if (init_list(&shell_list) == OK)
	    (void) insert_list(&shell_list, topshell);
    }

    return  topshell;
}

void application_main_loop()
{
    XtRealizeWidget(topshell);
    XtAppMainLoop(application);
}

void de_iconify_application()
{
    XMapWindow(display, XtWindow(topshell));
}

void manage_widget(Widget widget)
{
    XtManageChild(widget);
}

void realize_widget(Widget widget)
{
    XtRealizeWidget(widget);
}

void popup(Widget widget)
{
/*
    XtPopup(widget, XtGrabNone);
*/
    XtManageChild(widget);
    XtPopup(XtParent(widget), XtGrabNone);
}

void popdown(Widget widget)
{
/*
    XtPopdown(widget);
*/
    XtPopdown(XtParent(widget));
    XtUnmanageChild(widget);
}

Widget get_topshell()
{
    return  topshell;
}

Widget top_shell(Widget widget)
{
    while (widget && !XtIsWMShell(widget))
	widget = XtParent(widget);

    return  widget;
}

void change_delete_protocol(Widget shell, Callback_proc callback)
{
    Atom protocol;

    protocol = XmInternAtom(display, "WM_DELETE_WINDOW", False);
    XmAddWMProtocols(shell, &protocol, 1);
    XmAddWMProtocolCallback(shell, protocol, callback, NULL);
}

void do_nothing_callback(Widget widget, Callback_ptr client_data,
							Callback_ptr cbs)
{
    /* do nothing */
}

void not_implemented_callback(Widget widget, Callback_ptr client_data,
							Callback_ptr cbs)
{
    ERROR_AND_RETURN("feature not yet implemented");
}

void attachments(Widget widget, Widget top, Widget bottom,
					Widget left, Widget right)
{
    if (top == FORM_ATTACH)
	XtVaSetValues(widget,
		XmNtopAttachment,	XmATTACH_FORM,
		NULL);
    else if (top != NO_ATTACH)
	XtVaSetValues(widget,
		XmNtopAttachment,	XmATTACH_WIDGET,
		XmNtopWidget,		top,
		NULL);

    if (bottom == FORM_ATTACH)
	XtVaSetValues(widget,
		XmNbottomAttachment,	XmATTACH_FORM,
		NULL);
    else if (bottom != NO_ATTACH)
	XtVaSetValues(widget,
		XmNbottomAttachment,	XmATTACH_WIDGET,
		XmNbottomWidget,	bottom,
		NULL);

    if (left == FORM_ATTACH)
	XtVaSetValues(widget,
		XmNleftAttachment,	XmATTACH_FORM,
		NULL);
    else if (left != NO_ATTACH)
	XtVaSetValues(widget,
		XmNleftAttachment,	XmATTACH_WIDGET,
		XmNleftWidget,		left,
		NULL);

    if (right == FORM_ATTACH)
	XtVaSetValues(widget,
		XmNrightAttachment,	XmATTACH_FORM,
		NULL);
    else if (right != NO_ATTACH)
	XtVaSetValues(widget,
		XmNrightAttachment,	XmATTACH_WIDGET,
		XmNrightWidget,		right,
		NULL);
}

void positions(Widget widget, int top, int bottom, int left, int right)
{
    if (top != NO_POSITION)
	XtVaSetValues(widget,
		XmNtopAttachment,	XmATTACH_POSITION,
		XmNtopPosition,		top,
		NULL);

    if (bottom != NO_POSITION)
	XtVaSetValues(widget,
		XmNbottomAttachment,	XmATTACH_POSITION,
		XmNbottomPosition,	bottom,
		NULL);

    if (left != NO_POSITION)
	XtVaSetValues(widget,
		XmNleftAttachment,	XmATTACH_POSITION,
		XmNleftPosition,	left,
		NULL);

    if (right != NO_POSITION)
	XtVaSetValues(widget,
		XmNrightAttachment,	XmATTACH_POSITION,
		XmNrightPosition,	right,
		NULL);
}

void offsets(Widget widget, int top, int bottom, int left, int right)
{
    XtVaSetValues(widget,
		XmNtopOffset,		top,
		XmNbottomOffset,	bottom,
		XmNleftOffset,		left,
		XmNrightOffset,		right,
		NULL);
}

Bool toggle_being_set(Callback_ptr cbs)
{
    XmToggleButtonCallbackStruct *tbcbs = (XmToggleButtonCallbackStruct *) cbs;

    if (tbcbs->set)
	return  TRUE;
    else
	return  FALSE;
}

int get_radio_state(Widget *boxes, int nboxes)
{
    int i;

    for (i = 0; i < nboxes; i++)
        if (XmToggleButtonGadgetGetState(boxes[i]))
	    return  i;

    return -1;
}

void set_radio_state(Widget *boxes, int nboxes, int to_set)
{
    if ((to_set < 0) || (to_set >= nboxes))
	return;

    XmToggleButtonGadgetSetState(boxes[to_set], True, True);
}

int get_check_state(Widget *boxes, int nboxes)
{
    int i, n;

    n = 0;
    for (i = nboxes-1; i >= 0; i--)
    {
	n <<= 1;
        if (XmToggleButtonGadgetGetState(boxes[i]))
	    n++;
    }

    return  n;
}

void set_check_state(Widget *boxes, int nboxes, int to_set)
{
    int i;

    if (to_set <= 0)
	return;

    for (i = 0; i < nboxes; i++)
    {
	if (to_set & 1)
    	    XmToggleButtonGadgetSetState(boxes[i], True, True);

	to_set >>= 1;
    }
}

void get_textfields(Textfield_item *textfield_items, int ntexts)
{
    int i;
    char *value;

    for (i = 0; i < ntexts; i++)
    {
	value = XmTextFieldGetString(*(textfield_items[i].field));
	strncpy(textfield_items[i].value, value, LINE_SIZE);
	XtFree(value);
    }
}

void set_textfields(Textfield_item *textfield_items, int ntexts)
{
    int i;

    for (i = 0; i < ntexts; i++)
    {
	XmTextFieldSetString(*(textfield_items[i].field),
						textfield_items[i].value);
    }
}

void set_text_edit_state(Widget widget, Bool state)
{
    XtVaSetValues(widget,
	XmNeditable,	state,
	NULL);
}

void set_label(Widget widget, String label)
{
    XmString string = XmStringCreateSimple(label);
    XtVaSetValues(widget,
	XmNlabelString,	string,
	XmNalignment,	XmALIGNMENT_CENTER,
	NULL);
    XmStringFree(string);
}

void print_text(Widget widget, Bool new_text, String message)
{
    static XmTextPosition position;

    if (new_text)
    {
	XmTextSetString(widget, message);
	position = 0;
    }
    else
    {
	XmTextInsert(widget, position, message);
    }

    position += strlen(message);
}

Status insert_in_list(Widget widget, String string, int position)
{
    XmString item = XmStringCreateSimple(string);

    if (!item)
	return  ERROR;
 
    XmListAddItemUnselected(widget, item, position);
    XmStringFree(item);

    return  OK;
}

Status replace_in_list(Widget widget, String string, int position)
{
    XmString item = XmStringCreateSimple(string);

    if (!item)
	return  ERROR;
 
    XmListReplaceItemsPos(widget, &item, 1, position);
    XmStringFree(item);

    return  OK;
}

void delete_from_list(Widget widget, int position)
{
    XmListDeletePos(widget, position);
}

void select_from_list(Widget widget, int position, Bool notify)
{
    XmListSelectPos(widget, position, (Boolean) notify);
}

void deselect_from_list(Widget widget, int position)
{
    XmListDeselectPos(widget, position);
}

Bool selection_from_list(Widget widget, int **selected, int *nselected)
{
    return  (Bool) XmListGetSelectedPos(widget, selected, nselected);
}

int list_position(Callback_ptr cbs)
{
    XmListCallbackStruct *lcbs = (XmListCallbackStruct *) cbs;

    return  (lcbs->item_position - 1);
}

void get_scale_value(Callback_ptr cbs, Scale_numbers *numbers)
{
    XmScaleCallbackStruct *scbs = (XmScaleCallbackStruct *) cbs;

    numbers->value = ((float) scbs->value) / numbers->multiplier;
}

void set_scale_value(Widget scale, Scale_numbers *numbers)
{
    float value;

    value = numbers->multiplier * numbers->value;

    XtVaSetValues(scale,
		XmNvalue,		(int) NEAREST_INTEGER(value),
		NULL);
}

static short scale_decimal(float minimum, float maximum)
{
    short d;
    double r;

    r = maximum - minimum;
    r = log(r) / log(10);  /* seems to be a bug with log10() here?!? */

    if (r >= 2)
	d = 0;
    else if (r >= 1)
	d = 1;
    else if (r >= 0)
	d = 2;
    else
	d = 2.9999 - r;

    return  d;
}

void set_scale_numbers(Widget scale, Scale_numbers *numbers)
{
    short n;
    int min, max, v, s;
    float minimum, maximum, value;

    n = scale_decimal(numbers->minimum, numbers->maximum);

    numbers->multiplier = pow(10.0, (double) n);
    minimum = numbers->multiplier * numbers->minimum;
    maximum = numbers->multiplier * numbers->maximum;
    value = numbers->multiplier * numbers->value;

    min = NEAREST_INTEGER(minimum);
    max = NEAREST_INTEGER(maximum);
    v = NEAREST_INTEGER(value);
    s = (max - min) / 10;  s = MAX(s, 1);

    if (v < min)
    {
	v = min;
	numbers->value = v / numbers->multiplier;
    }
    else if (v > max)
    {
	v = max;
	numbers->value = v / numbers->multiplier;
    }

    XtVaSetValues(scale,
		XmNdecimalPoints,	n,
		XmNminimum,		min,
		XmNmaximum,		max,
		XmNvalue,		v,
		XmNscaleMultiple,	s,
		NULL);
}

void get_widget_size(Widget widget, int *width, int *height)
{
    Dimension w, h;

    XtVaGetValues(widget,
		XmNwidth,	&w,
		XmNheight,	&h,
		NULL);

    *width = (int) w;  *height = (int) h;
}

void set_widget_size(Widget widget, int width, int height)
{
    Dimension w, h;

    w = (Dimension) width;  h = (Dimension) height;

    XtVaSetValues(widget,
		XmNwidth,	w,
		XmNheight,	h,
		NULL);
}

int get_scrollbar_value(Callback_ptr cbs)
{
    XmScrollBarCallbackStruct *sbcbs = (XmScrollBarCallbackStruct *) cbs;

    return  sbcbs->value;
}

void set_scrollbar_value(Widget scrollbar, int value)
{
    XtVaSetValues(scrollbar,
		XmNvalue,		value,
                NULL);
}

void destroy_widget(Widget widget)
{
    unregister_shell(widget);

    XtDestroyWidget(widget);
}

Widget create_popup(Widget parent, String name)
{
    Widget popup;

    popup = XtVaCreatePopupShell(name,
		xmDialogShellWidgetClass,	parent,
		NULL);

    if (popup)
	register_shell(popup);

    return  popup;
}

Widget create_iconifiable_popup(Widget parent, String name)
{
    Widget popup;

    popup = XtVaCreatePopupShell(name,
		topLevelShellWidgetClass,	parent,
		NULL);

    if (popup)
	register_shell(popup);

    return  popup;
}

Widget create_drawing(Widget parent, Drawing_area_info *drawing_info)
{
    Widget drawing;

    drawing = XtVaCreateManagedWidget("Drawing",
                xmDrawingAreaWidgetClass,      parent,
		XmNunitType,	XmPIXELS,
		XmNwidth,	(Dimension) drawing_info->width,
		XmNheight,	(Dimension) drawing_info->height,
                NULL);

    if (drawing)
    {
	if (drawing_info->expose)
	    XtAddCallback(drawing, XmNexposeCallback,
						drawing_info->expose, NULL);

	if (drawing_info->resize)
	    XtAddCallback(drawing, XmNresizeCallback,
						drawing_info->resize, NULL);
    }

    return  drawing;
}

void update_scrollbar(Widget scrollbar, int view_size)
{
    XtVaSetValues(scrollbar,
		XmNsliderSize,		view_size,
		XmNincrement,		MAX(1,view_size/40),
		XmNpageIncrement,	MAX(1,view_size/2),
                NULL);
}

Widget create_scrolled_drawing(Widget parent,
			Scrolled_drawing_area_info *drawing_info)
{
    Widget scrolled_window, drawing;

    scrolled_window = XtVaCreateManagedWidget("Scrolled_window",
		xmScrolledWindowWidgetClass,	parent,
		XmNscrollingPolicy,		XmAPPLICATION_DEFINED,
		XmNvisualPolicy,		XmVARIABLE,
/* wrong in spirit really, but it does not seem to work just on drawing */
		XmNwidth,	20 + (Dimension) drawing_info->view_width,
		XmNheight,	20 + (Dimension) drawing_info->view_height,
		NULL);

    CHECK_WIDGET(scrolled_window);

    drawing = XtVaCreateManagedWidget("Drawing",
                xmDrawingAreaWidgetClass,      scrolled_window,
		XmNunitType,	XmPIXELS,
		XmNwidth,	(Dimension) drawing_info->view_width,
		XmNheight,	(Dimension) drawing_info->view_height,
                NULL);

    CHECK_WIDGET_DESTROY(drawing, scrolled_window);

    if (drawing_info->expose)
	XtAddCallback(drawing, XmNexposeCallback, drawing_info->expose, NULL);

    if (drawing_info->resize)
	XtAddCallback(drawing, XmNresizeCallback, drawing_info->resize, NULL);

    drawing_info->bars[0] = XtVaCreateManagedWidget("Scrollbar",
		xmScrollBarWidgetClass,	scrolled_window,
		XmNorientation,		XmHORIZONTAL,
		XmNminimum,		0,
		XmNmaximum,		drawing_info->width,
                NULL);

    CHECK_WIDGET_DESTROY(drawing_info->bars[0], scrolled_window);

    update_scrollbar(drawing_info->bars[0], drawing_info->width);

    drawing_info->bars[1] = XtVaCreateManagedWidget("Scrollbar",
		xmScrollBarWidgetClass,	scrolled_window,
		XmNorientation,		XmVERTICAL,
		XmNminimum,		0,
		XmNmaximum,		drawing_info->height,
                NULL);

    CHECK_WIDGET_DESTROY(drawing_info->bars[1], scrolled_window);

    update_scrollbar(drawing_info->bars[1], drawing_info->height);

    XmScrolledWindowSetAreas(scrolled_window, drawing_info->bars[0],
					drawing_info->bars[1],	drawing);

    if (drawing_info->scroll)
    {
	XtAddCallback(drawing_info->bars[0],
			XmNvalueChangedCallback, drawing_info->scroll, NULL);
	XtAddCallback(drawing_info->bars[0],
			XmNdragCallback, drawing_info->scroll, NULL);
	XtAddCallback(drawing_info->bars[1],
			XmNvalueChangedCallback, drawing_info->scroll, NULL);
	XtAddCallback(drawing_info->bars[1],
			XmNdragCallback, drawing_info->scroll, NULL);
    }

    return  drawing;
}

Widget create_scrolled_window(Widget parent,
				Scrolled_window_info *scrolled_info)
{
    Widget scrolled_window;

    scrolled_window = XtVaCreateManagedWidget("Scrolled_window",
		xmScrolledWindowWidgetClass,	parent,
		XmNscrollingPolicy,		XmAUTOMATIC,
		XmNscrollBarDisplayPolicy,	XmAS_NEEDED,
		XmNwidth,	(Dimension) scrolled_info->width,
		XmNheight,	(Dimension) scrolled_info->height,
		NULL);

    return  scrolled_window;
}

Widget create_form(Widget parent)
{
    return  XtVaCreateWidget("Form",
                xmFormWidgetClass,      parent,
                NULL);
}

Widget create_rowcol(Widget parent)
{
    return  XtVaCreateWidget("Rowcol",
                xmRowColumnWidgetClass,      parent,
                NULL);
}

Widget create_separator(Widget parent)
{
    return  XtVaCreateManagedWidget("Separator",
                xmSeparatorWidgetClass,      parent,
                NULL);
}

Widget create_label(Widget parent, String string)
{
    Widget label = (Widget) NULL;
    XmString str = XmStringCreateSimple(string);

    if (str)
    {
	label= XtVaCreateManagedWidget("Label",
		xmLabelWidgetClass,	parent,
		XmNlabelString,		str,
		NULL);

	XmStringFree(str);
    }

    return  label;
}

Widget create_pixmap_button(Widget parent, Button_item *button_item,
								Pixmap pixmap)
{
    Widget button;

    button = XtVaCreateManagedWidget("Button",
		xmPushButtonWidgetClass,	parent,
		XmNlabelType,		XmPIXMAP,
		XmNlabelPixmap,		pixmap,
		NULL);

    CHECK_WIDGET(button);

    if (button_item->callback)
    {
	XtAddCallback(button, XmNactivateCallback,
		button_item->callback, (XtPointer) button_item->data);
    }

    return  button;
}

#define  TIGHTNESS  20

Widget create_horizontal_buttons(Widget parent, Button_item *button_items,
								int nbuttons)
{
    int i;
    Widget form, button;
    XmString string;

    form = XtVaCreateWidget("Buttons",
		xmFormWidgetClass,	parent,
		XmNfractionBase,	TIGHTNESS*nbuttons - 1,
		XmNleftOffset,		10,
		XmNrightOffset,		10,
		NULL);

    CHECK_WIDGET(form);

    for (i = 0; i < nbuttons; i++)
    {
	string = XmStringCreateSimple(button_items[i].label);
	CHECK_WIDGET_DESTROY(string, form);

        button = XtVaCreateManagedWidget("Button",
		xmPushButtonWidgetClass,	form,
		XmNlabelString,		string,
		XmNleftAttachment,	i ? XmATTACH_POSITION : XmATTACH_FORM,
		XmNleftPosition,	TIGHTNESS*i,
		XmNtopAttachment,	XmATTACH_FORM,
		XmNbottomAttachment,	XmATTACH_FORM,
		XmNrightAttachment,
			i != nbuttons-1  ? XmATTACH_POSITION : XmATTACH_FORM,
		XmNrightPosition,	TIGHTNESS*i + (TIGHTNESS-1),
/*
		XmNshowAsDefault,	i == 0,
		XmNdefaultButtonShadowThickness,	1,
*/
		NULL);

	XmStringFree(string);

	CHECK_WIDGET_DESTROY(button, form);

        if (button_items[i].callback)
	{
	    XtAddCallback(button, XmNactivateCallback,
		button_items[i].callback, (XtPointer) button_items[i].data);
	}
    }

    XtManageChild(form);

    return  form;
}

Widget create_vertical_buttons(Widget parent, Button_item *button_items,
								int nbuttons)
{
    int i;
    Widget form, button;
    XmString string;

    form = XtVaCreateWidget("Buttons",
		xmFormWidgetClass,	parent,
		XmNfractionBase,	TIGHTNESS*nbuttons - 1,
		XmNtopOffset,		10,
		XmNbottomOffset,	10,
		NULL);

    CHECK_WIDGET(form);

    for (i = 0; i < nbuttons; i++)
    {
	string = XmStringCreateSimple(button_items[i].label);
	CHECK_WIDGET_DESTROY(string, form);

        button = XtVaCreateManagedWidget("Button",
		xmPushButtonWidgetClass,	form,
		XmNlabelString,		string,
		XmNtopAttachment,	i ? XmATTACH_POSITION : XmATTACH_FORM,
		XmNtopPosition,	TIGHTNESS*i,
		XmNleftAttachment,	XmATTACH_FORM,
		XmNrightAttachment,	XmATTACH_FORM,
		XmNbottomAttachment,
			i != nbuttons-1  ? XmATTACH_POSITION : XmATTACH_FORM,
		XmNbottomPosition,	TIGHTNESS*i + (TIGHTNESS-1),
		NULL);

	XmStringFree(string);

	CHECK_WIDGET_DESTROY(button, form);

        if (button_items[i].callback)
	{
	    XtAddCallback(button, XmNactivateCallback,
		button_items[i].callback, (XtPointer) button_items[i].data);
	}
    }

    XtManageChild(form);

    return  form;
}

#undef  TIGHTNESS

void initialize_boxes(Radiobox_item *items, String *labels,
                                                Widget *boxes, int nboxes)
{
    int i;

    for (i = 0; i < nboxes; i++)
    {
	items[i].label = labels[i];
	items[i].box = boxes + i;
	items[i].callback = NULL;
    }
}

Widget create_radiobox(Widget parent, Radiobox_item *radiobox_items,
				int nboxes, int item_set, int orientation)
{
    int i;
    Widget box, toggle;
    XmString string;

    box = XmCreateSimpleRadioBox(parent, "Radiobox", NULL, 0);
    CHECK_WIDGET(box);

    for (i = 0; i < nboxes; i++)
    {
	string = XmStringCreateSimple(radiobox_items[i].label);
	CHECK_WIDGET_DESTROY(string, box);

        toggle = XtVaCreateManagedWidget("Toggle",
		xmToggleButtonGadgetClass,	box,
		XmNlabelString,			string,
		NULL);

	XmStringFree(string);

	CHECK_WIDGET_DESTROY(toggle, box);

	if (i == item_set)
	    XmToggleButtonGadgetSetState(toggle, True, False);

        if (radiobox_items[i].callback)
	{
	    XtAddCallback(toggle, XmNvalueChangedCallback,
		radiobox_items[i].callback, (XtPointer) radiobox_items[i].data);
	}

	*(radiobox_items[i].box) = toggle;
    }

    XtVaSetValues(box, XmNorientation, (unsigned char) orientation, NULL);

    XtManageChild(box);

    return  box;
}

Widget create_checkbox(Widget parent, Checkbox_item *checkbox_items,
				int nboxes, int items_set, int orientation)
{
    int i;
    Widget box, toggle;
    XmString string;

    box = XmCreateSimpleCheckBox(parent, "Checkbox", NULL, 0);
    CHECK_WIDGET(box);

    for (i = 0; i < nboxes; i++)
    {
	string = XmStringCreateSimple(checkbox_items[i].label);
	CHECK_WIDGET_DESTROY(string, box);

        toggle = XtVaCreateManagedWidget("Toggle",
		xmToggleButtonGadgetClass,	box,
		XmNlabelString,			string,
		NULL);

	XmStringFree(string);

	CHECK_WIDGET_DESTROY(toggle, box);

	if (items_set % 2)
	    XmToggleButtonGadgetSetState(toggle, True, False);

	items_set >>= 1;

        if (checkbox_items[i].callback)
	{
	    XtAddCallback(toggle, XmNvalueChangedCallback,
		checkbox_items[i].callback, (XtPointer) checkbox_items[i].data);
	}

	*(checkbox_items[i].box) = toggle;
    }

    XtVaSetValues(box, XmNorientation, (unsigned char) orientation, NULL);

    XtManageChild(box);

    return  box;
}

Widget create_label_radiobox(Widget parent, String string,
		Radiobox_item *radiobox_items, int nboxes, int item_set)
{
    Widget form, label, radio;

    form = create_form(parent);
    CHECK_WIDGET(form);

    label = create_label(form, string);
    CHECK_WIDGET_DESTROY(label, form);
    attachments(label, FORM_ATTACH, FORM_ATTACH, FORM_ATTACH, NO_ATTACH);

    radio = create_radiobox(form, radiobox_items, nboxes, item_set,
								HORIZONTAL);
    CHECK_WIDGET_DESTROY(radio, form);
    attachments(radio, FORM_ATTACH, FORM_ATTACH, label, FORM_ATTACH);

    XtManageChild(form);

    return  form;
}

Widget create_label_checkbox(Widget parent, String string,
		Checkbox_item *checkbox_items, int nboxes, int item_set)
{
    Widget form, label, check;

    form = create_form(parent);
    CHECK_WIDGET(form);

    label = create_label(form, string);
    CHECK_WIDGET_DESTROY(label, form);
    attachments(label, FORM_ATTACH, FORM_ATTACH, FORM_ATTACH, NO_ATTACH);

    check = create_checkbox(form, checkbox_items, nboxes, item_set,
								HORIZONTAL);
    CHECK_WIDGET_DESTROY(check, form);
    attachments(check, FORM_ATTACH, FORM_ATTACH, label, FORM_ATTACH);

    XtManageChild(form);

    return  form;
}

Widget create_alternating_pair(Widget parent, Alternating_info *info)
{
    int i, top, bottom, h0, h1, base = 200;
    float r, s;
    Widget rowcol, form, label, w0, w1, separator;

    rowcol = create_rowcol(parent);
    CHECK_WIDGET(rowcol);

    form = create_form(rowcol);
    CHECK_WIDGET_DESTROY(form, rowcol);

    XtVaSetValues(form,
		XmNfractionBase,	2,
		XmNwidth,		info->width,
		NULL);

    label = create_label(form, info->label[0]);
    CHECK_WIDGET_DESTROY(label, rowcol);
    positions(label, NO_POSITION, 2, 0, 1);

    XtVaSetValues(label,
		XmNalignment,		XmALIGNMENT_BEGINNING,
		NULL);

    label = create_label(form, info->label[1]);
    CHECK_WIDGET_DESTROY(label, rowcol);
    positions(label, NO_POSITION, 2, 1, 2);

    XtVaSetValues(label,
		XmNalignment,		XmALIGNMENT_BEGINNING,
		NULL);

    XtManageChild(form);

    separator = create_separator(rowcol);
    CHECK_WIDGET_DESTROY(separator, rowcol);

    form = create_form(rowcol);
    CHECK_WIDGET_DESTROY(form, rowcol);

    XtVaSetValues(form,
		XmNfractionBase,	base,
		XmNwidth,		info->width,
		NULL);

    r = ((float) base) / info->n;

    w0 = (*(info->func[0]))(form, 0);
    CHECK_WIDGET_DESTROY(w0, rowcol);
    top = 0;
    bottom = NEAREST_INTEGER(r) - 1;
    positions(w0, top, bottom, 0, base/2);

    for (i = 1; i < info->n; i++)
    {
	w0 = (*(info->func[0]))(form, i);
	CHECK_WIDGET_DESTROY(w0, rowcol);

	top = NEAREST_INTEGER(i * r);
	bottom = NEAREST_INTEGER((i+1) * r) - 1;
	positions(w0, top, bottom, 0, base/2);
    }

    XtVaGetValues(w0,
		XmNheight,	&h0,
		NULL);

    for (i = 1; i < info->n; i++)
    {
	w1 = (*(info->func[1]))(form, i);
	CHECK_WIDGET_DESTROY(w1, rowcol);

	XtVaGetValues(w1,
		XmNheight,	&h1,
		NULL);

	s = (0.5 * h1) / h0;
	s = MIN(s, 0.5);

	top = NEAREST_INTEGER((i-s) * r);
	bottom = NEAREST_INTEGER((i+s) * r) - 1;
	positions(w1, top, bottom, base/2, base);
    }

    XtManageChild(form);
    XtManageChild(rowcol);

    return  rowcol;
}

static void next_tab_callback(Widget widget, XtPointer client_data,
							XtPointer cbs)
{
    (void) XmProcessTraversal(widget, XmTRAVERSE_NEXT_TAB_GROUP);
}

Widget create_textfields(Widget parent, Textfield_item *textfield_items,
								int ntexts)
{
    int i;
    Widget rowcol, form, label, text;
    XmString string;

    rowcol = XtVaCreateWidget("Textfields",
		xmRowColumnWidgetClass,	parent,
		NULL);

    CHECK_WIDGET(rowcol);

    for (i = 0; i < ntexts; i++)
    {
	form = XtVaCreateWidget("Form",
		xmFormWidgetClass,	rowcol,
		XmNfractionBase,	2,
		NULL);

	CHECK_WIDGET_DESTROY(form, rowcol);

	string = XmStringCreateSimple(textfield_items[i].label);
	CHECK_WIDGET_DESTROY(string, rowcol);

        label = XtVaCreateManagedWidget("Label",
		xmLabelWidgetClass,	form,
		XmNlabelString,		string,
		XmNtopAttachment,	XmATTACH_FORM,
		XmNbottomAttachment,	XmATTACH_FORM,
		XmNleftAttachment,	XmATTACH_FORM,
		XmNrightAttachment,	XmATTACH_POSITION,
		XmNrightPosition,	1,
		XmNalignment,		XmALIGNMENT_END,
		NULL);

	XmStringFree(string);

	CHECK_WIDGET_DESTROY(label, rowcol);

	text = XtVaCreateManagedWidget("Textfield",
		xmTextFieldWidgetClass,	form,
		XmNvalue,		textfield_items[i].value,
		XmNtraversalOn,		True,
		XmNtopAttachment,	XmATTACH_FORM,
		XmNbottomAttachment,	XmATTACH_FORM,
		XmNleftAttachment,	XmATTACH_POSITION,
		XmNleftPosition,	1,
		XmNrightAttachment,	XmATTACH_FORM,
		NULL);

	CHECK_WIDGET_DESTROY(text, rowcol);

	if (textfield_items[i].callback)
	{
/*
	    XtAddCallback(text, XmNactivateCallback,
*/
	    XtAddCallback(text, XmNvalueChangedCallback,
				textfield_items[i].callback,
				(XtPointer) textfield_items[i].data);
	}

	XtAddCallback(text, XmNactivateCallback, next_tab_callback, NULL);

	*(textfield_items[i].field) = text;

	XtManageChild(form);
    }

    XtManageChild(rowcol);

    return  rowcol;
}

Widget create_scrolled_text(Widget parent, Scrolled_text_info *text_info)
{
    Widget scrolled_text;

    scrolled_text = XmCreateScrolledText(parent, "Scrolled_text", NULL, 0);
    CHECK_WIDGET(scrolled_text);

    XtVaSetValues(scrolled_text,
		XmNrows,		text_info->nrows,
		XmNcolumns,		text_info->ncolumns,
		XmNeditable,		(Boolean) text_info->editable,
		XmNeditMode,		text_info->edit_mode,
                NULL);

    XtManageChild(scrolled_text);

    return  scrolled_text;
}

static XmStringTable create_items(int nitems, String_item_func func,
							Callback_ptr data)
{
    int i;
    XmStringTable items = (XmStringTable) NULL;

    if (nitems > 0)
    {
	items = (XmStringTable) XtMalloc(nitems * sizeof(XmString *));

	if (items)
	{
	    for (i = 0; i < nitems; i++)
		items[i] = XmStringCreateSimple((*func)(i, data));
	}
    }

    return  items;
}

static void destroy_items(int nitems, XmStringTable items)
{
    int i;

    if (items)
    {
	for (i = 0; i < nitems; i++)
	    XmStringFree(items[i]);

	XtFree((XtPointer) items);
    }
}

Widget create_scrolled_list(Widget parent, Scrolled_list_info *list_info)
{
    int n;
    Widget scrolled_list;
    XmStringTable items;

    items = create_items(list_info->nitems, list_info->func,
						list_info->func_data);
    n = items  ?  list_info->nitems  :  0;

    scrolled_list = XmCreateScrolledList(parent, "Scrolled_list", NULL, 0);

    if (scrolled_list)
    {
	XtVaSetValues(scrolled_list,
		XmNselectionPolicy,	list_info->selection_policy,
		XmNitems,		items,
		XmNitemCount,		n,
		XmNvisibleItemCount,	list_info->nvisible_items,
                NULL);
    }

    destroy_items(n, items);

    CHECK_WIDGET(scrolled_list);

    if (list_info->callback)
    {
	switch(list_info->selection_policy)
	{
	    case XmSINGLE_SELECT:
		XtAddCallback(scrolled_list, XmNsingleSelectionCallback,
		    list_info->callback, (XtPointer) list_info->callback_data);
		break;

	    case XmBROWSE_SELECT:
		XtAddCallback(scrolled_list, XmNbrowseSelectionCallback,
		    list_info->callback, (XtPointer) list_info->callback_data);
		break;

	    case XmMULTIPLE_SELECT:
		XtAddCallback(scrolled_list, XmNmultipleSelectionCallback,
		    list_info->callback, (XtPointer) list_info->callback_data);
		break;

	    case XmEXTENDED_SELECT:
		XtAddCallback(scrolled_list, XmNextendedSelectionCallback,
		    list_info->callback, (XtPointer) list_info->callback_data);
		break;
	}
    }

    XtManageChild(scrolled_list);

    return  scrolled_list;
}

static void dbl_scroll_action(Widget widget, XtPointer data, XtPointer cbs)
{
    int top;
    Widget *widgets = (Widget *) data;

    XtVaGetValues(widgets[DBL_SINGLE_LIST],
	XmNtopItemPosition,	&top,
	NULL);

    XmListSetPos(widgets[DBL_MULTI_LIST], top);
}

Widget create_scrolled_dbl_list(Widget parent,
					Scrolled_dbl_list_info *list_info)
{
    Widget form, single_list, multi_list, scrolled_window, sb;
    Widget *widgets = (Widget *) (list_info->widgets);
    XtPointer data_ptr;
    int n;
    XmStringTable items;

    form = create_form(parent);
    CHECK_WIDGET(form);

    items = create_items(list_info->nitems,
				list_info->funcs[DBL_MULTI_LIST],
				list_info->func_data[DBL_MULTI_LIST]);
    n = items  ?  list_info->nitems  :  0;

    multi_list = XtVaCreateManagedWidget("List",
		xmListWidgetClass,	form,
		XmNitems,		items,
		XmNitemCount,		n,
		XmNvisibleItemCount,	list_info->nvisible_items,
		XmNselectionPolicy,	XmMULTIPLE_SELECT,
		NULL);

    destroy_items(n, items);

    CHECK_WIDGET_DESTROY(multi_list, form);
    attachments(multi_list, FORM_ATTACH, FORM_ATTACH, FORM_ATTACH, NO_ATTACH);

    scrolled_window = XtVaCreateManagedWidget("Scrolled_window",
		xmScrolledWindowWidgetClass,	form,
		XmNscrollingPolicy,		XmAPPLICATION_DEFINED,
		NULL);

    CHECK_WIDGET_DESTROY(scrolled_window, form);
    attachments(scrolled_window, FORM_ATTACH, FORM_ATTACH, multi_list,
								FORM_ATTACH);

    items = create_items(list_info->nitems,
				list_info->funcs[DBL_SINGLE_LIST],
				list_info->func_data[DBL_SINGLE_LIST]);
    n = items  ?  list_info->nitems  :  0;

    single_list = XtVaCreateManagedWidget("Single_list",
		xmListWidgetClass,	scrolled_window,
		XmNitems,		items,
		XmNitemCount,		n,
		XmNvisibleItemCount,	list_info->nvisible_items,
		XmNselectionPolicy,	XmSINGLE_SELECT,
		XmNlistSizePolicy,	XmRESIZE_IF_POSSIBLE,
		NULL);

    destroy_items(n, items);

    CHECK_WIDGET_DESTROY(single_list, form);

    widgets[DBL_MULTI_LIST] = multi_list;
    widgets[DBL_SINGLE_LIST] = single_list;
    data_ptr = (XtPointer) widgets;

    XtVaGetValues(XtParent(single_list), XmNverticalScrollBar, &sb, NULL);

    XtAddCallback(sb, XmNvalueChangedCallback, dbl_scroll_action, data_ptr);
    XtAddCallback(sb, XmNdragCallback, dbl_scroll_action, data_ptr);
    XtAddCallback(sb, XmNincrementCallback, dbl_scroll_action, data_ptr);
    XtAddCallback(sb, XmNdecrementCallback, dbl_scroll_action, data_ptr);
    XtAddCallback(sb, XmNpageIncrementCallback, dbl_scroll_action, data_ptr);
    XtAddCallback(sb, XmNpageDecrementCallback, dbl_scroll_action, data_ptr);
    XtAddCallback(sb, XmNtoTopCallback, dbl_scroll_action, data_ptr);
    XtAddCallback(sb, XmNtoBottomCallback, dbl_scroll_action, data_ptr);

    if (list_info->callbacks[DBL_MULTI_LIST])
	XtAddCallback(multi_list, XmNmultipleSelectionCallback,
		list_info->callbacks[DBL_MULTI_LIST],
		(XtPointer) (list_info->callback_data[DBL_MULTI_LIST]));

    if (list_info->callbacks[DBL_SINGLE_LIST])
	XtAddCallback(single_list, XmNsingleSelectionCallback,
		list_info->callbacks[DBL_SINGLE_LIST],
		(XtPointer) (list_info->callback_data[DBL_SINGLE_LIST]));

    XtManageChild(form);

    return  form;
}

static void dropdown_name_callback(Widget widget, Callback_ptr data,
							Callback_ptr cbs)
{
    Callback_proc callback = (Callback_proc) data;
    String name = XtName(widget);

    (*callback)(widget, (Callback_ptr) name, cbs);
}

static void dropdown_number_callback(Widget widget, Callback_ptr data,
							Callback_ptr cbs)
{
    int i, n;
    Callback_proc callback = (Callback_proc) data;
    Widget parent = XtParent(widget);
    WidgetList children;

    XtVaGetValues(parent, XmNnumChildren, &n, NULL);
    XtVaGetValues(parent, XmNchildren, &children, NULL);

    /* not very efficient but not sure how to do otherwise */
    for (i = 0; i < n; i++)
    {
        if (children[i] == widget)
	    break;
    }

    if (i < n) /* protection, should never happen */
	(*callback)(widget, (Callback_ptr) &i, cbs);
}

Widget create_dropdown_list(Widget parent, Dropdown_list_info *list_info)
{
    int i, nargs = 0;
    Arg args[2];
    XmString label;
    Widget dropdown_list, button, menu;

    menu = XmCreatePulldownMenu(parent, "Dropdown_menu", NULL, 0);
    CHECK_WIDGET(menu);

    XtSetArg(args[nargs], XmNsubMenuId, menu); /* cannot do nargs++ */
    nargs++;

    if (list_info->label)
    {
	label = XmStringCreateSimple(list_info->label);
	XtSetArg(args[nargs], XmNlabelString, label);
	nargs++;
    }

    dropdown_list = XmCreateOptionMenu(parent, "Dropdown_list", args, nargs);
    CHECK_WIDGET_DESTROY(dropdown_list, menu);

    for (i = 0; i < list_info->nitems; i++)
    {
	button = XtVaCreateManagedWidget(list_info->items[i],
				xmPushButtonGadgetClass, menu, NULL);
	CHECK_WIDGET_DESTROY(button, dropdown_list);

	if (list_info->callback_names)
	    XtAddCallback(button, XmNactivateCallback,
		dropdown_name_callback, (Callback_ptr) list_info->callback);
	else
	    XtAddCallback(button, XmNactivateCallback,
		dropdown_number_callback, (Callback_ptr) list_info->callback);
    }

    XtManageChild(dropdown_list);

    if (list_info->label)
	XmStringFree(label);

    return  dropdown_list;
}

Widget create_scale(Widget parent, Scale_info *scale_info)
{
    Widget scale;

    scale = XtVaCreateWidget("Scale",
		xmScaleWidgetClass,	parent,
		XmNshowValue,		(Boolean) scale_info->show_value,
		XmNorientation,	(unsigned char) scale_info->orientation,
		NULL);

    CHECK_WIDGET(scale);
    set_scale_numbers(scale, scale_info->numbers);

    if (scale_info->title)
    {
	XtVaSetValues(scale,
		XtVaTypedArg,	XmNtitleString,		XmRString,
			scale_info->title,	strlen(scale_info->title),
		NULL);
/*  seems to be a bug with below (title does not appear in text_scale)
	string = XmStringCreateSimple(scale_info->title);

	CHECK_WIDGET_DESTROY(string, scale);

	XtVaSetValues(scale,
		XmNtitle,		string,
		NULL);

	XmStringFree(string);
*/
    }

    if (scale_info->value_changed)
    {
	XtAddCallback(scale, XmNvalueChangedCallback,
				scale_info->value_changed,
				(XtPointer) scale_info->data);
    }

    if (scale_info->dragged)
    {
	XtAddCallback(scale, XmNdragCallback,
				scale_info->dragged,
				(XtPointer) scale_info->data);
    }

    XtManageChild(scale);

    return  scale;
}

Widget create_text_scale(Widget parent, Text_scale_info *text_scale_info)
{
    Widget form, scale, min_text, max_text;
    Textfield_item *min, *max;

    form = create_form(parent);
    CHECK_WIDGET(form);

    min = text_scale_info->limits + 0;
    min_text = XtVaCreateManagedWidget("Text_scale",
		xmTextFieldWidgetClass,	form,
		XmNvalue,		min->value,
		XmNcolumns,		10,
		NULL);

    CHECK_WIDGET_DESTROY(min_text, form);
    attachments(min_text, FORM_ATTACH, NO_ATTACH, FORM_ATTACH, NO_ATTACH);
    *(min->field) = min_text;

    max = text_scale_info->limits + 1;
    max_text = XtVaCreateManagedWidget("Text_scale",
		xmTextFieldWidgetClass,	form,
		XmNvalue,		max->value,
		XmNcolumns,		10,
		NULL);

    CHECK_WIDGET_DESTROY(max_text, form);
    attachments(max_text, FORM_ATTACH, NO_ATTACH, NO_ATTACH, FORM_ATTACH);
    *(max->field) = max_text;

    scale = create_scale(form, text_scale_info->scale_info);
    CHECK_WIDGET_DESTROY(scale, form);
    attachments(scale, FORM_ATTACH, FORM_ATTACH, min_text, max_text);

    *(text_scale_info->scale) = scale;

    XtManageChild(form);

    return  form;
}

Widget create_text_buttons(Widget parent, Text_buttons_info *text_buttons_info)
{
    Widget form, text, button, label;
    XmString string;
    Textfield_item *textfield_item;

    form = create_form(parent);
    CHECK_WIDGET(form);

    textfield_item = text_buttons_info->text;

    string = XmStringCreateSimple(textfield_item->label);
    CHECK_WIDGET_DESTROY(string, form);

    label = XtVaCreateManagedWidget("Label",
		xmLabelWidgetClass,	form,
		XmNlabelString,		string,
		NULL);

    XmStringFree(string);

    CHECK_WIDGET_DESTROY(label, form);

    text = XtVaCreateManagedWidget("Textfield",
		xmTextFieldWidgetClass,	form,
		XmNvalue,		textfield_item->value,
		XmNwidth,		100,
		NULL);

    CHECK_WIDGET_DESTROY(text, form);

    if (textfield_item->callback)
    {
	XtAddCallback(text, XmNactivateCallback,
				textfield_item->callback,
				(XtPointer) textfield_item->data);
    }

    XtAddCallback(text, XmNactivateCallback, next_tab_callback, NULL);

    *(textfield_item->field) = text;

    button = create_horizontal_buttons(form, text_buttons_info->buttons,
						text_buttons_info->nbuttons);

    CHECK_WIDGET_DESTROY(button, form);

    attachments(label, FORM_ATTACH, FORM_ATTACH, NO_ATTACH, NO_ATTACH);
    attachments(text, FORM_ATTACH, FORM_ATTACH, label, NO_ATTACH);
    attachments(button, FORM_ATTACH, FORM_ATTACH, text, FORM_ATTACH);

    XtManageChild(form);

    return  form;
}

static YesNo yes_no;

static void force_callback(Widget widget, XtPointer data, XtPointer cbs)
{
    XmPushButtonCallbackStruct *pbcbs = (XmPushButtonCallbackStruct *) cbs;

    if (pbcbs->reason == XmCR_OK)
    	yes_no = YES;
    else if (pbcbs->reason == XmCR_CANCEL)
    	yes_no = NO;
    else if (pbcbs->reason == XmCR_HELP)
    	yes_no = CANCEL;
}

static void make_beep(int percent)
{
    XBell(XtDisplay(topshell), percent);
    XFlush(XtDisplay(topshell));
}

static YesNo screen_question(String msg, String yes, String no, String cancel)
{
    char ch;
    YesNo yes_no = NEITHER;

    make_beep(100);
    printf("Panic: could not create dialogue box to ask question!\n");
    printf("%s\n", msg);

    while (yes_no == NEITHER)
    {
	if (yes)
	    printf("\t%s: type '%c'\n", yes, yes[0]);

	if (no)
	    printf("\t%s: type '%c'\n", no, no[0]);

	if (cancel)
	    printf("\t%s: type '%c'\n", cancel, cancel[0]);

	printf("Choice: ");
	scanf("%c", &ch);

	if (yes && (ch == yes[0]))
	    yes_no = YES;

	if (no && (ch == no[0]))
	    yes_no = NO;

	if (cancel && (ch == cancel[0]))
	    yes_no = CANCEL;
    }

    return  yes_no;
}

YesNo force_question(String msg, String yes, String no, String cancel)
{
    static Widget dialog = NULL;
    XmString string;

    dialog = XmCreateQuestionDialog(topshell, "dialog", NULL, 0);

    if (!dialog)
	return  screen_question(msg, yes, no, cancel);

    string = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);

    if (!string)
    {
	destroy_widget(XtParent(dialog));
	return  screen_question(msg, yes, no, cancel);
    }

    XtVaSetValues(dialog,
		XmNmessageString,	string,
		XmNdialogStyle,		XmDIALOG_FULL_APPLICATION_MODAL,
		NULL);

    XmStringFree(string);

    if (yes)
    {
    	string = XmStringCreateSimple(yes);

	if (!string)
	{
	    destroy_widget(XtParent(dialog));
	    return  screen_question(msg, yes, no, cancel);
	}

    	XtVaSetValues(dialog,
		XmNokLabelString,	string,
		NULL);

    	XtAddCallback(dialog, XmNokCallback, force_callback, NULL);

    	XmStringFree(string);
    }
    else
    {
    	XtUnmanageChild(XmMessageBoxGetChild(dialog, XmDIALOG_OK_BUTTON));
    }

    if (no)
    {
    	string = XmStringCreateSimple(no);

	if (!string)
	{
	    destroy_widget(XtParent(dialog));
	    return  screen_question(msg, yes, no, cancel);
	}

    	XtVaSetValues(dialog,
		XmNcancelLabelString,	string,
		NULL);

    	XtAddCallback(dialog, XmNcancelCallback, force_callback, NULL);

    	XmStringFree(string);
    }
    else
    {
    	XtUnmanageChild(XmMessageBoxGetChild(dialog, XmDIALOG_CANCEL_BUTTON));
    }

    if (cancel)
    {
    	string = XmStringCreateSimple(cancel);

	if (!string)
	{
	    destroy_widget(XtParent(dialog));
	    return  screen_question(msg, yes, no, cancel);
	}

    	XtVaSetValues(dialog,
		XmNhelpLabelString,	string,
		XmNautoUnmanage,	True,
		NULL);

    	XtAddCallback(dialog, XmNhelpCallback, force_callback, NULL);

    	XmStringFree(string);
    }
    else
    {
    	XtUnmanageChild(XmMessageBoxGetChild(dialog, XmDIALOG_HELP_BUTTON));
    }

    yes_no = NEITHER;

    popup(dialog);

    while (yes_no == NEITHER)
	check_for_event();

    XtDestroyWidget(XtParent(dialog));

    return yes_no;
}

static void file_ok_callback(Widget widget, XtPointer data, XtPointer cbs)
{
    char *file, *dir, *fullfile = (char *) data;
    XmFileSelectionBoxCallbackStruct *fcbs = 
		(XmFileSelectionBoxCallbackStruct *) cbs;

    if (!XmStringGetLtoR(fcbs->value, XmSTRING_DEFAULT_CHARSET, &file))
	return;

    if (*file == '/')
    {
	strncpy(fullfile, file, LINE_SIZE+1);
    }
    else
    {
	/* if it's not a directory, determine the full pathname
	   of the selection by concatenating it to the "dir" part */

	if (XmStringGetLtoR(fcbs->dir, XmSTRING_DEFAULT_CHARSET, &dir))
	{
	    strncpy(fullfile, dir, LINE_SIZE+1);

            if (strlen(dir) < LINE_SIZE)
	    {
		strcat(fullfile, "/");
		strncat(fullfile, file, LINE_SIZE-strlen(dir));
	    }

	    XtFree(dir);
	}
	else
	{
	    strncpy(fullfile, file, LINE_SIZE+1);
	}
    }

    XtFree(file);

    yes_no = YES;
}

static void file_cancel_callback(Widget widget, XtPointer data, XtPointer cbs)
{
    yes_no = CANCEL;
}

YesNo select_file(String file)
{
    static Widget dialog = NULL;

    dialog = XmCreateFileSelectionDialog(topshell, "dialog", NULL, 0);

    if (!dialog)
	return  NO;

    XtVaSetValues(dialog,
		XmNdialogStyle,		XmDIALOG_FULL_APPLICATION_MODAL,
		NULL);

    XtAddCallback(dialog, XmNokCallback, file_ok_callback, (XtPointer) file);
    XtAddCallback(dialog, XmNcancelCallback, file_cancel_callback, NULL);

    yes_no = NEITHER;

    popup(dialog);

    while (yes_no == NEITHER)
	check_for_event();

    XtDestroyWidget(XtParent(dialog));

    return yes_no;
}

void check_for_event()
{
    XtAppProcessEvent(XtWidgetToApplicationContext(topshell), XtIMAll);
}

static void screen_error_message(String message)
{
    make_beep(100);
    printf("Panic: could not create dialogue box to print error message!\n");
    printf("%s\n", message);
}

void print_error_message(String message)
{
    static Widget dialog = NULL;
    XmString string1;
    XmString string2;

    if (!dialog)
    {
    	dialog = XmCreateErrorDialog(topshell, "dialog", NULL, 0);

    	XtAddCallback(dialog, XmNokCallback, force_callback, NULL);
    	XtUnmanageChild(XmMessageBoxGetChild(dialog, XmDIALOG_CANCEL_BUTTON));
    	XtUnmanageChild(XmMessageBoxGetChild(dialog, XmDIALOG_HELP_BUTTON));

/*
    	XtManageChild(dialog);
*/
    }

    if (!dialog)
    {
	screen_error_message(message);
	return;
    }

    string1 = XmStringCreateSimple(message);

    if (!string1)
    {
	screen_error_message(message);
	return;
    }

    string2 = XmStringCreateSimple("ok");

    if (!string2)
    {
	XmStringFree(string1);
	screen_error_message(message);
	return;
    }

    XtVaSetValues(dialog,
	XmNmessageString,	string1,
	XmNokLabelString,	string2,
	XmNdialogStyle,		XmDIALOG_FULL_APPLICATION_MODAL,
	NULL);

    XmStringFree(string1);
    XmStringFree(string2);

    yes_no = NEITHER;
/*
    popup(XtParent(dialog));
*/
    popup(dialog);

    while (yes_no == NEITHER)
    {
	XtAppProcessEvent(XtWidgetToApplicationContext(topshell), XtIMAll);
    }
}

void serious_warning(String message)
{
    Line error_msg;

    sprintf(error_msg, "%s:  please exit program", message);
    print_error_message(error_msg);
}

void force_update(Widget shell)
{
    Window window, topwindow;
    Display *display;
    XtAppContext context;
    XWindowAttributes attr;
    XEvent event;

    if (XtIsRealized(shell) && XtIsRealized(topshell))
    {
    	window = XtWindow(shell);
    	topwindow = XtWindow(topshell);
    	display = XtDisplay(topshell);
    	context = XtWidgetToApplicationContext(shell);

    	while(XGetWindowAttributes(display, window, &attr) &&
					(attr.map_state != IsViewable))
    	{
    	    XGetWindowAttributes(display, topwindow, &attr);
	    if (attr.map_state != IsViewable)  break;

	    XtAppNextEvent(context, &event);
	    XtDispatchEvent(&event);
    	}
    }

    if (XtIsRealized(topshell))
	XmUpdateDisplay(topshell);
}

void register_shell(Widget shell)
{
    if (shell_list)
	(void) insert_list(&shell_list, shell);
}

static Bool same_shell(Generic_ptr s1, Generic_ptr s2)
{
    Widget shell1 = (Widget) s1;
    Widget shell2 = (Widget) s2;

    return  (shell1 == shell2)  ?  TRUE  :  FALSE;
}

void unregister_shell(Widget shell)
{
    List list;

    if (shell_list && (find_key(shell_list, shell, same_shell, &list) == OK))
    	(void) delete_node(&shell_list, list);
}

static Status change_shell_cursor(Generic_ptr s)
{
    Widget shell = (Widget) s;

    XDefineCursor(display, XtWindow(shell), cursor);

    return  OK;
}

void change_cursor(Cursor cur)
{
    cursor = cur;

    if (shell_list)
	(void) traverse_list(shell_list, change_shell_cursor);
}

Cursor create_busy_cursor(Display *display)
{
    return  XCreateFontCursor(display, XC_watch);
}

void flush_display(Display *display)
{
    XFlush(display);
}
