#include "track_popup.h"

#include "tracking.hlp"

#include "callback.h"
#include "color.h"
#include "data.h"
#include "display.h"
#include "event_handle.h"
#include "info_popup.h"
#include "param.h"
#include "ref.h"
#include "region_func.h"
#include "region_popup.h"
#include "window.h"

#define  MIN_WIDTH	200
#define  MIN_HEIGHT	200

typedef struct
{
    int x0, y0, x1, y1;
}   Tracking_rectangle;

typedef struct
{
    float x0, y0, x1, y1;
}   Tracking_limits;

static Widget tracking_popup = (Widget) NULL;
static Widget tracking_area;
static Widget tracking_form;

static Display *tracking_display = (Display *) NULL;
static Window tracking_window;
static GC tracking_gc = (GC) NULL;

static int width = MIN_WIDTH;
static int height = MIN_HEIGHT;

static int ref_type;

static Bool have_valid_tracking = FALSE;
static Bool have_new_rectangle = FALSE;

static Tracking_rectangle current_rectangle;
static Tracking_rectangle new_rectangle;

static Tracking_limits limits;

static int cursor_x;
static int cursor_y;

void update_tracking_params()
{
    if (tracking_popup)
    {
	get_widget_size(tracking_area, &width, &height);
	sprintf(tracking_width, "%d", width);
	sprintf(tracking_height, "%d", height);
    }
    else
    {
	if (*tracking_width)
	    width = atoi(tracking_width);

	width = MAX(width, MIN_WIDTH);

	if (*tracking_height)
	    height = atoi(tracking_height);

	height = MAX(height, MIN_HEIGHT);
    }
}

static void clear_tracking()
{
    set_gc_color(tracking_display, tracking_gc, WHITE);
    clear_region(tracking_display, tracking_window, tracking_gc,
							0, 0, width, height);
}

static void do_xor_rectangle(Tracking_rectangle *rect)
{
    set_gc_function(tracking_display, tracking_gc, GXxor);

    set_gc_color(tracking_display, tracking_gc, GREEN);
    display_rectangle(tracking_display, tracking_window, tracking_gc, 
				rect->x0, rect->y0, rect->x1, rect->y1);

    set_gc_function(tracking_display, tracking_gc, GXcopy);
}

static void do_rectangle(int color, Tracking_rectangle *rect)
{
    set_gc_color(tracking_display, tracking_gc, color);

    display_rectangle(tracking_display, tracking_window, tracking_gc, 
				rect->x0, rect->y0, rect->x1, rect->y1);
}

static void do_rectangles()
{
    do_rectangle(BLACK, &current_rectangle);

    if (have_new_rectangle)
	do_rectangle(RED, &new_rectangle);
}

static void transform_to_tracking(Tracking_rectangle *rect,
						float *lower, float *upper)
{
    float c, d;

    c = (width - 1) / (limits.x1 - limits.x0);
    d = - c * limits.x0;

    rect->x0 = c*lower[0] + d;
    rect->x1 = c*upper[0] + d;

    c = (1 - height) / (limits.y1 - limits.y0);
    d = - c * limits.y1;

    rect->y0 = c*lower[1] + d;
    rect->y1 = c*upper[1] + d;
}

static void transform_from_tracking(Tracking_rectangle *rect,
						float *lower, float *upper)
{
    float c, d;

    if (rect->x0 > rect->x1)
	SWAP(rect->x0, rect->x1, float);

    if (rect->y0 < rect->y1)
	SWAP(rect->y0, rect->y1, float);

    c = (limits.x1 - limits.x0) / (width - 1);
    d = limits.x0;

    lower[0] = c*rect->x0 + d;
    upper[0] = c*rect->x1 + d;

    c = (limits.y1 - limits.y0) / (1 - height);
    d = limits.y1;

    lower[1] = c*rect->y0 + d;
    upper[1] = c*rect->y1 + d;
}

static void change_tracking_rectangle()
{
    float lower[DISPLAY_DIM], upper[DISPLAY_DIM];

    if (!have_valid_tracking || !have_new_rectangle)
	return;

    have_new_rectangle = FALSE;

    transform_from_tracking(&new_rectangle, lower, upper);

    check_orientation(ref_type, DISPLAY_DIM, lower, upper);
    set_region_func(ref_type, lower, upper);
    set_region(ref_type);
}

static Bool inside_rectangle(int x, int y)
{
    float lower[DISPLAY_DIM], upper[DISPLAY_DIM];

    if (!have_valid_tracking)
	return  FALSE;

    lower[0] = current_rectangle.x0;  upper[0] = current_rectangle.x1;
    lower[1] = current_rectangle.y0;  upper[1] = current_rectangle.y1;
    check_orientation(ref_type, DISPLAY_DIM, lower, upper);

    if ((x < lower[0]) || (x > upper[0]))
	return  FALSE;

    if ((y > lower[1]) || (y < upper[1]))
	return  FALSE;
/*
    if ((x < current_rectangle.x0) || (x > current_rectangle.x1))
	return  FALSE;

    if ((y > current_rectangle.y0) || (y < current_rectangle.y1))
	return  FALSE;
*/

    return  TRUE;
}

static void translate_rectangle(int x, int y)
{
    new_rectangle.x0 = current_rectangle.x0 + x - cursor_x;
    new_rectangle.y0 = current_rectangle.y0 + y - cursor_y;
    new_rectangle.x1 = current_rectangle.x1 + x - cursor_x;
    new_rectangle.y1 = current_rectangle.y1 + y - cursor_y;
}

static void apply_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    change_tracking_rectangle();
}

static void tracking_expose_callback(Widget widget, Callback_ptr data,
							Callback_ptr cbs)
{
    if (have_valid_tracking)
	do_tracking_drawing();
    else
	clear_tracking();
}

static void tracking_resize_callback(Widget widget, Callback_ptr data,
							Callback_ptr cbs)
{
    update_tracking_params();
}

static void button_press(Button button, int x, int y, Time time)
{
    if (button == Button1)
    {
	if (have_new_rectangle)
	{
	    change_tracking_rectangle();
	}
	else if (inside_rectangle(x, y))
	{
	    have_new_rectangle = TRUE;

	    cursor_x = x;
	    cursor_y = y;

	    translate_rectangle(cursor_x, cursor_y);
	    do_xor_rectangle(&new_rectangle);
	}
    }
    else if (button == Button2)
    {
	if (have_new_rectangle)
	{
	    have_new_rectangle = FALSE;

	    if (have_valid_tracking)
		do_tracking_drawing();
	    else
		clear_tracking();
	}

	if (have_valid_tracking)
	{
	    have_new_rectangle = TRUE;

	    new_rectangle.x0 = x;
	    new_rectangle.y0 = y;
	    new_rectangle.x1 = x;
	    new_rectangle.y1 = y;

	    do_xor_rectangle(&new_rectangle);
	}
    }
}

static void button_release(Button button, int x, int y, Time time)
{
    if (button == Button1)
    {
	if (have_new_rectangle)
	{
	    do_xor_rectangle(&new_rectangle);
	    translate_rectangle(x, y);

	    if ((cursor_x != x) || (cursor_y != y))
		do_rectangle(RED, &new_rectangle);
	    else
		have_new_rectangle = FALSE;
	}
    }
    else if (button == Button2)
    {
	if (have_new_rectangle)
	{
	    do_xor_rectangle(&new_rectangle);
	    new_rectangle.x1 = x;
	    new_rectangle.y1 = y;

	    if ((new_rectangle.x0 == new_rectangle.x1) ||
				(new_rectangle.y0 == new_rectangle.y1))
		have_new_rectangle = FALSE;
	    else
		do_rectangle(RED, &new_rectangle);
	}
    }
}

static void button_motion(Button button, int x, int y, Time time)
{
    if (button == Button1)
    {
	if (have_new_rectangle)
	{
	    do_xor_rectangle(&new_rectangle);
	    translate_rectangle(x, y);
	    do_xor_rectangle(&new_rectangle);
	}
    }
    else if (button == Button2)
    {
	if (have_new_rectangle)
	{
	    do_xor_rectangle(&new_rectangle);
	    new_rectangle.x1 = x;
	    new_rectangle.y1 = y;
	    do_xor_rectangle(&new_rectangle);
	}
    }
}

static Event_handlers handlers =
{   button_press, button_release, button_motion,
    no_button_handler, no_button_handler, no_key_handler, no_key_handler   };

static void create_tracking_popup(Widget parent)
{
    Widget buttons;
    Drawing_area_info drawing_info;
    Apply_dismiss_help_info adh_info;
    char apply_label[] = "apply";
    static Event_handle handle =
    { ButtonPressMask | ButtonReleaseMask | PointerMotionMask, &handlers };

    update_tracking_params();

    tracking_popup = create_popup(parent, "Tracking Display");
    CHECK_WIDGET_WARNING(tracking_popup);

    tracking_form = create_form(tracking_popup);
    CHECK_WIDGET_DESTROY_WARNING(tracking_form, tracking_popup);

    adh_info.apply_label = apply_label;
    adh_info.apply_callback = apply_callback;
    adh_info.dismiss_form = tracking_form;
    adh_info.help_message = tracking_help;

    buttons = create_apply_dismiss_help(tracking_form, &adh_info);
    CHECK_WIDGET_DESTROY_WARNING(buttons, tracking_popup);
    attachments(buttons, NO_ATTACH, FORM_ATTACH, FORM_ATTACH, FORM_ATTACH);

    drawing_info.width = width;
    drawing_info.height = height;
    drawing_info.expose = tracking_expose_callback;
    drawing_info.resize = tracking_resize_callback;

    tracking_area = create_drawing(tracking_form, &drawing_info);
    CHECK_WIDGET_DESTROY_WARNING(tracking_area, tracking_popup);
    attachments(tracking_area, FORM_ATTACH, buttons, FORM_ATTACH, FORM_ATTACH);

    add_event_handler(tracking_area, &handle);

    manage_widget(tracking_form);

    tracking_display = WIDGET_DISPLAY(tracking_area);
    initialize_display(tracking_display);

    tracking_window = WIDGET_WINDOW(tracking_area);
    tracking_gc = create_gc(tracking_display, tracking_window);

    clear_tracking();
}

void tracking_popup_callback(Widget parent, Callback_ptr data, Callback_ptr cbs)
{
    if (!tracking_popup)
        create_tracking_popup(parent);

    if (tracking_popup)
    {
	popup(tracking_form);
	do_tracking_drawing();
    }
}

void do_tracking_drawing()
{
    int i, ndata_sets;
    int fold_type[DISPLAY_DIM], flip_type[DISPLAY_DIM];
    float lower[DISPLAY_DIM], upper[DISPLAY_DIM];
    float bottom[DISPLAY_DIM], top[DISPLAY_DIM];
    float min0, min1, max0, max1;
    Data_info *data_info;
    Line error_msg;

    have_valid_tracking = FALSE;

    if (!tracking_popup)
	return;
/*
	create_tracking_popup(get_topshell());
*/

    clear_tracking();

    if (region_apply(&ref_type, lower, upper, fold_type, flip_type,
							error_msg) == ERROR)
	ERROR_AND_RETURN(error_msg);

    if (get_data_info(&ndata_sets, &data_info, error_msg) == ERROR)
	ERROR_AND_RETURN(error_msg);

    if (ndata_sets == 0)
	return;

    min0 = min1 = LARGE_FLOAT;
    max0 = max1 = - LARGE_FLOAT;

    for (i = 0; i < ndata_sets; i++)
    {
	bottom[0] = bottom[1] = 1;
	convert_from_points(ref_type, DISPLAY_DIM,
			data_info[i].npoints, data_info[i].ref, bottom);

	top[0] = data_info[i].npoints[0];  top[1] = data_info[i].npoints[1];
	convert_from_points(ref_type, DISPLAY_DIM,
			data_info[i].npoints, data_info[i].ref, top);

	check_orientation(ref_type, DISPLAY_DIM, bottom, top);

	min0 = MIN(min0, bottom[0]);
	min1 = MIN(min1, bottom[1]);
	max0 = MAX(max0, top[0]);
	max1 = MAX(max1, top[1]);
    }

    if ((min0 >= max0) || (min1 >= max1))
	return;

    bottom[0] = min0;  bottom[1] = min1;
    top[0] = max0;  top[1] = max1;

    check_orientation(ref_type, DISPLAY_DIM, bottom, top);

    limits.x0 = bottom[0];  limits.y0 = bottom[1];
    limits.x1 = top[0];  limits.y1 = top[1];

    transform_to_tracking(&current_rectangle, lower, upper);
    do_rectangles();

    have_valid_tracking = TRUE;
}

Status tracking_popup_command(String value, Generic_ptr data, String error_msg)
{
    tracking_popup_callback(get_topshell(), (Callback_ptr) NULL,
							(Callback_ptr) NULL);

    return  OK;
}
