#include "slice_popup.h"

#include "baseline.h"
#include "baseline_popup.h"
#include "display.h"
#include "event_handle.h"
#include "info_popup.h"
#include "param.h"
#include "object.h"
#include "object_popup.h"
#include "phase_popup.h"
#include "rowcol_popup.h"
#include "slice.h"
#include "utility.h"
#include "window.h"

#define  MIN_WIDTH	200
#define  MIN_HEIGHT	200

static Widget slice_popup = (Widget) NULL;
static Widget slice_area;

static Display *slice_display = (Display *) NULL;
static Window slice_window;
static GC slice_window_gc = (GC) NULL;
static GC slice_pixmap_gc = (GC) NULL;
static Pixmap slice_pixmap = (Pixmap) NULL;
static Drawable slice_drawable;
static GC slice_gc;

static int width = MIN_WIDTH;
static int height = MIN_HEIGHT;
static int depth = 1;

static float ax, bx, ay, by;

static Bool have_valid_slice = FALSE;

static float rowcol_mult = 1.5;

static Slice_info slice_info;

static Bool have_limit0 = FALSE;
static Bool have_limit1 = FALSE;

static int limit0;
static int limit1;

static Bool have_left_shift = FALSE;
static Bool have_right_shift = FALSE;

void update_slice_params()
{
    if (slice_popup)
    {
	get_widget_size(slice_area, &width, &height);
	sprintf(slice_width, "%d", width);
	sprintf(slice_height, "%d", height);
    }
    else
    {
	if (*slice_width)
	    width = atoi(slice_width);

	width = MAX(width, MIN_WIDTH);

	if (*slice_height)
	    height = atoi(slice_height);

	height = MAX(height, MIN_HEIGHT);
    }
}

static Bool inside_limits(int x, int *begin, int *end)
{
    if (have_limit0 && have_limit1)
    {
	*begin = MIN(limit0, limit1);
	*end = MAX(limit0, limit1);
    }
    else if (have_limit0)
    {
	*begin = limit0;
	*end = width;
    }
    else if (have_limit1)
    {
	*begin = 0;
	*end = limit1;
    }
    else
    {
	*begin = 0;
	*end = width;
    }

    if (x < *begin)
	return  FALSE;

    if (x > *end)
	return  FALSE;

    return  TRUE;
}

void slice_limits(int *size, int *begin, int *end)
{
    (void) inside_limits(0, begin, end);
    *size = width;
}

static void do_limits()
{
    if (have_limit0)
	display_line(slice_display, slice_window, slice_window_gc,
						limit0, 0, limit0, height);

    if (have_limit1)
	display_line(slice_display, slice_window, slice_window_gc,
						limit1, 0, limit1, height);
}

static void do_crosshair(int x, int y)
{
    if (x < 0)
	return;

    display_line(slice_display, slice_window, slice_window_gc, 0, y, width, y);
    display_line(slice_display, slice_window, slice_window_gc, x, 0, x, height);
}

static void clear_slice()
{
    set_gc_black(slice_display, slice_pixmap_gc); /* backwards */
    clear_region(slice_display, slice_pixmap, slice_pixmap_gc,
							0, 0, width, height);
}

static void show_slice(int x, int y)
{
    copy_plane(slice_display, slice_pixmap, slice_window, slice_window_gc,
						0, 0, width, height, 0, 0);
    do_limits();
    do_crosshair(x, y);
}

static void insert_shift_point(int x)
{
    float point;
    Line error_msg;

    if (!have_valid_slice)
	ERROR_AND_RETURN("no slice for baseline insertion");

    point = calculate_slice_point(x, width, &slice_info);

    if (insert_baseline_point(point, slice_info.npoints, error_msg) == ERROR)
	ERROR_AND_RETURN(error_msg);

    do_slice_drawing(TRUE);
}

static void delete_shift_point(int x)
{
    float point;
    Line error_msg;

    if (!have_valid_slice)
	ERROR_AND_RETURN("no slice for baseline deletion");

    point = calculate_slice_point(x, width, &slice_info);

    if (delete_baseline_point(point, slice_info.npoints, error_msg) == ERROR)
	ERROR_AND_RETURN(error_msg);

    do_slice_drawing(TRUE);
}

static void start_slice(Generic_ptr data)
{
    slice_drawable = slice_pixmap;
    slice_gc = slice_pixmap_gc;
    clear_slice();
}

static void end_slice();  /* forward reference */

static void new_slice_range(float x0, float y0, float x1, float y1, Bool clip)
{
    ax = width / (x1 - x0);  bx = - ax * x0;
    ay = - height / (y1 - y0);  by = - ay * y1;
	/* y formulas different from x ones because y = 0 on top of screen */
}

#define  CONVERT_X(x)  (ax*(x) + bx)
#define  CONVERT_Y(y)  (ay*(y) + by)

static void do_slice_line(float x0, float y0, float x1, float y1)
{
    int w0, w1, h0, h1;

    w0 = CONVERT_X(x0);
    w1 = CONVERT_X(x1);
    h0 = CONVERT_Y(y0);
    h1 = CONVERT_Y(y1);

    display_line(slice_display, slice_drawable, slice_gc, w0, h0, w1, h1);
}

static void do_slice_text(String text, float x, float y, float a, float b)
{
    int w, h;

    w = CONVERT_X(x);
    h = CONVERT_Y(y);

    display_text(slice_display, slice_drawable, slice_gc, w, h, a, b, text);
}

static void set_slice_color(int color)
{
    set_gc_white(slice_display, slice_gc); /* backwards */
}

static void set_slice_font(String name, int size)
{
}

static void set_slice_line_style(int line_style)
{
}

static Draw_funcs draw_funcs = { SCREEN_DISPLAY, (Generic_ptr) NULL,
	start_slice, end_slice, new_slice_range, do_slice_line, do_slice_text,
	set_slice_color, set_slice_font, set_slice_line_style };

static void end_slice()
{
    float lower[DISPLAY_DIM], upper[DISPLAY_DIM];

    COPY_VECTOR(lower, slice_info.lower, DISPLAY_DIM);
    COPY_VECTOR(upper, slice_info.upper, DISPLAY_DIM);

    convert_range_to_points(slice_info.ref_type, 1, &slice_info.npoints,
						slice_info.ref, lower, upper);

    draw_baseline(&draw_funcs, slice_info.npoints, slice_info.data,
								lower, upper);

    show_slice(-1, -1);
}

static void slice_expose_callback(Widget widget, Callback_ptr data,
							Callback_ptr cbs)
{
    show_slice(-1, -1);
}

static void slice_resize_callback(Widget widget, Callback_ptr data,
							Callback_ptr cbs)
{
    free_pixmap(slice_display, slice_pixmap);
    update_slice_params();
    slice_pixmap = create_pixmap(slice_display, slice_window,
						width, height, depth);

    if (have_valid_slice)
    {
	do_slice(&draw_funcs, &slice_info);
    }
    else
    {
	clear_slice();
	show_slice(-1, -1);
    }
}

static void button_press(unsigned int button, int x, int y, Time time)
{
    int begin, end;
    int point[DISPLAY_DIM], size[DISPLAY_DIM];
    Line error_msg;
    static Print_funcs print_funcs = { start_info, print_info, end_info };

    if (button == Button1)
    {
	if (have_left_shift || have_right_shift)
	{
	    insert_shift_point(x);
	    show_slice(x, y);
	}
	else if (have_limit0 || have_limit1)
	{
	    if (inside_limits(x, &begin, &end))
		change_point_rowcol(width, begin, end, EXPAND_ROWCOL);
	    else
		change_point_rowcol(width, begin, end, CONTRACT_ROWCOL);

	    have_limit0 = have_limit1 = FALSE;

	    do_slice_drawing(TRUE);
	    show_slice(x, y);
	}
    }
    else if (button == Button2)
    {
	have_limit0 = have_limit1 = FALSE;
	show_slice(x, y);
    }
    else if (button == Button3)
    {
	if (have_left_shift || have_right_shift)
	{
	    delete_shift_point(x);
	    show_slice(x, y);
	}
	else
	{
	    size[0] = width;  size[1] = height;
	    point[0] = x;  point[1] = height - 1 - y;

	    (void) rowcol_point_stats(&print_funcs, size, point, error_msg);
	}
    }
}

static void button_motion(unsigned int button, int x, int y, Time time)
{
    show_slice(x, y);
}

static void button_leave(unsigned int button, int x, int y, Time time)
{
    show_slice(-1, -1);
}

static void key_press(Key_symbol key, int x, int y, Time time)
{
    if (key == UP_KEY)
    {
	change_value_rowcol(1.0/rowcol_mult);
	do_slice_drawing(TRUE);
	show_slice(x, y);
    }
    else if (key == DOWN_KEY)
    {
	change_value_rowcol(rowcol_mult);
	do_slice_drawing(TRUE);
	show_slice(x, y);
    }
    else if (key == LEFT_KEY)
    {
	have_limit0 = TRUE;
	limit0 = x;
	show_slice(x, y);
    }
    else if (key == RIGHT_KEY)
    {
	have_limit1 = TRUE;
	limit1 = x;
	show_slice(x, y);
    }
    else if (key == LEFT_SHIFT_KEY)
    {
	have_left_shift = TRUE;
    }
    else if (key == RIGHT_SHIFT_KEY)
    {
	have_right_shift = TRUE;
    }
}

static void key_release(Key_symbol key, int x, int y, Time time)
{
    if (key == LEFT_SHIFT_KEY)
    {
	have_left_shift = FALSE;
    }
    else if (key == RIGHT_SHIFT_KEY)
    {
	have_right_shift = FALSE;
    }
}

static void quit_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    free_pixmap(slice_display, slice_pixmap);
    free_gc(slice_display, slice_window_gc);
    free_gc(slice_display, slice_pixmap_gc);
    destroy_widget(slice_popup);
    slice_popup = (Widget) NULL;
    slice_window_gc = (GC) NULL;
    slice_pixmap_gc = (GC) NULL;
}

static Event_handlers handlers =
{   button_press, no_button_handler, button_motion,
    no_button_handler, button_leave, key_press, key_release   };

static void create_slice_popup(Widget parent)
{
    Drawing_area_info drawing_info;
    static Event_handle handle =
    { ButtonPressMask | PointerMotionMask | LeaveWindowMask
				| KeyPressMask | KeyReleaseMask, &handlers };

    update_slice_params();

    slice_popup = create_iconifiable_popup(parent, "1D Display");
    CHECK_WIDGET_WARNING(slice_popup);
    change_delete_protocol(slice_popup, quit_callback);

    drawing_info.width = width;
    drawing_info.height = height;
    drawing_info.expose = slice_expose_callback;
    drawing_info.resize = slice_resize_callback;

    slice_area = create_drawing(slice_popup, &drawing_info);
    CHECK_WIDGET_DESTROY_WARNING(slice_area, slice_popup);

    realize_widget(slice_popup);

    add_event_handler(slice_area, &handle);

    slice_display = WIDGET_DISPLAY(slice_area);
    initialize_font(slice_display);

    slice_window = WIDGET_WINDOW(slice_area);
    slice_pixmap = create_pixmap(slice_display, slice_window,
						width, height, depth);
    slice_window_gc = create_gc(slice_display, slice_window);
    slice_pixmap_gc = create_gc(slice_display, slice_pixmap);

    clear_slice();
}

void slice_popup_callback(Widget parent, Callback_ptr data, Callback_ptr cbs)
{
    if (!slice_popup)
        create_slice_popup(parent);

    if (slice_popup)
	popup(slice_area);
}

static void draw_bounding_box()
{
    display_line(slice_display, slice_drawable, slice_gc,
                                        0, 0, width-1, 0);
    display_line(slice_display, slice_drawable, slice_gc,
                                        width-1, 0, width-1, height-1);
    display_line(slice_display, slice_drawable, slice_gc,
                                        width-1, height-1, 0, height-1);
    display_line(slice_display, slice_drawable, slice_gc,
                                        0, height-1, 0, 0);
}

void do_slice_drawing(Bool refresh)
{
    int ndata_sets;
    Data_info *data_info;
    Global_info global_info;
    Line error_msg;

    if (slice_popup && refresh)
    {
	clear_slice();
	show_slice(-1, -1);
    }

    if (!have_valid_slice || refresh)
    {
	have_valid_slice = FALSE;

	if (objects_apply(&ndata_sets, &data_info, &global_info, error_msg)
								== ERROR)
	    ERROR_AND_RETURN(error_msg);

	if (rowcol_apply(&slice_info, data_info, error_msg) == ERROR)
	    ERROR_AND_RETURN(error_msg);

	if (phase_apply(&slice_info, error_msg) == ERROR)
	    ERROR_AND_RETURN(error_msg);

	COPY_VECTOR(slice_info.major_ticks, global_info.major_ticks, DISPLAY_DIM);
	COPY_VECTOR(slice_info.minor_ticks, global_info.minor_ticks, DISPLAY_DIM);

	have_valid_slice = TRUE;
    }

    if (baseline_apply(&slice_info, error_msg) == ERROR)
    {
	have_valid_slice = FALSE;
	ERROR_AND_RETURN(error_msg);
    }

    if (!slice_popup)
	create_slice_popup(get_topshell());

    popup(slice_area);

    do_slice(&draw_funcs, &slice_info);

    draw_bounding_box();
}

#ifdef USE_XPM
Status do_slice_dump(String file, String error_msg)
{
    XpmAttributes attributes;

    attributes.valuemask = 0;

    do_slice_drawing(TRUE);

    if (have_valid_slice)
        (void) XpmWriteFileFromPixmap(slice_display, file,
			slice_pixmap, (Pixmap) NULL, &attributes);

    return  OK;
}
#endif /* USE_XPM */

Status slice_popup_command(String value, Generic_ptr data, String error_msg)
{
    slice_popup_callback(get_topshell(), (Callback_ptr) NULL,
							(Callback_ptr) NULL);

    return  OK;
}
