#include "region_func.h"

#include "data.h"
#include "fold_func.h"
#include "param.h"
#include "position.h"
#include "ref.h"
#include "region.h"

static int ref_type = REF_POINTS;

static Region_info region_info =
{ (float *) NULL, (float *) NULL, x_region, y_region };

static int npositions = -1;
static int current_position = -1;

static FILE *fp_stats = (FILE *) NULL;
static Line stats_file;

void set_region_func(int type, float *lower, float *upper)
{
    ref_type = type;

    sprintf(x_region, "%3.2f %3.2f", lower[0], upper[0]);
    sprintf(y_region, "%3.2f %3.2f", lower[1], upper[1]);
}

static void update_region_data()
{
    if (*region_ref)
	ref_type = atoi(region_ref);
}

Status find_region_func(int *p_ref_type, float *lower, float *upper,
							String error_msg)
{
    update_region_data();

    *p_ref_type = ref_type;

    region_info.lower = lower;
    region_info.upper = upper;

    CHECK_STATUS(find_region(&region_info, error_msg));

    return  OK;
}

Status region_apply_func(int *p_ref_type, float *lower, float *upper,
			int *fold_type, int *flip_type, String error_msg)
{
    CHECK_STATUS(find_region_func(p_ref_type, lower, upper, error_msg));
    CHECK_STATUS(find_fold_type(fold_type, flip_type, error_msg));

    return  OK;
}

static Status init_limits(float *lower, float *upper, String error_msg)
{
    region_info.lower = lower;
    region_info.upper = upper;

    CHECK_STATUS(find_region(&region_info, error_msg));

    check_orientation(ref_type, DISPLAY_DIM, lower, upper);

    return  OK;
}

static void end_limits(float *lower, float *upper)
{
    check_orientation(ref_type, DISPLAY_DIM, lower, upper);
    set_region_func(ref_type, lower, upper);
}

Status move_region_func(int *original, int *translation, String error_msg)
{
    int i;
    float c, lower[DISPLAY_DIM], upper[DISPLAY_DIM];

    CHECK_STATUS(init_limits(lower, upper, error_msg));

    for (i = 0; i < DISPLAY_DIM; i++)
    {
    	c = ((float) translation[i]) * (upper[i] - lower[i])
						/ ((float) original[i]);
	upper[i] += c;
	lower[i] += c;
    }

    end_limits(lower, upper);

    return  OK;
}

Status change_region_func(int *original, int *begin, int *end, int type,
							String error_msg)
{
    int i;
    float c, lower[DISPLAY_DIM], upper[DISPLAY_DIM];

    CHECK_STATUS(init_limits(lower, upper, error_msg));

    for (i = 0; i < DISPLAY_DIM; i++)
    {
	if (type == EXPAND_REGION)
	{
    	    c = (upper[i] - lower[i]) / ((float) original[i]);
	    upper[i] = c * end[i] + lower[i];
	    lower[i] += c * begin[i];
	}
	else  /* type == CONTRACT_REGION */
	{
	    c = (upper[i] - lower[i]) / ((float) (end[i] - begin[i]));
	    upper[i] = c * (original[i] - begin[i]) + lower[i];
	    lower[i] -= c * begin[i];
	}
    }

    end_limits(lower, upper);

    return  OK;
}

Status find_box_region(int *p_ref_type, String x_range, String y_range,
		int *original, int *begin, int *end, String error_msg)
{
    int i;
    float c, lower[DISPLAY_DIM], upper[DISPLAY_DIM];

    CHECK_STATUS(init_limits(lower, upper, error_msg));

    for (i = 0; i < DISPLAY_DIM; i++)
    {
    	c = (upper[i] - lower[i]) / ((float) original[i]);
	upper[i] = c * end[i] + lower[i];
	lower[i] += c * begin[i];
    }

    check_orientation(ref_type, DISPLAY_DIM, lower, upper);

    *p_ref_type = ref_type;

    sprintf(x_range, "%3.2f %3.2f", lower[0], upper[0]);
    sprintf(y_range, "%3.2f %3.2f", lower[1], upper[1]);

    return  OK;
}

Status find_point_stats(Print_funcs *print_funcs, int *size, int *point,
							String error_msg)
{
    int i;
    float c, lower[DISPLAY_DIM], upper[DISPLAY_DIM], desired_point[DISPLAY_DIM];

    CHECK_STATUS(init_limits(lower, upper, error_msg));
 
    for (i = 0; i < DISPLAY_DIM; i++)
    {
	c = point[i] * (upper[i] - lower[i]) / ((float) size[i]);
	desired_point[i] = lower[i] + c;
    }

    print_point_stats(ref_type, desired_point, print_funcs);

    return  OK;
}

static void check_init_position()
{
    if (npositions == -1)
	npositions = get_number_positions();
}

Status region_select_func(int position, String error_msg)
{
    check_init_position();

    if ((position < 0) || (position >= npositions))
	RETURN_ERROR_MSG("selected position out of range");

    if (current_position == position)
	current_position = -1;
    else
	current_position = position;

    return  OK;
}

Status region_insert_func(YesNo_func func, String error_msg)
{
    int type, position;
    float lower[DISPLAY_DIM], upper[DISPLAY_DIM];

    check_init_position();

    CHECK_STATUS(find_region_func(&type, lower, upper, error_msg));

    if (position_name_exists(position_name, &position))
    {
	if ((*func)("Position name exists, overwrite?",
					"Yes", "No", (String) NULL) == YES)
	    new_position_info(position, type, lower, upper);
    }
    else
    {
	CHECK_STATUS(initialize_position(position_name, type, lower, upper, error_msg));

	npositions++;
    }

    return  OK;
}

Status region_go_to_func(int *p_ref_type, String error_msg)
{
    Position_info *info;

    if (current_position == -1)
	RETURN_ERROR_MSG("no saved position selected");

    get_position_info(current_position, &info);

    set_region_func(info->ref_type, info->lower, info->upper);

    *p_ref_type = ref_type;

    return  OK;
}

Status region_delete_func(int *position, String error_msg)
{
    if (current_position == -1)
	RETURN_ERROR_MSG("no saved position selected");

    delete_position(current_position);
    *position = current_position;

    current_position = -1;

    return  OK;
}

Status region_stats_func(Print_funcs *print_funcs,
			Rectangle_func rectangle_func, String error_msg)
{
    int i, size[DISPLAY_DIM], begin[DISPLAY_DIM], end[DISPLAY_DIM];
    float c, lower[DISPLAY_DIM], upper[DISPLAY_DIM];

    update_region_data();

    CHECK_STATUS(init_limits(lower, upper, error_msg));
 
    if ((*rectangle_func)(size, begin, end))
    {
	for (i = 0; i < DISPLAY_DIM; i++)
	{
    	    c = (upper[i] - lower[i]) / ((float) size[i]);
	    upper[i] = c * end[i] + lower[i];
	    lower[i] += c * begin[i];
	}
    }

    print_region_stats(ref_type, lower, upper, print_funcs,
					fp_stats, stats_file);

    return  OK;
}

Status region_whole_func(String error_msg)
{
    int ndata_sets;
    Data_info *info;
    float lower[DISPLAY_DIM], upper[DISPLAY_DIM];

    update_region_data();

    CHECK_STATUS(get_data_info(&ndata_sets, &info, error_msg));

    if (ndata_sets == 0)
	return  OK;

    region_limits(ref_type, ndata_sets, info, lower, upper);
    set_region_func(ref_type, lower, upper);

    return  OK;
}

void region_stats_close_func()
{
    FCLOSE(fp_stats);
}

Status region_stats_open_func(String file, String error_msg)
{
    region_stats_close_func();

    CHECK_OPEN_FOR_WRITING(fp_stats, file);

    fprintf(fp_stats,
	"name\tx0\tx1\ty0\ty1\tnx\tny\tn\tmin\tmax\tsum\tavg\tstddev\n");

    strncpy(stats_file, file, LINE_SIZE);

    return  OK;
}
