#include "phase_popup.h"

#include "phase.hlp"

#include "callback.h"
#include "object.h"
#include "param.h"
#include "phase_func.h"
#include "slice.h"
#include "slice_popup.h"
#include "window.h"

static Widget phase_popup = (Widget) NULL;
static Widget phase_form;

#define  NON_OFF_TYPES  2

static Widget phase_on_off_boxes[NON_OFF_TYPES];
static Radiobox_item phase_on_off_items[] =
{
    { "on", NO_CALLBACK, NO_DATA, phase_on_off_boxes + VISIBILITY_ON },
    { "off", NO_CALLBACK, NO_DATA, phase_on_off_boxes + VISIBILITY_OFF }
};

static int phase_on_off_type = VISIBILITY_ON;
static int nphase_on_off_types = ARRAY_SIZE(phase_on_off_boxes);

static Widget pivot_field;

static Textfield_item textfield_items[] =
{
    { "pivot: ", phase_pivot, NO_CALLBACK, NO_DATA, &pivot_field }
};

static int ntexts = ARRAY_SIZE(textfield_items);

static Widget min0_field, max0_field;
static Widget min1_field, max1_field;

static Textfield_item phase0_items[] =
{
    { (char *) NULL, phase0_min, NO_CALLBACK, NO_DATA, &min0_field },
    { (char *) NULL, phase0_max, NO_CALLBACK, NO_DATA, &max0_field }
};

static Textfield_item phase1_items[] =
{
    { (char *) NULL, phase1_min, NO_CALLBACK, NO_DATA, &min1_field },
    { (char *) NULL, phase1_max, NO_CALLBACK, NO_DATA, &max1_field }
};

static int nlimits = ARRAY_SIZE(phase0_items);

static Scale_numbers numbers0;
static Scale_numbers numbers1;

static Widget scale0 = (Widget) NULL;
static Widget scale1 = (Widget) NULL;

static Bool have_valid_phasing = FALSE;

static Bool scale0_active = FALSE;
static Bool scale1_active = FALSE;

#define  VALUE_IN_RANGE(numbers) \
	 {   if ((numbers.value < numbers.minimum) || \
		 (numbers.value > numbers.maximum)) \
		 numbers.value = 0.5*(numbers.minimum + numbers.maximum);   }

#define  SET_VALUE(string, value) \
	 {   sprintf(string, "%7.2f", value);   }

static void set_phase_numbers()
{
    numbers0.minimum = atof(phase0_min);
    numbers0.maximum = atof(phase0_max);
    numbers0.value = atof(phase0_value);
    numbers1.minimum = atof(phase1_min);
    numbers1.maximum = atof(phase1_max);
    numbers1.value = atof(phase1_value);
}

static void set_phase_fields()
{
    if (phase_popup)
    {
	set_textfields(textfield_items, ntexts);
	set_textfields(phase0_items, nlimits);
	set_textfields(phase1_items, nlimits);
    }
}

static void get_phase_values()
{
    SET_VALUE(phase0_value, numbers0.value);
    SET_VALUE(phase1_value, numbers1.value);
}

void update_phase_params()
{
    if (phase_popup)
    {
	phase_on_off_type = get_radio_state(phase_on_off_boxes,
							nphase_on_off_types);
	get_textfields(textfield_items, ntexts);
	get_textfields(phase0_items, nlimits);
	get_textfields(phase1_items, nlimits);

	sprintf(phase_on_off, "%d", phase_on_off_type);
    }
    else
    {
	if (*phase_on_off)
	    phase_on_off_type = atoi(phase_on_off);

	set_phase_params();
    }

    set_phase_numbers();
}

static Status valid_phase_numbers(String error_msg)
{
    if (numbers0.minimum >= numbers0.maximum)
	RETURN_ERROR_MSG("phase0: minimum >= maximum");

    if (numbers1.minimum >= numbers1.maximum)
	RETURN_ERROR_MSG("phase1: minimum >= maximum");

    VALUE_IN_RANGE(numbers0);
    VALUE_IN_RANGE(numbers1);

    get_phase_values();

    if (scale0)
	set_scale_numbers(scale0, &numbers0);

    if (scale1)
	set_scale_numbers(scale1, &numbers1);

    return  OK;
}

Status phase_apply(Slice_info *info, String error_msg)
{
    update_phase_params();
    set_phase_fields();

    if (phase_on_off_type == VISIBILITY_OFF)
    {
	have_valid_phasing = TRUE;
	return  OK;
    }

    have_valid_phasing = FALSE;

    CHECK_STATUS(valid_phase_numbers(error_msg));

    CHECK_STATUS(phase_apply_func(info, error_msg));

    have_valid_phasing = TRUE;

    return  OK;
}

static void scale_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    if (widget == scale0)
    {
	if (!scale0_active)
	{
	    scale0_active = TRUE;
	    scale1_active = FALSE;
	    set_scale_numbers(scale0, &numbers0);

	    return;
	}

	get_scale_value(cbs, &numbers0);
    }
    else if (widget == scale1)
    {
	if (!scale1_active)
	{
	    scale1_active = TRUE;
	    scale0_active = FALSE;
	    set_scale_numbers(scale1, &numbers1);

	    return;
	}

	get_scale_value(cbs, &numbers1);
    }
    else
    {
	ERROR_AND_RETURN("should not be here in scale_callback");
    }

    get_phase_values();

    if (have_valid_phasing && (phase_on_off_type == VISIBILITY_ON))
    {
	phase_scale_func();
	do_slice_drawing(FALSE);
    }
}

static void on_off_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    if (!toggle_being_set(cbs))
	return;

    if (have_valid_phasing)
	do_slice_drawing(TRUE);
}

static Status apply_command(String error_msg)
{
    update_phase_params();
    set_phase_fields();

    if (valid_phase_numbers(error_msg) == ERROR)
	RETURN_ERROR_MSG(error_msg);

    do_slice_drawing(TRUE);

    return  OK;
}

static void apply_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    Line error_msg;

    if (apply_command(error_msg) == ERROR)
	ERROR_AND_RETURN(error_msg);
}

Status phase_apply_command(String value, Generic_ptr data, String error_msg)
{
    CHECK_STATUS(apply_command(error_msg));

    return  OK;
}

static void create_phase_popup(Widget parent)
{
    int i;
    Widget buttons, radio, texts, separator, scale;
    Apply_dismiss_help_info adh_info;
    char apply_label[] = "apply";
    static Scale_info scale0_info =
	{ HORIZONTAL, &numbers0, TRUE, "phase 0",
				scale_callback, scale_callback, NO_DATA };
    static Scale_info scale1_info =
	{ HORIZONTAL, &numbers1, TRUE, "phase 1",
				scale_callback, scale_callback, NO_DATA };
    static Text_scale_info phase0_info =
				{ &scale0_info, &scale0, phase0_items };
    static Text_scale_info phase1_info =
				{ &scale1_info, &scale1, phase1_items };
    Line error_msg;

    update_phase_params();

    if (valid_phase_numbers(error_msg) == ERROR)
    {
	default_phase_numbers();
	set_phase_numbers();
    }

    phase_popup = create_popup(parent, "Phase Slice");
    CHECK_WIDGET_WARNING(phase_popup);

    phase_form = create_form(phase_popup);
    CHECK_WIDGET_DESTROY_WARNING(phase_form, phase_popup);

    for (i = 0; i < nphase_on_off_types; i++)
	phase_on_off_items[i].callback = on_off_callback;

    radio = create_radiobox(phase_form, phase_on_off_items,
			nphase_on_off_types, phase_on_off_type, HORIZONTAL);
    CHECK_WIDGET_DESTROY_WARNING(radio, phase_popup);
    attachments(radio, FORM_ATTACH, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);

    separator = create_separator(phase_form);
    CHECK_WIDGET_DESTROY_WARNING(separator, phase_popup);
    attachments(separator, radio, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);

    texts = create_textfields(phase_form, textfield_items, ntexts);
    CHECK_WIDGET_DESTROY_WARNING(texts, phase_popup);
    attachments(texts, separator, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);

    separator = create_separator(phase_form);
    CHECK_WIDGET_DESTROY_WARNING(separator, phase_popup);
    attachments(separator, texts, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);

    scale = create_text_scale(phase_form, &phase0_info);
    CHECK_WIDGET_DESTROY_WARNING(scale, phase_popup);
    attachments(scale, separator, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);

    separator = create_separator(phase_form);
    CHECK_WIDGET_DESTROY_WARNING(separator, phase_popup);
    attachments(separator, scale, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);

    scale = create_text_scale(phase_form, &phase1_info);
    CHECK_WIDGET_DESTROY_WARNING(scale, phase_popup);
    attachments(scale, separator, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);

    separator = create_separator(phase_form);
    CHECK_WIDGET_DESTROY_WARNING(separator, phase_popup);
    attachments(separator, scale, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);

    adh_info.apply_label = apply_label;
    adh_info.apply_callback = apply_callback;
    adh_info.dismiss_form = phase_form;
    adh_info.help_message = phase_help;

    buttons = create_apply_dismiss_help(phase_form, &adh_info);
    CHECK_WIDGET_DESTROY_WARNING(buttons, phase_popup);
    attachments(buttons, separator, FORM_ATTACH, FORM_ATTACH, FORM_ATTACH);
    offsets(buttons, UNIT_OFFSET, UNIT_OFFSET, NO_OFFSET, NO_OFFSET);

    set_phase_fields();

    manage_widget(phase_form);
}

void phase_popup_callback(Widget parent, Callback_ptr data, Callback_ptr cbs)
{
    if (!phase_popup)
        create_phase_popup(parent);

    if (phase_popup)
    {
	scale0_active = scale1_active = FALSE;
        popup(phase_form);
    }
}

Status phase_popup_command(String value, Generic_ptr data, String error_msg)
{
    phase_popup_callback(get_topshell(), (Callback_ptr) NULL,
							(Callback_ptr) NULL);

    return  OK;
}
