#include "par.h"

#include "parse.h"
#include "ref.h"
#include "utility.h"

/*  Note memory is recycled every time read_par file is called, so
    any external routine must make its own copy to avoid problems.  */

#define  VARIAN_FILE_HEADER	8	/* 32 bytes */
#define  VARIAN_BLOCK_HEADER	7	/* 28 bytes */

static String file = NULL;

static int ndim;
static int *npoints = NULL;
static int *block_size = NULL;
static int header;

static Ref_info *ref = NULL;

static Bool swapped;
static Bool integer;
static Bool blocked;
static Bool deflated;
static Bool reflated;
static float level;
static Bool blocks;
static int byte_size;
static Bool big_endian;
static Bool little_endian;

static String par_file;

static int dim;
static Bool *dim_found = NULL;

static Bool allow_varian = FALSE;
static Bool have_varian;
static int *dim_order = NULL;
static int file_header = VARIAN_FILE_HEADER;
static int block_header = VARIAN_BLOCK_HEADER;
static char varian_header[BYTES_PER_WORD*VARIAN_FILE_HEADER];

/* NOTE: so far params only allowed for one dimension */
static int param_dim;
static float *params = NULL;

static int parse_int[] = { PARSE_INT };
static int parse_float[] = { PARSE_FLOAT };
static int parse_string[] = { PARSE_STRING };
static int parse_int_free[] = { PARSE_INT | PARSE_FREE };
static int parse_float_free[] = { PARSE_FLOAT | PARSE_FREE };

#define  CHECK_DIM_NOT_THERE(string) \
	 {   if (dim >= 0) \
	     {   sprintf(error_msg, \
				"in \"%s\" '%s' should be before any 'dim'", \
				par_file, string);  return  ERROR;   }   }

#define  CHECK_DIM_THERE(string) \
	 {   if (dim < 0) \
	     {   sprintf(error_msg, "in \"%s\" '%s' found before any 'dim'", \
				par_file, string);  return  ERROR;   }   }

#define  FOUND_TWICE(string) \
	 {   sprintf(error_msg, "in \"%s\" '%s' found twice", \
				par_file, string);  return  ERROR;   }

#define  FOUND_BEFORE(string1, string2) \
	 {   sprintf(error_msg, "in \"%s\" '%s' found before '%s'", \
				par_file, string1, string2);  return  ERROR;   }

#define  NOT_ALLOWED(string, condition) \
	 {   sprintf(error_msg, "in \"%s\" '%s' %s", \
				par_file, string, condition);  return  ERROR;   }

#define  NOT_ALLOWED_FOR_DIM(dim, string, condition) \
	 {   sprintf(error_msg, "in \"%s\" for dim %d '%s' %s", \
			par_file, dim+1, string, condition);  return  ERROR;   }

#define  NOT_FOUND(string) \
	 {   sprintf(error_msg, "in \"%s\" no '%s' found", \
				par_file, string);  return  ERROR;   }

#define  NOT_FOUND_FOR_DIM(string, dim) \
	 {   sprintf(error_msg, "in \"%s\" no '%s' found for dim %d", \
				par_file, string, dim+1);  return  ERROR;   }

static void free_par_memory()
{
    FREE(npoints, int);
    FREE(block_size, int);

    FREE(dim_found, Bool);

    FREE(file, char);

    FREE(dim_order, int);

    FREE(params, float);
}

static Status allocate_par_memory()
{
    MALLOC(npoints, int, ndim);
    MALLOC(block_size, int, ndim);

    MALLOC(dim_found, Bool, ndim);

    MALLOC(dim_order, int, ndim);

    return  OK;
}

static Status ndim_parse(Generic_ptr *var, String error_msg)
{
    int i;
    int *d = (int *) var[0];

    if (ndim > 0)
	FOUND_TWICE("ndim");

    ndim = *d;

    CHECK_STATUS(initialize_ref(ndim, &ref, error_msg));

    if (allocate_par_memory() == ERROR)
    {
    	sprintf(error_msg, "allocating memory while reading '%s'", par_file);
	return  ERROR;
    }

    for (i = 0; i < ndim; i++)
    {
	dim_found[i] = FALSE;

	npoints[i] = 0;
	block_size[i] = 0;
    }

    return  OK;
}

static Status file_parse(Generic_ptr *var, String error_msg)
{
    String name = (char *) var[0];

    CHECK_DIM_NOT_THERE("file");

    if (file)
	FOUND_TWICE("file");

    if (!(*name))
    {
	sprintf(error_msg, "no name given for 'file'");
	return  ERROR;
    }

    sprintf(error_msg, "allocating memory while reading '%s'", par_file);
    MALLOC(file, char, strlen(name)+1);

    strcpy(file, name);

    return  OK;
}

static Status deflate_parse(Generic_ptr *var, String error_msg)
{
    level = *((float *) var[0]);

    CHECK_DIM_NOT_THERE("deflate");

    if (deflated)
	FOUND_TWICE("deflate");

    if (reflated)
    {
	sprintf(error_msg, "in \"%s\" found 'deflate' and 'reflate'", par_file);
	return  ERROR;
    }

    deflated = TRUE;

    return  OK;
}

static Status reflate_parse(Generic_ptr *var, String error_msg)
{
    level = *((float *) var[0]);

    CHECK_DIM_NOT_THERE("reflate");

    if (reflated)
	FOUND_TWICE("reflate");

    if (deflated)
    {
	sprintf(error_msg, "in \"%s\" found 'deflate' and 'reflate'", par_file);
	return  ERROR;
    }

    reflated = TRUE;

    return  OK;
}

static Status header_parse(Generic_ptr *var, String error_msg)
{
    int *d = (int *) var[0];

    CHECK_DIM_NOT_THERE("head");

    if (header > 0)
	FOUND_TWICE("head");

    if (*d <= 0)
	NOT_ALLOWED("head", "<= 0");

    header = *d;

    return  OK;
}

static Status int_parse(Generic_ptr *var, String error_msg)
{
    CHECK_DIM_NOT_THERE("int");

    if (integer)
	FOUND_TWICE("int");

    integer = TRUE;

    return  OK;
}

static Status swap_parse(Generic_ptr *var, String error_msg)
{
    CHECK_DIM_NOT_THERE("swap");

    if (swapped)
	FOUND_TWICE("swap");

    swapped = TRUE;

    return  OK;
}

static Status big_endian_parse(Generic_ptr *var, String error_msg)
{
    CHECK_DIM_NOT_THERE("big_endian");

    if (big_endian)
	FOUND_TWICE("big_endian");

    big_endian = TRUE;

    return  OK;
}

static Status little_endian_parse(Generic_ptr *var, String error_msg)
{
    CHECK_DIM_NOT_THERE("little_endian");

    if (little_endian)
	FOUND_TWICE("little_endian");

    little_endian = TRUE;

    return  OK;
}

static Status blocks_parse(Generic_ptr *var, String error_msg)
{
    int i, n;

    CHECK_DIM_NOT_THERE("blocks");

    if (blocks)
	FOUND_TWICE("blocks");

    if (ndim == 0)
	FOUND_BEFORE("blocks", "ndim");

    n = *((int *) var[0]);
    if (n != ndim)
    {
	sprintf(error_msg,
		"in \"%s\" 'blocks' does not have #dim = %d entries",
							par_file, ndim);
	return  ERROR;
    }

    for (i = 0; i < ndim; i++)
    {
	n = *((int *) var[i+1]);

	if (n < 1)
	{
	    sprintf(error_msg,
		"in \"%s\" 'blocks' size #%d is %d < 1", par_file, i+1, n);
	    return  ERROR;
	}

	block_size[i] = n;
    }

    blocks = TRUE;

    return  OK;
}

static Status dim_parse(Generic_ptr *var, String error_msg)
{
    int d = *((int *) var[0]);
    Line msg;

    dim = d;

    if (dim > ndim)
    {
	sprintf(error_msg, "in \"%s\" 'dim' = %d > 'ndim' = %d",
	                                        par_file, dim, ndim);
	return  ERROR;
    }

    if (dim < 1)
	NOT_ALLOWED("dim", "< 1");

    dim--;
    if (dim_found[dim] == TRUE)
    {
	sprintf(msg, "dim %d", dim+1);
	FOUND_TWICE(msg);
    }

    dim_found[dim] = TRUE;

    return  OK;
}

static Status npts_parse(Generic_ptr *var, String error_msg)
{
    int *d = (int *) var[0];

    CHECK_DIM_THERE("npts");

    if (*d < 1)
	NOT_ALLOWED_FOR_DIM(dim, "npts", "< 1");

    npoints[dim] = *d;

    return  OK;
}

static Status block_parse(Generic_ptr *var, String error_msg)
{
    int d = *((int *) var[0]);

    CHECK_DIM_THERE("block");

    if (d < 1)
	NOT_ALLOWED_FOR_DIM(dim, "block", "< 1");

    block_size[dim] = d;
    blocked = TRUE;

    return  OK;
}

static Status varian_parse(Generic_ptr *var, String error_msg)
{
    int i, j, n;

    if (!allow_varian)
    {
	sprintf(error_msg, "in \"%s\" found 'varian', not allowed", par_file);
	return  ERROR;
    }

    CHECK_DIM_NOT_THERE("varian");

    if (have_varian)
	FOUND_TWICE("varian");

    if (ndim == 0)
	FOUND_BEFORE("varian", "ndim");

    n = *((int *) var[0]);
    if (n != (ndim-1))
    {
	sprintf(error_msg,
		"in \"%s\" 'varian' does not have #dim = %d entries",
							par_file, ndim-1);
	return  ERROR;
    }

    dim_order[0] = 0;
    for (i = 1; i < ndim; i++)
    {
	n = *((int *) var[i]);

	if (n < 2)
	{
	    sprintf(error_msg,
		"in \"%s\" 'varian' dim. #%d is %d < 2", par_file, i, n);
	    return  ERROR;
	}

	if (n > ndim)
	{
	    sprintf(error_msg,
		"in \"%s\" 'varian' dim. #%d is %d > %d", par_file, i, n, ndim);
	    return  ERROR;
	}

	for (j = 1; j < i; j++)
	{
	    if (dim_order[j] == (n-1))
	    {
		sprintf(error_msg,
		    "in \"%s\" 'varian' dim. #%d is %d, same as #%d",
							par_file, i, n, j);
		return  ERROR;
	    }
	}

	dim_order[i] = n - 1;
    }

    have_varian = TRUE;

    return  OK;
}

static Status params_parse(Generic_ptr *var, String error_msg)
{
    int i, n;

    CHECK_DIM_THERE("params");

    if (params)
	FOUND_TWICE("params");

    if (npoints[dim] == 0)
	FOUND_BEFORE("params", "npts");

    n = *((int *) var[0]);

    if (n != npoints[dim])
    {
	sprintf(error_msg,
		"in \"%s\" 'params' does not have #pts = %d entries",
						par_file, npoints[dim]);
	return  ERROR;
    }

    sprintf(error_msg, "allocating memory for params");

    MALLOC(params, float, n);

    for (i = 0; i < n; i++)
	params[i] = *((float *) var[i+1]);

    param_dim = dim;

    return  OK;
}

#include "ref_parse.c"	/* bodge for now */

static Parse_line par_table[] =
{
    { "ndim",		1,	parse_int,		ndim_parse },
    { "file",		1,	parse_string,		file_parse },
    { "deflate",	1,	parse_float,		deflate_parse },
    { "reflate",	1,	parse_float,		reflate_parse },
    { "head",		1,	parse_int,		header_parse },
    { "int",		0,	(int *) NULL,		int_parse },
    { "swap",		0,	(int *) NULL,		swap_parse },
    { "big_endian",	0,	(int *) NULL,		big_endian_parse },
    { "little_endian",	0,	(int *) NULL,		little_endian_parse },
    { "blocks",		1,	parse_int_free,		blocks_parse },
    { "dim",		1,	parse_int,		dim_parse },
    { "npts",		1,	parse_int,		npts_parse },
    { "block",		1,	parse_int,		block_parse },
    { "varian",		1,	parse_int_free,		varian_parse },
    { "params",		1,	parse_float_free,	params_parse },
#include "ref_parse.h"	/* bodge for now */
    { (String) NULL,	0,	(int *) NULL,		no_parse_func }
};

static Status get_varian_header(FILE *fp, String error_msg)
{
    if (FREAD(varian_header, 1, BYTES_PER_WORD*VARIAN_FILE_HEADER, fp))
	RETURN_ERROR_MSG("cannot read Varian file header");
    
    return  OK;
}

static long get_varian_long(int offset)
{
    long x;

    if (swapped)
    {
	SWAP(varian_header[offset], varian_header[offset+3], char);
	SWAP(varian_header[offset+1], varian_header[offset+2], char);
    }

    x = *((long *) (varian_header + offset));

    return  x;
}

static short get_varian_short(int offset)
{
    short x;

    if (swapped)
	SWAP(varian_header[offset], varian_header[offset+1], char);

    x = *((short *) (varian_header + offset));

    return  x;
}

static Status check_varian_header(String error_msg)
{
    long nblocks, ntraces, np, ebytes, tbytes, /*bbytes,*/ d;
    short status;
    FILE *fp;

    CHECK_OPEN_FOR_BINARY_READING(fp, file);

    sprintf(error_msg, "in \"%s\": ", file);
    error_msg += strlen(error_msg);

    CHECK_STATUS(get_varian_header(fp, error_msg));

    FCLOSE(fp);

    /* numbers in parentheses are offsets into header */
    nblocks = get_varian_long(0);   /* number of blocks in file */
    ntraces = get_varian_long(4);   /* number of traces per block */
    np = get_varian_long(8);	    /* number of elements per trace */
    ebytes = get_varian_long(12);   /* number of bytes per element */
    tbytes = get_varian_long(16);   /* number of bytes per trace */
    /*bbytes = get_varian_long(20);*/   /* number of bytes per block */
	    /* short starting at 24 is software version */
    status = get_varian_short(26);  /* status of whole file */
	    /* long starting at 28 is number of block headers */

    if (ntraces != 1)
	RETURN_ERROR_MSG("can only process Varian data with #traces = 1");

    if (np != npoints[0])
    {
	sprintf(error_msg,
	    "Varian header has #points = %d, par file has %d in dim 1",
	    np, npoints[0]);
	return  ERROR;
    }

    VECTOR_PRODUCT(d, npoints+1, ndim-1);
    if (d != nblocks)
    {
	printf("Warning: Varian header has #points = %d, par file has %d in >= dim 2",
	    nblocks, d);
    }

    if (tbytes != (ebytes * np))
	RETURN_ERROR_MSG("Varian header has inconsistent byte and point counts");

    if (!(status & (1<<0)))
	RETURN_ERROR_MSG("Varian header has no data");

    if (status & (1<<3))
	integer = FALSE;
    else
	integer = TRUE;

    if (integer)
    {
	if (ebytes == 2)
	{
	    if (status & (1<<2))
		RETURN_ERROR_MSG("Varian header has inconsistent byte information");
	}
	else if (ebytes == 4)
	{
	    if (!(status & (1<<2)))
		RETURN_ERROR_MSG("Varian header has inconsistent byte information");
	}
	else
	{
	    sprintf(error_msg,
		"Varian header has bytes per element %d, should be 2 or 4",
		ebytes);
	    return  ERROR;
	}
    }
    else /* !integer, i.e. floating point */
    {
	if (ebytes != 4)
	    RETURN_ERROR_MSG("Varian header has inconsistent byte information");
    }

    byte_size = ebytes;

    return  OK;
}

static Status check_par_params(String error_msg)
{
    int i;
    Line msg;

    if (!file)
        NOT_FOUND("file");

    if (ndim == 0)
        NOT_FOUND("ndim");

    if (swapped && big_endian)
    {
	sprintf(error_msg,
		"in \"%s\" have 'swap' and 'big_endian', only one allowed",
		par_file);
	return  ERROR;
    }

    if (swapped && little_endian)
    {
	sprintf(error_msg,
		"in \"%s\" have 'swap' and 'little_endian', only one allowed",
		par_file);
	return  ERROR;
    }

    if (big_endian && little_endian)
    {
	sprintf(error_msg,
	    "in \"%s\" have 'big_endian' and 'little_endian', only one allowed",
	    par_file);
	return  ERROR;
    }

#ifdef BIG_ENDIAN_DATA
    if (!little_endian && is_little_endian())
	swapped = !swapped;
#else /* !BIG_ENDIAN_DATA */
#ifndef LITTLE_ENDIAN_DATA
    if (big_endian && is_little_endian())
	swapped = TRUE;
#endif /* !LITTLE_ENDIAN_DATA */
#endif /* BIG_ENDIAN_DATA */

#ifdef LITTLE_ENDIAN_DATA
    if (!big_endian && is_big_endian())
	swapped = !swapped;
#else /* !LITTLE_ENDIAN_DATA */
#ifndef BIG_ENDIAN_DATA
    if (little_endian && is_big_endian())
	swapped = TRUE;
#endif /* !BIG_ENDIAN_DATA */
#endif /* LITTLE_ENDIAN_DATA */

    if (blocks && blocked)
    {
	sprintf(error_msg,
		"in \"%s\" have 'blocks', but data already blocked", par_file);
	return  ERROR;
    }

    if (deflated && !blocked)
    {
	sprintf(error_msg,
		"in \"%s\" have non-blocked data, but deflated", par_file);
	return  ERROR;
    }

    for (i = 0; i < ndim; i++)
    {
        if (dim_found[i] == FALSE)
        {
            sprintf(msg, "dim %d", i+1);
            NOT_FOUND(msg);
        }

        if (npoints[i] == 0)
            NOT_FOUND_FOR_DIM("npts", i);

        if (blocked && (block_size[i] == 0))
            NOT_FOUND_FOR_DIM("block", i);
    }

    if (have_varian)
    {
	for (i = 1; i < ndim; i++)
	{
	    if (npoints[i] % 2)
	    {
		sprintf(error_msg,
			"npts #%d = %d, must be even for varian data",
							i+1, npoints[i]);
		return  ERROR;
	    }
	}

	CHECK_STATUS(check_varian_header(error_msg));
    }

    return  OK;
}

Status read_par_file(String name, Par_info *par_info, String error_msg)
{
    int i, nblocks, size;

    free_par_memory();

    par_file = name;

    ndim = 0;
    dim = -1;
    swapped = integer = blocked = FALSE;
    big_endian = little_endian = FALSE;
    header = 0;
    deflated = reflated = FALSE;
    level = 0;
    blocks = FALSE;
    have_varian = FALSE;
    byte_size = BYTES_PER_WORD;
    param_dim = -1;

    CHECK_STATUS(parse_file(par_file, par_table, TRUE, error_msg));
    CHECK_STATUS(check_par_params(error_msg));

#ifdef FLIP_SWAP
    swapped = !swapped;
#endif /* FLIP_SWAP */

    par_info->file = file;
    par_info->ndim = ndim;
    par_info->block_size = block_size;
    par_info->npoints = npoints;
    par_info->ref = ref;
    par_info->swapped = swapped;
    par_info->integer = integer;
    par_info->blocked = blocked;
    par_info->header = header;
    par_info->deflated = deflated;
    par_info->level = level;
    par_info->byte_size = byte_size;

    if (deflated)
    {
	VECTOR_PRODUCT(size, block_size, ndim);

	nblocks = 1;
	for (i = 0; i < ndim; i++)
	    nblocks *= BLOCK(npoints[i], block_size[i]);

/*  make dir_size multiple of size (needed for Ansig)  */
	par_info->dir_size = BLOCK(nblocks, size);
	par_info->dir_size *= size;
    }
    else
    {
	par_info->dir_size = 0;
    }

    par_info->param_dim = param_dim;
    par_info->params = params;

    return  OK;
}

FILE *open_par_file(String name, String file, String error_msg)
{
    char *ptr;
    Line default_name;
    FILE *fp = NULL;

    if (!name)
    {
#define  PAR_FILE_EXTENSION  ".par"
	sprintf(default_name, "%s%s", file, PAR_FILE_EXTENSION);

	name = default_name;
	if (ptr = strrchr(name, DIRECTORY_SYMBOL))
	{
	    ptr++;

	    if (OPEN_FOR_WRITING(fp, ptr))
		printf("Could not open par file '%s', using full path\n", ptr);
	}
    }

    if (!fp)
    {
	ptr = name;

	if (OPEN_FOR_WRITING(fp, ptr))
	{
	    sprintf(error_msg, "opening par file '%s'", ptr);
	    return  (FILE *) NULL;
	}
    }

    fprintf(fp, "! this file = %s\n", ptr);

    return  fp;
}

void write_par_comment(FILE *fp, String comment)
{
    fprintf(fp, "! %s\n", comment);
}

void write_par_info(FILE *fp, Par_info *par_info)
{
    int i, j;

    fprintf(fp, "\n");
    fprintf(fp, "ndim %d\n", par_info->ndim);
    fprintf(fp, "file %s\n", par_info->file);

#ifdef WRITE_ENDIAN_PAR

#ifdef BIG_ENDIAN_DATA
    fprintf(fp, "big_endian\n");
#else /* !BIG_ENDIAN_DATA */
#ifndef LITTLE_ENDIAN_DATA
    if (is_big_endian())
	fprintf(fp, "big_endian\n");
#endif /* !LITTLE_ENDIAN_DATA */
#endif /* BIG_ENDIAN_DATA */

#ifdef LITTLE_ENDIAN_DATA
    fprintf(fp, "little_endian\n");
#else /* !LITTLE_ENDIAN_DATA */
#ifndef BIG_ENDIAN_DATA
    if (is_little_endian())
	fprintf(fp, "little_endian\n");
#endif /* !BIG_ENDIAN_DATA */
#endif /* LITTLE_ENDIAN_DATA */

#endif /* WRITE_ENDIAN_PAR */

    if (par_info->level > 0)
    {
	if (par_info->deflated)
	    fprintf(fp, "deflate %f\n", par_info->level);
	else /* reflated, supposedly */
	    fprintf(fp, "reflate %f\n", par_info->level);
    }

    for (i = 0; i < par_info->ndim; i++)
    {
        fprintf(fp, "\n");
        fprintf(fp, "dim %d\n", i+1);
        fprintf(fp, "npts %d\n", par_info->npoints[i]);

	if (par_info->blocked)
            fprintf(fp, "block %d\n", par_info->block_size[i]);
	
	write_ref(fp, &(par_info->ref[i]));

	if (i == par_info->param_dim)
	{
	    fprintf(fp, "params");
	    for (j = 0; j < par_info->npoints[i]; j++)
		fprintf(fp, " %6.4f", par_info->params[j]);
	    fprintf(fp, "\n");
	}
    }
}

Status write_par_file(String name, Par_info *par_info, String error_msg)
{
    FILE *fp;

    if ((fp = open_par_file(name, par_info->file, error_msg)) == NULL)
	return  ERROR;

    write_par_info(fp, par_info);

    fclose(fp);

    return  OK;
}

void find_block_sizes(int ndim, int size_of_block, int *npoints,
					int *block_size, Bool initialized)
{
    int s, i, i_max;
    float r, r_max;

    if (!initialized)
    {
	if (size_of_block < (1 << ndim))  /* unlikely! */
	{
	    for (i = 0; i < ndim; i++)
		block_size[i] = 1;
	}
	else
	{
	    for (i = 0; i < ndim; i++)
		block_size[i] = 2;
	}
    }

    VECTOR_PRODUCT(s, block_size, ndim);

    for( ; s < size_of_block; s *= 2)
    {
	r_max = 0;
	for (i = 0; i < ndim; i++)
	{
	    r = ((float) npoints[i]) / ((float) block_size[i]);
	    if (r > r_max)
	    {
		i_max = i;
		r_max = r;
	    }
	}

	block_size[i_max] *= 2;
    }
}

void set_varian()
{
    allow_varian = TRUE;
}

void get_varian(Varian_info *varian_info)
{
    if (allow_varian)
    {
	varian_info->have_varian = have_varian;
	varian_info->dim_order = dim_order;
	varian_info->file_header = file_header;
	varian_info->block_header = block_header;

	if (byte_size == 2) /* number of words should be doubled */
	    varian_info->block_header *= 2;
    }
    else
    {
	varian_info->have_varian = FALSE;
    }
}
