#include "fit_func.h"

#include "data.h"
#include "fit.h"
#include "nonlinear_model.h"
#include "param.h"
#include "peak.h"

static int current_peak_set = -1;
static int npeak_sets = 0;
static Peak_set **peak_sets;

static int method = LINEAR_FIT;

static int nalloc = 0;
static float *x = NULL;
static float *y = NULL;
static float *y_fit = NULL;

static int nalloc_params = 0;
static float *params_fit = NULL;

static void free_fit_memory()
{
    FREE(x, float);
    FREE(y, float);
    FREE(y_fit, float);

    nalloc = 0;
}

static void free_params_fit_memory()
{
    FREE(params_fit, float);

    nalloc_params = 0;
}

static Status alloc_fit_memory(int n, String error_msg)
{
    sprintf(error_msg, "allocating peak fitting memory");

    if (n < nalloc)
	return  OK;

    free_fit_memory();

    MALLOC(x, float, n);
    MALLOC(y, float, n);
    MALLOC(y_fit, float, n);

    nalloc = n;

    return  OK;
}

static Status alloc_params_fit_memory(int n, String error_msg)
{
    sprintf(error_msg, "allocating peak fitting parameter memory");

    if (n < nalloc_params)
	return  OK;

    free_params_fit_memory();

    MALLOC(params_fit, float, n);

    nalloc_params = n;

    return  OK;
}

static void update_fit_data()
{
    if (*fit_method)
	method = atoi(fit_method);
}

void fit_set_peak_sets(int current, int nsets, Peak_set **sets)
{
    current_peak_set = current;
    npeak_sets = nsets;
    peak_sets = sets;
}

void fit_select_set_func(int *peak_set)
{
    if ((*peak_set < 0) || (*peak_set >= npeak_sets)
				|| (*peak_set == current_peak_set))
	*peak_set = current_peak_set = -1;
    else
	current_peak_set = *peak_set;
}

static void cycle_peak_set(int cycle)
{
    if ((npeak_sets < 2) || (current_peak_set == -1))
        return;
 
    current_peak_set = (current_peak_set + npeak_sets + cycle) % npeak_sets;
}

void fit_next_set_func(int *peak_set)
{
    cycle_peak_set(1);
    *peak_set = current_peak_set;
}

void fit_previous_set_func(int *peak_set)
{
    cycle_peak_set(-1);
    *peak_set = current_peak_set;
}

static Status get_data_param(Peak *peak, String error_msg)
{
    int data_set;
    Data_info *data_info;

    if (!data_name_exists(peak->name, &data_set))
    {
	sprintf(error_msg, "peak '%s' has no associated plane", peak->name);
	return  ERROR;
    }

    data_info = get_data_set(data_set);

    if (!data_info->have_param)
    {
	sprintf(error_msg, "peak '%s' has no parameter", peak->name);
	return  ERROR;
    }

    peak->param = data_info->param;

    return  OK;
}

Status fit_peaks_func(String error_msg)
{
    Peak_set *ps;
    int i, nreq, npeaks;
    Peak **peaks, *p;
    float chisq;

    if ((current_peak_set < 0) || (current_peak_set >= npeak_sets))
	RETURN_ERROR_MSG("invalid current peak set");

    ps = peak_sets[current_peak_set];
    npeaks = ps->npeaks;
    peaks = ps->peaks;

    update_fit_data();

    nreq = get_method_nparams(method);

    if (npeaks < nreq)
	RETURN_ERROR_MSG("not enough peaks to do fit");

    for (i = 0; i < npeaks; i++)
	CHECK_STATUS(get_data_param(peaks[i], error_msg));

    CHECK_STATUS(alloc_fit_memory(npeaks, error_msg));
    CHECK_STATUS(alloc_params_fit_memory(nreq, error_msg));

    for (i = 0; i < npeaks; i++)
    {
	p = peaks[i];

	x[i] = p->param;
	y[i] = p->extremum;
    }

    CHECK_STATUS(fit_data(method, npeaks, x, y, params_fit, y_fit, &chisq, error_msg));

    if (nreq >= 1)
	ps->param1 = params_fit[0];
    else
	ps->param1 = 0;

    if (nreq >= 2)
	ps->param2 = params_fit[1];
    else
	ps->param2 = 0;

    if (nreq >= 3)
	ps->param3 = params_fit[2];
    else
	ps->param3 = 0;

    ps->method = method;
    ps->chisq = chisq;

    for (i = 0; i < npeaks; i++)
    {
	p = peaks[i];

	p->have_fit = TRUE;
	p->fit_extremum = y_fit[i];
    }

    return  OK;
}

float calculate_fit_func(float x)
{
    int nreq;
    Peak_set *ps;
/*
    Line error_msg;
*/

    if ((current_peak_set < 0) || (current_peak_set >= npeak_sets))
	return 0; /* arbitrary */

    ps = peak_sets[current_peak_set];

    update_fit_data();

    nreq = get_method_nparams(method);

/*  if (alloc_params_fit_memory(nreq, error_msg) == ERROR)
	return 0;  arbitrary */

    if (nreq >= 1)
	params_fit[0] = ps->param1;

    if (nreq >= 2)
	params_fit[1] = ps->param2;

    if (nreq >= 3)
	params_fit[2] = ps->param3;

    return calculate_fit(method, x, params_fit);
}
