#include "extract.h"

#include "par.h"
#include "ref.h"
#include "utility.h"

static int ndim;
static int *npoints;
static int thickness[MAX_NDIM];
static float point[MAX_NDIM];
static int dim_extr[MAX_NDIM];
static int first[MAX_NDIM];
static int last[MAX_NDIM];
static int step[MAX_NDIM];

static Ref_info *ref;

static Status do_non_dirns(String error_msg)
{
    int i, n, p, pt;
    float t;
    int dim_done[MAX_NDIM];

    ZERO_VECTOR(dim_done, ndim);

    for (i = 0; i < EXTRACT_DIM; i++)
	dim_done[dim_extr[i]] = 1;

    for (i = 0; i < ndim; i++)
    {
	if (dim_done[i] == 1)
	    continue;

	p = NEAREST_INTEGER(point[i]);
	n = npoints[i];

	if (p < 1)
	{
	    sprintf(error_msg, "component %d of point is %d < 1", i+1, p);
	    return  ERROR;
	}

	if (p > n)
	{
	    sprintf(error_msg, "component %d of point is %d > %d = npts",
								i+1, p, n);
	    return  ERROR;
	}

	t = 0.5 * ((float) thickness[i]);

	pt = p - t;
	first[i] = MAX(0, pt);

	pt = pt + thickness[i];
	last[i] = MIN(pt, n);

	step[i] = 1;
    }

    return  OK;
}

static Status do_param(Par_info *par_info, Param_info *param_info,
						String error_msg)
{
    int i;
    int dim = par_info->param_dim;

    if (!par_info->params)
    {
	param_info->have_param = FALSE;

	return  OK;
    }

    dim = par_info->param_dim;

    for (i = 0; i < EXTRACT_DIM; i++)
    {
	if (dim_extr[i] == dim)
	{
	    param_info->have_param = FALSE;

	    return  OK;
	}
    }

    param_info->have_param = TRUE;
    param_info->param = par_info->params[first[dim]];
	/* choice of first is arbitrary */
	/* should really only be using param with thickness == 1 */

    return  OK;
}

Status get_extract_param(Extract_param *param, Ref_info **ref_info,
			Size_info *size_info, File_info *file_info,
			Extract_info *extract_info, Param_info *param_info,
			String error_msg)
{
    int i, j;
    int f[EXTRACT_DIM], l[EXTRACT_DIM], s[EXTRACT_DIM];
    Par_info par_info;
    Bool thickness_flag;
    Line msg;

    CHECK_STATUS(read_par_file(param->param_file, &par_info, error_msg));

    if (par_info.deflated)
	printf("Warning: data file '%s' is deflated\n", par_info.file);
/*
	RETURN_ERROR_MSG("data file cannot be deflated");
*/

    *ref_info = ref = par_info.ref;

    ndim = size_info->ndim = par_info.ndim;
    npoints = size_info->npoints = par_info.npoints;
    size_info->block_size = par_info.block_size;

    file_info->name = par_info.file;
    file_info->swapped = par_info.swapped;
    file_info->integer = par_info.integer;
    file_info->blocked = par_info.blocked;
    file_info->header = par_info.header;
    file_info->deflated = par_info.deflated;
    file_info->dir_size = par_info.dir_size;
    file_info->byte_size = par_info.byte_size;

    extract_info->dim_extr = dim_extr;
    extract_info->first = first;
    extract_info->last = last;
    extract_info->step = step;

    if (get_integers(EXTRACT_DIM, dim_extr, param->dirns_string) == ERROR)
    {
	sprintf(error_msg, "plane directions must be a vector of length %d",
								EXTRACT_DIM);
	return  ERROR;
    }

    for (i = 0; i < EXTRACT_DIM; i++)
    {
	if (dim_extr[i] < 1)
	{
	    sprintf(error_msg, "plane direction #%d is %d, must be >= 1",
							i+1, dim_extr[i]);
	    return  ERROR;
	}

	if (dim_extr[i] > ndim)
	{
	    sprintf(error_msg,
	    "plane direction #%d is %d, must be <= #dimensions, which is %d",
						i+1, dim_extr[i], ndim);
	    return  ERROR;
	}
    }

    for (i = 0; i < EXTRACT_DIM; i++)
    {
	for (j = i+1; j < EXTRACT_DIM; j++)
	{
	    if (dim_extr[i] == dim_extr[j])
	    {
		sprintf(error_msg,
		    "plane directions #%d and #%d are the same, i.e. %d",
						i+1, j+1, dim_extr[i]);
	    	return  ERROR;
	    }
	}
    }

    for (i = 0; i < EXTRACT_DIM; i++)
	dim_extr[i]--;

    if (get_some_integers(ndim, EXTRACT_DIM, dim_extr, thickness,
					param->thick_string) == ERROR)
    {
	thickness_flag = FALSE;

	if (get_integers(1, thickness, param->thick_string) == ERROR)
	{
	    sprintf(error_msg, "must have integral plane thickness");
	    return  ERROR;
	}

	for (i = 1; i < ndim; i++)
	    thickness[i] = thickness[0];
    }
    else
    {
	thickness_flag = TRUE;
    }

    for (i = 0; i < EXTRACT_DIM; i++)
	thickness[dim_extr[i]] = 1;

    for (i = 0; i < ndim; i++)
    {
	if (thickness[i] < 1)
	{
	    if (thickness_flag)
		sprintf(msg, "#%d ", i+1);
	    else
		msg[0] = 0;

	    sprintf(error_msg, "plane thickness %sis %d, must be >= 1", 
							msg, thickness[i]);
	    return  ERROR;
	}
    }

    if (get_some_floats(ndim, EXTRACT_DIM, dim_extr, point,
						param->point_string) == ERROR)
    {
	sprintf(error_msg, "plane point must be a vector of length %d", ndim);
	return  ERROR;
    }

    convert_to_points(param->ref_type, ndim, npoints, ref, point);

    CHECK_STATUS(do_non_dirns(error_msg));

    if (param->subset_type == SUBSET_ALL)
    {
	for (i = 0; i < EXTRACT_DIM; i++)
	{
	    j = dim_extr[i];
	    first[j] = 0;
	    last[j] = npoints[j];
	    step[j] = 1;
	}
    }
    else /* == SUBSET_SOME */
    {
    	if (get_integers(EXTRACT_DIM, f, param->first_string) == ERROR)
    	{
	    sprintf(error_msg, "first points must be a vector of length %d",
							EXTRACT_DIM);
	    return  ERROR;
    	}

    	if (get_integers(EXTRACT_DIM, l, param->last_string) == ERROR)
    	{
	    sprintf(error_msg, "last points must be a vector of length %d",
							EXTRACT_DIM);
	    return  ERROR;
    	}

    	if (get_integers(EXTRACT_DIM, s, param->step_string) == ERROR)
    	{
	    sprintf(error_msg, "steps must be a vector of length %d",
							EXTRACT_DIM);
	    return  ERROR;
    	}

    	for (i = 0; i < EXTRACT_DIM; i++)
    	{
	    j = dim_extr[i];
	    first[j] = f[i] - 1;
	    last[j] = l[i];
	    step[j] = s[i];

	    if (f[i] < 1)
	    {
	    	sprintf(error_msg, "first point #%d is %d, must be >= 1",
								i+1, f[i]);
	    	return  ERROR;
	    }

	    if (l[i] < f[i])
	    {
	    	sprintf(error_msg,
		"last point #%d is %d, must be >= first point #%d, which is %d",
						i+1, l[i], i+1, f[i]);
	    	return  ERROR;
	    }

	    if (s[i] < 1)
	    {
	    	sprintf(error_msg, "step #%d is %d, must be >= 1",
								i+1, s[i]);
	    	return  ERROR;
	    }

	    if (l[i] > npoints[j])
	    {
	    	sprintf(error_msg,
		    "last point #%d is %d, must be <= npts #%d, which is %d",
					i+1, l[i], j+1, npoints[j]);
	    	return  ERROR;
	    }
    	}
    }

    CHECK_STATUS(do_param(&par_info, param_info, error_msg));

    return  OK;
}
