#include "display.h"

#include "color.h"

static unsigned long default_pixels[2];
static unsigned long *pixels;

static int npixels;
static int npixels_alloc = 0;

static XFontStruct *font_info = NULL;

void clear_region(Display *display, Drawable drawable, GC gc,
				int x, int y, int width, int height)
{
    XFillRectangle(display, drawable, gc, x, y,
			(unsigned int) width, (unsigned int) height);
}

void copy_region(Display *display, Drawable src, Drawable dest, GC gc,
	int src_x, int src_y, int width, int height, int dest_x, int dest_y)
{
    XCopyArea(display, src, dest, gc, src_x, src_y,
					width, height, dest_x, dest_y);
}

void copy_plane(Display *display, Drawable src, Drawable dest, GC gc,
	int src_x, int src_y, int width, int height, int dest_x, int dest_y)
{
    XCopyPlane(display, src, dest, gc, src_x, src_y,
					width, height, dest_x, dest_y, 1L);
}

GC create_gc(Display *display, Drawable drawable)
{
    XGCValues gcv;

    gcv.foreground = BlackPixelOfScreen(DefaultScreenOfDisplay(display));
    gcv.background = WhitePixelOfScreen(DefaultScreenOfDisplay(display));

    return  XCreateGC(display, drawable, GCForeground | GCBackground, &gcv);
}

void free_gc(Display *display, GC gc)
{
    XFreeGC(display, gc);
}

void free_pixmap(Display *display, Pixmap pixmap)
{
    if (pixmap)
    	XFreePixmap(display, pixmap);
}

Pixmap create_pixmap(Display *display, Window window,
				int width, int height, int depth)
{
    return  XCreatePixmap(display, window,
	(unsigned int) width, (unsigned int) height, (unsigned int) depth);
}

void display_line(Display *display, Drawable drawable, GC gc,
					int x1, int y1, int x2, int y2)
{
    XDrawLine(display, drawable, gc, x1, y1, x2, y2);
}

/*
void display_center_text(Display *display, Drawable drawable, GC gc,
					int x, int y, String text)
{
    XDrawString(display, drawable, gc, x, y, text, strlen(text));
}
*/

void display_text(Display *display, Drawable drawable, GC gc,
				int x, int y, float a, float b, String text)
{
    int w = 0, h = 0;

    if (font_info)
    {
	int direction, ascent, descent;
	XCharStruct overall;

	XTextExtents(font_info, text, strlen(text), &direction,
					&ascent, &descent, &overall);

	w = a * overall.width;
	h = b * (overall.ascent + overall.descent);
    }

    XDrawString(display, drawable, gc, x-w, y+h, text, strlen(text));
}

void display_rectangle(Display *display, Drawable drawable, GC gc,
					int x1, int y1, int x2, int y2)
{
    unsigned int w, h;

    if (x1 > x2)
	SWAP(x1, x2, int);

    if (y1 > y2)
	SWAP(y1, y2, int);

    w = x2 - x1;  h = y2 - y1;

    XDrawRectangle(display, drawable, gc, x1, y1, w, h);
}

void display_circle(Display *display, Drawable drawable, GC gc,
						int x, int y, int r)
{
    static int angle1 = 0, angle2 = 360*64;

    XDrawArc(display, drawable, gc, x-r, y-r, 2*r, 2*r, angle1, angle2);
}

void set_gc_function(Display *display, GC gc, int function)
{
    XSetFunction(display, gc, function);
}

void set_gc_color(Display *display, GC gc, int color)
{
    if (color < npixels)
	XSetForeground(display, gc, pixels[color]);
    else
	XSetForeground(display, gc, pixels[BLACK]);
}

void set_gc_white(Display *display, GC gc)
{
    XSetForeground(display, gc,
		WhitePixelOfScreen(DefaultScreenOfDisplay(display)));
}

void set_gc_black(Display *display, GC gc)
{
    XSetForeground(display, gc,
		BlackPixelOfScreen(DefaultScreenOfDisplay(display)));
}

void set_gc_dash(Display *display, GC gc, int ndashes, char *dash_list)
{
    int dash_offset = 0;
    XGCValues gcv;

    gcv.line_style = LineOnOffDash;
    XChangeGC(display, gc, GCLineStyle, &gcv);
    XSetDashes(display, gc, dash_offset, dash_list, ndashes);
}

void free_pixels()
{
    FREE(pixels, unsigned long);
    npixels_alloc = npixels = 0;
}

static Status allocate_pixels(int n)
{
    MALLOC(pixels, unsigned long, n);
    npixels_alloc = npixels = n;

    return  OK;
}

void initialize_display(Display *display)
{
    int i, ncolors;
    String name;
    Colormap cmap = DefaultColormapOfScreen(DefaultScreenOfDisplay(display));
    XColor col, unused;

    if (npixels_alloc > 0)
	return;

    ncolors = get_ncolors();

    if (allocate_pixels(ncolors) == ERROR)
    {
	free_pixels();
	pixels = default_pixels;
	npixels = 2;

	default_pixels[BLACK] =
			BlackPixelOfScreen(DefaultScreenOfDisplay(display));
	default_pixels[WHITE] =
			WhitePixelOfScreen(DefaultScreenOfDisplay(display));

	return;
    }

    for (i = 0; i < npixels; i++)
    {
	name = get_color_name(i);

        if (!XAllocNamedColor(display, cmap, name, &col, &unused))
        {
            if (i != WHITE)
                pixels[i] = BlackPixelOfScreen(DefaultScreenOfDisplay(display));            else
                pixels[i] = WhitePixelOfScreen(DefaultScreenOfDisplay(display));        }
        else
        {
            pixels[i] = col.pixel;
    	}
    }
}

void initialize_font(Display *display)
{
    if (!font_info)
	font_info = XLoadQueryFont(display, "9x15");
}
