#include "block.h"

#include "block_io.h"

#define  TIMER_UPDATES  32

static int ndim;
static int *npoints_in;
static int *block_size;

static float *data;

static int *dim_extr;
static int *first;
static int *last;
static int *step;

static int ndim_rest;
static int npts_rest;
static int base_point;
static int jump_in0;
static int jump_in1;
static int jump_out;
static int offset_in;
static int offset_out;
static int dim0;
static int dim1;
static int npts0;
static int npts1;
static int npoints_to_do;
static int nblocks_to_do;
static int size_of_block;

static int dim_rest[MAX_NDIM];
static int npoints_out[MAX_NDIM];
static int nblocks[MAX_NDIM];
static int npts[MAX_NDIM];
static int low[MAX_NDIM];
static int base_block[MAX_NDIM];
static int cum_block_size[MAX_NDIM];
static int cum_nblocks[MAX_NDIM];
static int cum_nblocks_to_do[MAX_NDIM];
static int cum_npts_rest[MAX_NDIM];
static int cum_block_size_rest[MAX_NDIM];
static int array[MAX_NDIM];

static Bool dim_extracted[MAX_NDIM];

static int nwork = 0;
static float *work;

static int ndirectory = 0;
static int *directory;

static Block_IO block_io;

static void init_arrays()
{
    int i, j, b, n;

    ndim_rest = ndim - 2;

    dim0 = dim_extr[0];
    dim1 = dim_extr[1];

    for (i = 0; i < ndim; i++)
	dim_extracted[i] = FALSE;

    dim_extracted[dim0] = TRUE;
    dim_extracted[dim1] = TRUE;

    for (i = 0, j = 0; i < ndim; i++)
    {
	if (!dim_extracted[i])
	    dim_rest[j++] = i;
    }

    for (i = 0; i < ndim; i++)
	npoints_out[i] = (last[i] - first[i] + step[i] - 1) / step[i];

    BLOCKS(nblocks, npoints_in, block_size, ndim);
    CUMULATIVE(cum_nblocks, nblocks, n, ndim);
    CUMULATIVE(cum_block_size, block_size, size_of_block, ndim);

    nblocks_to_do = 1;
    for (i = 0; i < ndim; i++)
    {
	base_block[i] = first[i] / block_size[i];
	cum_nblocks_to_do[i] = nblocks_to_do;
	b = (last[i] - 1) / block_size[i];
	nblocks_to_do *= b - base_block[i] + 1;
    }

    for (i = 0; i < ndim_rest; i++)
    {
	j = dim_rest[i];
	cum_block_size_rest[i] = cum_block_size[j];
    }

    npoints_to_do = npoints_out[dim0] * npoints_out[dim1];

    jump_in0 = cum_block_size[dim0] * step[dim0];
}

void free_block_memory()
{
    FREE(work, float);

    nwork = 0;
}

static Status alloc_block_memory()
{
    if (size_of_block > nwork)
    {
	free_block_memory();
	MALLOC(work, float, size_of_block);

	nwork = size_of_block;
    }

    return  OK;
}

void free_directory_memory()
{
    FREE(directory, int);

    ndirectory = 0;
}

static Status alloc_directory_memory(int dir_size)
{
    if (dir_size > ndirectory)
    {
	free_directory_memory();
	MALLOC(directory, int, dir_size);

	ndirectory = dir_size;
    }

    return  OK;
}

static void do_plane(int plane_to_do)
{
    int i, j, i0, i1;
    float *d, *w;

    ARRAY_OF_INDEX(array, plane_to_do, cum_npts_rest, ndim_rest);

    for (i = 0; i < ndim_rest; i++)
    {
	j = dim_rest[i];
	array[i] *= step[j];
    }

    INDEX_OF_ARRAY(offset_in, array, cum_block_size_rest, ndim_rest);
    offset_in += base_point;

    d = data + offset_out;
    w = work + offset_in;

    for (i1 = 0; i1 < npts1; i1++)
    {
	for (i0 = 0; i0 < npts0; i0++)
	{
	    *(d++) += *w;
	    w += jump_in0;
	}

	d += jump_out;
	w += jump_in1;
    }
}

static Status do_block(int block_to_do, String error_msg)
{
    int i, j, m, n0, n1, block;

    ARRAY_OF_INDEX(array, block_to_do, cum_nblocks_to_do, ndim);
    ADD_VECTORS(array, array, base_block, ndim);
    INDEX_OF_ARRAY(block, array, cum_nblocks, ndim);

    for (i = 0; i < ndim; i++)
    {
	m = array[i]*block_size[i] - first[i];

/*  second formula for n0 gives 1 when m < 1 and not 0 because   */
/*  C rounds upwards when converting a negative float to an int  */

	if (m < 1)
	    n0 = 0;
	else
	    n0 = 1 + (m - 1) / step[i];

/*  m >= - (block_size[i] - 1), so do not have above problem  */

	n1 = 1 + (m + block_size[i] - 1) / step[i];
	n1 = MIN(npoints_out[i], n1);

	low[i] = n0;
	npts[i] = n1 - n0;

	if (npts[i] == 0)
	    return  OK;
    }

    for (i = 0; i < ndim; i++)
	array[i] = (first[i] + low[i]*step[i]) % block_size[i];

    INDEX_OF_ARRAY(base_point, array, cum_block_size, ndim);

    npts_rest = 1;
    for (i = 0; i < ndim_rest; i++)
    {
	j = dim_rest[i];
	cum_npts_rest[i] = npts_rest;
	npts_rest *= npts[j];
    }

    npts0 = npts[dim0];
    npts1 = npts[dim1];

    offset_out = low[dim0] + low[dim1]*npoints_out[dim0];

    jump_in1 = cum_block_size[dim1] * step[dim1] - npts0 * jump_in0;
    jump_out = npoints_out[dim0] - npts0;

    CHECK_STATUS(read_file_block(&block_io, block, work, error_msg));

    for (i = 0; i < npts_rest; i++)
	do_plane(i);

    return  OK;
}

Status block_process(Size_info *size_info, Store_info *store_info,
			File_info *file_info, Extract_info *extract_info,
			Timer_funcs *timer_funcs, String error_msg)
{
    int i, update_freq;
    float fraction;
    Line message;
    Status status = OK;
    char *ptr;

    ndim = size_info->ndim;
    npoints_in = size_info->npoints;
    block_size = size_info->block_size;

    data = store_info->data;

    dim_extr = extract_info->dim_extr;
    first = extract_info->first;
    last = extract_info->last;
    step = extract_info->step;

    if (!(file_info->blocked))
/*
        RETURN_ERROR_MSG("input file must be blocked");
*/
    {
	block_size[0] = npoints_in[0];

	for (i = 1; i < ndim; i++)
	    block_size[i] = 1;
    }

    init_arrays();

    if (alloc_block_memory() == ERROR)
        RETURN_ERROR_MSG("allocating memory for block");

    if (file_info->deflated &&
		(alloc_directory_memory(file_info->dir_size) == ERROR))
        RETURN_ERROR_MSG("allocating memory for directory");

    block_io.name = file_info->name;
    block_io.file = file_info->file;
    block_io.swapped = file_info->swapped;
    block_io.integer = file_info->integer;
    block_io.deflated = file_info->deflated;
    block_io.header = file_info->header;
    block_io.dir_size = file_info->dir_size;
    block_io.directory = directory;
    block_io.block_size = size_of_block;
    block_io.byte_size = file_info->byte_size;

    CHECK_STATUS(init_block_read(&block_io, error_msg));

    ZERO_VECTOR(data, npoints_to_do);

    if ((ptr = strrchr(file_info->name, DIRECTORY_SYMBOL)) == NULL)
	ptr = file_info->name;
    else
	ptr++;

    sprintf(message, "Reading %s", ptr);

    (*(timer_funcs->start_timer))(message);

    update_freq = nblocks_to_do / TIMER_UPDATES;
    update_freq = MAX(update_freq, 1);

    for (i = 0; i < nblocks_to_do; i++)
    {
	if (!(i % update_freq))
	{
	    fraction = ((float) i) / ((float) nblocks_to_do);

	    if ((*(timer_funcs->update_timer))(fraction) == ABORT)
	    {
		status = OTHER;
		break;
	    }
	}

	if ((status = do_block(i, error_msg)) == ERROR)
	    break;
    }

    (*(timer_funcs->stop_timer))(status);

    return  status;
}
