#include "region.h"

#include "data.h"
#include "input.h"
#include "ref.h"
#include "utility.h"

#define  SMALL_RANGE  (1.0e-4)

static void data_range(float *dmin, float *dmax, int n, float *data,
						float scale, float offset)
{
    int i;
    float min, max;

    min = LARGE_FLOAT;
    max = - LARGE_FLOAT;

    for (i = 0; i < n; i++)
    {
	min = MIN(min, data[i]);
	max = MAX(max, data[i]);
    }

    min = scale*min + offset;
    max = scale*max + offset;

    *dmin = MIN(*dmin, min);
    *dmax = MAX(*dmax, max);
}

static Status fit_x_data(int ref_type, float *x, String error_msg)
{
    int i, ndata_sets = get_ndata_sets();
    float xmin, xmax, scale, offset, min, max;
    Data_info *info = get_data_info();

    if (ndata_sets == 0)
    {
	x[0] = 1;  x[1] = 10;  /* arbitrary */
	return  OK;
    }

    xmin = LARGE_FLOAT;
    xmax = - LARGE_FLOAT;

    for (i = 0; i < ndata_sets; i++)
    {
	scale = info[i].property.x_scale;
	offset = info[i].property.x_offset;

	if (info[i].x)
	{
	    data_range(&xmin, &xmax, info[i].x->ndata,
			(float *) (info[i].x->data), scale, offset);
	}
	else
	{
	    if ((ref_type != REF_POINTS) && !(info[i].ref))
		/* will never happen since all data comes from read_par */
	    {
		sprintf(error_msg,
		  "cannot use %s because '%s' missing referencing information",
				ref_names[ref_type], get_data_name(i));
		return  ERROR;
	    }

	    min = scale + offset;
	    max = scale * (info[i].y->ndata) + offset;

	    convert_range_from_points(ref_type, 1, &(info[i].y->ndata),
						info[i].ref, &min, &max);
	    xmin = MIN(xmin, min);
	    xmax = MAX(xmax, max);
	}
    }

    x[0] = xmin;
    x[1] = xmax;

    return  OK;
}

static Status fit_y_data(float *y, String error_msg)
{
    int i, ndata_sets = get_ndata_sets();
    float ymin, ymax, scale, offset;
    Data_info *info = get_data_info();

    if (ndata_sets == 0)
    {
	y[0] = 1;  y[1] = 10;  /* arbitrary */
	return  OK;
    }

    ymin = LARGE_FLOAT;
    ymax = - LARGE_FLOAT;

    for (i = 0; i < ndata_sets; i++)
    {
	scale = info[i].property.y_scale;
	offset = info[i].property.y_offset;

	data_range(&ymin, &ymax, info[i].y->ndata,
			(float *) (info[i].y->data), scale, offset);
    }

    y[0] = ymin;
    y[1] = ymax;

    return  OK;
}

static Status fit_region(float *range, String region, String msg,
							String error_msg)
{
    float d;

    if (get_floats(DISPLAY_DIM, range, region) == ERROR)
    {
	sprintf(error_msg, "need lower and upper values for %s region", msg);
	return  ERROR;
    }

    d = range[1] - range[0];

    if (d < SMALL_RANGE)
    {
	sprintf(error_msg, "%s region range = %7.2e, must be > %7.2e",
							msg, d, SMALL_RANGE);
	return  ERROR;
    }

    return  OK;
}

#define  FIT_ADDITIONAL  0.03

Status find_region(int ref_type, Region_info *info, String error_msg)
{
    float d;

    if (info->x_fit == FIT_DATA)
    {
	CHECK_STATUS(fit_x_data(ref_type, info->x, error_msg));

	d = info->x[1] - info->x[0];
	d = MAX(d, SMALL_RANGE);
	d *= FIT_ADDITIONAL;
	    
	info->x[1] += d;
	info->x[0] -= d;

    }
    else /* (info->x_fit == FIT_REGION) */
    {
	CHECK_STATUS(fit_region(info->x, info->x_region, "x", error_msg));
    }

    if (info->y_fit == FIT_DATA)
    {
	CHECK_STATUS(fit_y_data(info->y, error_msg));

	d = info->y[1] - info->y[0];
	d = MAX(d, SMALL_RANGE);
	d *= FIT_ADDITIONAL;

	info->y[1] += d;
	info->y[0] -= d;
    }
    else /* (info->y_fit == FIT_REGION) */
    {
	CHECK_STATUS(fit_region(info->y, info->y_region, "y", error_msg));
    }

    return  OK;
}
