#include "display_popup.h"

#include "color.h"
#include "display.h"
#include "draw.h"
#include "event_handle.h"
#include "object.h"
#include "object_popup.h"
#include "param.h"
#include "region_popup.h"
#include "script.h"
#include "utility.h"
#include "window.h"

#define  MIN_WIDTH	50
#define  MIN_HEIGHT	50

#define  UP_DOWN_SCALE	0.7

static Widget display_popup = NULL;
static Widget display_area;

static Display *display_display = (Display *) NULL;
static Window display_window;
static GC display_gc = (GC) NULL;
static Pixmap display_pixmap = (Pixmap) NULL;
static Drawable display_drawable;

static int width = MIN_WIDTH;
static int height = MIN_HEIGHT;
static int depth = 1;

static float ax, bx, ay, by;
static Draw_info draw_info;

static Bool have_valid_display = FALSE;

static Global_info *global_info = NULL;

static Bool have_limit0 = FALSE;
static Bool have_limit1 = FALSE;

static int limit0;
static int limit1;

void update_display_params()
{
    if (display_popup)
    {
	get_widget_size(display_area, &width, &height);
	sprintf(display_width, "%d", width);
	sprintf(display_height, "%d", height);
    }
    else
    {
	if (*display_width)
	    width = atoi(display_width);

	width = MAX(width, MIN_WIDTH);

	if (*display_height)
	    height = atoi(display_height);

	height = MAX(height, MIN_HEIGHT);
    }
}

static Bool inside_limits(int x, int *begin, int *end)
{
    if (have_limit0 && have_limit1)
    {
	*begin = MIN(limit0, limit1);
	*end = MAX(limit0, limit1);
    }
    else if (have_limit0)
    {
	*begin = limit0;
	*end = width - 1;
    }
    else if (have_limit1)
    {
	*begin = 0;
	*end = limit1;
    }
    else
    {
	*begin = 0;
	*end = width - 1;
    }

    if (x < *begin)
	return  FALSE;

    if (x > *end)
	return  FALSE;

    return  TRUE;
}

void display_limits(int *size, int *begin, int *end)
{
    (void) inside_limits(0, begin, end);
    *size = width;
}

static void do_limits()
{
    if (have_limit0)
	display_line(display_display, display_window, display_gc,
						limit0, 0, limit0, height);

    if (have_limit1)
	display_line(display_display, display_window, display_gc,
						limit1, 0, limit1, height);
}

static void do_crosshair(int x, int y)
{
    display_line(display_display, display_window, display_gc, 0, y, width, y);
    display_line(display_display, display_window, display_gc, x, 0, x, height);
}

static void clear_display()
{
    set_gc_color(display_display, display_gc, WHITE);
    clear_region(display_display, display_pixmap, display_gc,
						0, 0, width, height);
}

static void show_display(int x, int y)
{
    copy_region(display_display, display_pixmap, display_window, display_gc,
						0, 0, width, height, 0, 0);

    if (global_info)
    {
	set_gc_color(display_display, display_gc,
				global_info->color[CROSSHAIR_OBJECT]);
	do_limits();

	if ((x >= 0) &&
		(global_info->visibility[CROSSHAIR_OBJECT] == VISIBILITY_ON))
	    do_crosshair(x, y);
    }
}

static void start_display(Generic_ptr data)
{
    display_drawable = display_pixmap;
    clear_display();
}

static void end_display()
{
    show_display(-1, -1);
}

static void new_display_range(float x0, float y0, float x1, float y1, Bool clip)
{
    ax = width / (x1 - x0);  bx = - ax * x0;
    ay = - height / (y1 - y0);  by = - ay * y1;
	/* y formulas different from x ones because y = 0 on top of screen */
}

#define  CONVERT_X(x)  (ax*(x) + bx)
#define  CONVERT_Y(y)  (ay*(y) + by)

static void do_display_line(float x0, float y0, float x1, float y1)
{
    int w0, w1, h0, h1;

    w0 = CONVERT_X(x0);
    w1 = CONVERT_X(x1);
    h0 = CONVERT_Y(y0);
    h1 = CONVERT_Y(y1);

    display_line(display_display, display_drawable, display_gc, w0, h0, w1, h1);
}

static void do_display_text(String text, float x, float y, float a, float b)
{
    int w, h;

    w = CONVERT_X(x);
    h = CONVERT_Y(y);

    display_text(display_display, display_drawable, display_gc,
							w, h, a, b, text);
}

static void set_display_color(int color)
{
    set_gc_color(display_display, display_gc, color);
}

static void set_display_font(String name, int size)
{
}

static void set_display_line_style(int line_style)
{
}

static Draw_funcs draw_funcs = { SCREEN_DISPLAY, (Generic_ptr) NULL,
	start_display, end_display, new_display_range,
	do_display_line, do_display_text, set_display_color,
	set_display_font, set_display_line_style };

Status do_display(int start, String error_msg)
{
    have_valid_display = FALSE;

    initialize_properties();
    CHECK_STATUS(run_script(start, error_msg));

    CHECK_STATUS(do_drawing(start, &draw_info, &draw_funcs, error_msg));

    have_valid_display = TRUE;

    return  OK;
}

static void display_expose_callback(Widget widget, Callback_ptr data,
							Callback_ptr cbs)
{
    show_display(-1, -1);
}

static void display_resize_callback(Widget widget, Callback_ptr data,
							Callback_ptr cbs)
{
    Line error_msg;

    free_pixmap(display_display, display_pixmap);
    update_display_params();
    display_pixmap = create_pixmap(display_display, display_window,
						width, height, depth);

    if (have_valid_display)
    {
	if (do_display(0, error_msg) == ERROR)
	    ERROR_AND_RETURN(error_msg);
    }
    else
    {
    	clear_display();
	show_display(-1, -1);
    }
}

static void button_press(unsigned int button, int x, int y, Time time)
{
    int begin[DISPLAY_DIM], end[DISPLAY_DIM];
    int point[DISPLAY_DIM], size[DISPLAY_DIM];
    Line error_msg;
    static Bool done[] = { TRUE, FALSE };

    if (button == Button1)
    {
	if (have_limit0 || have_limit1)
	{
	    size[0] = width;

	    if (inside_limits(x, begin, end))
		change_region(size, begin, end, done, EXPAND_REGION);
	    else
		change_region(size, begin, end, done, CONTRACT_REGION);

	    have_limit0 = have_limit1 = FALSE;

	    if (do_display_drawing(0, error_msg) == ERROR)
		ERROR_AND_RETURN(error_msg);

	    show_display(x, y);
	}
    }
    else if (button == Button2)
    {
	have_limit0 = have_limit1 = FALSE;
	show_display(x, y);
    }
    else if (button == Button3)
    {
	size[0] = width;  size[1] = height;
	point[0] = x;  point[1] = height - 1 - y;

	find_point_stats(size, point);
    }
}

static void button_motion(unsigned int button, int x, int y, Time time)
{
    show_display(x, y);
}

static void button_leave(Button button, int x, int y, Time time)
{
    show_display(-1, -1);
}

static void key_press(Key_symbol key, int x, int y, Time time)
{
    int size[DISPLAY_DIM], begin[DISPLAY_DIM], end[DISPLAY_DIM];
    float r, a[LIMITS_DIM], b[LIMITS_DIM];
    Line error_msg;
    static Bool done[] = { FALSE, TRUE };
 
    if ((key == UP_KEY) || (key == DOWN_KEY))
    {
	size[1] = height;

	if (get_region(a, b, error_msg) == ERROR)
	    ERROR_AND_RETURN(error_msg);

	r = (height - 1) / (b[1] - b[0]);

	begin[1] = r * b[0] * (UP_DOWN_SCALE-1);
	end[1] = r * (b[1]*UP_DOWN_SCALE - b[0]);
    }

    if (key == UP_KEY)
    {
	change_region(size, begin, end, done, EXPAND_REGION);

	if (do_display_drawing(0, error_msg) == ERROR)
	    ERROR_AND_RETURN(error_msg);
    }
    else if (key == DOWN_KEY)
    {
	change_region(size, begin, end, done, CONTRACT_REGION);

	if (do_display_drawing(0, error_msg) == ERROR)
	    ERROR_AND_RETURN(error_msg);
    }
    else if (key == LEFT_KEY)
    {
	have_limit0 = TRUE;
	limit0 = x;
    }
    else if (key == RIGHT_KEY)
    {
	have_limit1 = TRUE;
	limit1 = x;
    }
    else
    {
	return;
    }

    show_display(x, y);
}

static void quit_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    free_pixmap(display_display, display_pixmap);
    free_gc(display_display, display_gc);
    destroy_widget(display_popup);
    display_popup = (Widget) NULL;
    display_gc = (GC) NULL;
}

static Event_handlers handlers =
{   button_press, no_button_handler, button_motion,
    no_button_handler, button_leave, key_press, no_key_handler   };

static void create_display_popup(Widget parent)
{
    Drawing_area_info drawing_info;
    static Event_handle handle =
    { ButtonPressMask | PointerMotionMask | LeaveWindowMask | KeyPressMask,
							&handlers };

    update_display_params();

    display_popup = create_iconifiable_popup(parent, "Display");
    CHECK_WIDGET_WARNING(display_popup);
    change_delete_protocol(display_popup, quit_callback);

    drawing_info.width = width;
    drawing_info.height = height;
    drawing_info.expose = display_expose_callback;
    drawing_info.resize = display_resize_callback;

    display_area = create_drawing(display_popup, &drawing_info);
    CHECK_WIDGET_DESTROY_WARNING(display_area, display_popup);

    realize_widget(display_popup);

    add_event_handler(display_area, &handle);

    display_display = WIDGET_DISPLAY(display_area);
    initialize_display(display_display);
    initialize_font(display_display);

    depth = DISPLAY_DEPTH(display_display);

    display_window = WIDGET_WINDOW(display_area);
    display_pixmap = create_pixmap(display_display, display_window,
							width, height, depth);
    display_gc = create_gc(display_display, display_window);

    global_info = get_global_info();

    clear_display();
}

void display_popup_callback(Widget parent, Callback_ptr data, Callback_ptr cbs)
{
    Line error_msg;

    if (!display_popup)
        create_display_popup(parent);

    if (display_popup)
    {
	popup(display_area);

	if (have_valid_display)
	{
	    if (do_display(0, error_msg) == ERROR)
		ERROR_AND_RETURN(error_msg);
	}
    }
}

Status do_display_drawing(int start, String error_msg)
{
    have_valid_display = FALSE;

    initialize_properties();
    CHECK_STATUS(run_script(start, error_msg));

    if (region_apply(&(draw_info.ref_type), draw_info.x, draw_info.y,
							error_msg) == ERROR)
	ERROR_AND_RETURN_ERROR(error_msg);

    if (objects_apply(error_msg) == ERROR)
	ERROR_AND_RETURN_ERROR(error_msg);

    if (!display_popup)
	create_display_popup(get_topshell());

    if (!display_popup)
	return  OK;  /* error message already displayed */

    popup(display_area);

    CHECK_STATUS(do_drawing(start, &draw_info, &draw_funcs, error_msg));

    have_valid_display = TRUE;

    return  OK;
}
