#include "arrange.h"

#include "parser.h"
#include "script.h"

#define  ARRANGE_V	2

#define  ARRANGE_V1	2
#define  ARRANGE_V2	3

#define  ARRANGE_VR	4
#define  ARRANGE_VI	5

#define  SETUP_ARRANGE(m) \
	 {   n = store[INPUT_X]->ndata; \
	     store_int_to_float(store[INPUT_X]); \
	     data_in = (float *) (store[INPUT_X]->data); \
	     store_type_float(store[OUTPUT_X]); \
	     CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], m, error_msg)); \
	     data_out = (float *) (store[OUTPUT_X]->data); \
	     complex = (store[INPUT_X]->data_type & PARSER_COMPLEX);   }

static int n;
static int complex;
static float *data_in;
static float *data_out;

static Status range_check(int v1, int v2, int n, String error_msg)
{
    if (v1 < 0)
    {
	sprintf(error_msg, "value #1 = %d, must be > 0", v1+1);
	return  ERROR;
    }

    if (v1 >= v2)
    {
	sprintf(error_msg, "value #1 = %d must be <= value #2 = %d", v1+1, v2);
	return  ERROR;
    }

    if (v2 > n)
    {
	sprintf(error_msg, "value #2 = %d, must be <= %d", v2, n);
	return  ERROR;
    }

    return  OK;
}

static Status bound_check(int v, int n, String error_msg)
{
    if (v < 0)
    {
	sprintf(error_msg, "value = %d, must be > 0", v+1);
	return  ERROR;
    }

    if (v >= n)
    {
	sprintf(error_msg, "value = %d, must be <= %d", v+1, n);
	return  ERROR;
    }

    return  OK;
}

static Status do_cycle(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, j, v;

    store_float_to_int(store[ARRANGE_V]);

    v = *((int *) store[ARRANGE_V]->data);

    SETUP_ARRANGE(n);

    if (v < 0)
    {
	v = - v;
	v %= n;
	v = - v;
    }
    else
    {
	v %= n;
    }

    if (complex)
    {
	n *= 2;
	v *= 2;
    }

    for (i = 0; i < n; i++)
    {
	j = (i + n + v) % n;
	data_out[j] = data_in[i];
    }

    return  OK;
}

static Status do_lower(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int v;

    store_float_to_int(store[ARRANGE_V]);

    v = *((int *) store[ARRANGE_V]->data);
    v--;

    SETUP_ARRANGE(n-v);

    CHECK_STATUS(bound_check(v, n, error_msg));

    if (complex)
    {
	n *= 2;
	v *= 2;
    }

    n -= v;
    data_in += v;

    COPY_VECTOR(data_out, data_in, n);

    return  OK;
}

static Status do_range(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int v1, v2;

    store_float_to_int(store[ARRANGE_V1]);
    store_float_to_int(store[ARRANGE_V2]);

    v1 = *((int *) store[ARRANGE_V1]->data);
    v2 = *((int *) store[ARRANGE_V2]->data);
    v1--;

    SETUP_ARRANGE(v2-v1);

    CHECK_STATUS(range_check(v1, v2, n, error_msg));

    if (complex)
    {
	v1 *= 2;
	v2 *= 2;
    }

    v2 -= v1;
    data_in += v1;

    COPY_VECTOR(data_out, data_in, v2);

    return  OK;
}

static Status do_reverse(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, j, k, step;

    SETUP_ARRANGE(n);

    if (complex)
	step = 2;
    else
	step = 1;

    n *= step;

    for (i = 0; i < n; i += step)
    {
	j = n - step - i;

	for (k = 0; k < step; k++)
	    data_out[i+k] = data_in[j+k];
    }

    return  OK;
}

static Status do_shift(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int v;

    store_float_to_int(store[ARRANGE_V]);

    v = *((int *) store[ARRANGE_V]->data);

    SETUP_ARRANGE(n+v);

    CHECK_STATUS(bound_check(v, v+1, error_msg));

    if (complex)
    {
	n *= 2;
	v *= 2;
    }

    ZERO_VECTOR(data_out, v);

    data_out += v;

    COPY_VECTOR(data_out, data_in, n);

    return  OK;
}

static Status do_upper(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int v;

    store_float_to_int(store[ARRANGE_V]);

    v = *((int *) store[ARRANGE_V]->data);

    SETUP_ARRANGE(v);

    CHECK_STATUS(bound_check(v-1, n, error_msg));

    if (complex)
	v *= 2;

    COPY_VECTOR(data_out, data_in, v);

    return  OK;
}

static Status do_zerofill(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int m, v;

    store_float_to_int(store[ARRANGE_V]);

    v = *((int *) store[ARRANGE_V]->data);

    if ((v < 0) || (v > 4))
	RETURN_ERROR_MSG("can only zerofill between 0 and 4 times");

    v = (1 << v);

    SETUP_ARRANGE(v*n);

    m = v * n;

    if (complex)
    {
	n *= 2;
	m *= 2;
    }

    COPY_VECTOR(data_out, data_in, n);

    data_out += n;
    m -= n;

    ZERO_VECTOR(data_out, m);

    return  OK;
}

static Status do_scale(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int v1, v2;
    float vr;

    store_float_to_int(store[ARRANGE_V1]);
    store_float_to_int(store[ARRANGE_V2]);
    store_int_to_float(store[ARRANGE_VR]);

    v1 = *((int *) store[ARRANGE_V1]->data);
    v2 = *((int *) store[ARRANGE_V2]->data);
    v1--;

    vr = *((float *) store[ARRANGE_VR]->data);

    SETUP_ARRANGE(n);

    CHECK_STATUS(range_check(v1, v2, n, error_msg));

    COPY_VECTOR(data_out, data_in, v1);
    SCALE_VECTOR(data_out+v1, data_in+v1, vr, v2 - v1);
    COPY_VECTOR(data_out+v2, data_in+v2, n-v2);

    return  OK;
}

static Status do_scale2(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, v1, v2;
    float vr, vi;

    store_float_to_int(store[ARRANGE_V1]);
    store_float_to_int(store[ARRANGE_V2]);
    store_int_to_float(store[ARRANGE_VR]);
    store_int_to_float(store[ARRANGE_VI]);

    v1 = *((int *) store[ARRANGE_V1]->data);
    v2 = *((int *) store[ARRANGE_V2]->data);
    v1--;

    vr = *((float *) store[ARRANGE_VR]->data);
    vi = *((float *) store[ARRANGE_VI]->data);

    SETUP_ARRANGE(n);

    CHECK_STATUS(range_check(v1, v2, n, error_msg));

    n *= 2;  v1 *= 2;  v2 *= 2;

    COPY_VECTOR(data_out, data_in, v1);

    for (i = v1; i < v2; i += 2)
    {
	data_out[i] = vr*data_in[i] - vi*data_in[i+1];
	data_out[i+1] = vr*data_in[i+1] + vi*data_in[i];
    }

    COPY_VECTOR(data_out+v2, data_in+v2, n-v2);

    return  OK;
}

static Status do_set(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, v1, v2;
    float vr;

    store_float_to_int(store[ARRANGE_V1]);
    store_float_to_int(store[ARRANGE_V2]);
    store_int_to_float(store[ARRANGE_VR]);

    v1 = *((int *) store[ARRANGE_V1]->data);
    v2 = *((int *) store[ARRANGE_V2]->data);
    v1--;

    vr = *((float *) store[ARRANGE_VR]->data);

    SETUP_ARRANGE(n);

    CHECK_STATUS(range_check(v1, v2, n, error_msg));

    COPY_VECTOR(data_out, data_in, v1);

    for (i = v1; i < v2; i++)
	data_out[i] = vr;

    COPY_VECTOR(data_out+v2, data_in+v2, n-v2);

    return  OK;
}

static Status do_set2(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, v1, v2;
    float vr, vi;

    store_float_to_int(store[ARRANGE_V1]);
    store_float_to_int(store[ARRANGE_V2]);
    store_int_to_float(store[ARRANGE_VR]);
    store_int_to_float(store[ARRANGE_VI]);

    v1 = *((int *) store[ARRANGE_V1]->data);
    v2 = *((int *) store[ARRANGE_V2]->data);
    v1--;

    vr = *((float *) store[ARRANGE_VR]->data);
    vi = *((float *) store[ARRANGE_VI]->data);

    SETUP_ARRANGE(n);

    CHECK_STATUS(range_check(v1, v2, n, error_msg));

    n *= 2;  v1 *= 2;  v2 *= 2;

    COPY_VECTOR(data_out, data_in, v1);

    for (i = v1; i < v2; i += 2)
    {
	data_out[i] = vr;
	data_out[i+1] = vi;
    }

    COPY_VECTOR(data_out+v2, data_in+v2, n-v2);

    return  OK;
}

static Status do_mask_ppmm(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i;

    SETUP_ARRANGE(n);

    if (complex)
	n *= 2;

    if (n % 4)
    {
	sprintf(error_msg,
		"number of (real) points = %d, must be multiple of 4", n);
	return  ERROR;
    }

    for (i = 0; i < n; i += 4)
    {
	data_out[i] = data_in[i];
	data_out[i+1] = data_in[i+1];
	data_out[i+2] = - data_in[i+2];
	data_out[i+3] = - data_in[i+3];
    }

    return  OK;
}

static Status do_mirror(int v, int nstore, Parser_store **store,
							String error_msg)
{
    int i, m;

    SETUP_ARRANGE(2*n-v);

    n *= 2;  v *= 2;
    m = n - v;

/*  first copy input data to upper part of output data  */

    COPY_VECTOR(data_out+m, data_in, n);

/*  then mirror conjugate input data to lower part of output data  */

    for (i = 0; i < m; i += 2)
    {
        data_out[m-i-2] = data_in[i+v];
        data_out[m-i-1] = - data_in[i+v+1];
    }

    return  OK;
}

static Status do_mirror_zero(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    return  do_mirror(1, nstore, store, error_msg);
}

static Status do_mirror_half(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    return  do_mirror(0, nstore, store, error_msg);
}

Status init_cycle(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "cycle", do_cycle, error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_lower(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "lower", do_lower, error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_range(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "range", do_range, error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_reverse(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "reverse", do_reverse, error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_shift(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "shift", do_shift, error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_upper(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "upper", do_upper, error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_zerofill(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "zerofill", do_zerofill,
							error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_scale(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "scale", do_scale, error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_scale2(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "scale2", do_scale2, error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_set(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "set", do_set, error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_set2(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "set2", do_set2, error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_mask_ppmm(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "mask_ppmm", do_mask_ppmm,
							error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_mirror_zero(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "mirror_zero", do_mirror_zero,
							error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_mirror_half(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "mirror_half", do_mirror_half,
							error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}
