#include "arith.h"

#include "parser.h"
#include "script.h"

#define  INPUT_Y	2

#define  SMALL_NUMBER	10e-8
#define  SMALL_NUMBER2	(SMALL_NUMBER * SMALL_NUMBER)

static int nx;
static int ny;
static int n;
static int step;
static int mx;
static int my;
static int sx;
static int sy;
static float *x;
static float *y;
static float *z;

static Status setup_maths_store(Parser_store **store, String error_msg)
{
    store_int_to_float(store[INPUT_X]);
    store_int_to_float(store[INPUT_Y]);

    store_type_float(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], n, error_msg));

    x = (float *) (store[INPUT_X]->data);
    y = (float *) (store[INPUT_Y]->data);
    z = (float *) (store[OUTPUT_X]->data);

    return  OK;
}

static Status consistent_size(Parser_store **store, String error_msg)
{
    nx = store[INPUT_X]->ndata;
    ny = store[INPUT_Y]->ndata;

    if ((nx != ny) && (nx != 1) && (ny != 1))
    {
	sprintf(error_msg,
		"argument #1 has size %d, argument #2 has size %d", nx, ny);
	return  ERROR;
    }

    if (store[INPUT_X]->data_type & PARSER_REAL)
	sx = 0;
    else
	sx = 1;

    if (store[INPUT_Y]->data_type & PARSER_REAL)
	sy = 0;
    else
	sy = 1;

    step = MAX(sx, sy) + 1;

    if (nx == 1)
	mx = 0;
    else
	mx = sx + 1;

    if (ny == 1)
	my = 0;
    else
	my = sy + 1;

    n = MAX(nx, ny);

    return  OK;
}

static Status do_add(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, j;

    CHECK_STATUS(consistent_size(store, error_msg));
    CHECK_STATUS(setup_maths_store(store, error_msg));

    n *= step;

    for (i = 0; i < n; i += step, x += mx, y += my)
	for (j = 0; j < step; j++)
	    z[i+j] = x[j*sx] + y[j*sy];

    return  OK;
}

static Status do_subtract(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, j;

    CHECK_STATUS(consistent_size(store, error_msg));
    CHECK_STATUS(setup_maths_store(store, error_msg));

    n *= step;

    for (i = 0; i < n; i += step, x += mx, y += my)
	for (j = 0; j < step; j++)
	    z[i+j] = x[j*sx] - y[j*sy];

    return  OK;
}

static Status do_multiply(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, j;
    float a, b, c, d;

    CHECK_STATUS(consistent_size(store, error_msg));
    CHECK_STATUS(setup_maths_store(store, error_msg));

    n *= step;

    if ((sx == 0) || (sy == 0))
    {
	for (i = 0; i < n; i += step, x += mx, y += my)
	    for (j = 0; j < step; j++)
		z[i+j] = x[j*sx] * y[j*sy];
    }
    else
    {
	for (i = 0; i < n; i += 2, x += mx, y += my)
	{
	    a = x[0];  b = x[1];
	    c = y[0];  d = y[1];
	    z[i] = a*c - b*d;
	    z[i+1] = a*d + b*c;
	}
    }

    return  OK;
}

static Status do_divide(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, j;
    float a, b, c, d, e;

    CHECK_STATUS(consistent_size(store, error_msg));
    CHECK_STATUS(setup_maths_store(store, error_msg));

    n *= step;

    if ((sx == 0) && (sy == 0))
    {
	for (i = 0; i < n; i++, x += mx, y += my)
	{
	    if (ABS(y[0]) < SMALL_NUMBER)
	    {
		sprintf(error_msg, "point %d is too close to 0", i+1);
		return  ERROR;
	    }

	    z[i] = x[0] / y[0];
	}
    }
    else if (sy == 0)
    {
	for (i = 0; i < n; i += step, x += mx, y += my)
	{
	    if (ABS(y[0]) < SMALL_NUMBER)
	    {
		sprintf(error_msg, "point %d is too close to 0", i+1);
		return  ERROR;
	    }

	    for (j = 0; j < step; j++)
		z[i+j] = x[j] / y[0];
	}
    }
    else if (sx == 0)
    {
	for (i = 0; i < n; i += step, x += mx, y += my)
	{
	    c = y[0];  d = y[1];
	    e = c*c + d*d;

	    if (ABS(e) < SMALL_NUMBER2)
	    {
		sprintf(error_msg, "point %d is too close to 0", i+1);
		return  ERROR;
	    }

	    z[i] = (c * x[0]) / e;
	    z[i+1] = (-d * x[0]) / e;
	}
    }
    else
    {
	for (i = 0; i < n; i += 2, x += mx, y += my)
	{
	    a = x[0];  b = x[1];
	    c = y[0];  d = y[1];
	    e = c*c + d*d;

	    if (ABS(e) < SMALL_NUMBER2)
	    {
		sprintf(error_msg, "point %d is too close to 0", i+1);
		return  ERROR;
	    }

	    z[i] = (a*c + b*d) / e;
	    z[i+1] = (-a*d + b*c) / e;
	}
    }

    return  OK;
}

static void output_data_type(Parser_store **store)
{
    int type = PARSER_NONE;

    if ((store[INPUT_X]->data_type & PARSER_REAL) &&
				(store[INPUT_Y]->data_type & PARSER_REAL))
	type |= PARSER_REAL;
    else
	type |= PARSER_COMPLEX;

    if ((store[INPUT_X]->data_type & PARSER_SCALAR) &&
				(store[INPUT_Y]->data_type & PARSER_SCALAR))
	type |= PARSER_SCALAR;
    else
	type |= PARSER_ARRAY;

    type |= PARSER_INT | PARSER_FLOAT;

    store[OUTPUT_X]->data_type = type;
}

Status init_add(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "add", do_add, error_msg) == ERROR)
	return  ERROR;

    output_data_type(store);

    return  OK;
}

Status init_subtract(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "subtract", do_subtract,
							error_msg) == ERROR)
	return  ERROR;

    output_data_type(store);

    return  OK;
}

Status init_multiply(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "multiply", do_multiply,
							error_msg) == ERROR)
	return  ERROR;

    output_data_type(store);

    return  OK;
}

Status init_divide(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "divide", do_divide, error_msg) == ERROR)
	return  ERROR;

    output_data_type(store);

    return  OK;
}
