#include "script_fit.h"

#include "command.h"
#include "par.h"
#include "parse.h"
#include "ref.h"

static int ndim;
static int *block_size;
static int *npoints;
static int data_type[MAX_NDIM];

static Ref_info *ref;

static char *input_data_file;
static char *input_peak_file = NULL;
static char *output_peak_file = NULL;
static char *output_ansig_file = NULL;
static char *output_ideal_file = NULL;
static char *output_rest_file = NULL;
static char *par_ideal_file = NULL;
static char *par_rest_file = NULL;
static Bool swapped;
static Bool integer;
static Bool blocked;
static int header;
static Bool deflated;
static int dir_size;
static int byte_size;

static String script_file;

static Bool input_found;
static Bool range_found;

static int dim;
static Bool dim_found[MAX_NDIM];
static Bool script_found[MAX_NDIM];
static Bool freq_found[MAX_NDIM];

static Bool fixed_phase[MAX_NDIM];
static Bool fixed_decay[MAX_NDIM];

static float chisq;
static Bool subtract;
static Bool group;
static float level;
static int nfitted;
static int nparams;
static int dim_fitted[MAX_NDIM];
static Script script[MAX_NDIM];
static int npts[MAX_NDIM];
static int width[MAX_NDIM];
static Bool periodic[MAX_NDIM];
static Bool complex[MAX_NDIM];
static float freq_a[MAX_NDIM];
static float freq_b[MAX_NDIM];
static int first[MAX_NDIM];
static int last[MAX_NDIM];
static float phase[MAX_NDIM];
static float decay[MAX_NDIM];

static int parse_string[] = { PARSE_STRING };
static int parse_int[] = { PARSE_INT };
static int parse_int2[] = { PARSE_INT, PARSE_INT };
static int parse_float[] = { PARSE_FLOAT };
static int parse_float2[] = { PARSE_FLOAT, PARSE_FLOAT };
static int parse_filed[] = { PARSE_FILE };

#define  CHECK_DIM_NOT_THERE(string) \
	 {   if (dim >= 0) \
	     {   sprintf(error_msg, \
				"in \"%s\" '%s' should be before any 'dim'", \
				script_file, string);  return  ERROR;   }   }

#define  CHECK_DIM_THERE(string) \
	 {   if (dim < 0) \
	     {   sprintf(error_msg, "in \"%s\" '%s' found before any 'dim'", \
				script_file, string);  return  ERROR;   }   }

#define  CHECK_NPTS_THERE(string, dim) \
	 {   if (npts[dim] == 0) \
	     {   sprintf(error_msg, \
			"in \"%s\" for dim %d '%s' found before 'npts'", \
			script_file, dim+1, string);  return  ERROR;   }   }

#define  FOUND_TWICE(string) \
	 {   sprintf(error_msg, "in \"%s\" '%s' found twice", \
				script_file, string);  return  ERROR;   }

#define  FOUND_TWICE_FOR_DIM(string, dim) \
	 {   sprintf(error_msg, "in \"%s\" for dim %d '%s' found twice", \
			script_file, dim+1, string);  return  ERROR;   }

#define  FOUND_BEFORE(string1, string2) \
	 {   sprintf(error_msg, "in \"%s\" '%s' found before '%s'", \
			script_file, string1, string2);  return  ERROR;   }

#define  NOT_ALLOWED(string, condition) \
	 {   sprintf(error_msg, "in \"%s\" '%s' %s", \
			script_file, string, condition);  return  ERROR;   }

#define  NOT_ALLOWED_FOR_DIM(dim, string, condition) \
	 {   sprintf(error_msg, "in \"%s\" for dim %d '%s' %s", \
		script_file, dim+1, string, condition);  return  ERROR;   }

#define  NOT_FOUND(string) \
	 {   sprintf(error_msg, "in \"%s\" no '%s' found", \
				script_file, string);  return  ERROR;   }

#define  NOT_FOUND_FOR_DIM(string, dim) \
	 {   sprintf(error_msg, "in \"%s\" no '%s' found for dim %d", \
			script_file, string, dim+1);  return  ERROR;   }

static Status input_data_parse(Generic_ptr *var, String error_msg)
{
    int i;
    String input_par_file = (String) (*var);
    Par_info par_info;

    if (input_found)
	FOUND_TWICE("input_data");

    input_found = TRUE;

    CHECK_STATUS(read_par_file(input_par_file, &par_info, error_msg));

    input_data_file = par_info.file;
    ndim = par_info.ndim;
    npoints = par_info.npoints;
    block_size = par_info.block_size;
    ref = par_info.ref;
    swapped = par_info.swapped;
    integer = par_info.integer;
    blocked = par_info.blocked;
    header = par_info.header;
    deflated = par_info.deflated;
    dir_size = par_info.dir_size;
    byte_size = par_info.byte_size;

    if (par_info.deflated)
    {
/*
	sprintf(error_msg, "input data file \"%s\" must not be deflated",
							input_data_file);
	return  ERROR;
*/
	printf("Warning: input data file \"%s\" deflated\n", input_data_file);
    }

    if (!blocked)
    {
	sprintf(error_msg, "input data file \"%s\" must be blocked",
							input_data_file);
	return  ERROR;
    }

    for (i = 0; i < ndim; i++)
    {
	data_type[i] = REAL_DATA;
	dim_found[i] = FALSE;
	script_found[i] = FALSE;
	freq_found[i] = FALSE;

	npts[i] = 0;
	periodic[i] = TRUE;
	complex[i] = FALSE;
	freq_a[i] = npoints[i] / 2.0;  freq_b[i] = TWOPI / npoints[i];
	first[i] = 0;  last[i] = npoints[i];

	fixed_phase[i] = FALSE;
	fixed_decay[i] = FALSE;
    }

    return  OK;
}

static Status input_peak_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (!input_found)
	FOUND_BEFORE("input_peak", "input_data");

    if (input_peak_file)
	FOUND_TWICE("input_peak");

    CHECK_DIM_NOT_THERE("input_peak");

    sprintf(error_msg, "allocating memory while reading '%s'", script_file);
    MALLOC(input_peak_file, char, strlen(name)+1);

    strcpy(input_peak_file, name);

    return  OK;
}

static Status output_peak_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (!input_found)
	FOUND_BEFORE("output_peak", "input_data");

    if (output_peak_file)
	FOUND_TWICE("output_peak");

    CHECK_DIM_NOT_THERE("output_peak");

    sprintf(error_msg, "allocating memory while reading '%s'", script_file);
    MALLOC(output_peak_file, char, strlen(name)+1);

    strcpy(output_peak_file, name);

    return  OK;
}

static Status output_ansig_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (!input_found)
	FOUND_BEFORE("output_ansig", "input_data");

    if (output_ansig_file)
	FOUND_TWICE("output_ansig");

    CHECK_DIM_NOT_THERE("output_ansig");

    sprintf(error_msg, "allocating memory while reading '%s'", script_file);
    MALLOC(output_ansig_file, char, strlen(name)+1);

    strcpy(output_ansig_file, name);

    return  OK;
}

static Status output_ideal_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (!input_found)
	FOUND_BEFORE("output_ideal", "input_data");

    if (output_ideal_file)
	FOUND_TWICE("output_ideal");

    CHECK_DIM_NOT_THERE("output_ideal");

    sprintf(error_msg, "allocating memory while reading '%s'", script_file);
    MALLOC(output_ideal_file, char, strlen(name)+1);

    strcpy(output_ideal_file, name);

    return  OK;
}

static Status output_rest_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (!input_found)
	FOUND_BEFORE("output_rest", "input_data");

    if (output_rest_file)
	FOUND_TWICE("output_rest");

    CHECK_DIM_NOT_THERE("output_rest");

    sprintf(error_msg, "allocating memory while reading '%s'", script_file);
    MALLOC(output_rest_file, char, strlen(name)+1);

    strcpy(output_rest_file, name);

    return  OK;
}

static Status par_ideal_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (!input_found)
	FOUND_BEFORE("par_ideal", "input_data");

    if (par_ideal_file)
	FOUND_TWICE("par_ideal");

    CHECK_DIM_NOT_THERE("par_ideal");

    sprintf(error_msg, "allocating memory while reading '%s'", script_file);
    MALLOC(par_ideal_file, char, strlen(name)+1);

    strcpy(par_ideal_file, name);

    return  OK;
}

static Status par_rest_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (!input_found)
	FOUND_BEFORE("par_rest", "input_data");

    if (par_rest_file)
	FOUND_TWICE("par_rest");

    CHECK_DIM_NOT_THERE("par_rest");

    sprintf(error_msg, "allocating memory while reading '%s'", script_file);
    MALLOC(par_rest_file, char, strlen(name)+1);

    strcpy(par_rest_file, name);

    return  OK;
}

static Status chisq_parse(Generic_ptr *var, String error_msg)
{
    float x = *((float *) var[0]);

    if (!input_found)
	FOUND_BEFORE("chisq", "input_data");

    CHECK_DIM_NOT_THERE("chisq");

    if (chisq > 0)
	FOUND_TWICE("chisq");

    if (x <= 0)
	NOT_ALLOWED("chisq", "<= 0");

    chisq = x;

    return  OK;
}

static Status subtract_parse(Generic_ptr *var, String error_msg)
{
    if (!input_found)
	FOUND_BEFORE("subtract", "input_data");

    if (subtract)
	FOUND_TWICE("subtract");

    CHECK_DIM_NOT_THERE("subtract");

    subtract = TRUE;

    return  OK;
}

static Status group_parse(Generic_ptr *var, String error_msg)
{
    if (!input_found)
	FOUND_BEFORE("group", "input_data");

    if (group)
	FOUND_TWICE("group");

    CHECK_DIM_NOT_THERE("group");

    group = TRUE;

    return  OK;
}

static Status deflate_parse(Generic_ptr *var, String error_msg)
{
    float x = *((float *) var[0]);

    if (!input_found)
	FOUND_BEFORE("deflate", "input_data");

    CHECK_DIM_NOT_THERE("deflate");

    if (level > 0)
	FOUND_TWICE("deflate");

    if (x <= 0)
	NOT_ALLOWED("deflate", "<= 0");

    level = x;

    return  OK;
}

static Status dim_parse(Generic_ptr *var, String error_msg)
{
    int d = *((int *) var[0]);
    Line msg;

    dim = d;

    if (dim > ndim)
    {
	sprintf(error_msg, "in \"%s\" 'dim' = %d > 'ndim' = %d",
	                                        script_file, dim+1, ndim);
	return  ERROR;
    }

    if (dim < 1)
	NOT_ALLOWED("dim", "< 1");

    dim--;
    if (dim_found[dim] == TRUE)
    {
	sprintf(msg, "dim %d", dim+1);
	FOUND_TWICE(msg);
    }

    dim_found[dim] = TRUE;
    range_found = FALSE;

    width[dim] = 0;

    return  OK;
}

static Status npts_parse(Generic_ptr *var, String error_msg)
{
    int d = *((int *) var[0]);

    CHECK_DIM_THERE("npts");

    if (d < 1)
	NOT_ALLOWED_FOR_DIM(dim, "npts", "< 1");

    npts[dim] = d;

    return  OK;
}

static Status complex_parse(Generic_ptr *var, String error_msg)
{
    CHECK_DIM_THERE("complex");

    complex[dim] = TRUE;

    return  OK;
}

static Status width_parse(Generic_ptr *var, String error_msg)
{
    int d = *((int *) var[0]);

    CHECK_DIM_THERE("width");

    if (width[dim] > 0)
	FOUND_TWICE_FOR_DIM("width", dim);

    if (d < 1)
	NOT_ALLOWED_FOR_DIM(dim, "width", "< 1");

    if (d > (npoints[dim]/2 - 1))
	NOT_ALLOWED_FOR_DIM(dim, "width", "> #points/2 - 1");

    width[dim] = d;

    return  OK;
}

static Status freq_parse(Generic_ptr *var, String error_msg)
{
    float a = *((float *) var[0]);
    float b = *((float *) var[1]);

    CHECK_DIM_THERE("freq");

    if (freq_found[dim])
	FOUND_TWICE_FOR_DIM("freq", dim);

    freq_found[dim] = TRUE;

    freq_a[dim] = a - 1;
    freq_b[dim] = TWOPI * b / npoints[dim];

    if (b != 1)
	periodic[dim] = FALSE;

    return  OK;
}

static Status range_parse(Generic_ptr *var, String error_msg)
{
    int f = *((int *) var[0]);
    int l = *((int *) var[1]);

    CHECK_DIM_THERE("range");

    if (range_found)
        FOUND_TWICE_FOR_DIM("range", dim);

    if ((f < 1) || (f > l) || (l > npoints[dim]))
    {
	sprintf(error_msg,
"for dim %d 'range' has illegal arguments %d, %d (should be in range 1 to %d)",
                                                dim+1, f, l, npoints[dim]);
	return  ERROR;
    }

    range_found = TRUE;

    first[dim] = f - 1;
    last[dim] = l;

    return  OK;
}

static Status phase_parse(Generic_ptr *var, String error_msg)
{
    float phi = *((float *) var[0]);

    CHECK_DIM_THERE("phase");

    if (fixed_phase[dim])
        FOUND_TWICE_FOR_DIM("phase", dim);

    fixed_phase[dim] = TRUE;
    phase[dim] = RADIAN_SCALE * phi;

    return  OK;
}

static Status decay_parse(Generic_ptr *var, String error_msg)
{
    float d = *((float *) var[0]);

    CHECK_DIM_THERE("decay");

    if (fixed_decay[dim])
        FOUND_TWICE_FOR_DIM("decay", dim);

    if (d <= 0)
    {
	sprintf(error_msg,
		"for dim %d 'decay' has illegal argument %4.3f (should be > 0)",
		                                                dim+1, d);
	return  ERROR;
    }

    fixed_decay[dim] = TRUE;
    decay[dim] = d;  /* still needs transforming */

    return  OK;
}

static Status start_script(int *n, String msg, String error_msg)
{
    CHECK_DIM_THERE(msg);
    CHECK_NPTS_THERE(msg, dim);

    if (script_found[dim])
	FOUND_TWICE_FOR_DIM(msg, dim);

    script_found[dim] = TRUE;

    data_type[dim] = COMPLEX_DATA;  /* data starts out as complex */
    *n = 2 * npts[dim];  /* npts is number of complex points */

    script[dim].ndim = 1;
    script[dim].dims[0] = dim;

    dim_fitted[nfitted++] = dim;

    return  OK;
}

static Status end_script(int n, int nmax, String error_msg)
{
    script[dim].npts[0] = n;
    script[dim].npts_max = nmax;

    if (data_type[dim] == COMPLEX_DATA)
	n /= 2;

    if (n != npoints[dim])
    {
	sprintf(error_msg,
"for dim %d #points in input file (%d) inconsistent with script output (%d)",
						dim+1, npoints[dim], n);
	return  ERROR;
    }

    return  OK;
}

static Status script_parse(Generic_ptr *var, String error_msg)
{
    char *file = (char *) var[0];
    int n, nmax;
    Line err_msg;

    CHECK_STATUS(start_script(&n, "script", error_msg));

    if (read_command_file(&n, &nmax, data_type+dim, file, script+dim,
							err_msg) == ERROR)
    {
	sprintf(error_msg, "script for dimension %d, file '%s': %s",
						dim+1, file, err_msg);
	return  ERROR;
    }

    CHECK_STATUS(end_script(n, nmax, error_msg));

    return  OK;
}

static Status script_com_parse(Generic_ptr *var, String error_msg)
{
    FILE *fp = (FILE *) var[0];
    int n, nmax;
    Line file, err_msg;

    CHECK_STATUS(start_script(&n, "script_com", error_msg));

    sprintf(file, "%s script_com", script_file);

    if (read_command_subfile(&n, &nmax, data_type+dim, file, fp, script+dim,
							err_msg) == ERROR)
    {
	sprintf(error_msg, "script_com for dimension %d: %s", dim+1, err_msg);
	return  ERROR;
    }

    CHECK_STATUS(end_script(n, nmax, error_msg));

    return  OK;
}

static Status check_fit_validity(String error_msg)
{
    int i, j;

    if (!input_data_file)
	NOT_FOUND("input_data");

    if (!input_peak_file)
	NOT_FOUND("input_peak");

    if (chisq == 0)
	NOT_FOUND("chisq");

    nparams = 1;  /* magnitude */
    for (i = 0; i < nfitted; i++)
    {
	j = dim_fitted[i];

	if (width[j] == 0)
	    NOT_FOUND_FOR_DIM("width", j);

	if (npts[j] == 0)
	    NOT_FOUND_FOR_DIM("npts", j);

	if (fixed_decay[j])
	{
	    if (npts[j] == 1)
		decay[j] = 0;
	    else
		decay[j] = - log((double) decay[j]) / (npts[j] - 1);
	}

	nparams++;  /* center */

	if (!fixed_phase[j])
	    nparams++;

	if (!fixed_decay[j])
	    nparams++;
    }

    sprintf(error_msg, "in \"%s\": ", script_file);
    error_msg += strlen(error_msg);

    if (par_ideal_file && !output_ideal_file)
	RETURN_ERROR_MSG("cannot have 'par_ideal' without 'output_ideal'");

    if (par_rest_file && !output_rest_file)
	RETURN_ERROR_MSG("cannot have 'par_rest' without 'output_rest'");

    if (!output_peak_file && !output_ansig_file &&
				!output_ideal_file && !output_rest_file)
	RETURN_ERROR_MSG("must have some output file");

    if (nfitted == 0)
	RETURN_ERROR_MSG("must have at least one dim. fitted");

    if (nfitted != ndim)
    {
	if (output_rest_file)
	   RETURN_ERROR_MSG("cannot have 'output_rest' unless all dim. fitted");

	if (subtract)
	    RETURN_ERROR_MSG("cannot have 'subtract' unless all dim. fitted");
    }

    for (i = 0; i < nfitted; i++)
    {
	j = dim_fitted[i];

	if ((data_type[j] == COMPLEX_DATA) && !complex[j])
	{
	    sprintf(error_msg,
			"for dim %d script output complex, must be real", j+1);
	    return  ERROR;
	}
	else if ((data_type[j] == REAL_DATA) && complex[j])
	{
	    sprintf(error_msg,
			"for dim %d script output real, must be complex", j+1);
	    return  ERROR;
	}
    }

    return  OK;
}

static Parse_line fit_script_table[] =
{
    { "input_data",	1,	parse_string,		input_data_parse },
    { "input_peak",	1,	parse_string,		input_peak_parse },
    { "output_peak",	1,	parse_string,		output_peak_parse },
    { "output_ansig",	1,	parse_string,		output_ansig_parse },
    { "output_ideal",	1,	parse_string,		output_ideal_parse },
    { "output_rest",	1,	parse_string,		output_rest_parse },
    { "par_ideal",	1,	parse_string,		par_ideal_parse },
    { "par_rest",	1,	parse_string,		par_rest_parse },
    { "chisq",		1,	parse_float,		chisq_parse },
    { "subtract",	0,	(int *) NULL,		subtract_parse },
    { "group",		0,	(int *) NULL,		group_parse },
    { "deflate",	1,	parse_float,		deflate_parse },
    { "dim",		1,	parse_int,		dim_parse },
    { "npts",		1,	parse_int,		npts_parse },
    { "complex",	0,	(int *) NULL,		complex_parse },
    { "script",		1,	parse_string,		script_parse },
    { "script_com",	1,	parse_filed,		script_com_parse },
    { "width",		1,	parse_int,		width_parse },
    { "freq",		2,	parse_float2,		freq_parse },
    { "range",		2,	parse_int2,		range_parse },
    { "phase",		1,	parse_float,		phase_parse },
    { "decay",		1,	parse_float,		decay_parse },
    { (String) NULL,	0,	(int *) NULL,		no_parse_func }
};

Status read_script_file(String file, Size_info *size_info,
			Ref_info **p_ref, File_info *file_info,
			Fit_param *fit_param, String error_msg)
{
    script_file = file;

    FREE(input_peak_file, char);
    FREE(output_peak_file, char);
    FREE(output_ansig_file, char);
    FREE(output_ideal_file, char);
    FREE(output_rest_file, char);
    FREE(par_ideal_file, char);
    FREE(par_rest_file, char);

    input_found = FALSE;
    subtract = FALSE;
    group = FALSE;
    dim = -1;
    nfitted = 0;
    level = 0;

    CHECK_STATUS(parse_file(script_file, fit_script_table, TRUE, error_msg));

    CHECK_STATUS(check_fit_validity(error_msg));

    size_info->ndim = ndim;
    size_info->block_size = block_size;
    size_info->npoints = npoints;

    *p_ref = ref;

    file_info->input_data_file = input_data_file;
    file_info->input_peak_file = input_peak_file;
    file_info->output_peak_file = output_peak_file;
    file_info->output_ansig_file = output_ansig_file;
    file_info->output_ideal_file = output_ideal_file;
    file_info->output_rest_file = output_rest_file;
    file_info->par_ideal_file = par_ideal_file;
    file_info->par_rest_file = par_rest_file;
    file_info->swapped = swapped;
    file_info->integer = integer;
    file_info->blocked = blocked;
    file_info->header = header;
    file_info->deflated = deflated;
    file_info->dir_size = dir_size;
    file_info->byte_size = byte_size;

    fit_param->chisq = chisq;
    fit_param->subtract = subtract;
    fit_param->group = group;
    fit_param->level = level;
    fit_param->nfitted = nfitted;
    fit_param->dim_fitted = dim_fitted;
    fit_param->script = script;
    fit_param->npts = npts;
    fit_param->width = width;
    fit_param->periodic = periodic;
    fit_param->complex = complex;
    fit_param->freq_a = freq_a;
    fit_param->freq_b = freq_b;
    fit_param->first = first;
    fit_param->last = last;
    fit_param->fixed_phase = fixed_phase;
    fit_param->phase = phase;
    fit_param->fixed_decay = fixed_decay;
    fit_param->decay = decay;
    fit_param->nparams = nparams;

    return  OK;
}

Ref_info *get_reference()
{
    return  (ref + dim);
}
