#include "peak_fit.h"

#include "data.h"
#include "fit.h"
#include "group.h"
#include "output.h"
#include "ref.h"
#include "script_fit.h"

#define  MEGAWORD		(1024 * 1024)
#define  DEFAULT_STORE		(2 * MEGAWORD)

#define  NUMBER_TYPE_STRING	"N"

static void print_peak(int ndim, Bool subtract, int nfitted, int *dim_fitted,
	int *npoints, int *npts, FILE *fp, Ref_info *ref_info, Fit_info *info)
{
    int i, j;
    float ppm, d;

    fprintf(fp, "%1.0f", info->before.value);

    for (i = 0; i < ndim; i++)
    {
	ppm = info->before.position[i] + 1;
	convert_from_point(REF_PPM, npoints[i], ref_info+i, &ppm);

	fprintf(fp, "\t%d\t%4.3f", info->before.position[i]+1, ppm);
    }

    if (subtract)
    {
	fprintf(fp, "\t%1.0f", info->after.value);

	for (i = 0; i < ndim; i++)
	{
	    ppm = info->after.position[i] + 1;
	    convert_from_point(REF_PPM, npoints[i], ref_info+i, &ppm);

	    fprintf(fp, "\t%d\t%4.3f", info->after.position[i]+1, ppm);
	}
    }

    fprintf(fp, "\t%2.1f\t%3.2f", info->peak->magnitude, 100*info->peak->chisq);

    for (i = 0; i < nfitted; i++)
    {
	j = dim_fitted[i];
	ppm = info->peak->center[i] + 1;
	convert_from_point(REF_PPM, npoints[j], ref_info+j, &ppm);

	d = exp((double) (- (npts[j]-1) * info->peak->decay[i]));
/*
printf("i=%d, j=%d, npts=%d, dec=%3.2e, d=%3.2e\n", i, j, npts[j], info->peak->decay[i], d);
*/

	fprintf(fp, "\t%2.1f\t%4.3f\t%2.1f\t%4.3f",
				info->peak->phase[i] / RADIAN_SCALE,
				d, info->peak->center[i] + 1, ppm);
    }

    fprintf(fp, "\n");
}

static void output_peaks(int ndim, Bool subtract, int nfitted, int *dim_fitted,
			int *npoints, int *npts, FILE *fp,
			Ref_info *ref_info, int nfit, Fit_info **fit_info)
{
    int i, j;

    fprintf(fp, "extr");
    for (i = 0; i < ndim; i++)
	fprintf(fp, "\tpnt%d\tppm%d", i+1, i+1);

    if (subtract)
    {
	fprintf(fp, "\textrs");
	for (i = 0; i < ndim; i++)
	    fprintf(fp, "\tpnts%d\tppms%d", i+1, i+1);
    }

    fprintf(fp, "\tmagn\tchisq");
    for (i = 0; i < nfitted; i++)
    {
	j = dim_fitted[i] + 1;
	fprintf(fp, "\tphase%d\tdecay%d\tpntf%d\tppmf%d", j, j, j, j);
    }

    fprintf(fp, "\n");

    for (i = 0; i < 2*ndim+4*nfitted+3; i++)
    {
	if (i > 0)
	    fprintf(fp, "\t");

	fprintf(fp, NUMBER_TYPE_STRING);
    }

    if (subtract)
    {
	for (i = 0; i < 2*ndim+1; i++)
	{
	    fprintf(fp, "\t");

	    fprintf(fp, NUMBER_TYPE_STRING);
	}
    }

    fprintf(fp, "\n");

    for (i = 0; i < nfit; i++)
    {
	if (fit_info[i]->peak)
	    print_peak(ndim, subtract, nfitted, dim_fitted, npoints, npts,
						fp, ref_info, fit_info[i]);
    }
}

static void print_ansig(int ndim, Bool subtract, int nfitted, int *dim_fitted,
	int *npoints, FILE *fp, Ref_info *ref_info, Fit_info *info)
{
    int i, j;
    float ppm;

    fprintf(fp, "%1.0f\t%2.1f\t%3.2f", info->before.value,
				info->peak->magnitude, 100*info->peak->chisq);

    if (subtract)
	fprintf(fp, "\t%1.0f", info->after.value);

    fprintf(fp, "\n");

    for (i = 0; i < ndim; i++)
    {
	ppm = info->before.position[i] + 1;
	convert_from_point(REF_PPM, npoints[i], ref_info+i, &ppm);

	fprintf(fp, "\t%4.3f", ppm);
    }

    fprintf(fp, "\n");

    if (subtract)
    {
	for (i = 0; i < ndim; i++)
	{
	    ppm = info->after.position[i] + 1;
	    convert_from_point(REF_PPM, npoints[i], ref_info+i, &ppm);

	    fprintf(fp, "\t%4.3f", ppm);
	}

	fprintf(fp, "\n");
    }

    for (i = 0; i < nfitted; i++)
    {
	j = dim_fitted[i];
	ppm = info->peak->center[i] + 1;
	convert_from_point(REF_PPM, npoints[j], ref_info+j, &ppm);

	fprintf(fp, "\t%4.3f", ppm);
    }

    fprintf(fp, "\n");
}

static void output_ansig(int ndim, Bool subtract, int nfitted, int *dim_fitted,
			int *npoints, FILE *fp,
			Ref_info *ref_info, int nfit, Fit_info **fit_info)
{
    int i, j;

    fprintf(fp, "extr\tmagn\tchisq");

    if (subtract)
	fprintf(fp, "\textrs");

    fprintf(fp, "\n");

    for (i = 0; i < ndim; i++)
	fprintf(fp, "\tppm%d", i+1);

    fprintf(fp, "\n");

    if (subtract)
    {
	for (i = 0; i < ndim; i++)
	    fprintf(fp, "\tppms%d", i+1);

	fprintf(fp, "\n");
    }

    for (i = 0; i < nfitted; i++)
    {
	j = dim_fitted[i] + 1;
	fprintf(fp, "\tppmf%d", j);
    }

    fprintf(fp, "\n");

    for (i = 0; i < nfit; i++)
    {
	if (fit_info[i]->peak)
	    print_ansig(ndim, subtract, nfitted, dim_fitted, npoints,
						fp, ref_info, fit_info[i]);
    }
}

static Status alloc_peak_memory(int ndim, int n, Fit_info ***p_fit_info,
							String error_msg)
{
    int i;
    Fit_info **fit_info;

    sprintf(error_msg, "allocating peak memory");

    MALLOC(fit_info, Fit_info *, n);

    for (i = 0; i < n; i++)
    {
	MALLOC(fit_info[i], Fit_info, 1);
	MALLOC(fit_info[i]->before.position, int, ndim);
	MALLOC(fit_info[i]->after.position, int, ndim);
    }

    *p_fit_info = fit_info;

    return  OK;
}

static Status read_peak_file(FILE *fp, Size_info *size_info,
			int *p_nfit, Fit_info ***p_fit_info, String error_msg)
{
    int i, j, n, nfit, p;
    float c, ppm;
    Fit_info **fit_info;
    Line line;

    fgets(line, LINE_SIZE, fp);  /* two header lines */
    fgets(line, LINE_SIZE, fp);

    for (nfit = 0; fgets(line, LINE_SIZE, fp); nfit++)
	;

    if (nfit == 0)
	RETURN_ERROR_MSG("no extrema found in input peak file");

    printf("%d data row%s found in input peak file\n",
						nfit, (nfit == 1) ? "" : "s");

   CHECK_STATUS(alloc_peak_memory(size_info->ndim, nfit, &fit_info, error_msg));

    rewind(fp);

    fgets(line, LINE_SIZE, fp);  /* two header lines */
    fgets(line, LINE_SIZE, fp);

    n = 0;
    for (i = 0; i < nfit; i++)
    {
	/* protect against blank lines */
	if (fscanf(fp, "%f", &(fit_info[n]->before.value)) != 1)
	    continue;

	for (j = 0; j < size_info->ndim; j++)
	{
	    if (fscanf(fp, "%f %f", &c, &ppm) != 2)
	    {
		sprintf(error_msg,
			"reading point/ppm data for dim. %d of extremum %d",
								 j+1, n+1);
		return  ERROR;
	    }

	    if ((c < 1) || (c > size_info->npoints[j]))
	    {
		sprintf(error_msg,
	    "point (= %2.1f) for dim. %d of extremum %d out of range (1, %d)",
					 c, j+1, n+1, size_info->npoints[j]);
		return  ERROR;
	    }

	    p = c + 0.5;

	    fit_info[n]->before.position[j] = p - 1;
	    fit_info[n]->n = n;  /* needed for grouping & ungrouping code */
	}

	n++;
	fgets(line, LINE_SIZE, fp);  /* discard rest of line */
    }

    if (n == 0)
	RETURN_ERROR_MSG("no extrema read in input peak file");

    printf("%d extrem%s successfully read in input peak file\n",
						n, (n == 1) ? "um" : "a");

    for (i = n; i < nfit; i++)  /* free unused resources */
    {
	FREE(fit_info[i]->before.position, int);
	FREE(fit_info[i]->after.position, int);
	FREE(fit_info[i], Fit_info);
    }

    *p_nfit = n;
    *p_fit_info = fit_info;

    return  OK;
}

static FILE *open_peak_file(String name)
{
    Line error_msg;
    FILE *fp = (FILE *) NULL;

    if (name)
    {
	if (OPEN_FOR_WRITING(fp, name))
	{
	    sprintf(error_msg, "opening \"%s\" for writing", name);
	    ERROR_AND_EXIT(error_msg);
	}
    }

    return  fp;
}

void main(int argc, char **argv)
{
    int i, j, nstore, nfit, npts_max[MAX_NDIM];
    String script_file;
    Line error_msg;
    Size_info size_info;
    Ref_info *ref_info;
    File_info file_info;
    Fit_param fit_param;
    Fit_info **fit_info;
    Group_info group_info;

    printf(product);

    if (help_request(argc, argv, help_table))
	exit (0);

    if ((argc != 2) && (argc != 3))
    {
	sprintf(error_msg, "correct usage: %s", argv[0]);
	strcat(error_msg, " [<memory in Mwords>] <script file>");
	ERROR_AND_EXIT(error_msg);
    }

    if (argc == 3)
	nstore = MAX(1, atoi(argv[1])) * MEGAWORD;
    else
	nstore = DEFAULT_STORE;

    script_file = argv[argc-1];

    if (read_script_file(script_file, &size_info, &ref_info,
				&file_info, &fit_param, error_msg) == ERROR)
	ERROR_AND_EXIT(error_msg);

    if (OPEN_FOR_BINARY_READING(file_info.file_data_in, file_info.input_data_file))
    {
	sprintf(error_msg, "opening \"%s\" for reading",
						file_info.input_data_file);
	ERROR_AND_EXIT(error_msg);
    }

    if (OPEN_FOR_READING(file_info.file_peak_in, file_info.input_peak_file))
    {
	sprintf(error_msg, "opening \"%s\" for reading",
						file_info.input_peak_file);
	ERROR_AND_EXIT(error_msg);
    }

    if (read_peak_file(file_info.file_peak_in, &size_info, &nfit, &fit_info,
							error_msg) == ERROR)
	ERROR_AND_EXIT(error_msg);

    file_info.file_peak_out = open_peak_file(file_info.output_peak_file);
    file_info.file_ansig_out = open_peak_file(file_info.output_ansig_file);
    file_info.file_ideal_out = open_peak_file(file_info.output_ideal_file);
    file_info.file_rest_out = open_peak_file(file_info.output_rest_file);

    if (initialize_data(nstore, &size_info, &file_info, error_msg) == ERROR)
	ERROR_AND_EXIT(error_msg);

    for (i = 0; i < fit_param.nfitted; i++)
    {
	j = fit_param.dim_fitted[i];
	npts_max[i] = fit_param.script[j].npts_max;
    }

    if (determine_fit_groups(size_info.ndim, fit_param.width,
	    fit_param.group, nfit, fit_info, &group_info, error_msg) == ERROR)
	ERROR_AND_EXIT(error_msg);

    if (initialize_fit(fit_param.nfitted, fit_param.nparams, npts_max,
				group_info.group_max, error_msg) == ERROR)
	ERROR_AND_EXIT(error_msg);

    if (fit_peaks(&size_info, &fit_param, nfit, fit_info, &group_info,
							error_msg) == ERROR)
	ERROR_AND_EXIT(error_msg);

    if (file_info.output_peak_file)
    {
	output_peaks(size_info.ndim, fit_param.subtract,
			fit_param.nfitted, fit_param.dim_fitted,
			size_info.npoints, fit_param.npts,
			file_info.file_peak_out, ref_info, nfit, fit_info);
    }

    if (file_info.output_ansig_file)
    {
	output_ansig(size_info.ndim, fit_param.subtract,
			fit_param.nfitted, fit_param.dim_fitted,
			size_info.npoints,
			file_info.file_ansig_out, ref_info, nfit, fit_info);
    }

    if (file_info.output_ideal_file || file_info.output_rest_file)
    {
	if (output_spectra(&size_info, &fit_param, &file_info, ref_info,
					nfit, fit_info, error_msg) == ERROR)
	    ERROR_AND_EXIT(error_msg);
    }

    exit(0);
}
