#include "data.h"

#include "block_io.h"
#include "dbl_list.h"

typedef struct
{
    int block;
    float *data;
}   Data_pair;

static int ndim;
static int *block_size;
static int *npoints;

static int used_store;

static Dbl_list block_list;
static Dbl_list *block_index;
static float *block_store;

static Dbl_list tail_list;
static float *data_ptr;

static int total_blocks;
static int size_of_block;
static int total_store;

static int nblocks[MAX_NDIM];
static int cum_blocks[MAX_NDIM];
static int cum_block_size[MAX_NDIM];
static int array[MAX_NDIM];

static int *directory;

static Block_IO block_io;

static void move_block_list(Dbl_list dbl_list)
{
    if (dbl_list == block_list)
	return;

    if (dbl_list == tail_list)
	tail_list = DBL_PREVIOUS(tail_list);

    move_dbl_list_to_front(&block_list, dbl_list);
}

static Status get_unloaded_block(int block, String error_msg)
{
    Data_pair *data_pair;

    if (used_store < total_store)
    {
	sprintf(error_msg, "allocating data pair memory");
	MALLOC(data_pair, Data_pair, 1);

	sprintf(error_msg, "inserting data pair in dbl_list");
	CHECK_STATUS(insert_dbl_list(&block_list, (Generic_ptr) data_pair));

	data_pair->data = block_store + size_of_block*used_store;

	if (used_store == 0)
	    tail_list = block_list;

	used_store++;
    }
    else
    {
	data_pair = (Data_pair *) DBL_DATA(tail_list);
	block_index[data_pair->block] = (Dbl_list) NULL;

	move_block_list(tail_list);
    }

    data_pair->block = block;
    block_index[block] = block_list;

    CHECK_STATUS(read_file_block(&block_io, block, data_pair->data, error_msg));

    data_ptr = data_pair->data;

    return  OK;
}

static void get_loaded_block(int block)
{
    Data_pair *data_pair = (Data_pair *) DBL_DATA(block_index[block]);

    data_ptr = data_pair->data;

    move_block_list(block_index[block]);
}

Status data_value(float *value, int *point, String error_msg)
{
    int i, pnt, block;

    for (i = 0; i < ndim; i++)
	array[i] = point[i] / block_size[i];

    INDEX_OF_ARRAY(block, array, cum_blocks, ndim);

    if (block_index[block])
	get_loaded_block(block);
    else
	CHECK_STATUS(get_unloaded_block(block, error_msg));

    for (i = 0; i < ndim; i++)
	array[i] = point[i] % block_size[i];

    INDEX_OF_ARRAY(pnt, array, cum_block_size, ndim);
    *value = data_ptr[pnt];

    return  OK;
}

static void init_arrays(int nstore)
{
    BLOCKS(nblocks, npoints, block_size, ndim);

    CUMULATIVE(cum_blocks, nblocks, total_blocks, ndim);
    CUMULATIVE(cum_block_size, block_size, size_of_block, ndim);

    total_store = nstore / size_of_block;
}

static Status alloc_data_memory(String error_msg)
{
    int i, n;

    sprintf(error_msg, "initializing data dbl_list");
    CHECK_STATUS(init_dbl_list(&block_list));

    sprintf(error_msg, "allocating data memory");

    n = total_store * size_of_block;
    MALLOC(block_store, float, n);

    used_store = 0;

    MALLOC(block_index, Dbl_list, total_blocks);

    for (i = 0; i < total_blocks; i++)
	block_index[i] = (Dbl_list) NULL;

    return  OK;
}

static Status alloc_directory_memory(int dir_size, String error_msg)
{
    sprintf(error_msg, "allocating directory memory");

    MALLOC(directory, int, dir_size);

    return  OK;
}

Status initialize_data(int nstore, Size_info *size_info, File_info *file_info,
							String error_msg)
{
    ndim = size_info->ndim;
    block_size = size_info->block_size;
    npoints = size_info->npoints;

    if (!(file_info->blocked))
        RETURN_ERROR_MSG("input file must be blocked");

    init_arrays(nstore);

    CHECK_STATUS(alloc_data_memory(error_msg));

    if (file_info->deflated)
	CHECK_STATUS(alloc_directory_memory(file_info->dir_size, error_msg));

    block_io.name = file_info->input_data_file;
    block_io.file = file_info->file_data_in;
    block_io.swapped = file_info->swapped;
    block_io.integer = file_info->integer;
    block_io.deflated = file_info->deflated;
    block_io.header = file_info->header;
    block_io.dir_size = file_info->dir_size;
    block_io.directory = directory;
    block_io.byte_size = file_info->byte_size;
    block_io.block_size = size_of_block;

    CHECK_STATUS(init_block_read(&block_io, error_msg));

    return  OK;
}

static void adjust_point(int ndim, int *point, int *npoints)
{
    int i;

    for (i = 0; i < ndim; i++)
    {
	if (point[i] < 0)
	    point[i] += npoints[i];
	else if (point[i] >= npoints[i])
	    point[i] -= npoints[i];
    }
}

void find_point(int ndim, int p, int *point, int *cum_points,
				int *base_point, int *npoints, Bool flag)
{
    ARRAY_OF_INDEX(point, p, cum_points, ndim);
    ADD_VECTORS(point, point, base_point, ndim);

    if (flag)
	adjust_point(ndim, point, npoints);
}
