#include "contours.h"

#include "block.h"
#include "list.h"
#include "par.h"
#include "parse.h"
#include "ref.h"

#define  MAX_NLEVELS  20

#define  SIZE_OF_BLOCK  4096

static int ndim;
static int *block_size;
static int *npoints;

static int nstore;
static int ncol_store;
static int nrow_store;
static int nwork;
static float *store;
static float *col_store;
static float *row_store;
static float *work;

static int nlevels;
static float levels[MAX_NLEVELS];
static List vertices[MAX_NLEVELS];

static int dim0;
static int dim1;
static int nsub_blocks0;
static int nsub_blocks1;

static Ref_info *ref;
static int dir_size_out;
static int *directory_out;
static int out_size;
static float *output;

static float lower[MAX_NDIM];
static float upper[MAX_NDIM];

static char *input_file;
static Line output_file;
static FILE *file_in;
static FILE *file_out;
static Bool swapped;
static Bool integer;
static Bool blocked;
static int header;
static Bool deflated;
static int dir_size_in;
static int *directory_in;

static Bool input_found;
static Bool output_found;
static char *contours_file;

static int dim;
static float pnt0;
static float pnt1;
static Bool dim_found[MAX_NDIM];
static int first[MAX_NDIM];
static int last[MAX_NDIM];

static Exclude_info exclude_info[MAX_NDIM];
static Diagonal_info diagonal_info;

static int parse_string[] = { PARSE_STRING };
static int parse_int2[] = { PARSE_INT, PARSE_INT };
static int parse_int_float2[] = { PARSE_INT, PARSE_FLOAT, PARSE_FLOAT };
static int parse_int2_float[] = { PARSE_INT, PARSE_INT, PARSE_FLOAT };
static int parse_float_free[] = { PARSE_FLOAT | PARSE_FREE };
static int parse_float_int_float[] = { PARSE_FLOAT, PARSE_INT, PARSE_FLOAT };

#define  FOUND_TWICE(string) \
	 {   sprintf(error_msg, "in \"%s\" '%s' found twice", \
				contours_file, string);  return  ERROR;   }

#define  FOUND_BEFORE(string1, string2) \
	 {   sprintf(error_msg, "in \"%s\" '%s' found before '%s'", \
			contours_file, string1, string2);  return  ERROR;   }

#define  NOT_FOUND(string) \
	 {   sprintf(error_msg, "in \"%s\" no '%s' found", \
				contours_file, string);  return  ERROR;   }

static Status allocate_memory(String error_msg)
{
    sprintf(error_msg, "allocating memory for store");
    MALLOC(store, float, nstore);

    sprintf(error_msg, "allocating memory for row_store");
    MALLOC(row_store, float, nrow_store);

    sprintf(error_msg, "allocating memory for col_store");
    MALLOC(col_store, float, ncol_store);

    sprintf(error_msg, "allocating memory for work");
    MALLOC(work, float, nwork);

    sprintf(error_msg, "allocating memory for output directory");
    MALLOC(directory_out, int, dir_size_out);

    sprintf(error_msg, "allocating memory for output");
    MALLOC(output, float, out_size);

    if (deflated)
    {
	sprintf(error_msg, "allocating memory for input directory");
	MALLOC(directory_in, int, dir_size_in);
    }

    return  OK;
}

#define  PLANE_SIZE  32

static void determine_params()
{
    int i, n, size, nsub_planes, nblocks[MAX_NDIM];
    int nsub_chunks0, nsub_points0, nsub_chunks1, nsub_points1;

    BLOCKS(nblocks, npoints, block_size, ndim);

    nsub_blocks0 = PLANE_SIZE / block_size[dim0];
    nsub_blocks0 = MIN(nsub_blocks0, nblocks[dim0]);
    nsub_blocks0 = MAX(nsub_blocks0, 1);

    nsub_blocks1 = PLANE_SIZE / block_size[dim1];
    nsub_blocks1 = MIN(nsub_blocks1, nblocks[dim1]);
    nsub_blocks1 = MAX(nsub_blocks1, 1);

    n = (last[dim0]-1)/block_size[dim0] + 1;
    n -= first[dim0]/block_size[dim0];
    nsub_chunks0 = BLOCK(n, nsub_blocks0);

    n = (last[dim1]-1)/block_size[dim1] + 1;
    n -= first[dim1]/block_size[dim1];
    nsub_chunks1 = BLOCK(n, nsub_blocks1);

    nsub_points0 = nsub_blocks0 * block_size[dim0];
    nsub_points1 = nsub_blocks1 * block_size[dim1];

    VECTOR_PRODUCT(size, block_size, ndim);
    nsub_planes = size / (block_size[dim0] * block_size[dim1]);

    nstore = nsub_points0 * nsub_points1 * nsub_planes;
    nrow_store = (nsub_points0+1) * nsub_chunks0 * nsub_planes;
    ncol_store = nsub_points1 * nsub_planes;
    nwork = nsub_points0 + 1;

    n = 1;
    for (i = 0; i < ndim; i++)
    {
	if ((i == dim0) || (i == dim1))
	    continue;

	n *= last[i] - first[i];
    }

    dir_size_out = 2 * n * nsub_chunks0 * nsub_chunks1;
    dir_size_out += 4*ndim + 4;
/*
    dir_size_out = size * BLOCK(dir_size_out, size);
*/
    dir_size_out = SIZE_OF_BLOCK * BLOCK(dir_size_out, SIZE_OF_BLOCK);

/*
    out_size = size;
*/
    out_size = SIZE_OF_BLOCK;

    for (i = 0; i < ndim; i++)
    {
	lower[i] = 1;
	upper[i] = npoints[i];
    }

    convert_from_points(REF_PPM, ndim, npoints, ref, lower);
    convert_from_points(REF_PPM, ndim, npoints, ref, upper);
}

static Status input_parse(Generic_ptr *var, String error_msg)
{
    String par_file = (String) (*var);
    Par_info par_info;

    if (input_found)
	FOUND_TWICE("input");

    CHECK_STATUS(read_par_file(par_file, &par_info, error_msg));

    input_file = par_info.file;
    ndim = par_info.ndim;
    npoints = par_info.npoints;
    block_size = par_info.block_size;
    ref = par_info.ref;
    swapped = par_info.swapped;
    integer = par_info.integer;
    blocked = par_info.blocked;
    header = par_info.header;
    deflated = par_info.deflated;
    dir_size_in = par_info.dir_size;

    if (!blocked)
    {
	sprintf(error_msg, "input file \"%s\" must be blocked", input_file);
	return  ERROR;
    }

    if (deflated)
    {
/*
	sprintf(error_msg, "input file \"%s\" must not be deflated",
								input_file);
	return  ERROR;
*/
	printf("Warning: input file \"%s\" deflated", input_file);
    }

    input_found = TRUE;

    ZERO_VECTOR(first, ndim);
    COPY_VECTOR(last, npoints, ndim);

    return  OK;
}

static Status output_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (!input_found)
	FOUND_BEFORE("output", "input");

    if (output_found)
	FOUND_TWICE("output");

    strcpy(output_file, name);

    output_found = TRUE;

    return  OK;
}

static Status dims_parse(Generic_ptr *var, String error_msg)
{
    dim0 = *((int *) var[0]);
    dim1 = *((int *) var[1]);

    if (!output_found)
	FOUND_BEFORE("dims", "output");

    if ((dim0 > ndim) || (dim1 > ndim))
    {
	sprintf(error_msg, "in \"%s\" have 'dims %d %d' but 'ndim' = %d",
					contours_file, dim0, dim1, ndim);
	return  ERROR;
    }

    if ((dim0 < 1) || (dim1 < 1) || (dim0 == dim1))
    {
	sprintf(error_msg, "in \"%s\" have 'dims %d %d'",
					contours_file, dim0, dim1);
	return  ERROR;
    }

    if (dim0 > dim1)
    {
	sprintf(error_msg,
		"in \"%s\" have 'dims %d %d', must have 'dims %d %d'",
				contours_file, dim0, dim1, dim1, dim0);
	return  ERROR;
    }

    dim0--;  dim1--;

    return  OK;
}

static Status range_check(Generic_ptr *var, String msg, String error_msg)
{
    Line msg2;

    dim = *((int *) var[0]);
    pnt0 = *((float *) var[1]);
    pnt1 = *((float *) var[2]);

    if (!output_found)
	FOUND_BEFORE(msg, "output");

    dim--;

    if ((dim < 0) || (dim >= ndim))
    {
	sprintf(error_msg, "'dim' = %d in '%s', out of range", dim+1, msg);
	return  ERROR;
    }

    if (dim_found[dim])
    {
	sprintf(msg2, "'dim' = %d appears twice with '%s'", dim+1, msg);
	FOUND_TWICE(msg2);
    }

    dim_found[dim] = TRUE;

    if (pnt0 >= pnt1)
    {
	sprintf(error_msg, "for 'dim' %d in '%s' first point >= second point",
								dim+1, msg);
	return  ERROR;
    }

    return  OK;
}

static Status setup_range(String msg, String error_msg)
{
    if (pnt1 < 1)
    {
	sprintf(error_msg, "for 'dim' %d in '%s' second point < 1",
								dim+1, msg);
	return  ERROR;
    }

    if (pnt0 > npoints[dim])
    {
	sprintf(error_msg, "for 'dim' %d in '%s' first point > #points",
								dim+1, msg);
	return  ERROR;
    }

    first[dim] = NEAREST_INTEGER(pnt0) - 1;
    last[dim] = NEAREST_INTEGER(pnt1);

    first[dim] = MAX(first[dim], 0);
    last[dim] = MIN(last[dim], npoints[dim]);

    last[dim] = MAX(first[dim]+1, last[dim]);

    return  OK;
}

static Status pnt_range_parse(Generic_ptr *var, String error_msg)
{
    CHECK_STATUS(range_check(var, "pnt_range", error_msg));

    CHECK_STATUS(setup_range("pnt_range", error_msg));

    return  OK;
}

static Status ppm_range_parse(Generic_ptr *var, String error_msg)
{
    CHECK_STATUS(range_check(var, "ppm_range", error_msg));

    convert_range_to_points(REF_PPM, 1, npoints+dim, ref+dim, &pnt0, &pnt1);

    CHECK_STATUS(setup_range("ppm_range", error_msg));

    return  OK;
}

static Status exclude_check(Generic_ptr *var, String msg, String error_msg)
{
    dim = *((int *) var[0]);
    pnt0 = *((float *) var[1]);
    pnt1 = *((float *) var[2]);

    if (!output_found)
	FOUND_BEFORE(msg, "output");

    dim--;

    if ((dim < 0) || (dim >= ndim))
    {
	sprintf(error_msg, "'dim' = %d in '%s', out of range", dim+1, msg);
	return  ERROR;
    }

    if (pnt0 >= pnt1)
    {
	sprintf(error_msg, "for 'dim' %d in '%s' first point >= second point",
								dim+1, msg);
	return  ERROR;
    }

    return  OK;
}

static Status setup_exclude(String msg, String error_msg)
{
    int n;
    Exclude_info *e;

    e = exclude_info + dim;
    n = e->n;

    if (n >= MAX_EXCLUDE)
    {
	sprintf(error_msg, "too many exclusions in dim %d, limit of %d",
					dim+1, MAX_EXCLUDE);
	return  ERROR;
    }

    e->first[n] = pnt0 - 1;
    e->last[n] = pnt1 - 1;
    e->n = n + 1;

    return  OK;
}

static Status pnt_exclude_parse(Generic_ptr *var, String error_msg)
{
    CHECK_STATUS(exclude_check(var, "pnt_exclude", error_msg));

/*
    if ((dim == dim0) || (dim == dim1))
	convert_range_from_points(REF_PPM, 1, npoints+dim, ref+dim,
							&pnt0, &pnt1);
*/

    CHECK_STATUS(setup_exclude("pnt_exclude", error_msg));

    return  OK;
}

static Status ppm_exclude_parse(Generic_ptr *var, String error_msg)
{
    CHECK_STATUS(exclude_check(var, "ppm_exclude", error_msg));

/*
    if ((dim != dim0) && (dim != dim1))
*/
	convert_range_to_points(REF_PPM, 1, npoints+dim, ref+dim,
							&pnt0, &pnt1);

    CHECK_STATUS(setup_exclude("ppm_exclude", error_msg));

    return  OK;
}

static Status diag_exclude_parse(Generic_ptr *var, String error_msg)
{
    int i, n;
    Line msg;
    int d0 = *((int *) var[0]);
    int d1 = *((int *) var[1]);
    float delta = *((float *) var[2]);

    if (!output_found)
	FOUND_BEFORE("diag_exclude", "output");

    if ((d0 < 1) || (d0 > ndim))
    {
	sprintf(error_msg, "in \"%s\" diag_exclude 'dim1' = %d, out of range",
							contours_file, d0);
	return  ERROR;
    }

    if ((d1 < 1) || (d1 > ndim))
    {
	sprintf(error_msg, "in \"%s\" diag_exclude 'dim2' = %d, out of range",
							contours_file, d1);
	return  ERROR;
    }

    if (d0 == d1)
    {
	sprintf(error_msg, "in \"%s\" diag_exclude 'dim1' = 'dim2' = %d",
							contours_file, d0);
	return  ERROR;
    }

    if (delta <= 0)
    {
	sprintf(error_msg,
		"in \"%s\" diag_exclude 'delta' = %3.2e, must be > 0",
							contours_file, delta);
	return  ERROR;
    }

    d0--;  d1--;

    if (d0 > d1)
	SWAP(d0, d1, int);

    n = diagonal_info.n;

    for (i = 0; i < n; i++)
    {
	if ((diagonal_info.dim0[i] == d0) && (diagonal_info.dim1[i] == d1))
	{
	    sprintf(msg, "diag_exclude %d %d\n", d0+1, d1+1);
	    FOUND_TWICE(msg);
	}
    }

    diagonal_info.dim0[n] = d0;
    diagonal_info.dim1[n] = d1;
    diagonal_info.delta[n] = delta;
    diagonal_info.n = n + 1;

    return  OK;
}

static Status levels_parse(Generic_ptr *var, String error_msg)
{
    int i, n;

    if (!output_found)
	FOUND_BEFORE("levels", "output");

    n = *((int *) var[0]);

    if ((nlevels+n) > MAX_NLEVELS)
    {
	sprintf(error_msg,
	    "in \"%s\" have %d levels, cannot have more than %d",
				contours_file, nlevels+n, MAX_NLEVELS);
	return  ERROR;
    }

    for (i = 0; i < n; i++)
	levels[nlevels+i] = *((float *) var[i+1]);

    nlevels += n;

    return  OK;
}

static Status levels_mult_parse(Generic_ptr *var, String error_msg)
{
    int i, n;
    float base_level, level_mult, level;

    if (!output_found)
	FOUND_BEFORE("levels", "output");

    base_level = *((float *) var[0]);
    n = *((int *) var[1]);
    level_mult = *((float *) var[2]);

    if ((nlevels+n) > MAX_NLEVELS)
    {
	sprintf(error_msg,
	    "in \"%s\" have %d levels, cannot have more than %d",
				contours_file, nlevels+n, MAX_NLEVELS);
	return  ERROR;
    }

    level = base_level;

    for (i = 0; i < n; i++)
    {
	levels[nlevels+i] = level;
	level *= level_mult;
    }

    nlevels += n;

    return  OK;
}

static Status levels_add_parse(Generic_ptr *var, String error_msg)
{
    int i, n;
    float base_level, level_add, level;

    if (!output_found)
	FOUND_BEFORE("levels", "output");

    base_level = *((float *) var[0]);
    n = *((int *) var[1]);
    level_add = *((float *) var[2]);

    if ((nlevels+n) > MAX_NLEVELS)
    {
	sprintf(error_msg,
	    "in \"%s\" have %d levels, cannot have more than %d",
				contours_file, nlevels+n, MAX_NLEVELS);
	return  ERROR;
    }

    level = base_level;

    for (i = 0; i < n; i++)
    {
	levels[nlevels+i] = level;
	level += level_add;
    }

    nlevels += n;

    return  OK;
}

static Parse_line contours_table[] =
{
    { "input",		1,	parse_string,		input_parse },
    { "output",		1,	parse_string,		output_parse },
    { "dims",		2,	parse_int2,		dims_parse },
    { "pnt_range",	3,	parse_int_float2,	pnt_range_parse },
    { "ppm_range",	3,	parse_int_float2,	ppm_range_parse },
    { "pnt_exclude",	3,	parse_int_float2,	pnt_exclude_parse },
    { "ppm_exclude",	3,	parse_int_float2,	ppm_exclude_parse },
    { "diag_exclude",	3,	parse_int2_float,	diag_exclude_parse },
    { "levels",		1,	parse_float_free,	levels_parse },
    { "levels_mult",	3,	parse_float_int_float,	levels_mult_parse },
    { "levels_add",	3,	parse_float_int_float,	levels_add_parse },
    { (String) NULL,	0,	(int *) NULL,		no_parse_func }
};

static Status read_contours_file(String error_msg)
{
    int i;

    input_found = FALSE;
    output_found = FALSE;
    dim0 = dim1 = -1;
    nlevels = 0;

    for (i = 0; i < MAX_NDIM; i++)
	dim_found[i] = FALSE;

    for (i = 0; i < MAX_NDIM; i++)
	exclude_info[i].n = 0;

    diagonal_info.n = 0;

    CHECK_STATUS(parse_file(contours_file, contours_table, TRUE, error_msg));

    if (!input_found)
	NOT_FOUND("input");

    if (!output_found)
	NOT_FOUND("output");

    if ((dim0 < 0) || (dim1 < 0))
	NOT_FOUND("dims");

    if (nlevels < 1)
    {
	sprintf(error_msg, "in \"%s\" have no levels", contours_file);
	return  ERROR;
    }

    return  OK;
}

static void exclude_params()
{
    int i, j, n;
    Exclude_info *e;

    for (i = 0; i < ndim; i++)
    {
	if ((i == dim0) || (i == dim1)) /* convert to ppm */
	{
	    e = exclude_info + i;
	    n = e->n;

	    for (j = 0; j < n; j++)
	    {
		e->first[j] = e->first[j] + 1;
		e->last[j] = e->last[j] + 1;
		convert_range_from_points(REF_PPM, 1, npoints+i, ref+i,
						&e->first[j], &e->last[j]);
	    }
	}

    }
}

static void print_ppm_info()
{
    int i;
    float f, l;

    printf("\nppm range for data set:\n");

    for (i = 0; i < ndim; i++)
	printf("\tdim %d: (%5.2f, %5.2f)\n", i+1, upper[i], lower[i]);

    printf("\nppm range for contours:\n");

    for (i = 0; i < ndim; i++)
    {
	f = first[i]+1;  l = last[i];
	convert_from_point(REF_PPM, npoints[i], ref+i, &f);
	convert_from_point(REF_PPM, npoints[i], ref+i, &l);
	printf("\tdim %d: (%5.2f, %5.2f)\n", i+1, l, f);
    }

    printf("\n");
}

static void print_exclude_info()
{
    int i, j, n;
    float f, l;
    Exclude_info *e;

    n = 0;
    for (i = 0; i < ndim; i++)
    {
	e = exclude_info + i;
	n += e->n;
    }

    if (n == 0)
	return;

    printf("ppm exclusions for data set:\n");

    for (i = 0; i < ndim; i++)
    {
	printf("\tdim %d:", i+1);

	e = exclude_info + i;
	n = e->n;

	if (n == 0)
	{
	    printf(" none\n");
	}
	else
	{
	    for (j = 0; j < n; j++)
	    {
		if (j > 0)
		    printf(",");

		f = e->first[j] + 1;  l = e->last[j] + 1;
		convert_from_point(REF_PPM, npoints[i], ref+i, &f);
		convert_from_point(REF_PPM, npoints[i], ref+i, &l);
		printf(" (%5.2f, %5.2f)", l, f);
	    }

	    printf("\n");
	}
    }

    printf("\n");
}

static void print_diagonal_info()
{
    int i, n, d0, d1;
    float delta;

    n = diagonal_info.n;

    if (n == 0)
	return;

    printf("diagonal exclusions for data set:\n");

    for (i = 0; i < n; i++)
    {
	d0 = diagonal_info.dim0[i];
	d1 = diagonal_info.dim1[i];
	delta = diagonal_info.delta[i];

	printf("\tdims %d %d: %4.3f\n", d0+1, d1+1, delta);
    }
}

static void print_level_info()
{
    int i;

    printf("number of levels to contour = %d\n", nlevels);
    printf("levels =");

    for (i = 0; i < nlevels; i++)
    {
	if (ABS(levels[i]) > 10)
	    printf(" %3.0f", levels[i]);
	else if (ABS(levels[i]) > 1)
	    printf(" %3.1f", levels[i]);
	else
	    printf(" %f", levels[i]);
    }

    printf("\n\n");
}

void main(int argc, char **argv)
{
    Line error_msg;
    Size_info size_info;
    Store_info store_info;
    Level_info level_info;
    Plane_info plane_info;
    Output_info output_info;
    File_info file_info;
    Limit_info limit_info;

    printf(product);

    if (help_request(argc, argv, help_table))
        exit (0);

    if (argc != 2)
    {
        sprintf(error_msg, "correct usage: %s <contours file>", argv[0]);
        ERROR_AND_EXIT(error_msg);
    }

    contours_file = argv[1];

    if (read_contours_file(error_msg) == ERROR)
        ERROR_AND_EXIT(error_msg);

    determine_params();

    if (allocate_memory(error_msg) == ERROR)
        ERROR_AND_EXIT(error_msg);

    if (OPEN_FOR_BINARY_READING(file_in, input_file))
    {
	sprintf(error_msg, "opening \"%s\" for reading", input_file);
	ERROR_AND_EXIT(error_msg);
    }

    if (OPEN_FOR_BINARY_WRITING(file_out, output_file))
    {
	sprintf(error_msg, "opening \"%s\" for writing", output_file);
	ERROR_AND_EXIT(error_msg);
    }

    size_info.ndim = ndim;
    size_info.block_size = block_size;
    size_info.npoints = npoints;

    store_info.store = store;
    store_info.col_store = col_store;
    store_info.row_store = row_store;
    store_info.work = work;
    store_info.directory = directory_in;

    level_info.nlevels = nlevels;
    level_info.levels = levels;
    level_info.vertices = vertices;

    plane_info.dim0 = dim0;
    plane_info.dim1 = dim1;
    plane_info.nsub_blocks0 = nsub_blocks0;
    plane_info.nsub_blocks1 = nsub_blocks1;

    output_info.dir_size = dir_size_out;
    output_info.directory = directory_out;
    output_info.out_size = out_size;
    output_info.output = output;
    output_info.lower = lower;
    output_info.upper = upper;

    file_info.input_file = input_file;
    file_info.output_file = output_file;
    file_info.file_in = file_in;
    file_info.file_out = file_out;
    file_info.swapped = swapped;
    file_info.integer = integer;
    file_info.blocked = blocked;
    file_info.header = header;
    file_info.deflated = deflated;
    file_info.dir_size = dir_size_in;

    limit_info.first = first;
    limit_info.last = last;

    print_ppm_info();
    print_exclude_info();
    print_diagonal_info();
    print_level_info();
    FLUSH;

    exclude_params();

    if (block_process(&size_info, &store_info, &level_info, &output_info,
		&plane_info, &file_info, &limit_info, exclude_info,
		&diagonal_info, ref, error_msg) == ERROR)
	ERROR_AND_EXIT(error_msg);
}
