#include "block.h"

#include "block_io.h"
#include "principal.h"
#include "sorts.h"

static int ndim;
static int *npoints;
static int *block_size;

static float **store_in;
static float **store_out;
static float **matrix;
static float *eigenvalues;
static float *work;
static int **order;

static int ninput_files;
static int noutput_files;

static int ncomponents;
static float fraction;
static Bool correlation;

static int total_nblocks;
static int size_of_block;

static int nblocks[MAX_NDIM];
static int blocks[MAX_NDIM];
static int points[MAX_NDIM];
static int end_points[MAX_NDIM];
static int cum_nblocks[MAX_NDIM];
static int cum_block_size[MAX_NDIM];

static Block_IO *block_io_in;
static Block_IO *block_io_out;

static void init_arrays()
{
    BLOCKS(nblocks, npoints, block_size, ndim);

    CUMULATIVE(cum_nblocks, nblocks, total_nblocks, ndim);

    CUMULATIVE(cum_block_size, block_size, size_of_block, ndim);
}

static Status disk_to_store(int block, String error_msg)
{
    int i;
    float *s;

    for (i = 0; i < ninput_files; i++)
    {
	s = store_in[i];

	CHECK_STATUS(read_file_block(&(block_io_in[i]), block, s, error_msg));
    }

    return  OK;
}

static Status store_to_disk(int block, String error_msg)
{
    int i;
    float *s;

    for (i = 0; i < noutput_files; i++)
    {
	s = store_out[i];

	CHECK_STATUS(write_file_block(&(block_io_out[i]), block, s, error_msg));
    }

    return  OK;
}

static Bool find_end_points()
{
    int i;
    Bool boundary_case = FALSE;

    for (i = 0; i < ndim; i++)
    {
	if (blocks[i] == (nblocks[i]-1))
	{
	    end_points[i] = 1 + (npoints[i]-1) % block_size[i];

	    if (end_points[i] != block_size[i])
		boundary_case = TRUE;
	}
	else
	{
	    end_points[i] = block_size[i];
	}
    }

    return  boundary_case;
}

static Bool point_in_range()
{
    int i;

    for (i = 0; i < ndim; i++)
    {
	if (points[i] > end_points[i])
	    return  FALSE;
    }

    return  TRUE;
}

static void boundary_process()
{
    int i, j, k;

    for (i = 0; i < ninput_files; i++)
    {
	for (j = 0; j < size_of_block; j++)
	{
	    /* could do a row at a time but not worth the effort */
	    ARRAY_OF_INDEX(points, j, cum_block_size, ndim);

	    if (point_in_range())
	    {
		for (k = i; k < ninput_files; k++)
		    matrix[i][k] += store_in[i][j] * store_in[k][j];
	    }
	}
    }
}

static void normal_process()
{
    int i, k;
    float d;

    for (i = 0; i < ninput_files; i++)
    {
	for (k = i; k < ninput_files; k++)
	{
	    INNER_PRODUCT(d, store_in[i], store_in[k], size_of_block);
	    matrix[i][k] += d;
	}
    }
}

static Status find_components(String error_msg)
{
    int i, j;
    float d;

    for (i = 0; i < ninput_files; i++)
    {
	if (matrix[i][i] == 0)
	{
	    sprintf(error_msg, "input file %d has data all zero", i+1);
	    return  ERROR;
	}
    }

    if (correlation)
    {
	for (i = 0; i < ninput_files; i++)
	    work[i] = (float) sqrt((double) matrix[i][i]);

	for (i = 0; i < ninput_files; i++)
	    for (j = i; j < ninput_files; j++)
		matrix[i][j] /= work[i] * work[j];
    }
    else /* make largest entry 1 */
    {
	d = 0;

	for (i = 0; i < ninput_files; i++)
	    d = MAX(d, matrix[i][i]);

	for (i = 0; i < ninput_files; i++)
	    for (j = i; j < ninput_files; j++)
		matrix[i][j] /= d;
    }

    for (i = 0; i < ninput_files; i++)
	for (j = i+1; j < ninput_files; j++)
	    matrix[j][i] = matrix[i][j];

    return  principal_components(ninput_files, matrix,
					eigenvalues, work, error_msg);
}

static int cmp_eigenvalues(Generic_ptr d1, Generic_ptr d2)
{
    int i = *((int *) d1);
    int j = *((int *) d2);

    if (eigenvalues[i] < eigenvalues[j])
	return  -1;
    else if (eigenvalues[i] > eigenvalues[j])
	return  1;
    else
	return  0;
}

static void components_used()
{
    int i, n, o;
    float t;

    t = 0;
    for (i = 0; i < ninput_files; i++)
	t += eigenvalues[i];

    for (i = 0; i < ninput_files; i++)
	eigenvalues[i] /= t;

    n = 0;
    t = 0;

    do
    {
	o = *order[n];
	t += eigenvalues[o];

	n++;
    }   while ((n < ncomponents) || (t < fraction));

    ncomponents = n;
    fraction = t;
}

Status block_process(Size_info *size_info, Store_info *store_info,
    File_info *file_info, Components_info *components_info, String error_msg)
{
    int i, p;
    char *msg;

    ndim = size_info->ndim;
    npoints = size_info->npoints;
    block_size = size_info->block_size;

    store_in = store_info->store_in;
    matrix = store_info->matrix;
    eigenvalues = store_info->eigenvalues;
    work = store_info->work;
    order = store_info->order;

    ninput_files = file_info->ninput_files;

    ncomponents = MIN(components_info->ncomponents, ninput_files);
    fraction = MIN(1, components_info->fraction);
    correlation = components_info->correlation;

    if (!(file_info->blocked))
	RETURN_ERROR_MSG("input file must be blocked");

    init_arrays();

    sprintf(error_msg, "allocating memory for block_io_in");
    MALLOC(block_io_in, Block_IO, ninput_files);

    for (i = 0; i < ninput_files; i++)
    {
	block_io_in[i].name = file_info->input_files[i];
	block_io_in[i].file = file_info->file_in[i];
	block_io_in[i].swapped = file_info->swapped;
	block_io_in[i].integer = file_info->integer;
	block_io_in[i].deflated = FALSE;
	block_io_in[i].header = file_info->header;
	block_io_in[i].dir_size = 0;
	block_io_in[i].directory = (int *) NULL;
	block_io_in[i].block_size = size_of_block;
	block_io_in[i].byte_size = BYTES_PER_WORD;

	CHECK_STATUS(init_block_read(&(block_io_in[i]), error_msg));

	ZERO_VECTOR(matrix[i], ninput_files);
    }

    printf("finding matrix:\n");
    p = MAX(total_nblocks/32, 1);
    for (i = 0; i < total_nblocks; i++)
    {
	if (!(i % p))
	{
	    printf("\t... working on block %d of %d\n", i+1, total_nblocks);
	    FLUSH;
	}

	sprintf(error_msg, "block %d: ", i+1);
	msg = error_msg + strlen(error_msg);

	ARRAY_OF_INDEX(blocks, i, cum_nblocks, ndim);

	CHECK_STATUS(disk_to_store(i, msg));

	if (find_end_points())
	    boundary_process();
	else
	    normal_process();
    }

    FREE(block_io_in, Block_IO);

    printf("finding components\n");

    CHECK_STATUS(find_components(error_msg));
    heap_sort((Generic_ptr *) order, ninput_files, FALSE, cmp_eigenvalues);
    components_used();

    components_info->ncomponents = ncomponents;
    components_info->fraction = fraction;

    return  OK;
}

static void boundary_output()
{
    int i, j, k, o;

    for (i = 0; i < noutput_files; i++)
    {
	o = *order[i];

	for (j = 0; j < size_of_block; j++)
	{
	    ARRAY_OF_INDEX(points, j, cum_block_size, ndim);

	    if (point_in_range())
	    {
		for (k = 0; k < ninput_files; k++)
		    store_out[i][j] += store_in[k][j] * matrix[k][o];
	    }
	}
    }
}

static void normal_output()
{
    int i, j, k, o;

    for (i = 0; i < noutput_files; i++)
    {
	o = *order[i];

	for (j = 0; j < size_of_block; j++)
	{
	    for (k = 0; k < ninput_files; k++)
		store_out[i][j] += store_in[k][j] * matrix[k][o];
	}
    }
}

Status block_output(Size_info *size_info, Store_info *store_info,
    File_info *file_info, Components_info *components_info, String error_msg)
{
    int i, j, p;
    char *msg;

    ndim = size_info->ndim;
    npoints = size_info->npoints;
    block_size = size_info->block_size;

    store_in = store_info->store_in;
    store_out = store_info->store_out;
    matrix = store_info->matrix;
    eigenvalues = store_info->eigenvalues;
    order = store_info->order;

    ninput_files = file_info->ninput_files;
    ncomponents = MIN(components_info->ncomponents, ninput_files);
    noutput_files = MIN(file_info->noutput_files, ncomponents);

    if (!(file_info->blocked))
	RETURN_ERROR_MSG("input file must be blocked");

    init_arrays();

    sprintf(error_msg, "allocating memory for block_io_in");
    MALLOC(block_io_in, Block_IO, ninput_files);

    sprintf(error_msg, "allocating memory for block_io_out");
    MALLOC(block_io_out, Block_IO, noutput_files);

    for (i = 0; i < ninput_files; i++)
    {
	block_io_in[i].name = file_info->input_files[i];
	block_io_in[i].file = file_info->file_in[i];
	block_io_in[i].swapped = file_info->swapped;
	block_io_in[i].integer = file_info->integer;
	block_io_in[i].deflated = FALSE;
	block_io_in[i].header = file_info->header;
	block_io_in[i].dir_size = 0;
	block_io_in[i].directory = (int *) NULL;
	block_io_in[i].block_size = size_of_block;
	block_io_in[i].byte_size = BYTES_PER_WORD;

	CHECK_STATUS(init_block_read(&(block_io_in[i]), error_msg));
    }

    for (i = 0; i < noutput_files; i++)
    {
	block_io_out[i].name = file_info->output_files[i];
	block_io_out[i].file = file_info->file_out[i];
	block_io_out[i].block_size = size_of_block;
	block_io_out[i].deflated = FALSE;

	CHECK_STATUS(init_block_write(&(block_io_out[i]), error_msg));
    }

    printf("outputting data:\n");

    p = MAX(total_nblocks/32, 1);
    for (i = 0; i < total_nblocks; i++)
    {
	if (!(i % p))
	{
	    printf("\t... outputting block %d of %d\n", i+1, total_nblocks);
	    FLUSH;
	}

	sprintf(error_msg, "block %d: ", i+1);
	msg = error_msg + strlen(error_msg);

	ARRAY_OF_INDEX(blocks, i, cum_nblocks, ndim);

	CHECK_STATUS(disk_to_store(i, msg));

	for (j = 0; j < noutput_files; j++)
	    ZERO_VECTOR(store_out[j], size_of_block);

	if (find_end_points())
	    boundary_output();
	else
	    normal_output();

	CHECK_STATUS(store_to_disk(i, msg));
    }

    FREE(block_io_in, Block_IO);
    FREE(block_io_out, Block_IO);

    return  OK;
}
