// Copyright 2015-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License"). You may
// not use this file except in compliance with the License. A copy of the
// License is located at
//
//     http://aws.amazon.com/apache2.0/
//
// or in the "license" file accompanying this file. This file is distributed
// on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
// express or implied. See the License for the specific language governing
// permissions and limitations under the License.

package sts

import (
	log "github.com/sirupsen/logrus"
	"github.com/aws/amazon-ecs-cli/ecs-cli/modules/clients"
	"github.com/aws/amazon-ecs-cli/ecs-cli/modules/config"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sts"
	"github.com/aws/aws-sdk-go/service/sts/stsiface"
)

// Client sts interface
type Client interface {
	GetAWSAccountID() (string, error)
}

// stsClient implements Client
type stsClient struct {
	client stsiface.STSAPI
	config *config.CommandConfig
}

// NewClient Creates a new sts client
func NewClient(config *config.CommandConfig) Client {
	client := sts.New(session.New(config.Session.Config))
	client.Handlers.Build.PushBackNamed(clients.CustomUserAgentHandler())
	return newClient(config, client)
}

func newClient(config *config.CommandConfig, client stsiface.STSAPI) Client {
	return &stsClient{
		client: client,
		config: config,
	}
}

// GetAWSAccountID returns the accountId of the caller
func (c *stsClient) GetAWSAccountID() (string, error) {
	log.Info("Getting AWS account ID...")

	resp, err := c.client.GetCallerIdentity(&sts.GetCallerIdentityInput{})
	if err != nil {
		return "", err
	}
	return aws.StringValue(resp.Account), nil
}
