/*
 * Copyright David Leonard, 2000. All rights reserved. <Berkeley license>
 *
 * Based on an ancient game published in Australian Personal Computer 
 * Magazine (1986?) also called 'amaze'. Written in BASIC I recall. No idea
 * who the author was, I'm afraid. I just remember porting it from QBASIC
 * to AppleSoft BASIC, and simplifying the perspective line drawing code.
 */

#include <stdio.h>
#include <curses.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/time.h>

int show_compass = 0;			/* -c */
int show_map = 0;			/* -m */
int can_shoot = 0;			/* -s */
int remember_visited = 0;		/* -v */
int no_quit_on_win = 0;			/* -n */

chtype SPACE =		' ' | A_NORMAL;	/* Space you walk walk through */
chtype BLOCK =		' ' | A_REVERSE;/* A block that you can't walk through */
chtype OBSPACE =	'#' | A_NORMAL; /* Obscured space */
chtype VISITED =	'.' | A_NORMAL;	/* Visited space */
chtype GOAL =		'%' | A_NORMAL;	/* Exit from the maze */
chtype START =		'+' | A_NORMAL;	/* Starting point in the maze */
attr_t WIN =		      A_STANDOUT;
chtype SKY =		' ' | A_NORMAL;

enum dir { DIR_DOWN=0, DIR_RIGHT=1, DIR_UP=2, DIR_LEFT=3 };
#define _TOD(d) (enum dir)((d) % 4)
#define _TOI(d) (int)(d)
#define LEFT_OF(d) _TOD(_TOI(d) + 1)
#define REVERSE_OF(d) _TOD(_TOI(d) + 2)
#define RIGHT_OF(d) _TOD(_TOI(d) + 3)

chtype VLINE =		'|' | A_NORMAL;		/* | */
chtype HLINEt =		'_' | A_NORMAL;		/* _ at top */
chtype HLINEb =		'_' | A_NORMAL;		/* _ at bottom */
chtype DIAG1 =		'\\' | A_NORMAL;	/* \ */
chtype DIAG2 =		'/' | A_NORMAL;		/* / */

static struct { int y, x; } dirtab[4] = 
	{ { 1, 0 }, { 0, 1 }, { -1, 0 }, { 0, -1 } };
static chtype ptab[4] = { 'v' | A_BOLD, '>' | A_BOLD, 
			  '^' | A_BOLD, '<' | A_BOLD};
int px, py;		/* Player position */
enum dir pdir;		/* Player direction */
chtype punder;		/* character under player, if any */
int gx, gy;		/* Goal position */
int gdist;		/* Distance from start to goal */
int won = 0;		/* Reached goal */
int cheated = 0;	/* Cheated somehow */

WINDOW *map;		/* Actual maze */
WINDOW *umap;		/* Maze map that user sees */
WINDOW *view;		/* 3D perspective view player sees */
WINDOW *msg;		/* Message information window */

void drawview(void);
void mappmove(int, int, enum dir);

/* Look at position (y,x) in the maze map */
chtype
at(y, x)
	int y, x;
{
	int maxy, maxx;

	if (y == py && x == px)
		return punder;

	getmaxyx(map, maxy, maxx);
	if (y < 0 || y >= maxy || x < 0 || x >= maxx)
		return SPACE;
	else
		return mvwinch(map, y, x);
}

void
copyumap(y, x, fullvis)
	int y, x, fullvis;
{
	chtype c;

	c = at(y, x);
	if (!fullvis && c == SPACE && mvwinch(umap, y, x) != SPACE)
		c = OBSPACE;
	mvwaddch(umap, y, x, c);
}

struct path {
	int y, x;
	int ttl;		/* Time until this path stops */
	int spawns;		/* Max number of forks this path can do */
	int distance;		/* Distance from start */
	struct path *next;	
};

/*
 * A better maze-digging algorithm.
 * Simultaneously advance multiple digging paths through the map.
 * This is done by having a work queue of advancing paths. 
 * Occasionally a path can fork; thus adding more to the work
 * queue and diversifying the maze.
 */
void
eatmaze(starty, startx)
	int starty, startx;
{
	struct path *path_free, *path_head, *path_tail;
	struct path path_storage[2000];
	struct path *p, *s;
	int i, try;
	int y, x, dy, dx;
	int sdir;

	/* Set up the free list of path cells */
	for (i = 2; i < sizeof path_storage / sizeof path_storage[0]; i++)
		path_storage[i].next = &path_storage[i-1];
	path_storage[1].next = NULL;
	path_free = &path_storage[sizeof path_storage / sizeof path_storage[0] - 1];

	/* Set up the initial path cell */
	path_storage[0].y = starty;
	path_storage[0].x = startx;
	mvwaddch(map, starty, startx, SPACE);

	/* Set up the initial goal. It will move later. */
	gy = starty;
	gx = startx;
	gdist = 0;

	/* Initial properties of the root path */
	path_storage[0].ttl = 50;
	path_storage[0].spawns = 20;
	path_storage[0].distance = 0;

	/* Put the initial path into the queue */
	path_storage[0].next = NULL;
	path_head = path_tail = &path_storage[0];

	while (path_head != NULL) {
		/* Dequeue */
		p = path_head;
		path_head = p->next;
		if (path_head == NULL)
			path_tail = NULL;

		/* There's a large chance that some paths miss a turn */
		if (random() % 100 < 60)
			goto requeue;

		/* First thing we do is advance the path. */
		y = p->y;
		x = p->x;

		sdir = random() % 4;
		for (try = 0; try < 4; try ++) {
			dx = dirtab[(sdir + try) % 4].x;
			dy = dirtab[(sdir + try) % 4].y;

			/* Going back on ourselves? */
			if (at(y + dy, x + dx) != BLOCK)
				continue;

			/* Connecting to another path? */
			if (at(y + dy * 2, x + dx * 2) != BLOCK)
				continue;
			if (at(y + dy + dx, x + dx - dy) != BLOCK)
				continue;
			if (at(y + dy - dx, x + dx + dy) != BLOCK)
				continue;

			break;
		}
		if (try == 4 || p->ttl <= 0) {
			/* Failed: the path is placed on the free list. */
			p->next = path_free;
			path_free = p;
			continue;
		}

		/* Dig the path a bit */
		p->y = y + dy;
		p->x = x + dx;
		mvwaddch(map, p->y, p->x, SPACE);
		p->ttl--;
		p->distance++;

		if (p->distance > gdist) {
			gx = p->x;
			gy = p->y;
			gdist = p->distance;
		}

		/* Decide if we should spawn */
		if (/*random() % (p->ttl + 1) < p->spawns && */ path_free) {
			/* Take a new path element off the free list */
			s = path_free;
			path_free = s->next;


			/* Insert it at the tail of the queue */
			s->next = NULL;
			if (path_tail) path_tail->next = s;
			else           path_head = s;
			path_tail = s;

			/* Newly spawned path s will inherit most properties from p */
			s->y = p->y;
			s->x = p->x;
			s->ttl = p->ttl + random() % 10;
			s->spawns = p->spawns;
			s->distance = p->distance;

			/* p->spawns--; */
		}

	requeue:
		/* Put p onto the tail of the queue */
		p->next = NULL;
		if (path_tail) path_tail->next = p;
		else           path_head = p;
		path_tail = p;
	}
}

/* Move the player to a new position/direcion in the maze map */
void
mappmove(newpy, newpx, newpdir)
	int newpy, newpx;
	enum dir newpdir;
{
	mvwaddch(map, py, px, punder);
	copyumap(py, px, 1);
	punder = at(newpy, newpx);
	py = newpy;
	px = newpx;
	pdir = newpdir;
	copyumap(py, px, 1);
	mvwaddch(umap, py, px, ptab[(int)pdir]);
	wmove(umap, 0, 0);
	wnoutrefresh(umap);
}

void
clearmap(amap)
	WINDOW *amap;
{
	int maxy, maxx;
	int y, x;

	getmaxyx(map, maxy, maxx);
	werase(amap);
	for (y = 0; y < maxy; y++)
		for (x = 0; x < maxx; x++)
			mvwaddch(amap, y, x, BLOCK);
}

/* Reveal the solution to the user */
void
showmap()
{
	int maxy, maxx, y, x;
	chtype ch, och;

	getmaxyx(umap, maxy, maxx);
	for (y = 0; y < maxy; y++)
		for (x = 0; x < maxx; x++) {
			ch = at(y, x);
			if (ch == SPACE) {
				och = mvwinch(umap, y, x);
				if (och == BLOCK || och == OBSPACE)
					ch = OBSPACE;
			}
			mvwaddch(umap, y, x, ch);
		}
	mvwaddch(umap, py, px, ptab[(int)pdir]);
	wnoutrefresh(umap);
}

/*
 * Create a new maze map
 * The algorithm here is quite inferior to that presented in the
 * magazine. I could only remember the gist of it: recursively dig a
 * trail that doesn't touch any other part of the maze, keeping track
 * of all possible points where the path could fork. Later on try those
 * possible branches; put limits on the segment lengths etc.
 */
void
makemaze()
{
	int maxy, maxx;
	int i;

	/* Get the window dimensions */
	getmaxyx(map, maxy, maxx);

	clearmap(map);

	py = rand() % (maxy - 2) + 1; /* maxy/2 */
	px = rand() % (maxx - 2) + 1; /* maxx/2 */
	eatmaze(py, px);

	/* Face in an interesting direction: */
	pdir = DIR_UP;
	for (i = 0; 
	     i < 4 && at(py + dirtab[pdir].y, px + dirtab[pdir].x) == BLOCK;
	     i++)
		pdir = LEFT_OF(pdir);

	mvwaddch(map, py, px, START);
	mvwaddch(map, gy, gx, GOAL);
	punder = START;
	mappmove(py, px, pdir);
}

/* 
 * Position of the start of a vertical block edge
 * Also used for horizontal edges
 *
 * ___
 *  h |\h
 *    | \	f = front
 * f  | s 	s = side
 *    |         v = vertical edge
 *    v /       h = horizontal edge
 * ___|/
 */
#define MAXDIST 6
static int barx[MAXDIST + 2] = { -1, 3, 9, 13, 17, 19, 21, 21 };
static int bary[MAXDIST + 2] = { 0, 2, 5,  7,  9, 10, 11, 11 };

/* Draw the close vertical edge of a block */
void
draw_vert_edge(dist, right)
{
	int y;
	for (y = bary[dist]; y <= 22 - bary[dist]; y++)
		if (right)
			mvwaddch(view, y, 43 - barx[dist], VLINE);
		else
			mvwaddch(view, y, barx[dist], VLINE);
}

/* Draw the horizontal edge of a block */
void
draw_horiz_front(dist, right)
{
	int x;

	for (x = barx[dist-1] + 1; x < barx[dist]; x++)
		if (right) {
			mvwaddch(view, bary[dist] - 1, 43 - x, HLINEt);
			mvwaddch(view, 22 - bary[dist], 43 - x, HLINEb);
		} else {
			mvwaddch(view, bary[dist] - 1, x, HLINEt);
			mvwaddch(view, 22 - bary[dist], x, HLINEb);
		}
}

/* Draw the horiz edge of a wall in the way */
void
draw_horiz_wall(dist)
{
	int x;
	for (x = barx[dist] + 1; x <= 43 - (barx[dist] + 1); x++) {
		mvwaddch(view, bary[dist] - 1, x, HLINEt);
		mvwaddch(view, 22 - (bary[dist]), x, HLINEb);
	}
}

/* Draw the (visually) diagonal edge of a block */
void 
draw_horiz_side(dist, right)
	int dist, right;
{
	int y, x;

	for (y = bary[dist], x = barx[dist] + 1; x < barx[dist + 1]; y++, x+=2)
		if (right) {
			mvwaddch(view, y, 43 - x, DIAG2);
			mvwaddch(view, 22-y, 43 - x, DIAG1);
		} else {
			mvwaddch(view, y, x, DIAG1);
			mvwaddch(view, 22-y, x, DIAG2);
		}
}

/* Draw the floor in the centre of view */
void
draw_floor_centre(dist, ch)
	int dist;
	chtype ch;
{
	int y, x, xx;

	for (y = bary[dist], x = barx[dist] + 1; x < barx[dist + 1]; y++, x+=2)
		for (xx = x + 1; xx <= 43 - (x + 1); xx++) {
			mvwaddch(view, y, xx, SKY);
			mvwaddch(view, 22-y, xx, ch);
		}
}

/* Draw the floor to the side */
void
draw_floor_side(dist, right, ch)
	int dist;
	int right;
	chtype ch;
{
	int y, x, xx;

	for (y = bary[dist], x = barx[dist] + 1; x < barx[dist + 1]; y++, x+=2)
		for (xx = barx[dist]; xx <= x; xx++)
			if (right) {
				mvwaddch(view, y, 43-xx, SKY);
				mvwaddch(view, 22-y, 43-xx, ch);
			} else {
				mvwaddch(view, y, xx, SKY);
				mvwaddch(view, 22-y, xx, ch);
			}
}

/* Draw the view the player would see */
void
drawview()
{
	int dist;
	int dx, dy, x, y;
	int a, l, la, r, ra;
	int g, gl, gr;
	int lastdist;

	if (!show_map) {
		clearmap(umap);
		copyumap(gy, gx, 1);
	}


	dx = dirtab[(int)pdir].x;
	dy = dirtab[(int)pdir].y;

	for (dist = 1; dist < MAXDIST; dist++)
		if (at(py + dy * dist, px + dx * dist) == BLOCK)
			break;

	lastdist = dist - 1;
	werase(view);
	while (--dist >= 0) {
		x = px + dx * dist;
		y = py + dy * dist;

		/* XXX only looks one cell to the side */

		/* ahead */
		a = (at(y + dy, x + dx) == BLOCK);
		/* to the left */
		l = (at(y - dx, x + dy) == BLOCK);
		/* left, ahead */
		la = (at(y - dx + dy, x + dy + dx) == BLOCK);
		/* to the right */
		r = (at(y + dx, x - dy) == BLOCK);
		/* right, ahead */
		ra = (at(y + dx + dy, x - dy + dx) == BLOCK);
		/* floor */
		g = at(y, x);
		gl = at(y - dx, x + dy);
		gr = at(y + dx, x - dy);

		if (dist == lastdist) {
			if (a || la) draw_vert_edge(dist +1, 0);
			if (a || ra) draw_vert_edge(dist +1, 1);
		}

		if (g != BLOCK)
			draw_floor_centre(dist, g);
		if (gl != BLOCK)
			draw_floor_side(dist, 0, gl);
		if (gr != BLOCK)
			draw_floor_side(dist, 1, gr);

		if (l) {
			draw_horiz_side(dist, 0);
			if (dist != 0) draw_vert_edge(dist, 0);
			draw_vert_edge(dist + 1, 0);
		}
		if (r)  {
			draw_horiz_side(dist, 1);
			if (dist != 0) draw_vert_edge(dist, 1);
			draw_vert_edge(dist + 1, 1);
		}

		if (!l && la)
			draw_horiz_front(dist + 1, 0);
		if (!r && ra)
			draw_horiz_front(dist + 1, 1);
		if (a) {
			draw_horiz_wall(dist + 1);
		}

		copyumap(y + dy, x + dx, 0);	/* ahead */
		copyumap(y, x, 1);		/* here */
		copyumap(y - dx, x + dy, 0);	/* left */
		copyumap(y + dx, x - dy, 0);	/* right */
		if (!l)
			copyumap(y - dx + dy, x + dy + dx, 0);	/* left ahead */
		if (!r)
			copyumap(y + dx + dy, x - dy + dx, 0);	/* right ahead */
	}

	if (show_compass) {
		/* Provide a compass pointing 'north' */
		switch (pdir) {
		case DIR_UP:
			mvwaddch(view, 21, 21, '^');
			mvwaddch(view, 22, 21, '|');
			break;
		case DIR_DOWN:
			mvwaddch(view, 21, 21, '|');
			mvwaddch(view, 22, 21, 'v');
			break;
		case DIR_LEFT:
			mvwaddch(view, 21, 21, '-');
			mvwaddch(view, 21, 22, '>');
			break;
		case DIR_RIGHT:
			mvwaddch(view, 21, 20, '<');
			mvwaddch(view, 21, 21, '-');
			break;
		}
	}

	wnoutrefresh(view);
	mvwaddch(umap, py, px, ptab[(int)pdir]);
}

void
shoot()
{
	int x, y, dx, dy;
	int maxy, maxx;

	dx = dirtab[(int)pdir].x;
	dy = dirtab[(int)pdir].y;
	getmaxyx(map, maxy, maxx);

	x = px + dx;
	y = py + dy;
	while (at(y, x) != BLOCK) {
		x += dx;
		y += dy;
	}
	if (x == 0 || y == 0 || x == maxx - 1 || y == maxy - 1)
		beep();
	else
		mvwaddch(map, y, x, SPACE);
}

void
win()
{
	werase(msg);
	wattrset(msg, WIN);
	mvwprintw(msg, 0, 18, "amazing!");
	wattrset(msg, A_NORMAL);
	won++;
	wnoutrefresh(msg);
	showmap();
	show_map = 1;
}

/* Try to move the player in the direction given */
void
trymove(dir)
	enum dir dir;
{
	int nx, ny;

	ny = py + dirtab[(int)dir].y;
	nx = px + dirtab[(int)dir].x;

	if (at(ny, nx) == BLOCK) {
		beep();
		return;
	}

	if (at(ny, nx) == GOAL)
		win();

	mappmove(ny, nx, pdir);
	if (remember_visited && punder == SPACE)
		punder = VISITED;
	drawview();
}

void
walkleft()
{
	int a, l;
	int dx, dy;
	int owon = won;

	nodelay(umap, 1);
	while (1) {
		wmove(umap, 0, 0);
		doupdate();
		/* usleep(100000); */		/* slower walk */
		if (wgetch(umap) != ERR)
			break;
		if (won != owon)
			break;

		dx = dirtab[(int)pdir].x;
		dy = dirtab[(int)pdir].y;
		
		/* ahead */
		a = (at(py + dy, px + dx) == BLOCK);
		/* to the left */
		l = (at(py - dx, px + dy) == BLOCK);

		if (!l) {
			mappmove(py, px, LEFT_OF(pdir));
			drawview();
			doupdate();
usleep(100000);
			trymove(pdir);
			continue;
		}
		if (a) {
			mappmove(py, px, RIGHT_OF(pdir));
			drawview();
			continue;
		} 
		trymove(pdir);
	}
	nodelay(umap, 0);
}

int
amaze()
{
	int quitting;

	srandom(time(NULL));

	initscr();
	start_color();
	cbreak();
	noecho();
	nonl();
	intrflush(stdscr, FALSE);
	keypad(stdscr, TRUE);

	OBSPACE = ACS_CKBOARD;

	if (has_colors()) {
		short fg, bg;
		int i;

		/* Get the normal colours */
		pair_content(0, &fg, &bg);
		/* Make the goal character red on white if possible */
		init_pair(1, COLOR_RED, bg);
		GOAL |= COLOR_PAIR(1) | A_BOLD;
		BLOCK   = ' ' | A_REVERSE | A_BOLD;
		OBSPACE = ' ' | A_REVERSE | A_DIM;
		WIN = COLOR_PAIR(1) | A_BOLD;

		init_pair(2, COLOR_YELLOW, bg);
		for (i = 0; i < 4; i++)
			ptab[i] |= COLOR_PAIR(2);

		init_pair(3, fg, COLOR_BLUE);
		SKY = ' ' | COLOR_PAIR(3);

		VLINE = ACS_VLINE;
		HLINEt = ACS_S9 | COLOR_PAIR(3);
		HLINEb = ACS_S9;
	}

	view = newwin(23, 45, 0, 0);
	leaveok(view, 0);
	scrollok(view, 0);

	/* This window is never updated, used purely for storage */
	map = newwin(23, 79 - 45, 0, 45);

	umap = newwin(23, 79 - 45, 0, 45);
	leaveok(umap, 0);
	scrollok(umap, 0);

	clearmap(umap);

	msg = newwin(1, 78, 23, 0);
	leaveok(msg, 1);
	scrollok(msg, 0);

	makemaze(map);

	/* Show where the goal is */
	copyumap(gy, gx, 1);

	werase(msg);
	mvwprintw(msg, 0, 19, "amaze!");
	wnoutrefresh(msg);

	quitting = 0;

	mappmove(py, px, pdir);
	if (remember_visited)
		punder = VISITED;
	else
		punder = SPACE;
	drawview();

	while (!quitting) {
		wmove(umap, 0, 0);
		doupdate();
		nodelay(umap, 0);
		switch (wgetch(umap)) {
		case 'q': quitting = 1; break;
		case 'k': trymove(pdir); break;
		case 'j': trymove(REVERSE_OF(pdir)); break;
		case 'H': trymove(LEFT_OF(pdir)); break;
		case 'L': trymove(RIGHT_OF(pdir)); break;
		case 'h': 
			mappmove(py, px, LEFT_OF(pdir));
			drawview();
			break;
		case 'l': 
			mappmove(py, px, RIGHT_OF(pdir));
			drawview();
			break;
		case ' ':
			if (can_shoot) {
				shoot();
				cheated++;
				drawview();
			} else
				beep();
			break;
		case 'w':
			cheated++;
			walkleft();
			break;
		default: 
			beep();
			break;
		}
		if (won && !no_quit_on_win) {
			wmove(umap, 0, 0);
			doupdate();
			beep();
			sleep(1);
			break;
		}
	}

	endwin();

	if (won) {
		if (cheated) {
			printf("You cheated.\n");
			return 0;
		}
		printf("You win!\n");
		return 1;
	} else {
		printf("You lose.\n");
		return 0;
	}
}


int
main(argc, argv)
	int argc;
	char *argv[];
{
	int ch;
	extern int optind;

	while ((ch = getopt(argc, argv, "cmnsv")) != -1)
		switch (ch) {
		case 'c':	show_compass = 1; break;
		case 'm':	show_map = 1; break;
		case 'n':	no_quit_on_win = 1; break;
		case 's':	can_shoot = 1; break;
		case 'v':	remember_visited = 1; break;
		default:
			fprintf(stderr, "usage: %s [-cmnsv]\n", argv[0]);
			exit(1);
		}
	if (argv[optind])  {
		fprintf(stderr, "usage: %s [-cmnsv]\n", argv[0]);
		exit(1);
	}

	if (amaze())
		exit(0);	/* win */
	else
		exit(1);	/* lose */
}
