// directory.cpp
// Revision 14-may-2005

#include "directory.h"

#include <iostream>
#include <vector>
#include <algorithm>

#include <glob.h>
#include <assert.h>

#define ASSERT assert

using std::cerr;


namespace {


bool validcpmfilename (const std::string & result,
	std::string::size_type maxfilelen,
	std::string::size_type maxextlen)
{
	std::string file;
	std::string ext;
	const std::string::size_type point= result.find ('.');
	if (point != std::string::npos)
	{
		file= result.substr (0, point);
		ext= result.substr (point + 1);
	}
	else
		file= result;

	if (file.size () > maxfilelen)
		return false;
	if (ext.size () > maxextlen)
		return false;

	// Only one '.' allowed,
	if (ext.find ('.') != std::string::npos)
		return false;

	#if 0
	// No lowercase.
	for (std::string::size_type i= 0, l= file.size (); i < l; ++i)
		if (islower (file [i]) )
			return false;
	for (std::string::size_type i= 0, l= ext.size (); i < l; ++i)
		if (islower (ext [i]) )
			return false;
	#endif
	return true;
}

FcbFile createfcb (const std::string & s)
{
	const std::string::size_type point= s.find ('.');
	std::string file;
	std::string ext;
	if (point != std::string::npos)
	{
		file= s.substr (0, point);
		ext= s.substr (point + 1);
	}
	else
		file= s;
	FcbFile fcb;
	fcb.setfilename (file);
	fcb.setfileext (ext);
	return fcb;
}


} // namespace


//***********************************************
//		FindFile::Internal
//***********************************************


class FindFile::Internal {
public:
	Internal (const std::string & directory, bool debug);
	Internal (const std::string & directory,
		const std::string & mask, bool debug);
	Internal (const std::string & directory,
		FcbFile & fcbmask, bool debug);

	void addref ();
	void delref ();

	FindFile::Result::code findfirst
		(const std::string & mask, std::string & result);
	FindFile::Result::code findnext (std::string & result);
	FindFile::Result::code findfirst
		(FcbFile & fcbmask, FcbFile & result);
	FindFile::Result::code findnext (FcbFile & result);

	FindFile::Result::code get (std::string & result);
	FindFile::Result::code get (FcbFile & result);
private:
	class AvoidWarning { };
	friend class AvoidWarning;

	~Internal ();
	void operator = (const Internal &); // Forbidden.

	static std::string adjustdir (std::string directory);
	void closesearch ();

	void startsearch (std::string file, std::string ext);

	size_t refcount;
	const bool debug_flag;
	bool finding;
	const std::string dir;
	size_t n;

	std::vector <std::string> vfile;

	typedef FindFile::Result Result;
	FindFile::Result::code result;
};

FindFile::Internal::Internal (const std::string & directory, bool debug) :
	refcount (1),
	debug_flag (debug),
	finding (false),
	dir (adjustdir (directory) ),
	result (Result::NoFile)
{
}

FindFile::Internal::Internal (const std::string & directory,
		const std::string & mask, bool debug) :
	refcount (1),
	debug_flag (debug),
	finding (false),
	dir (adjustdir (directory) ),
	result (Result::NoFile)
{
	if (debug_flag)
		cerr << "FindFile mask: " << mask << "\r\n";

	std::string file;
	std::string ext;
	const std::string::size_type pos= mask.find ('.');
	if (pos == std::string::npos)
		file= mask;
	else
	{
		file= mask.substr (0, pos);
		ext= mask.substr (pos + 1);
	}

	startsearch (file, ext);
}

FindFile::Internal::Internal (const std::string & directory,
		FcbFile & fcbmask, bool debug) :
	refcount (1),
	debug_flag (debug),
	finding (false),
	dir (adjustdir (directory) ),
	result (Result::NoFile)
{
	startsearch (fcbmask.getfilename (), fcbmask.getfileext ());
}

FindFile::Internal::~Internal ()
{
	ASSERT (refcount == 0);
	if (finding)
		closesearch ();
}

void FindFile::Internal::addref ()
{
	++refcount;
}

void FindFile::Internal::delref ()
{
	if (--refcount == 0)
		delete this;
}


std::string FindFile::Internal::adjustdir (std::string directory)
{
	if (! directory.empty () )
	{
		if (directory [directory.size () - 1] != '/')
			directory+= '/';
	}
	return directory;
}

void FindFile::Internal::closesearch ()
{
	finding= false;
	result= Result::NoFile;
	vfile.clear ();
	n= 0;
}

void FindFile::Internal::startsearch (std::string file, std::string ext)
{
	if (debug_flag)
	{
		cerr << "File: '" << file << "' Ext: '" << ext <<
			"'\r\n";
	}

	closesearch ();

	// CP/M allows that a ? replaces an empty position,
	// glob does not. Thus we replace the last '?' with
	// a '*', but save the max length allowed to test
	// the results against.
	const std::string::size_type maxfilelen= file.size ();
	const std::string::size_type maxextlen= ext.size ();
	int l= file.size () - 1;
	if (l > 0 && file [l] == '?')
	{
		while (l >= 0 && file [l] == '?')
			--l;
		if (l >= 0)
		{
			file= file.substr (0, l + 1);
			if (file.size () < 8)
				file+= '*';
		}
		else
			file= "*";
	}
	l= ext.size () - 1;
	if (l > 0 && ext [l] == '?')
	{
		while (l >= 0 && ext [l] == '?')
			--l;
		if (l >= 0)
		{
			ext= ext.substr (0, l + 1);
			if (ext.size () < 3)
				ext+= '*';
		}
		else
			ext= "*";
	}

	std::string s= dir + file + '.' + ext;

	if (debug_flag)
		cerr << "FindFile path: " << s << "\r\n";

	int r;
	glob_t g;
	int flag= GLOB_NOSORT | GLOB_ERR; // Don't sort, we do it.
	switch ( (r= glob (s.c_str (), flag, NULL, & g) ) )
	{
	case 0:
	case GLOB_NOMATCH:
		break;
	case GLOB_ABORTED:
		result= Result::NoDisk;
		return;
	default:
		result= Result::NoDisk;
		return;
	}

	// I want that FILE is found when searching for FILE.*
	// or for "FILE.   ", thus we do a second search without
	// extension in that cases.

	if (ext.empty () || ext == "*")
	{
		int secondflag= flag;
		if (r == 0)
			secondflag|= GLOB_APPEND;
		s= dir + file;
		if (debug_flag)
			cerr << "FindFile second path: " << s << "\r\n";
		glob (s.c_str (), secondflag, NULL, &g);
	}

	if (g.gl_pathc == 0)
	{
		globfree (& g);
		result= Result::NoFile;
		return;
	}
	else
		result= Result::Ok;

	finding= true;
	for (size_t i= 0; i < static_cast <size_t> (g.gl_pathc); ++i)
	{
		std::string filename (g.gl_pathv [i] );
		const std::string::size_type pos= filename.rfind ('/');
		if (pos != std::string::npos)
			filename.erase (0, pos + 1);
		if (validcpmfilename (filename, maxfilelen, maxextlen) )
			vfile.push_back (filename);
	}

	globfree (& g);

	// As two glob operations were done, we need to
	// remove duplicates.	
	std::sort (vfile.begin (), vfile.end () );
	std::vector <std::string>::iterator it=
		unique (vfile.begin (), vfile.end () );
	vfile.erase (it, vfile.end () );

	// TODO: still a problem, if FILE and FILE. exists
	// CP/M see the two but can open only one.
}

FindFile::Result::code FindFile::Internal::findfirst
		(const std::string & mask, std::string & result)
{
	if (debug_flag)
		cerr << "FindFile mask: " << mask << "\r\n";

	std::string file;
	std::string ext;
	const std::string::size_type pos= mask.find ('.');
	if (pos == std::string::npos)
		file= mask;
	else
	{
		file= mask.substr (0, pos);
		ext= mask.substr (pos + 1);
	}

	startsearch (file, ext);

	return findnext (result);
}

FindFile::Result::code FindFile::Internal::findnext (std::string & result)
{
	if (! finding)
		return Result::NoFile;

	if (n >= vfile.size () )
	{
		closesearch ();
		return Result::NoFile;
	}
	result= vfile [n++];
	return Result::Ok;
}

FindFile::Result::code FindFile::Internal::findfirst
		(FcbFile & fcbmask, FcbFile & result)
{
	std::string file;
	std::string ext;

	file= fcbmask.getfilename ();
	ext= fcbmask.getfileext ();

	startsearch (file, ext);

	return findnext (result);
}

FindFile::Result::code FindFile::Internal::findnext (FcbFile & result)
{
	if (! finding)
		return Result::NoFile;

	if (n >= vfile.size () )
	{
		closesearch ();
		return Result::NoFile;
	}
	result.setfile (vfile [n++] );
	return Result::Ok;
}

FindFile::Result::code FindFile::Internal::get (std::string & found)
{
	if (! finding)
		return result;

	if (n >= vfile.size () )
	{
		closesearch ();
		return Result::NoFile;
	}
	found= vfile [n++];
	return Result::Ok;
}

FindFile::Result::code FindFile::Internal::get (FcbFile & found)
{
	if (! finding)
		return result;

	if (n >= vfile.size () )
	{
		closesearch ();
		return Result::NoFile;
	}
	found.setfile (vfile [n++] );
	return Result::Ok;
}


//***********************************************
//		FindFile
//***********************************************


FindFile::FindFile (const FindFile & ff) :
	pin (ff.pin)
{
	pin->addref ();
}

FindFile::FindFile (const std::string & directory, bool debug) :
	pin (new Internal (directory, debug) )
{
}

FindFile::FindFile (const std::string & directory,
		const std::string & mask, bool debug) :
	pin (new Internal (directory, mask, debug) )
{
}

FindFile::FindFile (const std::string & directory,
		FcbFile & fcbmask, bool debug) :
	pin (new Internal (directory, fcbmask, debug) )
{
}

FindFile::~FindFile ()
{
	//delete pin;
	pin->delref ();
}

FindFile::Result::code FindFile::findfirst
		(const std::string & mask, std::string & result)
{
	return pin->findfirst (mask, result);
}

FindFile::Result::code FindFile::findnext (std::string & result)
{
	return pin->findnext (result);
}

FindFile::Result::code FindFile::findfirst
	(FcbFile & fcbmask, FcbFile & result)
{
	return pin->findfirst (fcbmask, result);
}

FindFile::Result::code FindFile::findnext (FcbFile & result)
{
	return pin->findnext (result);
}

FindFile::Result::code FindFile::get (std::string & found)
{
	return pin->get (found);
}

FindFile::Result::code FindFile::get (FcbFile & found)
{
	return pin->get (found);
}


// End of directory.cpp
