/* key.c, Ait, BSD 3-Clause, Kevin Bloom, 2023-2025,
   Derived from: Atto January 2017
   Derived from: Anthony's Editor January 93
*/

#include <sys/types.h>
#include "header.h"
#include "termbox.h"
#include "util.h"

/* desc, keys, func */
keymap_t keymap[] = {
	{"C-a beginning-of-line    ", TB_MOD_CTRL, (int[2]){TB_KEY_CTRL_A, -1}, lnbegin, UA_REPEAT },
	{"C-b backward-char        ", TB_MOD_CTRL, (int[2]){TB_KEY_CTRL_B, -1}, left, UA_REPEAT },
	{"C-d delete               ", TB_MOD_CTRL, (int[2]){TB_KEY_CTRL_D, -1}, delete, UA_REPEAT },
	{"C-e end-of-line          ", TB_MOD_CTRL, (int[2]){TB_KEY_CTRL_E, -1}, lnend, UA_REPEAT },
	{"C-f foward-char          ", TB_MOD_CTRL, (int[2]){TB_KEY_CTRL_F, -1}, right, UA_REPEAT },
	{"C-h backspace            ", TB_MOD_CTRL, (int[2]){TB_KEY_BACKSPACE, -1}, backsp, UA_REPEAT },
	{"C-h backspace            ", TB_MOD_CTRL, (int[2]){TB_KEY_BACKSPACE2, -1}, backsp, UA_REPEAT },
	{"C-i indent               ", TB_MOD_CTRL, (int[2]){TB_KEY_CTRL_I, -1}, insertspacetab, UA_REPEAT },
	{"C-k kill-to-eol          ", TB_MOD_CTRL, (int[2]){TB_KEY_CTRL_K, -1}, killtoeol, UA_REPEAT },
	{"C-l refresh              ", TB_MOD_CTRL, (int[2]){TB_KEY_CTRL_L, -1}, recenter, UA_NOTHING },
	{"C-x u undo               ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, 'u', -1}, undo, UA_NOTHING },
	{"C-/   undo               ", TB_MOD_CTRL, (int[2]){TB_KEY_CTRL_SLASH, -1}, undo, UA_NOTHING },
	{"C-n next-line            ", TB_MOD_CTRL, (int[2]){TB_KEY_CTRL_N, -1}, down, UA_REPEAT },
	{"C-m newline              ", TB_MOD_CTRL, (int[2]){TB_KEY_CTRL_M, -1}, insertnewline, UA_REPEAT },
	{"C-p previous-line        ", TB_MOD_CTRL, (int[2]){TB_KEY_CTRL_P, -1}, up, UA_REPEAT },
	{"C-q insert-control-char  ", TB_MOD_CTRL, (int[2]){TB_KEY_CTRL_Q, -1}, insert_control_char, UA_PREVENT },
	{"C-r search               ", TB_MOD_CTRL, (int[2]){TB_KEY_CTRL_R, -1}, search_rev, UA_PREVENT },
	{"C-o newline-below        ", TB_MOD_CTRL, (int[2]){TB_KEY_CTRL_O, -1}, insertnewlinebelow, UA_REPEAT },
	{"C-s search               ", TB_MOD_CTRL, (int[2]){TB_KEY_CTRL_S, -1}, search_fwd, UA_PREVENT },
	{"C-t transpose            ", TB_MOD_CTRL, (int[2]){TB_KEY_CTRL_T, -1}, transpose, UA_REPEAT },
	{"C-u universal-argument   ", TB_MOD_CTRL, (int[2]){TB_KEY_CTRL_U, -1}, universal_argument_load, UA_NOTHING },
	{"C-v forward-page         ", TB_MOD_CTRL, (int[2]){TB_KEY_CTRL_V, -1}, pgdown, UA_REPEAT },
	{"C-w kill-region          ", TB_MOD_CTRL, (int[2]){TB_KEY_CTRL_W, -1}, cut, UA_CUSTOM },
	{"C-y yank                 ", TB_MOD_CTRL, (int[2]){TB_KEY_CTRL_Y, -1}, paste, UA_CUSTOM },
	{"C-z suspend              ", TB_MOD_CTRL, (int[2]){TB_KEY_CTRL_Z, -1}, suspend, UA_PREVENT },
	{"C-space set-mark         ", TB_MOD_CTRL, (int[2]){TB_KEY_CTRL_TILDE, -1}, iblock, UA_NOTHING },
	{"C-g remove-mark          ", TB_MOD_CTRL, (int[2]){TB_KEY_CTRL_G, -1}, unmark, UA_NOTHING },
	{"C-x 0 delete-other-window", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, '0', -1}, delete_other_windows, UA_NOTHING },
	{"C-x 1 delete-other-window", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, '1', -1}, delete_other_windows, UA_NOTHING },
	{"C-x 2 split-window       ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, '2', -1}, split_window, UA_NOTHING },
	{"C-x 3 chop-window        ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, '3', -1}, chop_window, UA_NOTHING },
	{"C-x 4 tri-split-window   ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, '4', -1}, tri_split_window, UA_NOTHING },
	{"C-x 5 tri-chop-window    ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, '5', -1}, tri_chop_window, UA_NOTHING },
	{"C-x 6 fib-right-window   ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, '6', -1}, fib_right, UA_NOTHING },
	{"C-x 7 fib-left-window    ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, '7', -1}, fib_left, UA_NOTHING },
	{"C-x 8 quad-window        ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, '8', -1}, quad_window, UA_NOTHING },
	{"C-x o other-window       ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, 'o', -1}, next_window, UA_NOTHING },
	{"C-x = cursor-position    ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, '=', -1}, showpos, UA_NOTHING },
	{"C-x i insert-file        ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, 'i', -1}, insertfile, UA_PREVENT },
	{"C-x k kill-buffer        ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, 'k', -1}, killbuffer, UA_PREVENT },
	{"C-x C-n next-buffer      ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, TB_KEY_CTRL_N, -1}, next_buffer, UA_NOTHING },
	{"C-x n next-buffer        ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, 'n', -1}, next_buffer, UA_NOTHING },
	{"C-x p prev-buffer        ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, 'p', -1}, prev_buffer, UA_NOTHING },
	{"C-x b switch-to-buffer   ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, 'b', -1}, switch_buffer, UA_NOTHING },
	{"C-x ( start-kbd-macro    ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, '(', -1}, start_kbd_macro, UA_PREVENT },
	{"C-x ) start-kbd-macro    ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, ')', -1}, end_kbd_macro, UA_PREVENT },
    /* UA_PREVENT is misleading here, since it uses the numeric_argument in the get_key function */
	{"C-x e run-kbd-macro      ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, 'e', -1}, run_kbd_macro, UA_PREVENT },
	{"C-x C-f find-file        ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, TB_KEY_CTRL_F, -1}, readfile, UA_NOTHING },
	{"C-x C-s save-buffer      ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, TB_KEY_CTRL_S, -1}, savebuffer, UA_NOTHING },
	{"C-x C-g umark            ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, TB_KEY_CTRL_G, -1}, unmark, UA_NOTHING },
	{"C-x C-w write-file       ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, TB_KEY_CTRL_W, -1}, writefile, UA_NOTHING },
	{"C-x C-c exit             ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, TB_KEY_CTRL_C, -1}, quit_ask, UA_NOTHING },
	{"C-x C-x pop-to-mark      ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, TB_KEY_CTRL_X, -1}, poptomark, UA_NOTHING },
	{"C-x C-; comment          ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, ';', -1}, comment, UA_NOTHING },
	{"C-x : comment-at-eol     ", TB_MOD_CTRL, (int[3]){TB_KEY_CTRL_X, ':', -1}, comment_at_eol, UA_NOTHING },
	{"esc 0 numeric-arg-0      ", TB_MOD_ALT, (int[2]){'0', -1}, numeric_argument_load, UA_PREVENT },
	{"esc 1 numeric-arg-1      ", TB_MOD_ALT, (int[2]){'1', -1}, numeric_argument_load, UA_PREVENT },
	{"esc 2 numeric-arg-2      ", TB_MOD_ALT, (int[2]){'2', -1}, numeric_argument_load, UA_PREVENT },
	{"esc 3 numeric-arg-3      ", TB_MOD_ALT, (int[2]){'3', -1}, numeric_argument_load, UA_PREVENT },
	{"esc 4 numeric-arg-4      ", TB_MOD_ALT, (int[2]){'4', -1}, numeric_argument_load, UA_PREVENT },
	{"esc 5 numeric-arg-5      ", TB_MOD_ALT, (int[2]){'5', -1}, numeric_argument_load, UA_PREVENT },
	{"esc 6 numeric-arg-6      ", TB_MOD_ALT, (int[2]){'6', -1}, numeric_argument_load, UA_PREVENT },
	{"esc 7 numeric-arg-7      ", TB_MOD_ALT, (int[2]){'7', -1}, numeric_argument_load, UA_PREVENT },
	{"esc 8 numeric-arg-8      ", TB_MOD_ALT, (int[2]){'8', -1}, numeric_argument_load, UA_PREVENT },
	{"esc 9 numeric-arg-9      ", TB_MOD_ALT, (int[2]){'9', -1}, numeric_argument_load, UA_PREVENT },
	{"esc b back-word          ", TB_MOD_ALT, (int[2]){'b', -1}, wleft, UA_REPEAT },
	{"esc bksp back-word-delete", TB_MOD_CTRL | TB_MOD_ALT, (int[2]){TB_KEY_BACKSPACE2, -1}, wleftdelete, UA_REPEAT },
	{"esc bksp back-word-delete", TB_MOD_CTRL | TB_MOD_ALT, (int[2]){TB_KEY_BACKSPACE, -1}, wleftdelete, UA_REPEAT },
	{"esc f fwd-word           ", TB_MOD_ALT, (int[2]){'f', -1}, wright, UA_REPEAT },
	{"esc d fwd-word-delete    ", TB_MOD_ALT, (int[2]){'d', -1}, wrightdelete, UA_REPEAT },
	{"esc x execute-shell-cmd  ", TB_MOD_ALT, (int[2]){'x', -1}, insert_from_shell, UA_PREVENT },
	{"esc g gotoline           ", TB_MOD_ALT, (int[2]){'g', -1}, gotoline, UA_PREVENT },
	{"esc G gotocolumn         ", TB_MOD_ALT, (int[2]){'G', -1}, gotocolumn, UA_PREVENT },
	{"esc r jumptorow          ", TB_MOD_ALT, (int[2]){'r', -1}, jumptorow, UA_PREVENT },
	{"esc j jumpword           ", TB_MOD_ALT, (int[2]){'j', -1}, jumpword, UA_PREVENT },
	{"esc i un-indent          ", TB_MOD_ALT, (int[2]){'i', -1}, inserttab_negated, UA_REPEAT },
	{"esc k kill-region        ", TB_MOD_ALT, (int[2]){'k', -1}, cut, UA_NOTHING },
	{"esc m back-to-indentation", TB_MOD_ALT, (int[2]){'m', -1}, back_to_indentation, UA_NOTHING },
	{"esc n negate             ", TB_MOD_ALT, (int[2]){'-', -1}, negate, UA_NOTHING },
	{"esc o open-shell-cmd     ", TB_MOD_ALT, (int[2]){'o', -1}, open_file_from_shell, UA_PREVENT },
	{"esc % query-replace      ", TB_MOD_ALT, (int[2]){'%', -1}, query_replace, UA_NOTHING },
	{"esc v backward-page      ", TB_MOD_ALT, (int[2]){'v', -1}, pgup, UA_REPEAT },
	{"esc w copy-region        ", TB_MOD_ALT, (int[2]){'w', -1}, copy, UA_NOTHING },
	{"esc @ set-mark           ", TB_MOD_ALT, (int[2]){'@', -1}, iblock, UA_NOTHING },
	{"esc < beg-of-buf         ", TB_MOD_ALT, (int[2]){'<', -1}, top, UA_NOTHING },
	{"esc > end-of-buf         ", TB_MOD_ALT, (int[2]){'>', -1}, bottom, UA_NOTHING },
	{"esc \\ delete-between    ", TB_MOD_ALT, (int[2]){'\\', -1}, delete_between, UA_CUSTOM },
	{"esc / redo               ", TB_MOD_ALT, (int[2]){'/', -1}, redo, UA_NOTHING },
	{"esc ^ indent-to          ", TB_MOD_ALT, (int[2]){'^', -1}, indent_to_above_or_below, UA_PREVENT },
	{"esc t transpose word     ", TB_MOD_ALT, (int[2]){'t', -1}, transposeword, UA_REPEAT },
	{"esc l lowercase-word     ", TB_MOD_ALT, (int[2]){'l', -1}, lowercaseword, UA_REPEAT },
	{"esc c capitalize-word    ", TB_MOD_ALT, (int[2]){'c', -1}, capitalizeword, UA_REPEAT },
	{"esc u uppercase-word     ", TB_MOD_ALT, (int[2]){'u', -1}, uppercaseword, UA_REPEAT },
	{"esc ; jump-to-char       ", TB_MOD_ALT, (int[2]){';', -1}, jumptochar, UA_REPEAT },
	{"esc : jump-to-char       ", TB_MOD_ALT, (int[2]){':', -1}, negated_jumptochar, UA_REPEAT },
	{"esc z zap-to-char        ", TB_MOD_ALT, (int[2]){'z', -1}, zaptochar, UA_CUSTOM },
	{"esc Z zap-to-char        ", TB_MOD_ALT, (int[2]){'Z', -1}, negated_zaptochar, UA_CUSTOM },
	{"esc . dynamically-expand ", TB_MOD_ALT, (int[2]){'.', -1}, dynamically_expand, UA_NOTHING },
	{"esc home, beg-of-buf     ", 0, (int[2]){TB_KEY_HOME, -1}, top, UA_NOTHING },
	{"esc end, end-of-buf      ", 0, (int[2]){TB_KEY_END, -1}, bottom, UA_NOTHING },
	{"esc esc show-version     ", 0, (int[2]){TB_KEY_ESC, -1}, version, UA_NOTHING },
	{"ins toggle-overwrite-mode", 0, (int[2]){TB_KEY_INSERT, -1}, toggle_overwrite_mode , UA_NOTHING },
	{"del forward-delete-char  ", 0, (int[2]){TB_KEY_DELETE, -1}, delete, UA_REPEAT },
	{"backspace delete-left    ", 0, (int[2]){TB_KEY_BACKSPACE, -1}, backsp, UA_REPEAT },
	{"backspace delete-left    ", 0, (int[2]){TB_KEY_BACKSPACE2, -1}, backsp, UA_REPEAT },
	{"up previous-line         ", 0, (int[2]){TB_KEY_ARROW_UP, -1}, up, UA_REPEAT },
	{"down next-line           ", 0, (int[2]){TB_KEY_ARROW_DOWN, -1}, down, UA_REPEAT },
	{"left backward-character  ", 0, (int[2]){TB_KEY_ARROW_LEFT, -1}, left, UA_REPEAT },
	{"right forward-character  ", 0, (int[2]){TB_KEY_ARROW_RIGHT, -1}, right, UA_REPEAT },
	{"pgup backward-page       ", 0, (int[2]){TB_KEY_PGUP, -1}, pgup , UA_REPEAT },
	{"pgdn forward-page        ", 0, (int[2]){TB_KEY_PGDN, -1}, pgdown, UA_REPEAT },
  //	{"resize resize-terminal   ", 0, (int[2]){TB_KEY_},     resize_terminal , UA_NOTHING },
	{"C-M-f forward-bracket    ", TB_MOD_CTRL | TB_MOD_ALT, (int[2]){TB_KEY_CTRL_F, -1}, forward_bracket, UA_REPEAT },
	{"C-M-b backward-bracket   ", TB_MOD_CTRL | TB_MOD_ALT, (int[2]){TB_KEY_CTRL_B, -1}, backward_bracket, UA_REPEAT },
	{"K_ERROR                  ", 0, NULL, NULL, UA_NOTHING }
};

keymap_t extraneous_keymap[] = {
	{"clipboard                ", 0, (int[2]){0, -1}, clipboard, UA_NOTHING},
	{"K_ERROR                  ", 0, NULL, NULL, UA_NOTHING }
};

char_t *get_key(keymap_t *keys, keymap_t **key_return)
{
  keymap_t *k;
  int ret;
  static char_t buffer[K_BUFFER_LENGTH];
  static char_t *record = buffer;
  struct tb_event ev, ev1;

  *key_return = NULL;

  /* if recorded bytes remain, return next recorded byte. */
  if (*record != '\0') {
    *key_return = NULL;
    return record++;
  }
  /* reset record buffer. */
  record = buffer;

  if(execute_kbd_macro) {
    use_kbd_macro(&ev);
    ret = TB_OK;
  } else {
    ret = tb_poll_event(&ev);
  }

  if(ret == TB_OK) {
    assert(K_BUFFER_LENGTH > record - buffer);

    if(ev.type == TB_EVENT_RESIZE) {
      resize_terminal();
      *key_return = NULL;
      return (record++);
    } else if(ev.type == TB_EVENT_MOUSE) {
      *key_return = NULL;
      return (record++);
    }
    *record++ = (unsigned)ev.ch;
    *record = '\0';

    /* When text is input via the system clipboard it is rapidly
       input into the editor. Using the tb_peek_event with a very
       small timeout seems to work at grabbing all the contents from
       the clipboard in. This does two things:
        1. Prevents the editor from inputting one char at a time
        2. Prevents auto-indent from indenting consecutively during
           paste.
    */
    if(!execute_kbd_macro) {
      int ci = 0, size = 1;
      if(gtemp == NULL) {
        gtemp = calloc(TEMPBUF+1, sizeof(char));
        ngtemp = TEMPBUF+1;
      }
      unicode_buf[tb_utf8_unicode_to_char(unicode_buf, ev.ch)] = '\0';
      gtemp[ci] = ev.key == TB_KEY_CTRL_M || ev.key == TB_KEY_CTRL_J ?
         '\n' : unicode_buf[0];
      ret = tb_peek_event(&ev1, 0);
      if(record_input) {
        record_buffer[record_buffer_index] = ev1;
        record_buffer_index++;
      }
      ci++;
      while(ret == TB_OK) {
        int maxsize = TEMPBUF*size;
        if(ci >= maxsize) {
          char *p2;
          int newsize = TEMPBUF*(++size)+1;
          if ((p2 = calloc(newsize, sizeof(char))) == NULL) {
            if (gtemp != NULL)
              free(gtemp);
            gtemp = NULL;
            return NULL;
          }
          memcpy(p2, gtemp, ngtemp);
          free(gtemp);
          gtemp = NULL;
          gtemp = p2;
          ngtemp = newsize;
        }
        int uni_size = tb_utf8_unicode_to_char(unicode_buf, ev1.ch);
        unicode_buf[uni_size] = '\0';
        unicode_buf[0] =
          ev1.key == TB_KEY_CTRL_M || ev1.key == TB_KEY_CTRL_J ?
            '\n' : unicode_buf[0];
        strncat(gtemp, unicode_buf, uni_size);
        ci += uni_size;
        ret = tb_peek_event(&ev1, 0);
        if(record_input) {
          record_buffer[record_buffer_index] = ev1;
          record_buffer_index++;
        }
      }
      gtemp[ci] = '\0';
      if(ci > 2) {
        k = extraneous_keymap;
        *key_return = k;
        record = buffer;
        *record = '\0';
        return record;
      }
    }
    if(record_input) {
      record_buffer[record_buffer_index] = ev;
      record_buffer_index++;
    }

    /* if recorded bytes match any multi-byte sequence... */
    for (k = keys; k->key_bytes != NULL; ++k) {
      if(k->key_mod == ev.mod &&
         (k->key_bytes[submatch] == ev.key ||
          (ev.ch != 0 && k->key_bytes[submatch] == (int)ev.ch))) {
        if(k->key_bytes[submatch+1] != -1) {
          submatch++;
          msg("C-x");
          break;
        } else {
          record = buffer;
          *record = '\0';
          *key_return = k;
          input_char = ev.ch;
          return record;
        }
      }
      if(submatch > 0 && k->key_bytes[submatch] == (int)ev.ch) {
        record = buffer;
        *record = '\0';
        *key_return = k;
        input_char = ev.ch;
        return record;
      }
    }
    unicode_buf[tb_utf8_unicode_to_char(unicode_buf, ev.ch)] = '\0';
    /* If no match is found, kill the loop. */
    if(key_return == NULL) {
      submatch = 0;
    }
  }
  /* nothing matched, return recorded bytes. */
  record = buffer;
  return (record++);
}

int getinput(char *prompt, char *buf, int nbuf, int flag, int allowblank)
{
  int cpos = 0;
  int c;
  int start_col = strlen(prompt);
  struct tb_event ev;

  print_to_msgline(prompt);
  clrtoeol(start_col, MSGLINE);

  if (flag == F_CLEAR) buf[0] = '\0';

  /* if we have a default value print it and go to end of it */
  if (buf[0] != '\0') {
    addstr(buf);
    cpos = strlen(buf);
  }

  for (;;) {
    tb_present();
    if(execute_kbd_macro) {
      use_kbd_macro(&ev);
    } else if(tb_poll_event(&ev) != TB_OK)
      return 0;

    if(msgline_editor(ev, prompt, buf, nbuf, &cpos))
      continue;

    if(!ev.mod)
      c = ev.ch;
    else
      c = ev.key;

    if(record_input) {
      record_buffer[record_buffer_index] = ev;
      record_buffer_index++;
    }

    /* ignore control keys other than backspace, cr, lf */
    if (c < 32 &&
        c != TB_KEY_CTRL_G &&
        c != TB_KEY_BACKSPACE &&
        c != TB_KEY_CTRL_J &&
        c != TB_KEY_CTRL_I &&
        c != TB_KEY_TAB &&
        c != TB_KEY_ENTER)
      continue;

    switch(c) {
      case TB_KEY_CTRL_J: /* cr, lf */
      case TB_KEY_ENTER: {
        int z = 0;
        while(buf[z] != 0)
           z++;
        buf[z] = '\0';
        return (cpos > 0 ? TRUE : allowblank);
      }
      case TB_KEY_CTRL_G: /* ctrl-g */
        clrtoeol(0, MSGLINE);
        return FALSE;

      case TB_KEY_BACKSPACE2: /* del, erase */
      case TB_KEY_BACKSPACE: /* backspace */
        if (cpos == 0)
          continue;

        tb_set_cursor(start_col + cpos - 1, MSGLINE);
        addch(' ');
        tb_set_cursor(start_col + cpos - 1, MSGLINE);
        buf[--cpos] = '\0';
        break;

      default:
        if (cpos < nbuf -1) {
          for(int i = strlen(buf); i > cpos; i--) {
            buf[i] = buf[i - 1];
          }
          buf[cpos] = c;
          tb_set_cursor(start_col, MSGLINE);
          addstr(buf);
          cpos++;
          tb_set_cursor(start_col + cpos, MSGLINE);
        }
        break;
      }
    }
}

int msgline_editor(
  struct tb_event ev, char *prompt, char *buf, int nbuf, int *cpos
)
{
  char full[PATH_MAX+1], b;
  int k = 0, i = 0, start_col = strlen(prompt);
  strcpy(full, prompt);
  if(ev.mod == TB_MOD_ALT || ev.mod == (TB_MOD_CTRL | TB_MOD_ALT)) {
      k = ev.ch;
      b = buf[*cpos];
      if(k == 'b') {
        if ((!isspace(b) || !is_symbol(b)) && *cpos != 0)
          *cpos = *cpos -1;
        for (;(isspace(b) || is_symbol(b)) && *cpos != 0; b = buf[*cpos])
          *cpos = *cpos -1;
        for (;!isspace(b) && !is_symbol(b) && *cpos != 0; b = buf[*cpos])
          *cpos = *cpos -1;
        tb_set_cursor(start_col + *cpos, MSGLINE);
        return 1;
      } else if(k == 'f') {
        if ((!isspace(b) || !is_symbol(b)) && *cpos != strlen(buf))
          *cpos = *cpos + 1;
        for (;(isspace(b) || is_symbol(b))  && *cpos != strlen(buf); b = buf[*cpos])
          *cpos = *cpos + 1;
        for (;!isspace(b) && !is_symbol(b) && *cpos != strlen(buf); b = buf[*cpos])
          *cpos = *cpos + 1;
        tb_set_cursor(start_col + *cpos, MSGLINE);
        return 1;
      } else if(k == 'd') {
        if ((!isspace(b) || !is_symbol(b)) && *cpos != strlen(buf)) {
          for(i = *cpos; buf[i] != '\0'; i++) {
            buf[i] = buf[i+1];
          }
        }
        for (;(isspace(b) || is_symbol(b))  && *cpos != strlen(buf); b = buf[*cpos]) {
          for(i = *cpos; buf[i] != '\0'; i++) {
            buf[i] = buf[i+1];
          }
        }
        for (;!isspace(b) && !is_symbol(b) && *cpos != strlen(buf); b = buf[*cpos]) {
          for(i = *cpos; buf[i] != '\0'; i++) {
            buf[i] = buf[i+1];
          }
        }
        strcat(full, buf);
        tb_set_cursor(start_col, MSGLINE);
        addstr(buf);
        clrtoeol(start_col + *cpos, MSGLINE);
        tb_set_cursor(start_col + *cpos, MSGLINE);
        full[start_col + *cpos] = '\0';
        return 1;
      } else if(k == 'y') {
        int ondef = FALSE, pastdef = FALSE;
        for(int i = 0, ii = 8; prompt[i] != '\0'; i++) {
          if(prompt[i] == '(')
            ondef = TRUE;
          if(pastdef) {
            buf[*cpos] = prompt[i];
            *cpos += 1;
          }
          if(ondef && !pastdef) {
            if(ii == 0)
              pastdef = TRUE;
            ii--;
          }
        }
        /* Since the default value can contain anything, we have to
           determine we're at the end of the default value we go all
           the way to the null terminator. Once there, we know that
           there is always a two char postfix of "):" To not copy
           this into the msgline buffer, we delete those by setting
           them to null terminator.

           We have to set i to 3 here because of cpos being 1 ahead
           of the length everything.
        */
        for(int i = 3; i > 0; i--)
          buf[--*cpos] = '\0';
        strcat(full, buf);
        tb_set_cursor(start_col, MSGLINE);
        addstr(buf);
        clrtoeol(start_col + *cpos, MSGLINE);
        tb_set_cursor(start_col + *cpos, MSGLINE);
        full[start_col + *cpos] = '\0';
        return 1;
      } else if(ev.key == TB_KEY_BACKSPACE || ev.key == TB_KEY_BACKSPACE2) {
        b = buf[*cpos-1];
        if ((!isspace(b) || !is_symbol(b)) && *cpos != 0) {
          *cpos = *cpos -1;
          for(i = *cpos; buf[i] != '\0'; i++) {
            buf[i] = buf[i+1];
          }
        }
        for (;(isspace(b) || is_symbol(b))  && *cpos != 0; b = buf[*cpos-1]) {
          *cpos = *cpos -1;
          for(i = *cpos; buf[i] != '\0'; i++) {
            buf[i] = buf[i+1];
          }
        }
        for (;!isspace(b) && !is_symbol(b) && *cpos != 0; b = buf[*cpos-1]) {
          *cpos = *cpos -1;
          for(i = *cpos; buf[i] != '\0'; i++) {
            buf[i] = buf[i+1];
          }
        }
        strcat(full, buf);
        tb_set_cursor(start_col, MSGLINE);
        addstr(buf);
        clrtoeol(start_col + *cpos, MSGLINE);
        tb_set_cursor(start_col + *cpos, MSGLINE);
        full[start_col + *cpos] = '\0';
        return 1;
      }
    } else
      k = ev.key;
  switch(k) {
    case TB_KEY_CTRL_B:
      if(*cpos != 0)
        *cpos = *cpos - 1;
      tb_set_cursor(start_col + *cpos, MSGLINE);
      return 1;

    case TB_KEY_CTRL_F:
      if(*cpos != strlen(buf))
        *cpos = *cpos + 1;
      tb_set_cursor(start_col + *cpos, MSGLINE);
      return 1;

    case TB_KEY_HOME:
    case TB_KEY_CTRL_A:
      *cpos = 0;
      tb_set_cursor(start_col + *cpos, MSGLINE);
      return 1;

    case TB_KEY_END:
    case TB_KEY_CTRL_E:
      *cpos = strlen(buf);
      tb_set_cursor(start_col + *cpos, MSGLINE);
      return 1;

    case TB_KEY_CTRL_K:
      for(i = *cpos; buf[i] != '\0'; i++)
        buf[i] = '\0';
      strcat(full, buf);
      clrtoeol(start_col + *cpos, MSGLINE);
      return 1;

    case TB_KEY_DELETE:
    case TB_KEY_CTRL_D:
      if(*cpos == strlen(buf))
        break;
      for(i = *cpos; buf[i] != '\0'; i++) {
        buf[i] = buf[i+1];
      }
      strcat(full, buf);
      tb_set_cursor(start_col, MSGLINE);
      addstr(buf);
      clrtoeol(start_col + strlen(buf), MSGLINE);
      tb_set_cursor(start_col + *cpos, MSGLINE);
      full[start_col + *cpos] = '\0';
      return 1;
    case TB_KEY_BACKSPACE2: /* del, erase */
    case TB_KEY_BACKSPACE: /* backspace */
      if(*cpos == 0)
        return 1;
      *cpos = *cpos - 1;
      for(i = *cpos; buf[i] != '\0'; i++) {
        buf[i] = buf[i+1];
      }
      strcat(full, buf);
      tb_set_cursor(start_col, MSGLINE);
      addstr(buf);
      clrtoeol(start_col + strlen(buf), MSGLINE);
      tb_set_cursor(start_col + *cpos, MSGLINE);
      full[start_col + *cpos] = '\0';
      return 1;

    case TB_KEY_CTRL_U: {
      universal_argument++;
      return 1;
    }

    case TB_KEY_CTRL_Y: {
      char_t *oscrap, *new_scrap = NULL;
      int onscrap = 0;

      if(scrap.data == NULL)
        return 0;

      if(universal_argument > 0 && universal_argument-1 < KILLRING_SIZE) {
        oscrap = (char_t *)strndup((char *)scrap.data, scrap.len);
        onscrap = scrap.len;
        free(scrap.data);
        scrap.len = kill_ring[universal_argument-1].len;
        new_scrap = (char_t *) strndup(
          (const char *)kill_ring[universal_argument-1].data,
          scrap.len
        );
        scrap.data = new_scrap;
        universal_argument = 0;
      }

      int remaining = nbuf - strlen(buf) - 1;
      int size = scrap.len > remaining ? remaining : scrap.len;
      *cpos += size;
      strncat(buf, (const char *)scrap.data, size);
      remaining = PATH_MAX - strlen(full) - 1;
      size = strlen(buf) > remaining ? remaining : strlen(buf);
      strncat(full, buf, size);
      tb_set_cursor(start_col, MSGLINE);
      addstr(buf);
      clrtoeol(start_col + *cpos, MSGLINE);
      tb_set_cursor(start_col + *cpos, MSGLINE);
      full[start_col + *cpos] = '\0';
      if(onscrap > 0) {
        free(scrap.data);
        scrap.len = onscrap;
        scrap.data = oscrap;
      }
      return 1;
    }
    case TB_KEY_CTRL_Q: {
      struct tb_event ev1;
      if(tb_poll_event(&ev1) != TB_OK) return 0;
      if(ev1.key > 0x1a) {
        return 0;
      }
      buf[*cpos] = (char_t)ev1.key;
      *cpos += 1;
      tb_set_cursor(start_col + *cpos, MSGLINE);
      full[strlen(prompt)] = '\0';
      return 1;
    }
    default:
      return 0;
  }
  return 0;
}

void use_kbd_macro(struct tb_event *ev)
{
  *ev = record_buffer[run_buffer_index];
  if(run_buffer_index > record_buffer_index) {
    if(numeric_argument > 0) {
      numeric_argument--;
    } else {
      execute_kbd_macro = FALSE;
    }
    run_buffer_index = 0;
  } else {
    run_buffer_index++;
  }
  ignorenotbound = TRUE;
}
