/*  $Id: AfternoonStalkerEngine.h,v 1.45 2017/12/09 18:31:57 sarrazip Exp $
    AfternoonStalkerEngine.h - A robot-killing video game engine.

    afternoonstalker - A robot-killing video game.
    Copyright (C) 2001-2012 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public
    License along with this program; if not, write to the Free
    Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA  02110-1301, USA.
*/

#ifndef _H_AfternoonStalkerEngine
#define _H_AfternoonStalkerEngine

#define PAUL 12

#include "Controller.h"

#include <flatzebra/GameEngine.h>
#include <flatzebra/Sprite.h>
#include <flatzebra/SoundMixer.h>

class RobotSprite;

#include <string>
#include <vector>
#include <iostream>

using namespace flatzebra;


class AfternoonStalkerEngine : public GameEngine
/*  A robot-killing video game.
*/
{
public:

    enum { RIGHT, UP, LEFT, DOWN };
    /*  Possible directions of movement, in trigonometric order.
        This enumeration is garanteed to start at zero.
        The order of the constants is guaranteed, so they are 0, 1, 2, 3.
    */

    enum Rules { AS_RULES_1, AS_RULES_2 };
    /*  AS_RULES_1 = original Afternoon Stalker rules.
        AS_RULES_2 = 2008 Afternoon Stalker rules.
    */

    AfternoonStalkerEngine(const std::string &windowManagerCaption,
                            bool _useSound,
                            bool fullScreen,
                            size_t _mazeIndex,
                            Rules rules,
                            bool processActiveEvent);
    /*  See base class.

        'windowManagerCaption' must contain the title to give to the window.
        'fullScreen' attempts to display the fame in full screen mode
        instead of using an ordinary window.
        '_mazeIndex' must be a zero-based index of a .set file available
        in the game's PKGDATADIR.
        'rules': see manual page (doc/afternoonstalker.6.in).
        'processActiveEvent' determines if the game automatically pauses
        when the game window loses focus.

        Throws a string or an integer code if an error occurs.
    */

    virtual ~AfternoonStalkerEngine();
    /*  Nothing interesting.
    */

    const AfternoonStalkerEngine &getInstance() const;
    AfternoonStalkerEngine &getInstance();

    virtual void processKey(SDLKey keysym, bool pressed);
    /*  Inherited.
    */

    virtual void processActivation(bool appActive);
    /*  Inherited.
    */

    virtual bool tick();
    /*  Inherited.
    */

private:

    enum TileNo
    {
        // The first value must be zero
        WALL_TILE,

        // The CORNER* and WALLEND* names must be contiguous
        CORNER0_TILE,
        CORNER1_TILE,
        CORNER2_TILE,
        CORNER3_TILE,
        WALLEND0_TILE,
        WALLEND1_TILE,
        WALLEND2_TILE,
        WALLEND3_TILE,

        FLOOR_TILE,
        COBWEB_TILE,

        BUNKER_TILE,

        // The FLOOR[0-3]* names must be contiguous
        BUNKER0_TILE,
        BUNKER1_TILE,
        BUNKER2_TILE,
        BUNKER3_TILE,

        BUNKER_DOOR_TILE
    };

    typedef std::vector<TileNo> TileMatrixRow;
    typedef std::vector<TileMatrixRow> TileMatrix;


    enum Privilege
    // Used in positionIsAllowed()
    {
        PLAYER_PRIV,
        ENEMY_PRIV,
        BULLET_PRIV,
        PIERCING_BULLET_PRIV
    };


    class Menu
    {
    public:
        Menu() : title(), entries(), currentSelectionIndex(0), parentMenu(NULL) {}

        void setParentMenu(Menu *p) { parentMenu = p; }
        Menu *getParentMenu() const { return parentMenu; }

        void setTitle(const std::string &t) { title = t; }
        const std::string &getTitle() const { return title; }

        void addEntry(const std::string &entry) { entries.push_back(entry); }
        const std::string &getEntry(size_t index) const { return entries[index]; }

        void setSelectionIndex(size_t i) { currentSelectionIndex = i; }
        void nextSelection() { currentSelectionIndex = (currentSelectionIndex + 1) % entries.size(); }
        void prevSelection() { currentSelectionIndex = (currentSelectionIndex + entries.size() - 1) % entries.size(); }
        size_t getSelectionIndex() const { return currentSelectionIndex; }

        size_t getNumEntries() const { return entries.size(); }

    private:
        std::string title;
        std::vector<std::string> entries;
        size_t currentSelectionIndex;
        Menu *parentMenu;

        // Forbidden operations:
        Menu(const Menu &);
        Menu operator = (const Menu &);
    };


    ///////////////////////////////////////////////////////////////////////////
    //
    //  DATA MEMBERS
    //
    //


    bool paused;
    unsigned long tickCount;
    bool useSound;
    bool soundChunksInitialized;
    Rules rules;
    size_t mazeIndex;


    // SDL color values (see SDL_MapRGB()):
    Uint32 blackColor;

    /*  SETTING:
    */
    PixmapArray tilePA;
    TileMatrix  tileMatrix;


    /*  SCORE BOARD:
    */
    long theScore;


    /*  PLAYER:
    */
    PixmapArray playerPA;
    Sprite *playerSprite;
    size_t numPlayerLives;
    Couple initPlayerPos;
    size_t numPlayerBullets;
    int lastPlayerDir;
    int lastRequestedDir;
    long playerParalysisTime;

    PixmapArray playerBulletPA;
    SpriteList playerBulletSprites;

    PixmapArray playerExplosionPA;
    SpriteList playerExplosionSprites;
            // when this list is not empty, the player is dying

    PixmapArray gunPA;
    SpriteList gunSprites;


    /*  ENEMIES:

        CONVENTION: all robot sprites are the same size as the human.
    */
    PixmapArray grayRobotPA;
    PixmapArray blueRobotPA;
    PixmapArray whiteRobotPA;
    PixmapArray blackRobotPA;
    PixmapArray invisibleRobotPA;
    PixmapArray blinkPA;
    SpriteList robotSprites;
    long timeUntilNextRobot;
    bool sendOnlyBlueRobots;
    Couple initRobotPos;
    size_t numCreatedRobots;

    PixmapArray robotBulletPA;
    PixmapArray bigRobotBulletPA;
    SpriteList robotBulletSprites;

    PixmapArray grayRobotExplosionPA;
    PixmapArray blueRobotExplosionPA;
    PixmapArray whiteRobotExplosionPA;
    PixmapArray blackRobotExplosionPA;
    PixmapArray invisibleRobotExplosionPA;
    SpriteList robotExplosionSprites;

    PixmapArray spiderPA;
    PixmapArray batPA;
    SpriteList spiderSprites;
    SpriteList batSprites;
    Couple initSpiderPos;
    long timeUntilNextAnimal;

    PixmapArray spiderExplosionPA;
    SpriteList spiderExplosionSprites;
    PixmapArray batExplosionPA;
    SpriteList batExplosionSprites;

    /*  DIGITS:
    */
    PixmapArray digitPA;
    SpriteList scoreSprites;


    Controller controller;

    Menu mainMenu;
    Menu instructionsMenu;
    Menu optionsMenu;
    Menu rulesMenu;
    Menu mazeMenu;
    Menu soundsMenu;
    Menu pauseMenu;
    Menu *currentMenu;


    /*  SOUND EFFECTS:
    */
    SoundMixer *theSoundMixer;  // see method playSoundEffect()
    SoundMixer::Chunk gunPickupSound;
    SoundMixer::Chunk playerBulletSound;
    SoundMixer::Chunk shootingBlanksSound;
    SoundMixer::Chunk playerHitSound;
    SoundMixer::Chunk robotBulletSound;
    SoundMixer::Chunk robotHitSound;
    SoundMixer::Chunk batKilledSound;
    SoundMixer::Chunk spiderKilledSound;
    SoundMixer::Chunk newLifeSound;


    ///////////////////////////////////////////////////////////////////////////


    bool animatePlayer();
    int getRobotScore(const Sprite &robot) const;
    void makePlayerShoot(int shotDir);
    Couple attemptMove(const Sprite &s,
                        const bool attemptedDirections[4],
                        int speedFactor) const;
    Couple getSpeedFromAttemptedAndAllowedDirections(
                        const bool attemptedDirections[4],
                        const bool allowedDirections[4],
                        int speedFactor,
                        Couple delta) const;
    Couple determineAllowedDirections(const Sprite &s,
                                        bool forPlayer,
                                        int speedFactor, int tolerance,
                                        bool allowedDirections[4]) const;
    Couple getDistanceToPerfectPos(const Sprite &s) const;
    bool positionIsAllowed(int direction, Privilege priv,
                                Couple pos, Couple size) const;
    std::pair<TileNo, TileNo> getSideTilesAtPos(
                                int direction, Couple pos, Couple size) const;
    TileNo getTileAtPos(Couple pos) const;
    void setTileAtPos(Couple pos, TileNo t);

    void animateTemporarySprites(SpriteList &slist) const;
    void animateAutomaticCharacters();
    void killSpritesNearPosition(
                        SpriteList &slist, Couple pos, int minDistance);
    void moveExplosionSprites(SpriteList &slist);
    bool isOutOfSetting(const Sprite &s) const;
    size_t getNumToughRobots() const;
    void moveEnemyList(SpriteList &slist, int speedFactor);
    void moveBullets(SpriteList &slist);
    int chooseDirectionTowardTarget(Couple startPos,
                                    Couple targetPos,
                                    int speedFactor,
                                    const bool allowedDirections[4]) const;
    Couple chooseRandomAllowedTile();
    void makeRobotsShoot();

    void detectCollisions();
    void makePlayerDie();
    void endCurrentGame();
    void paralyzePlayer();
    void createRobotExplosion(const RobotSprite &r);
    void createAnimalExplosion(Couple pos,
                                    const PixmapArray &pa, SpriteList &slist);
    void killEnemy(const Sprite &s);
    void createPlayerExplosion(const Sprite &s);
    void addToScore(long n);
    void createScoreSprites(long n, Couple center);
    void loadPixmaps();
    void restoreBackground();
    void restoreForeground();
    void putSprite(const Sprite &s);
    void putSpriteList(const SpriteList &slist);
    void drawSprites();
    void drawScoreBoard();
    void displayMessage(int row, const char *msg);

    void displayErrorMessage(const std::string &msg);
    void initializeLevel();
    void initializePlayerLife();
    void setTimeUntilNextRobot();
    void setTimeUntilNextAnimal();
    void doOneTimeInitializations();
    void initSounds();
    void initMenus();

    enum MenuAction { START_GAME, QUIT_GAME, QUIT_PROGRAM, STAY_IN_MENUS };
    MenuAction processMenuMode();
    void loadLevel();

    void playSoundEffect(SoundMixer::Chunk &wb);
    void processJoystick();
    void setPauseMode();
    void displayCurrentMenu();
    bool findSecondCobwebTile(size_t &x, size_t &y) const;


    /*        Forbidden operations:
    */
    AfternoonStalkerEngine(const AfternoonStalkerEngine &);
    AfternoonStalkerEngine &operator = (const AfternoonStalkerEngine &);
};


#endif  /* _H_AfternoonStalkerEngine */
