/*
 * vim: ts=4 sw=4 et tw=0 wm=0
 *
 * libtopology - Classes used in generating and managing topology constraints.
 *
 * Copyright (C) 2007-2008  Monash University
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library in the file LICENSE; if not, 
 * write to the Free Software Foundation, Inc., 59 Temple Place, 
 * Suite 330, Boston, MA  02111-1307  USA
 *
*/

/**
 * Test of topology conserving constraints - no overlap constraints.
 * A number of rectangles is generated and overlaps between them 
 * removed using libvpsc.  For each drawing dimension dim,
 * an instance of TopologyConstraints 
 * in axis dim is generated for this set of non-overlapping rectangles and
 * hence a set of libvpsc::Constraints that should prevent overlap
 * if the rectangles move in dim.  The rectangles are then randomly perturbed in
 * dim and overlaps resolved again using libvpsc with the set of constraints
 * generated by TopologyConstraints.
 * Overlaps must then be non-existent.
 *
 * \file nooverlap.cpp
 * \author Tim Dwyer
 * \date Jan 2008
 */
#include <libvpsc/rectangle.h>
#include <libtopology/topology_constraints.h>
#include <libcola/cola.h>
#include <libcola/output_svg.h>
#include <libvpsc/variable.h>
#include <libvpsc/constraint.h>
#include <vector>
#include <iostream>
#include <sstream>
using namespace std;
using namespace topology;
#include "test.h"
struct Create {
    Create() : ctr(0) {}
    Node* operator() (vpsc::Rectangle* r) {
        return new Node(ctr,r,new vpsc::Variable(ctr));
        ++ctr;
    }
    unsigned ctr;
};
void testRectangles(vpsc::Rectangles& rs) {
    rs.push_back(new vpsc::Rectangle(5,15,14,52));
    rs.push_back(new vpsc::Rectangle(5,10,14,23));
    rs.push_back(new vpsc::Rectangle(5,35,13,39));
    rs.push_back(new vpsc::Rectangle(2,10,6,34));
    rs.push_back(new vpsc::Rectangle(5,9,2,22));
    rs.push_back(new vpsc::Rectangle(11,13,8,24));
    rs.push_back(new vpsc::Rectangle(7,35,10,27));
    rs.push_back(new vpsc::Rectangle(9,39,8,10));
    rs.push_back(new vpsc::Rectangle(11,38,11,20));
    rs.push_back(new vpsc::Rectangle(9,47,5,55));
}
void testCase() {
    string name("nooverlap");
    Edges es;
    vpsc::Rectangles rs;
    testRectangles(rs);
    const size_t V = rs.size();
    
    removeoverlaps(rs);
    noRectangleOverlaps(rs);
    Nodes nodes(V);
    transform(rs.begin(),rs.end(),nodes.begin(),Create());
    vpsc::Variables vs;
    getVariables(nodes,vs);

    writeFile(nodes,es,name+"-0.svg");

    try {
        vpsc::Constraints cs;
        Edges es; // not used in this test
        TopologyConstraints t(vpsc::XDIM,nodes,es,nullptr,vs,cs);
        for(Nodes::iterator v=nodes.begin();v!=nodes.end();++v) {
            (*v)->var->desiredPosition=getRand(5);
        }
        t.solve();
        writeFile(nodes,es,name+"-1.svg");

        noRectangleOverlaps(rs);

        for_each(cs.begin(),cs.end(),delete_object());
    } catch(const char *e) {
        fprintf(stderr,"ERROR: %s\n",e);
    }

    for_each(nodes.begin(),nodes.end(),delete_node());
    for_each(vs.begin(),vs.end(),delete_object());
}
void randomTest(vpsc::Dim dim) {
    string name("nooverlap");
    const size_t V = 10;
    vpsc::Rectangles rs, rs_bak;
    generateRandomRects(V,rs);
    for(vpsc::Rectangles::iterator i=rs.begin();i!=rs.end();++i) {
        rs_bak.push_back(new vpsc::Rectangle(**i));
    }
    static const double EXTRA_GAP=1e-10;
    vpsc::Rectangle::setXBorder(EXTRA_GAP);
    vpsc::Rectangle::setYBorder(EXTRA_GAP);
    removeoverlaps(rs);
    noRectangleOverlaps(rs);
    Nodes nodes(V);
    transform(rs.begin(),rs.end(),nodes.begin(),Create());
    vpsc::Variables vs;
    getVariables(nodes,vs);

    try {
        vpsc::Constraints cs;
        Edges es; // not used in this test
        TopologyConstraints t(dim,nodes,es,nullptr,vs,cs);
        for(Nodes::iterator v=nodes.begin();v!=nodes.end();++v) {
            (*v)->var->desiredPosition=getRand(5);
        }
        t.solve();
        vpsc::Rectangle::setXBorder(0);
        vpsc::Rectangle::setYBorder(0);

        noRectOverlaps(rs);

        for_each(cs.begin(),cs.end(),delete_object());
    } catch(const char *e) {
        fprintf(stderr,"ERROR: %s\n",e);
        printf("original:");
        for(vpsc::Rectangles::iterator i=rs_bak.begin();i!=rs_bak.end();++i) {
            cout << **i << endl;
        }
        printf("overlapping:");
        for(vpsc::Rectangles::iterator i=rs.begin();i!=rs.end();++i) {
            cout << **i << endl;
        }
    }

    for_each(nodes.begin(),nodes.end(),delete_node());
    for_each(vs.begin(),vs.end(),delete_object());
}

int main() {
    //testCase();
    //return 0;
    for(unsigned i=0;i<10000;++i) {
        if(!(i%1000)) {
            printf(" completed %d instances...\n",i);
        }
        randomTest(vpsc::HORIZONTAL);
        randomTest(vpsc::VERTICAL);
    }
    return 0;
}
// vim: filetype=cpp:expandtab:shiftwidth=4:tabstop=4:softtabstop=4:textwidth=80 :
