/*
 * Copyright (c) Citrix Systems, Inc.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 *   1) Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 * 
 *   2) Redistributions in binary form must reproduce the above
 *      copyright notice, this list of conditions and the following
 *      disclaimer in the documentation and/or other materials
 *      provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */



#include <stddef.h>
#include <stdlib.h>

#include "xen_internal.h"
#include "xen_string_vif_operations_map_internal.h"
#include "xen_vif_locking_mode_internal.h"
#include "xen_vif_operations_internal.h"
#include <xen/api/xen_common.h>
#include <xen/api/xen_network.h>
#include <xen/api/xen_string_string_map.h>
#include <xen/api/xen_string_vif_operations_map.h>
#include <xen/api/xen_vif.h>
#include <xen/api/xen_vif_metrics.h>
#include <xen/api/xen_vif_operations.h>
#include <xen/api/xen_vif_xen_vif_record_map.h>
#include <xen/api/xen_vm.h>


XEN_FREE(xen_vif)
XEN_SET_ALLOC_FREE(xen_vif)
XEN_ALLOC(xen_vif_record)
XEN_SET_ALLOC_FREE(xen_vif_record)
XEN_ALLOC(xen_vif_record_opt)
XEN_RECORD_OPT_FREE(xen_vif)
XEN_SET_ALLOC_FREE(xen_vif_record_opt)


static const struct_member xen_vif_record_struct_members[] =
    {
        { .key = "uuid",
          .type = &abstract_type_string,
          .offset = offsetof(xen_vif_record, uuid) },
        { .key = "allowed_operations",
          .type = &xen_vif_operations_set_abstract_type_,
          .offset = offsetof(xen_vif_record, allowed_operations) },
        { .key = "current_operations",
          .type = &string_vif_operations_map_abstract_type_,
          .offset = offsetof(xen_vif_record, current_operations) },
        { .key = "device",
          .type = &abstract_type_string,
          .offset = offsetof(xen_vif_record, device) },
        { .key = "network",
          .type = &abstract_type_ref,
          .offset = offsetof(xen_vif_record, network) },
        { .key = "VM",
          .type = &abstract_type_ref,
          .offset = offsetof(xen_vif_record, vm) },
        { .key = "MAC",
          .type = &abstract_type_string,
          .offset = offsetof(xen_vif_record, mac) },
        { .key = "MTU",
          .type = &abstract_type_int,
          .offset = offsetof(xen_vif_record, mtu) },
        { .key = "other_config",
          .type = &abstract_type_string_string_map,
          .offset = offsetof(xen_vif_record, other_config) },
        { .key = "currently_attached",
          .type = &abstract_type_bool,
          .offset = offsetof(xen_vif_record, currently_attached) },
        { .key = "status_code",
          .type = &abstract_type_int,
          .offset = offsetof(xen_vif_record, status_code) },
        { .key = "status_detail",
          .type = &abstract_type_string,
          .offset = offsetof(xen_vif_record, status_detail) },
        { .key = "runtime_properties",
          .type = &abstract_type_string_string_map,
          .offset = offsetof(xen_vif_record, runtime_properties) },
        { .key = "qos_algorithm_type",
          .type = &abstract_type_string,
          .offset = offsetof(xen_vif_record, qos_algorithm_type) },
        { .key = "qos_algorithm_params",
          .type = &abstract_type_string_string_map,
          .offset = offsetof(xen_vif_record, qos_algorithm_params) },
        { .key = "qos_supported_algorithms",
          .type = &abstract_type_string_set,
          .offset = offsetof(xen_vif_record, qos_supported_algorithms) },
        { .key = "metrics",
          .type = &abstract_type_ref,
          .offset = offsetof(xen_vif_record, metrics) },
        { .key = "MAC_autogenerated",
          .type = &abstract_type_bool,
          .offset = offsetof(xen_vif_record, mac_autogenerated) },
        { .key = "locking_mode",
          .type = &xen_vif_locking_mode_abstract_type_,
          .offset = offsetof(xen_vif_record, locking_mode) },
        { .key = "ipv4_allowed",
          .type = &abstract_type_string_set,
          .offset = offsetof(xen_vif_record, ipv4_allowed) },
        { .key = "ipv6_allowed",
          .type = &abstract_type_string_set,
          .offset = offsetof(xen_vif_record, ipv6_allowed) }
    };

const abstract_type xen_vif_record_abstract_type_ =
    {
       .typename = STRUCT,
       .struct_size = sizeof(xen_vif_record),
       .member_count =
           sizeof(xen_vif_record_struct_members) / sizeof(struct_member),
       .members = xen_vif_record_struct_members
    };


static const struct struct_member xen_vif_xen_vif_record_members[] =
{
    {
        .type = &abstract_type_string,
        .offset = offsetof(xen_vif_xen_vif_record_map_contents, key)
    },
    {
        .type = &xen_vif_record_abstract_type_,
        .offset = offsetof(xen_vif_xen_vif_record_map_contents, val)
    }
};

const abstract_type abstract_type_string_xen_vif_record_map =
{
    .typename = MAP,
    .struct_size = sizeof(xen_vif_xen_vif_record_map_contents),
    .members = xen_vif_xen_vif_record_members
};


void
xen_vif_record_free(xen_vif_record *record)
{
    if (record == NULL)
    {
        return;
    }
    free(record->handle);
    free(record->uuid);
    xen_vif_operations_set_free(record->allowed_operations);
    xen_string_vif_operations_map_free(record->current_operations);
    free(record->device);
    xen_network_record_opt_free(record->network);
    xen_vm_record_opt_free(record->vm);
    free(record->mac);
    xen_string_string_map_free(record->other_config);
    free(record->status_detail);
    xen_string_string_map_free(record->runtime_properties);
    free(record->qos_algorithm_type);
    xen_string_string_map_free(record->qos_algorithm_params);
    xen_string_set_free(record->qos_supported_algorithms);
    xen_vif_metrics_record_opt_free(record->metrics);
    xen_string_set_free(record->ipv4_allowed);
    xen_string_set_free(record->ipv6_allowed);
    free(record);
}


bool
xen_vif_get_record(xen_session *session, xen_vif_record **result, xen_vif vif)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif }
        };

    abstract_type result_type = xen_vif_record_abstract_type_;

    *result = NULL;
    XEN_CALL_("VIF.get_record");

    if (session->ok)
    {
       (*result)->handle = xen_strdup_((*result)->uuid);
    }

    return session->ok;
}


bool
xen_vif_get_by_uuid(xen_session *session, xen_vif *result, char *uuid)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = uuid }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("VIF.get_by_uuid");
    return session->ok;
}


bool
xen_vif_create(xen_session *session, xen_vif *result, xen_vif_record *record)
{
    abstract_value param_values[] =
        {
            { .type = &xen_vif_record_abstract_type_,
              .u.struct_val = record }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("VIF.create");
    return session->ok;
}

bool
xen_vif_create_async(xen_session *session, xen_task *result, xen_vif_record *record)
{
    abstract_value param_values[] =
        {
            { .type = &xen_vif_record_abstract_type_,
              .u.struct_val = record }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("Async.VIF.create");
    return session->ok;
}

bool
xen_vif_destroy(xen_session *session, xen_vif vif)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif }
        };

    xen_call_(session, "VIF.destroy", param_values, 1, NULL, NULL);
    return session->ok;
}

bool
xen_vif_destroy_async(xen_session *session, xen_task *result, xen_vif vif)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("Async.VIF.destroy");
    return session->ok;
}

bool
xen_vif_get_allowed_operations(xen_session *session, struct xen_vif_operations_set **result, xen_vif vif)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif }
        };

    abstract_type result_type = xen_vif_operations_set_abstract_type_;

    *result = NULL;
    XEN_CALL_("VIF.get_allowed_operations");
    return session->ok;
}


bool
xen_vif_get_current_operations(xen_session *session, xen_string_vif_operations_map **result, xen_vif vif)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif }
        };

    abstract_type result_type = string_vif_operations_map_abstract_type_;

    *result = NULL;
    XEN_CALL_("VIF.get_current_operations");
    return session->ok;
}


bool
xen_vif_get_device(xen_session *session, char **result, xen_vif vif)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("VIF.get_device");
    return session->ok;
}


bool
xen_vif_get_network(xen_session *session, xen_network *result, xen_vif vif)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("VIF.get_network");
    return session->ok;
}


bool
xen_vif_get_vm(xen_session *session, xen_vm *result, xen_vif vif)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("VIF.get_VM");
    return session->ok;
}


bool
xen_vif_get_mac(xen_session *session, char **result, xen_vif vif)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("VIF.get_MAC");
    return session->ok;
}


bool
xen_vif_get_mtu(xen_session *session, int64_t *result, xen_vif vif)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif }
        };

    abstract_type result_type = abstract_type_int;

    XEN_CALL_("VIF.get_MTU");
    return session->ok;
}


bool
xen_vif_get_other_config(xen_session *session, xen_string_string_map **result, xen_vif vif)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif }
        };

    abstract_type result_type = abstract_type_string_string_map;

    *result = NULL;
    XEN_CALL_("VIF.get_other_config");
    return session->ok;
}


bool
xen_vif_get_currently_attached(xen_session *session, bool *result, xen_vif vif)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif }
        };

    abstract_type result_type = abstract_type_bool;

    XEN_CALL_("VIF.get_currently_attached");
    return session->ok;
}


bool
xen_vif_get_status_code(xen_session *session, int64_t *result, xen_vif vif)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif }
        };

    abstract_type result_type = abstract_type_int;

    XEN_CALL_("VIF.get_status_code");
    return session->ok;
}


bool
xen_vif_get_status_detail(xen_session *session, char **result, xen_vif vif)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("VIF.get_status_detail");
    return session->ok;
}


bool
xen_vif_get_runtime_properties(xen_session *session, xen_string_string_map **result, xen_vif vif)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif }
        };

    abstract_type result_type = abstract_type_string_string_map;

    *result = NULL;
    XEN_CALL_("VIF.get_runtime_properties");
    return session->ok;
}


bool
xen_vif_get_qos_algorithm_type(xen_session *session, char **result, xen_vif vif)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("VIF.get_qos_algorithm_type");
    return session->ok;
}


bool
xen_vif_get_qos_algorithm_params(xen_session *session, xen_string_string_map **result, xen_vif vif)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif }
        };

    abstract_type result_type = abstract_type_string_string_map;

    *result = NULL;
    XEN_CALL_("VIF.get_qos_algorithm_params");
    return session->ok;
}


bool
xen_vif_get_qos_supported_algorithms(xen_session *session, struct xen_string_set **result, xen_vif vif)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif }
        };

    abstract_type result_type = abstract_type_string_set;

    *result = NULL;
    XEN_CALL_("VIF.get_qos_supported_algorithms");
    return session->ok;
}


bool
xen_vif_get_metrics(xen_session *session, xen_vif_metrics *result, xen_vif vif)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("VIF.get_metrics");
    return session->ok;
}


bool
xen_vif_get_mac_autogenerated(xen_session *session, bool *result, xen_vif vif)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif }
        };

    abstract_type result_type = abstract_type_bool;

    XEN_CALL_("VIF.get_MAC_autogenerated");
    return session->ok;
}


bool
xen_vif_get_locking_mode(xen_session *session, enum xen_vif_locking_mode *result, xen_vif vif)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif }
        };

    abstract_type result_type = xen_vif_locking_mode_abstract_type_;
    XEN_CALL_("VIF.get_locking_mode");
    return session->ok;
}


bool
xen_vif_get_ipv4_allowed(xen_session *session, struct xen_string_set **result, xen_vif vif)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif }
        };

    abstract_type result_type = abstract_type_string_set;

    *result = NULL;
    XEN_CALL_("VIF.get_ipv4_allowed");
    return session->ok;
}


bool
xen_vif_get_ipv6_allowed(xen_session *session, struct xen_string_set **result, xen_vif vif)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif }
        };

    abstract_type result_type = abstract_type_string_set;

    *result = NULL;
    XEN_CALL_("VIF.get_ipv6_allowed");
    return session->ok;
}


bool
xen_vif_set_other_config(xen_session *session, xen_vif vif, xen_string_string_map *other_config)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif },
            { .type = &abstract_type_string_string_map,
              .u.set_val = (arbitrary_set *)other_config }
        };

    xen_call_(session, "VIF.set_other_config", param_values, 2, NULL, NULL);
    return session->ok;
}


bool
xen_vif_add_to_other_config(xen_session *session, xen_vif vif, char *key, char *value)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif },
            { .type = &abstract_type_string,
              .u.string_val = key },
            { .type = &abstract_type_string,
              .u.string_val = value }
        };

    xen_call_(session, "VIF.add_to_other_config", param_values, 3, NULL, NULL);
    return session->ok;
}


bool
xen_vif_remove_from_other_config(xen_session *session, xen_vif vif, char *key)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif },
            { .type = &abstract_type_string,
              .u.string_val = key }
        };

    xen_call_(session, "VIF.remove_from_other_config", param_values, 2, NULL, NULL);
    return session->ok;
}


bool
xen_vif_set_qos_algorithm_type(xen_session *session, xen_vif vif, char *algorithm_type)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif },
            { .type = &abstract_type_string,
              .u.string_val = algorithm_type }
        };

    xen_call_(session, "VIF.set_qos_algorithm_type", param_values, 2, NULL, NULL);
    return session->ok;
}


bool
xen_vif_set_qos_algorithm_params(xen_session *session, xen_vif vif, xen_string_string_map *algorithm_params)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif },
            { .type = &abstract_type_string_string_map,
              .u.set_val = (arbitrary_set *)algorithm_params }
        };

    xen_call_(session, "VIF.set_qos_algorithm_params", param_values, 2, NULL, NULL);
    return session->ok;
}


bool
xen_vif_add_to_qos_algorithm_params(xen_session *session, xen_vif vif, char *key, char *value)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif },
            { .type = &abstract_type_string,
              .u.string_val = key },
            { .type = &abstract_type_string,
              .u.string_val = value }
        };

    xen_call_(session, "VIF.add_to_qos_algorithm_params", param_values, 3, NULL, NULL);
    return session->ok;
}


bool
xen_vif_remove_from_qos_algorithm_params(xen_session *session, xen_vif vif, char *key)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif },
            { .type = &abstract_type_string,
              .u.string_val = key }
        };

    xen_call_(session, "VIF.remove_from_qos_algorithm_params", param_values, 2, NULL, NULL);
    return session->ok;
}


bool
xen_vif_plug(xen_session *session, xen_vif self)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self }
        };

    xen_call_(session, "VIF.plug", param_values, 1, NULL, NULL);
    return session->ok;
}

bool
xen_vif_plug_async(xen_session *session, xen_task *result, xen_vif self)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("Async.VIF.plug");
    return session->ok;
}

bool
xen_vif_unplug(xen_session *session, xen_vif self)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self }
        };

    xen_call_(session, "VIF.unplug", param_values, 1, NULL, NULL);
    return session->ok;
}

bool
xen_vif_unplug_async(xen_session *session, xen_task *result, xen_vif self)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("Async.VIF.unplug");
    return session->ok;
}

bool
xen_vif_unplug_force(xen_session *session, xen_vif self)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self }
        };

    xen_call_(session, "VIF.unplug_force", param_values, 1, NULL, NULL);
    return session->ok;
}

bool
xen_vif_unplug_force_async(xen_session *session, xen_task *result, xen_vif self)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("Async.VIF.unplug_force");
    return session->ok;
}

bool
xen_vif_set_locking_mode(xen_session *session, xen_vif self, enum xen_vif_locking_mode value)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self },
            { .type = &xen_vif_locking_mode_abstract_type_,
              .u.enum_val = value }
        };

    xen_call_(session, "VIF.set_locking_mode", param_values, 2, NULL, NULL);
    return session->ok;
}

bool
xen_vif_set_locking_mode_async(xen_session *session, xen_task *result, xen_vif self, enum xen_vif_locking_mode value)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self },
            { .type = &xen_vif_locking_mode_abstract_type_,
              .u.enum_val = value }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("Async.VIF.set_locking_mode");
    return session->ok;
}

bool
xen_vif_set_ipv4_allowed(xen_session *session, xen_vif self, struct xen_string_set *value)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self },
            { .type = &abstract_type_string_set,
              .u.set_val = (arbitrary_set *)value }
        };

    xen_call_(session, "VIF.set_ipv4_allowed", param_values, 2, NULL, NULL);
    return session->ok;
}

bool
xen_vif_set_ipv4_allowed_async(xen_session *session, xen_task *result, xen_vif self, struct xen_string_set *value)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self },
            { .type = &abstract_type_string_set,
              .u.set_val = (arbitrary_set *)value }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("Async.VIF.set_ipv4_allowed");
    return session->ok;
}

bool
xen_vif_add_ipv4_allowed(xen_session *session, xen_vif self, char *value)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self },
            { .type = &abstract_type_string,
              .u.string_val = value }
        };

    xen_call_(session, "VIF.add_ipv4_allowed", param_values, 2, NULL, NULL);
    return session->ok;
}

bool
xen_vif_add_ipv4_allowed_async(xen_session *session, xen_task *result, xen_vif self, char *value)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self },
            { .type = &abstract_type_string,
              .u.string_val = value }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("Async.VIF.add_ipv4_allowed");
    return session->ok;
}

bool
xen_vif_remove_ipv4_allowed(xen_session *session, xen_vif self, char *value)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self },
            { .type = &abstract_type_string,
              .u.string_val = value }
        };

    xen_call_(session, "VIF.remove_ipv4_allowed", param_values, 2, NULL, NULL);
    return session->ok;
}

bool
xen_vif_remove_ipv4_allowed_async(xen_session *session, xen_task *result, xen_vif self, char *value)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self },
            { .type = &abstract_type_string,
              .u.string_val = value }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("Async.VIF.remove_ipv4_allowed");
    return session->ok;
}

bool
xen_vif_set_ipv6_allowed(xen_session *session, xen_vif self, struct xen_string_set *value)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self },
            { .type = &abstract_type_string_set,
              .u.set_val = (arbitrary_set *)value }
        };

    xen_call_(session, "VIF.set_ipv6_allowed", param_values, 2, NULL, NULL);
    return session->ok;
}

bool
xen_vif_set_ipv6_allowed_async(xen_session *session, xen_task *result, xen_vif self, struct xen_string_set *value)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self },
            { .type = &abstract_type_string_set,
              .u.set_val = (arbitrary_set *)value }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("Async.VIF.set_ipv6_allowed");
    return session->ok;
}

bool
xen_vif_add_ipv6_allowed(xen_session *session, xen_vif self, char *value)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self },
            { .type = &abstract_type_string,
              .u.string_val = value }
        };

    xen_call_(session, "VIF.add_ipv6_allowed", param_values, 2, NULL, NULL);
    return session->ok;
}

bool
xen_vif_add_ipv6_allowed_async(xen_session *session, xen_task *result, xen_vif self, char *value)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self },
            { .type = &abstract_type_string,
              .u.string_val = value }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("Async.VIF.add_ipv6_allowed");
    return session->ok;
}

bool
xen_vif_remove_ipv6_allowed(xen_session *session, xen_vif self, char *value)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self },
            { .type = &abstract_type_string,
              .u.string_val = value }
        };

    xen_call_(session, "VIF.remove_ipv6_allowed", param_values, 2, NULL, NULL);
    return session->ok;
}

bool
xen_vif_remove_ipv6_allowed_async(xen_session *session, xen_task *result, xen_vif self, char *value)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self },
            { .type = &abstract_type_string,
              .u.string_val = value }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("Async.VIF.remove_ipv6_allowed");
    return session->ok;
}

bool
xen_vif_get_all(xen_session *session, struct xen_vif_set **result)
{

    abstract_type result_type = abstract_type_string_set;

    *result = NULL;
    xen_call_(session, "VIF.get_all", NULL, 0, &result_type, result);
    return session->ok;
}


bool
xen_vif_get_all_records(xen_session *session, xen_vif_xen_vif_record_map **result)
{

    abstract_type result_type = abstract_type_string_xen_vif_record_map;

    *result = NULL;
    xen_call_(session, "VIF.get_all_records", NULL, 0, &result_type, result);
    return session->ok;
}


bool
xen_vif_get_uuid(xen_session *session, char **result, xen_vif vif)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("VIF.get_uuid");
    return session->ok;
}

