/*
 * Copyright (c) Citrix Systems, Inc.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 *   1) Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 * 
 *   2) Redistributions in binary form must reproduce the above
 *      copyright notice, this list of conditions and the following
 *      disclaimer in the documentation and/or other materials
 *      provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using System;
using System.Collections;
using System.Collections.Generic;

using CookComputing.XmlRpc;


namespace XenAPI
{
    public partial class VIF : XenObject<VIF>
    {
        public VIF()
        {
        }

        public VIF(string uuid,
            List<vif_operations> allowed_operations,
            Dictionary<string, vif_operations> current_operations,
            string device,
            XenRef<Network> network,
            XenRef<VM> VM,
            string MAC,
            long MTU,
            Dictionary<string, string> other_config,
            bool currently_attached,
            long status_code,
            string status_detail,
            Dictionary<string, string> runtime_properties,
            string qos_algorithm_type,
            Dictionary<string, string> qos_algorithm_params,
            string[] qos_supported_algorithms,
            XenRef<VIF_metrics> metrics,
            bool MAC_autogenerated,
            vif_locking_mode locking_mode,
            string[] ipv4_allowed,
            string[] ipv6_allowed)
        {
            this.uuid = uuid;
            this.allowed_operations = allowed_operations;
            this.current_operations = current_operations;
            this.device = device;
            this.network = network;
            this.VM = VM;
            this.MAC = MAC;
            this.MTU = MTU;
            this.other_config = other_config;
            this.currently_attached = currently_attached;
            this.status_code = status_code;
            this.status_detail = status_detail;
            this.runtime_properties = runtime_properties;
            this.qos_algorithm_type = qos_algorithm_type;
            this.qos_algorithm_params = qos_algorithm_params;
            this.qos_supported_algorithms = qos_supported_algorithms;
            this.metrics = metrics;
            this.MAC_autogenerated = MAC_autogenerated;
            this.locking_mode = locking_mode;
            this.ipv4_allowed = ipv4_allowed;
            this.ipv6_allowed = ipv6_allowed;
        }

        /// <summary>
        /// Creates a new VIF from a Proxy_VIF.
        /// </summary>
        /// <param name="proxy"></param>
        public VIF(Proxy_VIF proxy)
        {
            this.UpdateFromProxy(proxy);
        }

        public override void UpdateFrom(VIF update)
        {
            uuid = update.uuid;
            allowed_operations = update.allowed_operations;
            current_operations = update.current_operations;
            device = update.device;
            network = update.network;
            VM = update.VM;
            MAC = update.MAC;
            MTU = update.MTU;
            other_config = update.other_config;
            currently_attached = update.currently_attached;
            status_code = update.status_code;
            status_detail = update.status_detail;
            runtime_properties = update.runtime_properties;
            qos_algorithm_type = update.qos_algorithm_type;
            qos_algorithm_params = update.qos_algorithm_params;
            qos_supported_algorithms = update.qos_supported_algorithms;
            metrics = update.metrics;
            MAC_autogenerated = update.MAC_autogenerated;
            locking_mode = update.locking_mode;
            ipv4_allowed = update.ipv4_allowed;
            ipv6_allowed = update.ipv6_allowed;
        }

        internal void UpdateFromProxy(Proxy_VIF proxy)
        {
            uuid = proxy.uuid == null ? null : (string)proxy.uuid;
            allowed_operations = proxy.allowed_operations == null ? null : Helper.StringArrayToEnumList<vif_operations>(proxy.allowed_operations);
            current_operations = proxy.current_operations == null ? null : Maps.convert_from_proxy_string_vif_operations(proxy.current_operations);
            device = proxy.device == null ? null : (string)proxy.device;
            network = proxy.network == null ? null : XenRef<Network>.Create(proxy.network);
            VM = proxy.VM == null ? null : XenRef<VM>.Create(proxy.VM);
            MAC = proxy.MAC == null ? null : (string)proxy.MAC;
            MTU = proxy.MTU == null ? 0 : long.Parse((string)proxy.MTU);
            other_config = proxy.other_config == null ? null : Maps.convert_from_proxy_string_string(proxy.other_config);
            currently_attached = (bool)proxy.currently_attached;
            status_code = proxy.status_code == null ? 0 : long.Parse((string)proxy.status_code);
            status_detail = proxy.status_detail == null ? null : (string)proxy.status_detail;
            runtime_properties = proxy.runtime_properties == null ? null : Maps.convert_from_proxy_string_string(proxy.runtime_properties);
            qos_algorithm_type = proxy.qos_algorithm_type == null ? null : (string)proxy.qos_algorithm_type;
            qos_algorithm_params = proxy.qos_algorithm_params == null ? null : Maps.convert_from_proxy_string_string(proxy.qos_algorithm_params);
            qos_supported_algorithms = proxy.qos_supported_algorithms == null ? new string[] {} : (string [])proxy.qos_supported_algorithms;
            metrics = proxy.metrics == null ? null : XenRef<VIF_metrics>.Create(proxy.metrics);
            MAC_autogenerated = (bool)proxy.MAC_autogenerated;
            locking_mode = proxy.locking_mode == null ? (vif_locking_mode) 0 : (vif_locking_mode)Helper.EnumParseDefault(typeof(vif_locking_mode), (string)proxy.locking_mode);
            ipv4_allowed = proxy.ipv4_allowed == null ? new string[] {} : (string [])proxy.ipv4_allowed;
            ipv6_allowed = proxy.ipv6_allowed == null ? new string[] {} : (string [])proxy.ipv6_allowed;
        }

        public Proxy_VIF ToProxy()
        {
            Proxy_VIF result_ = new Proxy_VIF();
            result_.uuid = (uuid != null) ? uuid : "";
            result_.allowed_operations = (allowed_operations != null) ? Helper.ObjectListToStringArray(allowed_operations) : new string[] {};
            result_.current_operations = Maps.convert_to_proxy_string_vif_operations(current_operations);
            result_.device = (device != null) ? device : "";
            result_.network = (network != null) ? network : "";
            result_.VM = (VM != null) ? VM : "";
            result_.MAC = (MAC != null) ? MAC : "";
            result_.MTU = MTU.ToString();
            result_.other_config = Maps.convert_to_proxy_string_string(other_config);
            result_.currently_attached = currently_attached;
            result_.status_code = status_code.ToString();
            result_.status_detail = (status_detail != null) ? status_detail : "";
            result_.runtime_properties = Maps.convert_to_proxy_string_string(runtime_properties);
            result_.qos_algorithm_type = (qos_algorithm_type != null) ? qos_algorithm_type : "";
            result_.qos_algorithm_params = Maps.convert_to_proxy_string_string(qos_algorithm_params);
            result_.qos_supported_algorithms = qos_supported_algorithms;
            result_.metrics = (metrics != null) ? metrics : "";
            result_.MAC_autogenerated = MAC_autogenerated;
            result_.locking_mode = vif_locking_mode_helper.ToString(locking_mode);
            result_.ipv4_allowed = ipv4_allowed;
            result_.ipv6_allowed = ipv6_allowed;
            return result_;
        }

        /// <summary>
        /// Creates a new VIF from a Hashtable.
        /// </summary>
        /// <param name="table"></param>
        public VIF(Hashtable table)
        {
            uuid = Marshalling.ParseString(table, "uuid");
            allowed_operations = Helper.StringArrayToEnumList<vif_operations>(Marshalling.ParseStringArray(table, "allowed_operations"));
            current_operations = Maps.convert_from_proxy_string_vif_operations(Marshalling.ParseHashTable(table, "current_operations"));
            device = Marshalling.ParseString(table, "device");
            network = Marshalling.ParseRef<Network>(table, "network");
            VM = Marshalling.ParseRef<VM>(table, "VM");
            MAC = Marshalling.ParseString(table, "MAC");
            MTU = Marshalling.ParseLong(table, "MTU");
            other_config = Maps.convert_from_proxy_string_string(Marshalling.ParseHashTable(table, "other_config"));
            currently_attached = Marshalling.ParseBool(table, "currently_attached");
            status_code = Marshalling.ParseLong(table, "status_code");
            status_detail = Marshalling.ParseString(table, "status_detail");
            runtime_properties = Maps.convert_from_proxy_string_string(Marshalling.ParseHashTable(table, "runtime_properties"));
            qos_algorithm_type = Marshalling.ParseString(table, "qos_algorithm_type");
            qos_algorithm_params = Maps.convert_from_proxy_string_string(Marshalling.ParseHashTable(table, "qos_algorithm_params"));
            qos_supported_algorithms = Marshalling.ParseStringArray(table, "qos_supported_algorithms");
            metrics = Marshalling.ParseRef<VIF_metrics>(table, "metrics");
            MAC_autogenerated = Marshalling.ParseBool(table, "MAC_autogenerated");
            locking_mode = (vif_locking_mode)Helper.EnumParseDefault(typeof(vif_locking_mode), Marshalling.ParseString(table, "locking_mode"));
            ipv4_allowed = Marshalling.ParseStringArray(table, "ipv4_allowed");
            ipv6_allowed = Marshalling.ParseStringArray(table, "ipv6_allowed");
        }

        public bool DeepEquals(VIF other, bool ignoreCurrentOperations)
        {
            if (ReferenceEquals(null, other))
                return false;
            if (ReferenceEquals(this, other))
                return true;

            if (!ignoreCurrentOperations && !Helper.AreEqual2(this.current_operations, other.current_operations))
                return false;

            return Helper.AreEqual2(this._uuid, other._uuid) &&
                Helper.AreEqual2(this._allowed_operations, other._allowed_operations) &&
                Helper.AreEqual2(this._device, other._device) &&
                Helper.AreEqual2(this._network, other._network) &&
                Helper.AreEqual2(this._VM, other._VM) &&
                Helper.AreEqual2(this._MAC, other._MAC) &&
                Helper.AreEqual2(this._MTU, other._MTU) &&
                Helper.AreEqual2(this._other_config, other._other_config) &&
                Helper.AreEqual2(this._currently_attached, other._currently_attached) &&
                Helper.AreEqual2(this._status_code, other._status_code) &&
                Helper.AreEqual2(this._status_detail, other._status_detail) &&
                Helper.AreEqual2(this._runtime_properties, other._runtime_properties) &&
                Helper.AreEqual2(this._qos_algorithm_type, other._qos_algorithm_type) &&
                Helper.AreEqual2(this._qos_algorithm_params, other._qos_algorithm_params) &&
                Helper.AreEqual2(this._qos_supported_algorithms, other._qos_supported_algorithms) &&
                Helper.AreEqual2(this._metrics, other._metrics) &&
                Helper.AreEqual2(this._MAC_autogenerated, other._MAC_autogenerated) &&
                Helper.AreEqual2(this._locking_mode, other._locking_mode) &&
                Helper.AreEqual2(this._ipv4_allowed, other._ipv4_allowed) &&
                Helper.AreEqual2(this._ipv6_allowed, other._ipv6_allowed);
        }

        public override string SaveChanges(Session session, string opaqueRef, VIF server)
        {
            if (opaqueRef == null)
            {
                Proxy_VIF p = this.ToProxy();
                return session.proxy.vif_create(session.uuid, p).parse();
            }
            else
            {
                if (!Helper.AreEqual2(_other_config, server._other_config))
                {
                    VIF.set_other_config(session, opaqueRef, _other_config);
                }
                if (!Helper.AreEqual2(_qos_algorithm_type, server._qos_algorithm_type))
                {
                    VIF.set_qos_algorithm_type(session, opaqueRef, _qos_algorithm_type);
                }
                if (!Helper.AreEqual2(_qos_algorithm_params, server._qos_algorithm_params))
                {
                    VIF.set_qos_algorithm_params(session, opaqueRef, _qos_algorithm_params);
                }
                if (!Helper.AreEqual2(_locking_mode, server._locking_mode))
                {
                    VIF.set_locking_mode(session, opaqueRef, _locking_mode);
                }
                if (!Helper.AreEqual2(_ipv4_allowed, server._ipv4_allowed))
                {
                    VIF.set_ipv4_allowed(session, opaqueRef, _ipv4_allowed);
                }
                if (!Helper.AreEqual2(_ipv6_allowed, server._ipv6_allowed))
                {
                    VIF.set_ipv6_allowed(session, opaqueRef, _ipv6_allowed);
                }

                return null;
            }
        }

        public static VIF get_record(Session session, string _vif)
        {
            return new VIF((Proxy_VIF)session.proxy.vif_get_record(session.uuid, (_vif != null) ? _vif : "").parse());
        }

        public static XenRef<VIF> get_by_uuid(Session session, string _uuid)
        {
            return XenRef<VIF>.Create(session.proxy.vif_get_by_uuid(session.uuid, (_uuid != null) ? _uuid : "").parse());
        }

        public static XenRef<VIF> create(Session session, VIF _record)
        {
            return XenRef<VIF>.Create(session.proxy.vif_create(session.uuid, _record.ToProxy()).parse());
        }

        public static XenRef<Task> async_create(Session session, VIF _record)
        {
            return XenRef<Task>.Create(session.proxy.async_vif_create(session.uuid, _record.ToProxy()).parse());
        }

        public static void destroy(Session session, string _vif)
        {
            session.proxy.vif_destroy(session.uuid, (_vif != null) ? _vif : "").parse();
        }

        public static XenRef<Task> async_destroy(Session session, string _vif)
        {
            return XenRef<Task>.Create(session.proxy.async_vif_destroy(session.uuid, (_vif != null) ? _vif : "").parse());
        }

        public static string get_uuid(Session session, string _vif)
        {
            return (string)session.proxy.vif_get_uuid(session.uuid, (_vif != null) ? _vif : "").parse();
        }

        public static List<vif_operations> get_allowed_operations(Session session, string _vif)
        {
            return Helper.StringArrayToEnumList<vif_operations>(session.proxy.vif_get_allowed_operations(session.uuid, (_vif != null) ? _vif : "").parse());
        }

        public static Dictionary<string, vif_operations> get_current_operations(Session session, string _vif)
        {
            return Maps.convert_from_proxy_string_vif_operations(session.proxy.vif_get_current_operations(session.uuid, (_vif != null) ? _vif : "").parse());
        }

        public static string get_device(Session session, string _vif)
        {
            return (string)session.proxy.vif_get_device(session.uuid, (_vif != null) ? _vif : "").parse();
        }

        public static XenRef<Network> get_network(Session session, string _vif)
        {
            return XenRef<Network>.Create(session.proxy.vif_get_network(session.uuid, (_vif != null) ? _vif : "").parse());
        }

        public static XenRef<VM> get_VM(Session session, string _vif)
        {
            return XenRef<VM>.Create(session.proxy.vif_get_vm(session.uuid, (_vif != null) ? _vif : "").parse());
        }

        public static string get_MAC(Session session, string _vif)
        {
            return (string)session.proxy.vif_get_mac(session.uuid, (_vif != null) ? _vif : "").parse();
        }

        public static long get_MTU(Session session, string _vif)
        {
            return long.Parse((string)session.proxy.vif_get_mtu(session.uuid, (_vif != null) ? _vif : "").parse());
        }

        public static Dictionary<string, string> get_other_config(Session session, string _vif)
        {
            return Maps.convert_from_proxy_string_string(session.proxy.vif_get_other_config(session.uuid, (_vif != null) ? _vif : "").parse());
        }

        public static bool get_currently_attached(Session session, string _vif)
        {
            return (bool)session.proxy.vif_get_currently_attached(session.uuid, (_vif != null) ? _vif : "").parse();
        }

        public static long get_status_code(Session session, string _vif)
        {
            return long.Parse((string)session.proxy.vif_get_status_code(session.uuid, (_vif != null) ? _vif : "").parse());
        }

        public static string get_status_detail(Session session, string _vif)
        {
            return (string)session.proxy.vif_get_status_detail(session.uuid, (_vif != null) ? _vif : "").parse();
        }

        public static Dictionary<string, string> get_runtime_properties(Session session, string _vif)
        {
            return Maps.convert_from_proxy_string_string(session.proxy.vif_get_runtime_properties(session.uuid, (_vif != null) ? _vif : "").parse());
        }

        public static string get_qos_algorithm_type(Session session, string _vif)
        {
            return (string)session.proxy.vif_get_qos_algorithm_type(session.uuid, (_vif != null) ? _vif : "").parse();
        }

        public static Dictionary<string, string> get_qos_algorithm_params(Session session, string _vif)
        {
            return Maps.convert_from_proxy_string_string(session.proxy.vif_get_qos_algorithm_params(session.uuid, (_vif != null) ? _vif : "").parse());
        }

        public static string[] get_qos_supported_algorithms(Session session, string _vif)
        {
            return (string [])session.proxy.vif_get_qos_supported_algorithms(session.uuid, (_vif != null) ? _vif : "").parse();
        }

        public static XenRef<VIF_metrics> get_metrics(Session session, string _vif)
        {
            return XenRef<VIF_metrics>.Create(session.proxy.vif_get_metrics(session.uuid, (_vif != null) ? _vif : "").parse());
        }

        public static bool get_MAC_autogenerated(Session session, string _vif)
        {
            return (bool)session.proxy.vif_get_mac_autogenerated(session.uuid, (_vif != null) ? _vif : "").parse();
        }

        public static vif_locking_mode get_locking_mode(Session session, string _vif)
        {
            return (vif_locking_mode)Helper.EnumParseDefault(typeof(vif_locking_mode), (string)session.proxy.vif_get_locking_mode(session.uuid, (_vif != null) ? _vif : "").parse());
        }

        public static string[] get_ipv4_allowed(Session session, string _vif)
        {
            return (string [])session.proxy.vif_get_ipv4_allowed(session.uuid, (_vif != null) ? _vif : "").parse();
        }

        public static string[] get_ipv6_allowed(Session session, string _vif)
        {
            return (string [])session.proxy.vif_get_ipv6_allowed(session.uuid, (_vif != null) ? _vif : "").parse();
        }

        public static void set_other_config(Session session, string _vif, Dictionary<string, string> _other_config)
        {
            session.proxy.vif_set_other_config(session.uuid, (_vif != null) ? _vif : "", Maps.convert_to_proxy_string_string(_other_config)).parse();
        }

        public static void add_to_other_config(Session session, string _vif, string _key, string _value)
        {
            session.proxy.vif_add_to_other_config(session.uuid, (_vif != null) ? _vif : "", (_key != null) ? _key : "", (_value != null) ? _value : "").parse();
        }

        public static void remove_from_other_config(Session session, string _vif, string _key)
        {
            session.proxy.vif_remove_from_other_config(session.uuid, (_vif != null) ? _vif : "", (_key != null) ? _key : "").parse();
        }

        public static void set_qos_algorithm_type(Session session, string _vif, string _algorithm_type)
        {
            session.proxy.vif_set_qos_algorithm_type(session.uuid, (_vif != null) ? _vif : "", (_algorithm_type != null) ? _algorithm_type : "").parse();
        }

        public static void set_qos_algorithm_params(Session session, string _vif, Dictionary<string, string> _algorithm_params)
        {
            session.proxy.vif_set_qos_algorithm_params(session.uuid, (_vif != null) ? _vif : "", Maps.convert_to_proxy_string_string(_algorithm_params)).parse();
        }

        public static void add_to_qos_algorithm_params(Session session, string _vif, string _key, string _value)
        {
            session.proxy.vif_add_to_qos_algorithm_params(session.uuid, (_vif != null) ? _vif : "", (_key != null) ? _key : "", (_value != null) ? _value : "").parse();
        }

        public static void remove_from_qos_algorithm_params(Session session, string _vif, string _key)
        {
            session.proxy.vif_remove_from_qos_algorithm_params(session.uuid, (_vif != null) ? _vif : "", (_key != null) ? _key : "").parse();
        }

        public static void plug(Session session, string _self)
        {
            session.proxy.vif_plug(session.uuid, (_self != null) ? _self : "").parse();
        }

        public static XenRef<Task> async_plug(Session session, string _self)
        {
            return XenRef<Task>.Create(session.proxy.async_vif_plug(session.uuid, (_self != null) ? _self : "").parse());
        }

        public static void unplug(Session session, string _self)
        {
            session.proxy.vif_unplug(session.uuid, (_self != null) ? _self : "").parse();
        }

        public static XenRef<Task> async_unplug(Session session, string _self)
        {
            return XenRef<Task>.Create(session.proxy.async_vif_unplug(session.uuid, (_self != null) ? _self : "").parse());
        }

        public static void unplug_force(Session session, string _self)
        {
            session.proxy.vif_unplug_force(session.uuid, (_self != null) ? _self : "").parse();
        }

        public static XenRef<Task> async_unplug_force(Session session, string _self)
        {
            return XenRef<Task>.Create(session.proxy.async_vif_unplug_force(session.uuid, (_self != null) ? _self : "").parse());
        }

        public static void set_locking_mode(Session session, string _self, vif_locking_mode _value)
        {
            session.proxy.vif_set_locking_mode(session.uuid, (_self != null) ? _self : "", vif_locking_mode_helper.ToString(_value)).parse();
        }

        public static XenRef<Task> async_set_locking_mode(Session session, string _self, vif_locking_mode _value)
        {
            return XenRef<Task>.Create(session.proxy.async_vif_set_locking_mode(session.uuid, (_self != null) ? _self : "", vif_locking_mode_helper.ToString(_value)).parse());
        }

        public static void set_ipv4_allowed(Session session, string _self, string[] _value)
        {
            session.proxy.vif_set_ipv4_allowed(session.uuid, (_self != null) ? _self : "", _value).parse();
        }

        public static XenRef<Task> async_set_ipv4_allowed(Session session, string _self, string[] _value)
        {
            return XenRef<Task>.Create(session.proxy.async_vif_set_ipv4_allowed(session.uuid, (_self != null) ? _self : "", _value).parse());
        }

        public static void add_ipv4_allowed(Session session, string _self, string _value)
        {
            session.proxy.vif_add_ipv4_allowed(session.uuid, (_self != null) ? _self : "", (_value != null) ? _value : "").parse();
        }

        public static XenRef<Task> async_add_ipv4_allowed(Session session, string _self, string _value)
        {
            return XenRef<Task>.Create(session.proxy.async_vif_add_ipv4_allowed(session.uuid, (_self != null) ? _self : "", (_value != null) ? _value : "").parse());
        }

        public static void remove_ipv4_allowed(Session session, string _self, string _value)
        {
            session.proxy.vif_remove_ipv4_allowed(session.uuid, (_self != null) ? _self : "", (_value != null) ? _value : "").parse();
        }

        public static XenRef<Task> async_remove_ipv4_allowed(Session session, string _self, string _value)
        {
            return XenRef<Task>.Create(session.proxy.async_vif_remove_ipv4_allowed(session.uuid, (_self != null) ? _self : "", (_value != null) ? _value : "").parse());
        }

        public static void set_ipv6_allowed(Session session, string _self, string[] _value)
        {
            session.proxy.vif_set_ipv6_allowed(session.uuid, (_self != null) ? _self : "", _value).parse();
        }

        public static XenRef<Task> async_set_ipv6_allowed(Session session, string _self, string[] _value)
        {
            return XenRef<Task>.Create(session.proxy.async_vif_set_ipv6_allowed(session.uuid, (_self != null) ? _self : "", _value).parse());
        }

        public static void add_ipv6_allowed(Session session, string _self, string _value)
        {
            session.proxy.vif_add_ipv6_allowed(session.uuid, (_self != null) ? _self : "", (_value != null) ? _value : "").parse();
        }

        public static XenRef<Task> async_add_ipv6_allowed(Session session, string _self, string _value)
        {
            return XenRef<Task>.Create(session.proxy.async_vif_add_ipv6_allowed(session.uuid, (_self != null) ? _self : "", (_value != null) ? _value : "").parse());
        }

        public static void remove_ipv6_allowed(Session session, string _self, string _value)
        {
            session.proxy.vif_remove_ipv6_allowed(session.uuid, (_self != null) ? _self : "", (_value != null) ? _value : "").parse();
        }

        public static XenRef<Task> async_remove_ipv6_allowed(Session session, string _self, string _value)
        {
            return XenRef<Task>.Create(session.proxy.async_vif_remove_ipv6_allowed(session.uuid, (_self != null) ? _self : "", (_value != null) ? _value : "").parse());
        }

        public static List<XenRef<VIF>> get_all(Session session)
        {
            return XenRef<VIF>.Create(session.proxy.vif_get_all(session.uuid).parse());
        }

        public static Dictionary<XenRef<VIF>, VIF> get_all_records(Session session)
        {
            return XenRef<VIF>.Create<Proxy_VIF>(session.proxy.vif_get_all_records(session.uuid).parse());
        }

        private string _uuid;
        public virtual string uuid {
             get { return _uuid; }
             set { if (!Helper.AreEqual(value, _uuid)) { _uuid = value; Changed = true; NotifyPropertyChanged("uuid"); } }
         }

        private List<vif_operations> _allowed_operations;
        public virtual List<vif_operations> allowed_operations {
             get { return _allowed_operations; }
             set { if (!Helper.AreEqual(value, _allowed_operations)) { _allowed_operations = value; Changed = true; NotifyPropertyChanged("allowed_operations"); } }
         }

        private Dictionary<string, vif_operations> _current_operations;
        public virtual Dictionary<string, vif_operations> current_operations {
             get { return _current_operations; }
             set { if (!Helper.AreEqual(value, _current_operations)) { _current_operations = value; Changed = true; NotifyPropertyChanged("current_operations"); } }
         }

        private string _device;
        public virtual string device {
             get { return _device; }
             set { if (!Helper.AreEqual(value, _device)) { _device = value; Changed = true; NotifyPropertyChanged("device"); } }
         }

        private XenRef<Network> _network;
        public virtual XenRef<Network> network {
             get { return _network; }
             set { if (!Helper.AreEqual(value, _network)) { _network = value; Changed = true; NotifyPropertyChanged("network"); } }
         }

        private XenRef<VM> _VM;
        public virtual XenRef<VM> VM {
             get { return _VM; }
             set { if (!Helper.AreEqual(value, _VM)) { _VM = value; Changed = true; NotifyPropertyChanged("VM"); } }
         }

        private string _MAC;
        public virtual string MAC {
             get { return _MAC; }
             set { if (!Helper.AreEqual(value, _MAC)) { _MAC = value; Changed = true; NotifyPropertyChanged("MAC"); } }
         }

        private long _MTU;
        public virtual long MTU {
             get { return _MTU; }
             set { if (!Helper.AreEqual(value, _MTU)) { _MTU = value; Changed = true; NotifyPropertyChanged("MTU"); } }
         }

        private Dictionary<string, string> _other_config;
        public virtual Dictionary<string, string> other_config {
             get { return _other_config; }
             set { if (!Helper.AreEqual(value, _other_config)) { _other_config = value; Changed = true; NotifyPropertyChanged("other_config"); } }
         }

        private bool _currently_attached;
        public virtual bool currently_attached {
             get { return _currently_attached; }
             set { if (!Helper.AreEqual(value, _currently_attached)) { _currently_attached = value; Changed = true; NotifyPropertyChanged("currently_attached"); } }
         }

        private long _status_code;
        public virtual long status_code {
             get { return _status_code; }
             set { if (!Helper.AreEqual(value, _status_code)) { _status_code = value; Changed = true; NotifyPropertyChanged("status_code"); } }
         }

        private string _status_detail;
        public virtual string status_detail {
             get { return _status_detail; }
             set { if (!Helper.AreEqual(value, _status_detail)) { _status_detail = value; Changed = true; NotifyPropertyChanged("status_detail"); } }
         }

        private Dictionary<string, string> _runtime_properties;
        public virtual Dictionary<string, string> runtime_properties {
             get { return _runtime_properties; }
             set { if (!Helper.AreEqual(value, _runtime_properties)) { _runtime_properties = value; Changed = true; NotifyPropertyChanged("runtime_properties"); } }
         }

        private string _qos_algorithm_type;
        public virtual string qos_algorithm_type {
             get { return _qos_algorithm_type; }
             set { if (!Helper.AreEqual(value, _qos_algorithm_type)) { _qos_algorithm_type = value; Changed = true; NotifyPropertyChanged("qos_algorithm_type"); } }
         }

        private Dictionary<string, string> _qos_algorithm_params;
        public virtual Dictionary<string, string> qos_algorithm_params {
             get { return _qos_algorithm_params; }
             set { if (!Helper.AreEqual(value, _qos_algorithm_params)) { _qos_algorithm_params = value; Changed = true; NotifyPropertyChanged("qos_algorithm_params"); } }
         }

        private string[] _qos_supported_algorithms;
        public virtual string[] qos_supported_algorithms {
             get { return _qos_supported_algorithms; }
             set { if (!Helper.AreEqual(value, _qos_supported_algorithms)) { _qos_supported_algorithms = value; Changed = true; NotifyPropertyChanged("qos_supported_algorithms"); } }
         }

        private XenRef<VIF_metrics> _metrics;
        public virtual XenRef<VIF_metrics> metrics {
             get { return _metrics; }
             set { if (!Helper.AreEqual(value, _metrics)) { _metrics = value; Changed = true; NotifyPropertyChanged("metrics"); } }
         }

        private bool _MAC_autogenerated;
        public virtual bool MAC_autogenerated {
             get { return _MAC_autogenerated; }
             set { if (!Helper.AreEqual(value, _MAC_autogenerated)) { _MAC_autogenerated = value; Changed = true; NotifyPropertyChanged("MAC_autogenerated"); } }
         }

        private vif_locking_mode _locking_mode;
        public virtual vif_locking_mode locking_mode {
             get { return _locking_mode; }
             set { if (!Helper.AreEqual(value, _locking_mode)) { _locking_mode = value; Changed = true; NotifyPropertyChanged("locking_mode"); } }
         }

        private string[] _ipv4_allowed;
        public virtual string[] ipv4_allowed {
             get { return _ipv4_allowed; }
             set { if (!Helper.AreEqual(value, _ipv4_allowed)) { _ipv4_allowed = value; Changed = true; NotifyPropertyChanged("ipv4_allowed"); } }
         }

        private string[] _ipv6_allowed;
        public virtual string[] ipv6_allowed {
             get { return _ipv6_allowed; }
             set { if (!Helper.AreEqual(value, _ipv6_allowed)) { _ipv6_allowed = value; Changed = true; NotifyPropertyChanged("ipv6_allowed"); } }
         }


    }
}
