#include "game/los.h"

#include "game/room.h"
#include "global/const.h"
#include "util.h"

#include <stdint.h>

static int32_t LOS_CheckX(const GAME_VECTOR *start, GAME_VECTOR *target);
static int32_t LOS_CheckZ(const GAME_VECTOR *start, GAME_VECTOR *target);
static bool LOS_ClipTarget(
    const GAME_VECTOR *start, GAME_VECTOR *target, const FLOOR_INFO *floor);

static int32_t LOS_CheckX(
    const GAME_VECTOR *const start, GAME_VECTOR *const target)
{
    FLOOR_INFO *floor;

    int32_t dx = target->x - start->x;
    if (dx == 0) {
        return 1;
    }

    int32_t dy = ((target->y - start->y) << WALL_SHIFT) / dx;
    int32_t dz = ((target->z - start->z) << WALL_SHIFT) / dx;

    int16_t room_num = start->room_number;
    int16_t last_room;

    if (dx < 0) {
        int32_t x = start->x & ~(WALL_L - 1);
        int32_t y = start->y + ((dy * (x - start->x)) >> WALL_SHIFT);
        int32_t z = start->z + ((dz * (x - start->x)) >> WALL_SHIFT);

        while (x > target->x) {
            floor = Room_GetFloor(x, y, z, &room_num);
            if (y > Room_GetHeight(floor, x, y, z)
                || y < Room_GetCeiling(floor, x, y, z)) {
                target->x = x;
                target->y = y;
                target->z = z;
                target->room_number = room_num;
                return -1;
            }

            last_room = room_num;

            floor = Room_GetFloor(x - 1, y, z, &room_num);
            if (y > Room_GetHeight(floor, x - 1, y, z)
                || y < Room_GetCeiling(floor, x - 1, y, z)) {
                target->x = x;
                target->y = y;
                target->z = z;
                target->room_number = last_room;
                return 0;
            }

            x -= WALL_L;
            y -= dy;
            z -= dz;
        }
    } else {
        int32_t x = start->x | (WALL_L - 1);
        int32_t y = start->y + ((dy * (x - start->x)) >> WALL_SHIFT);
        int32_t z = start->z + ((dz * (x - start->x)) >> WALL_SHIFT);

        while (x < target->x) {
            floor = Room_GetFloor(x, y, z, &room_num);
            if (y > Room_GetHeight(floor, x, y, z)
                || y < Room_GetCeiling(floor, x, y, z)) {
                target->x = x;
                target->y = y;
                target->z = z;
                target->room_number = room_num;
                return -1;
            }

            last_room = room_num;

            floor = Room_GetFloor(x + 1, y, z, &room_num);
            if (y > Room_GetHeight(floor, x + 1, y, z)
                || y < Room_GetCeiling(floor, x + 1, y, z)) {
                target->x = x;
                target->y = y;
                target->z = z;
                target->room_number = last_room;
                return 0;
            }

            x += WALL_L;
            y += dy;
            z += dz;
        }
    }

    target->room_number = room_num;
    return 1;
}

static int32_t LOS_CheckZ(
    const GAME_VECTOR *const start, GAME_VECTOR *const target)
{
    FLOOR_INFO *floor;

    int32_t dz = target->z - start->z;
    if (dz == 0) {
        return 1;
    }

    int32_t dx = ((target->x - start->x) << WALL_SHIFT) / dz;
    int32_t dy = ((target->y - start->y) << WALL_SHIFT) / dz;

    int16_t room_num = start->room_number;
    int16_t last_room;

    if (dz < 0) {
        int32_t z = start->z & ~(WALL_L - 1);
        int32_t x = start->x + ((dx * (z - start->z)) >> WALL_SHIFT);
        int32_t y = start->y + ((dy * (z - start->z)) >> WALL_SHIFT);

        while (z > target->z) {
            floor = Room_GetFloor(x, y, z, &room_num);
            if (y > Room_GetHeight(floor, x, y, z)
                || y < Room_GetCeiling(floor, x, y, z)) {
                target->x = x;
                target->y = y;
                target->z = z;
                target->room_number = room_num;
                return -1;
            }

            last_room = room_num;

            floor = Room_GetFloor(x, y, z - 1, &room_num);
            if (y > Room_GetHeight(floor, x, y, z - 1)
                || y < Room_GetCeiling(floor, x, y, z - 1)) {
                target->x = x;
                target->y = y;
                target->z = z;
                target->room_number = last_room;
                return 0;
            }

            z -= WALL_L;
            x -= dx;
            y -= dy;
        }
    } else {
        int32_t z = start->z | (WALL_L - 1);
        int32_t x = start->x + ((dx * (z - start->z)) >> WALL_SHIFT);
        int32_t y = start->y + ((dy * (z - start->z)) >> WALL_SHIFT);

        while (z < target->z) {
            floor = Room_GetFloor(x, y, z, &room_num);
            if (y > Room_GetHeight(floor, x, y, z)
                || y < Room_GetCeiling(floor, x, y, z)) {
                target->x = x;
                target->y = y;
                target->z = z;
                target->room_number = room_num;
                return -1;
            }

            last_room = room_num;

            floor = Room_GetFloor(x, y, z + 1, &room_num);
            if (y > Room_GetHeight(floor, x, y, z + 1)
                || y < Room_GetCeiling(floor, x, y, z + 1)) {
                target->x = x;
                target->y = y;
                target->z = z;
                target->room_number = last_room;
                return 0;
            }

            z += WALL_L;
            x += dx;
            y += dy;
        }
    }

    target->room_number = room_num;
    return 1;
}

static bool LOS_ClipTarget(
    const GAME_VECTOR *const start, GAME_VECTOR *const target,
    const FLOOR_INFO *const floor)
{
    int32_t dx = target->x - start->x;
    int32_t dy = target->y - start->y;
    int32_t dz = target->z - start->z;

    int32_t height = Room_GetHeight(floor, target->x, target->y, target->z);
    if (target->y > height && start->y < height) {
        target->y = height;
        target->x = start->x + dx * (height - start->y) / dy;
        target->z = start->z + dz * (height - start->y) / dy;
        return false;
    }

    int32_t ceiling = Room_GetCeiling(floor, target->x, target->y, target->z);
    if (target->y < ceiling && start->y > ceiling) {
        target->y = ceiling;
        target->x = start->x + dx * (ceiling - start->y) / dy;
        target->z = start->z + dz * (ceiling - start->y) / dy;
        return false;
    }

    return true;
}

bool LOS_Check(const GAME_VECTOR *const start, GAME_VECTOR *const target)
{
    int32_t los1;
    int32_t los2;

    if (ABS(target->z - start->z) > ABS(target->x - start->x)) {
        los1 = LOS_CheckX(start, target);
        los2 = LOS_CheckZ(start, target);
    } else {
        los1 = LOS_CheckZ(start, target);
        los2 = LOS_CheckX(start, target);
    }

    if (!los2) {
        return false;
    }

    FLOOR_INFO *floor =
        Room_GetFloor(target->x, target->y, target->z, &target->room_number);

    return LOS_ClipTarget(start, target, floor) && los1 == 1 && los2 == 1;
}
