/*  SpiralSynth
 *  Copyleft (C) 2000 David Griffiths <dave@pawfal.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include "Synth.h"

#include <FL/Fl.H>
#include <FL/Enumerations.H>
#include <fstream>
#include <strstream>
#include <string>
#include <time.h>
#include <stdlib.h>

#include "SpiralSound/Oscillator.h"
#include "SpiralSound/Output.h"
#include "SpiralSound/Midi.h"

//#define PLUGIN

// midi controlled cutoff is set to effect 1
static const int MIDI_CUTOFF_CONTROL=91;

Synth::Synth():
m_Oct(5),
m_Osc1GUI(&m_Osc1),
m_Osc2GUI(&m_Osc2),
m_Osc3GUI(&m_Osc3),
m_Env1GUI(&m_Env1),
m_Env2GUI(&m_Env2),
m_Env3GUI(&m_Env3),
m_FilterGUI(&m_Filter),
m_MixerGUI(&m_Mixer),
m_Mixer2GUI(&m_Mixer2),
m_DelayGUI(&m_Delay),
m_FilterEnvGUI(&m_FilterEnv),
m_LFOGUI(&m_LFO),
m_OutGUI(OSSOutput::Get())
{ 		
	m_Databuf.Allocate(SpiralInfo::BUFSIZE);
	m_Databuf2.Allocate(SpiralInfo::BUFSIZE);
	m_Databuf3.Allocate(SpiralInfo::BUFSIZE);
	m_OscControlbuf1.Allocate(SpiralInfo::BUFSIZE);
	m_OscControlbuf2.Allocate(SpiralInfo::BUFSIZE);
	m_OscControlbuf3.Allocate(SpiralInfo::BUFSIZE);
	m_EnvControlbuf.Allocate(SpiralInfo::BUFSIZE);
    m_LFOControlbuf.Allocate(SpiralInfo::BUFSIZE);
	m_BothControlbuf.Allocate(SpiralInfo::BUFSIZE);

	m_XModbuf1 = new Sample[SpiralInfo::POLY];
	m_XModbuf2 = new Sample[SpiralInfo::POLY];

	m_VoiceNote =   new int[SpiralInfo::POLY];
	for (int n=0; n<SpiralInfo::POLY; n++)
	{
		m_VoiceNote[n] = -1;
		m_XModbuf1[n].Allocate(SpiralInfo::BUFSIZE);
		m_XModbuf2[n].Allocate(SpiralInfo::BUFSIZE);
	}	

	m_KeyVoice = new char[SpiralInfo::POLY];

	for (int n=0; n<SpiralInfo::POLY; n++)
	{
		m_KeyVoice[n]=' ';
	}

	LoadPatch(1);
	
	MidiDevice::Get()->SetDeviceName(SpiralInfo::MIDIFILE);
}

Synth::~Synth()
{ 	
}

Fl_Window *Synth::CreateWindow()
{
 Fl_Window* w;
  { Fl_Window* o = Window = new Fl_Window(754, 369, "SpiralSynth 2.0.0");
    w = o;
	o->color(SpiralInfo::GUIBG_COLOUR);
 
	// put part guis here
	m_Osc1GUI.CreateGUI(0,0,"Osc1");
	m_Osc2GUI.CreateGUI(0,110,"Osc2");
	m_Osc3GUI.CreateGUI(0,220,"Osc3");
	m_Env1GUI.CreateGUI(245,0,"Envelope1");
	m_Env2GUI.CreateGUI(245,110,"Envelope2");
	m_Env3GUI.CreateGUI(245,220,"Envelope3");
	m_FilterGUI.CreateGUI(465,0,"Filter");
	m_MixerGUI.CreateGUI(365,0,"Mixer 1&2");
	m_Mixer2GUI.CreateGUI(365,110,"Mixer [1,2]&3");
	m_DelayGUI.CreateGUI(585,0,"Delay");
	m_FilterEnvGUI.CreateGUI(465,110,"Extra Envelope");
	m_LFOGUI.CreateGUI(585,110,"LFO");
	m_OutGUI.CreateGUI(670,0,"Output");
	m_PatchBank.CreateGUI(0,330,"Patch Bank");
	m_LFORouteGUI.CreateGUI(585,275,"LFO Route");
	m_EnvRouteGUI.CreateGUI(585,220,"Envelope Route");
		
	m_Scope.CreateGUI(365,220,"Scope");
  }  
    	
 return w;
}

void Synth::Trigger(int note, float vol=0.001f)
{		
	// Move to the next voice
	m_CurrentVoice++;
	
	if (m_CurrentVoice>=SpiralInfo::POLY)
	{
		m_CurrentVoice=0;
	}
		
	// if it's not free
	if (m_VoiceNote[m_CurrentVoice]!=-1)
	{
		// look for a free voice
		for (int n=0; n<SpiralInfo::POLY; n++)
		{
			if (m_VoiceNote[n]==-1)
			{
				m_CurrentVoice=n;
			} 
		}
	}
			
	m_VoiceNote[m_CurrentVoice]=note;
	
	m_Osc1.NoteTrigger(m_CurrentVoice,note,vol); m_Env1.Trigger(m_CurrentVoice);
	m_Osc2.NoteTrigger(m_CurrentVoice,note,vol); m_Env2.Trigger(m_CurrentVoice);
	m_Osc3.NoteTrigger(m_CurrentVoice,note,vol); m_Env3.Trigger(m_CurrentVoice);
	m_FilterEnv.Trigger(m_CurrentVoice);
}

void Synth::UnTrigger(int note=-1)
{
	// clear all voices
	if (note==-1)
	{
		for (int n=0; n<SpiralInfo::POLY; n++)
		{
			m_Env1.UnTrigger(n);
			m_Env2.UnTrigger(n);
			m_Env3.UnTrigger(n);
			m_FilterEnv.UnTrigger(n);
			m_VoiceNote[n]=-1;
		}
	}
	else
	{	
		// look for the voice and clear it
		for (int n=0; n<SpiralInfo::POLY; n++)
		{
			if (m_VoiceNote[n]==note)
			{
				m_Env1.UnTrigger(n);
				m_Env2.UnTrigger(n);
				m_Env3.UnTrigger(n);
				m_FilterEnv.UnTrigger(n);
				m_VoiceNote[n]=-1;
			}
		}
	}

}

void Synth::PitchBend(int amount)
{	
	m_Osc1GUI.MidiPitchBend(amount);
	m_Osc2GUI.MidiPitchBend(amount);
	m_Osc3GUI.MidiPitchBend(amount);
}

void Synth::ChangeParameter(int CC, int amount)
{	
	if (CC==MIDI_CUTOFF_CONTROL)
	{
		m_FilterGUI.MidiCutoff(amount);
	}
}

void Synth::DoIdle()
{
	m_CurrentEvent=MidiDevice::Get()->GetEvent(0);
	
	int midicount=0;
	// get all the midi events since the last check
	while(m_CurrentEvent.GetType()!=MidiEvent::NONE)
	{
		if (m_CurrentEvent.GetType()==MidiEvent::ON)
		{
			Trigger(m_CurrentEvent.GetNote(),
					m_CurrentEvent.GetVolume()); 		
		}
			
		if (m_CurrentEvent.GetType()==MidiEvent::OFF)
		{
			UnTrigger(m_CurrentEvent.GetNote());
		}
		
		if (m_CurrentEvent.GetType()==MidiEvent::PITCHBEND)
		{
			PitchBend(m_CurrentEvent.GetVolume()); 		
		}
		
		if (m_CurrentEvent.GetType()==MidiEvent::PARAMETER)
		{
			ChangeParameter(m_CurrentEvent.GetNote(),
							m_CurrentEvent.GetVolume()); 		
		}
		
		m_CurrentEvent=MidiDevice::Get()->GetEvent(0);
		midicount++;
	}

	float f;
	m_LFO.Execute(0,m_LFOControlbuf);
	
	// run the synth!
	for (int Voice=0; Voice<SpiralInfo::POLY; Voice++)
	{
		// Xmod routing
		if (m_Mixer.GetType()==Mixer::XMOD)
		{
			m_Osc2.ModulateFreq(&m_XModbuf1[Voice]);		
		}
			
		if (m_Mixer2.GetType()==Mixer::XMOD)
		{
			m_Osc3.ModulateFreq(&m_XModbuf2[Voice]);		
		}		

		m_FilterEnv.Execute(Voice, m_EnvControlbuf);
	
		for (int n=0; n<SpiralInfo::BUFSIZE; n++)
			m_BothControlbuf[n] = m_LFOControlbuf[n]+m_EnvControlbuf[n];
	
		m_Osc1.Execute(Voice, m_Databuf);										
		m_Osc2.Execute(Voice, m_Databuf2);
		m_Osc3.Execute(Voice, m_Databuf3);					
		m_Env1.Execute(Voice, m_OscControlbuf1);
		m_Env2.Execute(Voice, m_OscControlbuf2);
		m_Env3.Execute(Voice, m_OscControlbuf3);
		m_OscAmp1.Execute(m_OscControlbuf1,m_Databuf);
		m_OscAmp2.Execute(m_OscControlbuf2,m_Databuf2);
		m_OscAmp3.Execute(m_OscControlbuf3,m_Databuf3);

		// store the buffer here for feeding back into the oscillator
		if (m_Mixer.GetType()==Mixer::XMOD)
		{
			for (int n=0; n<SpiralInfo::BUFSIZE; n++)
				m_XModbuf1[Voice].Set(n,m_Databuf[n]);
		}
		
		m_Mixer.Execute(m_Databuf2,m_Databuf);
		
		// store the buffer here for feeding back into the oscillator
		if (m_Mixer2.GetType()==Mixer::XMOD)
		{
			for (int n=0; n<SpiralInfo::BUFSIZE; n++)
				m_XModbuf2[Voice].Set(n,m_Databuf2[n]);
		}	
		
		m_Mixer2.Execute(m_Databuf3,m_Databuf);			
		
		if (m_Env1.IsOperating(Voice)|| 
			m_Env2.IsOperating(Voice)|| 
			m_Env3.IsOperating(Voice))
		{
			m_Filter.Execute(Voice,m_Databuf);
			OSSOutput::Get()->SendStereo(&m_Databuf,&m_Databuf);
		}		
	}
		
	m_Delay.GetOutput(OSSOutput::Get()->GetBuffer());
	
	m_Scope.Display(OSSOutput::Get()->GetBuffer());
	
	OSSOutput::Get()->Play();
	
	m_CurrentPatch=m_PatchBank.GetOutput();	
	
	if (m_CurrentPatch>=0) 
	{
		m_Filter.Reset();
		if (m_PatchBank.IsSaving()) 
		{
			SavePatch(m_CurrentPatch);
		}
		else
		{
			LoadPatch(m_CurrentPatch);
			UpdateValues();
		}
	}
	
	if (m_CurrentPatch==-2) 
	{
		Randomise();
	}
	
	Route();
	
	return;// OSSOutput::Get()->GetBuffer();
}

void Synth::SavePatch(int n)
{
	char buf[1024];
	ostrstream os(buf,1024,ios::out);
	os<<*this<<"| ";
	WritePatch(n,buf);	
}

void Synth::LoadPatch(int n)
{
	char buf[1024];
	if (ReadPatch(n,buf))
	{
		istrstream is(buf);
		is>>*this;
	}
}


int Synth::ReadPatch(int n, char *out)
{
	string Patchfn(SpiralInfo::Get()->GetHomeDir()+"/.SpiralPatches.bank");
	ifstream is(Patchfn.c_str());
	
	if (!is.good()) // make a patch file
	{
		cerr<<"SpiralPatches.bank not found - writing new patch file."<<endl;
		
		ofstream os(Patchfn.c_str());
		for (int n=0; n<NUM_PATCHES; n++)
			os<<*this<<"| ";
		
		return 0;
	}
	
	char c=0;
	int count=0;
	
	while(is.good() && count!=n)
	{
		is.get(c);
		//cerr<<"c="<<c<<" count="<<count<<endl;
		if (c=='|') count++;
	}
	
	if(is.good()) 
	{
		is.get(out,1024);
	}
	
	return 1;
}

void Synth::WritePatch(int n, char *in)
{
	// todo: get rid of this cheesy implementation
	string Patchfn(SpiralInfo::Get()->GetHomeDir()+"/.SpiralPatches.bank");
	string Tempfn(SpiralInfo::Get()->GetHomeDir()+"/temp");
	ifstream is(Patchfn.c_str());
	ofstream os(Tempfn.c_str());
	
	char c=0;
	int count=0;
	int count2=0;
	bool IgnorePatch=false;
	
	while(is.good())
	{
		is.get(c);		
		
		while (count==n) // this is the patch to overwrite
		{
			c=in[count2++];
			
			if (c=='|') // end of patch
			{
				IgnorePatch=true;
				count++;
				os<<c<<" ";
				is.get(c);
			}
			else os<<c;
		}
		
		if (IgnorePatch) // step through until end
		{
		
			if (c=='|') 
			{
				IgnorePatch=false;
				count++;
			}
		}
		else // normal write mode
		{
			if (c=='|') 
			{
				count++;
			}
			
			os<<c;
		}
		
	}
	
	string command("mv -f "+Tempfn+" "+Patchfn);
	system(command.c_str());
}

void Synth::Route()
{
	m_Osc1.ModulateFreq(NULL);
	m_Osc1.ModulatePulseWidth(NULL);
	m_Osc2.ModulateFreq(NULL);
	m_Osc2.ModulatePulseWidth(NULL);
	m_Filter.ModulateCutoff(NULL);
	m_Filter.ModulateResonance(NULL);

	if (m_LFORouteGUI.Query(RouteGUI::OSC1FREQ))
		m_Osc1.ModulateFreq(&m_LFOControlbuf);
		
	if (m_LFORouteGUI.Query(RouteGUI::OSC1PW))
		m_Osc1.ModulatePulseWidth(&m_LFOControlbuf);
	
	if (m_LFORouteGUI.Query(RouteGUI::OSC2FREQ))
		m_Osc2.ModulateFreq(&m_LFOControlbuf);
		
	if (m_LFORouteGUI.Query(RouteGUI::OSC2PW))
		m_Osc2.ModulatePulseWidth(&m_LFOControlbuf);
	
	if (m_LFORouteGUI.Query(RouteGUI::FILTERC))
		m_Filter.ModulateCutoff(&m_LFOControlbuf);
					
	if (m_LFORouteGUI.Query(RouteGUI::FILTERR))		
		m_Filter.ModulateResonance(&m_LFOControlbuf);
		
		
	if (m_EnvRouteGUI.Query(RouteGUI::OSC1FREQ))
		m_Osc1.ModulateFreq(&m_EnvControlbuf);
		
	if (m_EnvRouteGUI.Query(RouteGUI::OSC1PW))
		m_Osc1.ModulatePulseWidth(&m_EnvControlbuf);
	
	if (m_EnvRouteGUI.Query(RouteGUI::OSC2FREQ))
		m_Osc2.ModulateFreq(&m_EnvControlbuf);
		
	if (m_EnvRouteGUI.Query(RouteGUI::OSC2PW))
		m_Osc2.ModulatePulseWidth(&m_EnvControlbuf);
	
	if (m_EnvRouteGUI.Query(RouteGUI::FILTERC))
		m_Filter.ModulateCutoff(&m_EnvControlbuf);
					
	if (m_EnvRouteGUI.Query(RouteGUI::FILTERR))		
		m_Filter.ModulateResonance(&m_EnvControlbuf);
		
		
	if (m_LFORouteGUI.Query(RouteGUI::OSC1FREQ) && 
	    m_EnvRouteGUI.Query(RouteGUI::OSC1FREQ))
		m_Osc1.ModulateFreq(&m_BothControlbuf);
		
	if (m_LFORouteGUI.Query(RouteGUI::OSC1PW) && 
		m_EnvRouteGUI.Query(RouteGUI::OSC1PW))
		m_Osc1.ModulatePulseWidth(&m_BothControlbuf);
	
	if (m_LFORouteGUI.Query(RouteGUI::OSC2FREQ) &&
		m_EnvRouteGUI.Query(RouteGUI::OSC2FREQ))
		m_Osc2.ModulateFreq(&m_BothControlbuf);
		
	if (m_LFORouteGUI.Query(RouteGUI::OSC2PW) &&
		m_EnvRouteGUI.Query(RouteGUI::OSC2PW))
		m_Osc2.ModulatePulseWidth(&m_BothControlbuf);
	
	if (m_LFORouteGUI.Query(RouteGUI::FILTERC) &&
		m_EnvRouteGUI.Query(RouteGUI::FILTERC))
		m_Filter.ModulateCutoff(&m_BothControlbuf);
					
	if (m_LFORouteGUI.Query(RouteGUI::FILTERR) &&
		m_EnvRouteGUI.Query(RouteGUI::FILTERR))		
		m_Filter.ModulateResonance(&m_BothControlbuf);
}

void Synth::UpdateValues()
{
	m_Osc1GUI.UpdateValues();
	m_Osc2GUI.UpdateValues();
	m_Osc3GUI.UpdateValues();
	m_Env1GUI.UpdateValues();
	m_Env2GUI.UpdateValues();
	m_Env3GUI.UpdateValues();
	m_FilterGUI.UpdateValues();
	m_MixerGUI.UpdateValues();
	m_Mixer2GUI.UpdateValues();
	m_DelayGUI.UpdateValues();
	m_FilterEnvGUI.UpdateValues();
	m_LFOGUI.UpdateValues();
	m_LFORouteGUI.UpdateValues();
	m_EnvRouteGUI.UpdateValues();
}

void Synth::Randomise()
{
	//m_Osc1.Randomise();
	//m_Osc2.Randomise();
	//m_Osc3.Randomise();
	m_Env1.Randomise();
	m_Env2.Randomise();
	m_Env3.Randomise();
	m_Filter.Randomise();
	m_Mixer.Randomise();
	m_Mixer2.Randomise();
	m_FilterEnv.Randomise();
	//m_LFO.Randomise();
	m_LFORouteGUI.Randomise();
	m_EnvRouteGUI.Randomise();
	
	UpdateValues();
}

istream &operator>>(istream &s, Synth &o)
{	
	char t;
	s>>o.m_Osc1>>o.m_Osc2>>o.m_Osc3>>o.m_Env1>>o.m_Env2>>
	o.m_Env3>>o.m_Filter>>o.m_Mixer>>o.m_Mixer2>>o.m_Delay>>
	o.m_FilterEnv>>o.m_LFO>>o.m_LFORouteGUI>>o.m_EnvRouteGUI;
	return s;
}

ostream &operator<<(ostream &s, Synth &o)
{
	s<<" "<<o.m_Osc1<<o.m_Osc2<<o.m_Osc3<<o.m_Env1
	<<o.m_Env2<<o.m_Env3<<o.m_Filter<<o.m_Mixer
	<<o.m_Mixer2<<o.m_Delay<<o.m_FilterEnv<<o.m_LFO
	<<o.m_LFORouteGUI<<o.m_EnvRouteGUI<<endl;	
	return s;
}

#ifdef PLUGIN

Synth *synth;

void Create()
{
	cerr<<"Loading SpiralSynth plugin"<<endl;
	
	srand(time(NULL));
	SpiralInfo::Get()->LoadPrefs();

	synth = new Synth;

	Fl::visual(FL_DOUBLE|FL_RGB);
	synth->CreateWindow();
    synth->Window->show();
}

void Run()
{
	if (!synth)
	{
		cerr<<"Error in SpiralSynth plugin - not created!"<<endl;
		return 0;
	}
	
    Fl::check();
	synth->DoIdle();    	
}

void Destroy()
{
	if (!synth)
	{
		cerr<<"Error in SpiralSynth plugin - not created!"<<endl;
		return;
	}
	synth->Window->hide();
	delete synth->Window;
	delete synth;	
}

#else

int main(int argc, char **argv)
{	
	srand(time(NULL));
	SpiralInfo::Get()->LoadPrefs();
	
	Synth *synth=new Synth;
	
	Fl::visual(FL_DOUBLE|FL_RGB);
	synth->CreateWindow();
    synth->Window->show(argc, argv);
			
    for (;;) 
	{
    	if (!Fl::check()) break;
		synth->DoIdle();    	
  	}
	
	delete synth;	
	
	return 1;
}

#endif
