/*  SpiralSynth
 *  Copyleft (C) 2000 David Griffiths <dave@pawfal.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include "Oscillator.h"
#include "Output.h"
#include <stdlib.h>
#include <limits.h>

///////////////////////////////////////////////////////

Oscillator::Oscillator() :
m_Type(SQUARE),
m_Octave(0),
m_FineFreq(1.0f),
m_PulseWidth(0.5f),
m_SHLen(0),
m_ModAmount(1.0f),
m_Noisev(0),
m_FreqModBuf(NULL),
m_PulseWidthModBuf(NULL),
m_SHModBuf(NULL)
{
	m_CyclePos=new int[SpiralInfo::POLY];
	m_Note=new int[SpiralInfo::POLY];
	m_LastFreq=new float[SpiralInfo::POLY];
	m_Volume=new float[SpiralInfo::POLY];

	
	for (int n=0; n<SpiralInfo::POLY; n++)
	{
		m_CyclePos[n]=0;			
		m_Note[n]=0;
		m_LastFreq[n]=0;
		m_Volume[n]=1.0f;
	}
}

Oscillator::~Oscillator()
{
}

void Oscillator::NoteTrigger(int V, int s,float v) 
{
	m_Note[V]=s;
	m_Volume[V]=v;
}

void Oscillator::Execute(int V, Sample &data)
{	
	float Freq=0;
	float CycleLen=0;
	int samplelen, PW;
		
	switch (m_Type)
	{
	case SQUARE:
		for (int n=0; n<SpiralInfo::BUFSIZE; n++)
		{
			if (m_FreqModBuf) Freq=(NoteTable[m_Note[V]]*m_FineFreq)+((*m_FreqModBuf)[n]*m_ModAmount);					
			else Freq=NoteTable[m_Note[V]]*m_FineFreq;
								
			if (m_Octave>0) Freq*=1<<(m_Octave);
			if (m_Octave<0) Freq/=1<<(-m_Octave);
			CycleLen = SpiralInfo::SAMPLERATE/Freq;
			if (m_PulseWidthModBuf) PW = (int)(m_PulseWidth+fabs((*m_PulseWidthModBuf)[n])*m_ModAmount*CycleLen);
			else PW = (int)(m_PulseWidth * CycleLen);
	
			// calculate square wave pattern
			m_CyclePos[V]++;
			if (m_CyclePos[V]>CycleLen) m_CyclePos[V]=0;	
				
			if (m_CyclePos[V]<PW) data.Set(n,m_Volume[V]);
			else data.Set(n,-m_Volume[V]);						
		}
		break;
			
	case SAW:
		for (int n=0; n<SpiralInfo::BUFSIZE; n++)
		{
			if (m_FreqModBuf) Freq=(NoteTable[m_Note[V]]*m_FineFreq)+((*m_FreqModBuf)[n]*m_ModAmount);					
			else Freq=NoteTable[m_Note[V]]*m_FineFreq;

			if (m_Octave>0) Freq*=1<<(m_Octave);
			if (m_Octave<0) Freq/=1<<(-m_Octave);			
			CycleLen = SpiralInfo::SAMPLERATE/Freq;
			if (m_PulseWidthModBuf) PW = (int)(m_PulseWidth+fabs((*m_PulseWidthModBuf)[n])*m_ModAmount*CycleLen);
			else PW = (int)(m_PulseWidth * CycleLen);

			// get normailise position between cycle 			
			m_CyclePos[V]++;
			if (m_CyclePos[V]>CycleLen) m_CyclePos[V]=0;		
						
			if (m_CyclePos[V]<PW) 
			{
				// before pw -1->1	
				data.Set(n,Linear(0,PW,m_CyclePos[V],-m_Volume[V],m_Volume[V]));
			}
			else 
			{
				// after pw 1->-1
				data.Set(n,Linear(PW,CycleLen,m_CyclePos[V],m_Volume[V],-m_Volume[V]));
			}		
		}					
		break;
		
	case NOISE:
		for (int n=0; n<SpiralInfo::BUFSIZE; n++)
		{
			m_CyclePos[V]++;

			//modulate the sample & hold length
			samplelen = (int)(m_SHLen*SpiralInfo::SAMPLERATE);
		
			// do sample & hold on the noise
			if (m_CyclePos[V]>samplelen)
			{
				m_Noisev=(short)((rand()%SHRT_MAX*2)-SHRT_MAX);
				m_CyclePos[V]=0;
			}
			data.Set(n,m_Noisev/(float)SHRT_MAX*m_Volume[V]);
		}
		break;
		
	case NONE: break;
	}
}

void Oscillator::StreamOut(ostream &s)
{
	s<<1<<" "<<*this;
}

void Oscillator::StreamIn(istream &s)
{
	int version;
	s>>version>>*this;
}

istream &operator>>(istream &s, Oscillator &o)
{
	float dummy=0;
	s>>(int&)o.m_Type>>o.m_Octave>>o.m_FineFreq>>o.m_PulseWidth>>dummy>>
	o.m_SHLen>>o.m_ModAmount;
	return s;
}

ostream &operator<<(ostream &s, Oscillator &o)
{
	float dummy=0;
	s<<(int)o.m_Type<<" "<<o.m_Octave<<" "<<o.m_FineFreq<<" "<<o.m_PulseWidth<<" "<<
	dummy<<" "<<o.m_SHLen<<" "<<o.m_ModAmount<<" ";
	return s;
}

