/*
    AGMessage.cpp
    OpenAG, libOpenAG, OpenAG X
 
    Created by Eric Seidel on Fri Nov 02 2001.
 
    Copyright (c) 2001-2002 Eric Seidel. All rights reserved.
 
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "common.h"

#include <assert.h>
#include <signal.h>
#include <errno.h>
#include <arpa/inet.h>
#include "error.h"

#include "AGMessage.h"
#include "AGMain.h"
#include "socket.h"

int NextMessageID = 0;

/** INCOMING PACKET CONSTRUCTOR **/
AGMessage::AGMessage(char* ReadBuffer, unsigned int BufferLength, bool isServerMessage)
{	
    #if AGDEBUGLEVEL > 2
    printf("\nINITIALIZING INCOMING PACKET\n");
    #endif
    SendPacket = false;

    MessageStatus = AG_MESSAGE_INCOMPLETE;

    if (BufferLength < 12) // atleast 12
    {
        printf("not enough data in buffer to parse message, returning\n");
        DataBuffer = NULL; // so we know not to delete it later.
        return;
    }
    
    //temporarly set the buffer to be our given one:
    //temporary, to fool the read operations.
    BufferSize = BufferLength;
    DataBuffer = ReadBuffer;
    ReadPointer = DataBuffer;
    WritePointer = DataBuffer;
    
    #if AGDEBUGLEVEL > 6
    printf("About to read message packet header data\n");
    #endif
    StartSignal = read_int();
    
    #if AGDEBUGLEVEL > 6
    printf("returned from starsignal read\n");
    printf("Read StartSignal: %i\n", StartSignal);
    #endif
    if (StartSignal != CLIENT_SERVER_STARTSIGNAL)
    {
        dumpHex();
        err_print("Bad start signal: %i on packet.  Setting MessageStatus = BAD_START_SIGNAL, connection will be closed.\n",StartSignal);
        MessageStatus = AG_MESSAGE_BAD_STARTSIGNAL;
    }
    #if AGDEBUGLEVEL > 4
    printf("about to read in size and ID\n");
    #endif
    MessageSize = read_int();
    MessageID   = read_int();
    #if AGDEBUGLEVEL > 4
    printf("StartSignal: %i\n", StartSignal);
    printf("MessageSize: %i\n", MessageSize);
    printf("MessageID: %i\n", MessageID);
    #endif
    if (isServerMessage)
    {
        MessageType = read_short();
        #if AGDEBUGLEVEL > 2
        printf("MessageType: %i\n", MessageType);
        #endif
    }
    else
    {
        MessageType = -1;
        #if AGDEBUGLEVEL > 2
        printf("NO TYPE, peer-to-peer message\n");
        #endif
    }
    
    #if AGDEBUGLEVEL > 5
    printf("About to copy databuffer\n");
    #endif
    
    BufferSize = MessageSize + 4; // set it to the REAL value of the entire message;
    if (BufferLength < BufferSize)
    {
        #if AGDEBUGLEVEL > 5
        printf("Message is not complete requested: %i, only have %i, copying what we have...\n", BufferSize, BufferLength);
        #endif
        MessageStatus = AG_MESSAGE_INCOMPLETE; /*IMPORTANT HOW CAN WE FORCE THEM TO CHECK THIS? */
        BufferSize = BufferLength; // this way we don't get sementation faults...
    }
    else MessageStatus = AG_MESSAGE_COMPLETE;
    
    #if AGDEBUGLEVEL > 5
    printf("allocating buffer of size: %i\n", BufferSize);
    #endif
    // segmentation fault somewhere after here...
    
    DataBuffer = new char[BufferSize]; // allocate space for the copy.

    #if AGDEBUGLEVEL > 5
    printf("about to copy data into buffer at: %p of size: %i from: %p\n", 
	DataBuffer, BufferSize,ReadBuffer);
    #endif

    memcpy(DataBuffer, ReadBuffer, BufferSize);  //copy the whole message.
    
    /* set the write pointer to the full message length */
    WritePointer += BufferSize;
    
    #if AGDEBUGLEVEL > 5
    printf("done copying in data.\n");
    #endif
    
    // set ReadPointer at start of unread data.
    int HeaderSize = 4 + 4 + 4; // Signal, Size, ID
    if (isServerMessage) HeaderSize += 2; //for type
    ReadPointer = (DataBuffer + HeaderSize);
    // ReadBuffer will be deallocated by the calling function
    #if AGDEBUGLEVEL > 4
    // getting malloc error, (modification of freed object) sometime after here...
    printf("hexdump:\n");
    dumpHex();
    #endif
}

/** OUTGOING PACKET CONSTRUCTOR **/
AGMessage::AGMessage(int s)
{
    //printf("INITIALIZING OUTGOING PACKET\n");
    TheSocket 		= s;
    SendPacket		= true;
    
    BufferSize		= 200;
    DataBuffer		= new char[BufferSize];
    WritePointer	= DataBuffer; //set the write pointer to the beginning.
    
    bzero(DataBuffer, BufferSize); // makes cleaner hexDump() calls.
    
    StartSignal		= CLIENT_SERVER_STARTSIGNAL;
    MessageSize		= 4 + 4;  //self + MessageID
    MessageID		= NextMessageID++;
    MessageType		= -1;  //default... for debugging..
    //write them later.
    #if AGDEBUGLEVEL > 5
    printf("successfully initialized outgoing packet.\n");
    #endif
}

/** DESTRUCTOR **/
AGMessage::~AGMessage()
{
    #if AGDEBUGLEVEL > 5
    printf("AGMessage destructor called, deleting data at: %p\n", DataBuffer);
    #endif
    if (DataBuffer != NULL) // only if somehow we didn't succeed initialization.
        delete(DataBuffer);
    //do I need to delete the other two pointers?
}

/** IS THIS A SEND PACKET? **/
bool AGMessage::isSendPacket()
{
return SendPacket;
}

int AGMessage::getCompleteMessageSize()
{
    if (!(MessageStatus == AG_MESSAGE_COMPLETE) && !SendPacket)
        return (BufferSize); // return the whole thing.
    else
        return (MessageSize + 4); // more accurate than BufferSize.
}

int AGMessage::getCompleteMessageData(char* &deleteMe)  //correct?
{
    #if AGDEBUGLEVEL > 4
    printf("getting complete message data\n");
    #endif
    if (!SendPacket)
    {
        memcpy(deleteMe, DataBuffer, getCompleteMessageSize());
        return (getCompleteMessageSize());
    }
    else
    {
        int gCMSlength = getCompleteMessageSize();
        int ourTotalLength = 4+4+4+((getType()>-1)?2:0)+getDataLength();
        
        if (gCMSlength != ourTotalLength)
            printf("gCMS:%i  ourTotal:%i  -- THAT WILL CAUSE BAD MEMORY ERRORS.\n",
                    gCMSlength, ourTotalLength);
        
        char* writepos = deleteMe;
        int temp_int;
        short int temp_short;
        
        /*Some compilers don't like &(htonl(x))*/
        temp_int = htonl(StartSignal);
        memcpy(writepos, &temp_int, 4); writepos += 4;
        temp_int = htonl(MessageSize);
        memcpy(writepos, &temp_int, 4); writepos += 4;
        temp_int = htonl(MessageID);
        memcpy(writepos, &temp_int, 4); writepos += 4;
        //check to make sure it's got a type.
        
        temp_short = htons(MessageType);
        if (getType() >= 0) {memcpy(writepos, &temp_short, 2); writepos +=2;}
        
        #if AGDEBUGLEVEL > 4
        printf("about to copy over the databuffer of \"size\": %i\n", getDataLength());
        #endif
        memcpy(writepos, DataBuffer, getDataLength()); writepos += getDataLength();
        
        // could check ourTotalLength agains writepos - deleteMe
        
        return (writepos - deleteMe);
    }
}

int AGMessage::getDataLength()
{
    if (isSendPacket())
        return (WritePointer - DataBuffer);
    else
    {
        err_exit("This call should never be made for incoming packets.\n");
    }
    return -1; // SHOULD NEVER REACH!!!
}

/****** FOR INCOMING PACKETS **********/

/* FIX -- THERE HAS GOT TO BE A CLEANER WAY TO DO THESE!!! */

unsigned int AGMessage::read_int()
{
    char* temp = new char[4];
    read_string(temp, (unsigned short)4);
    unsigned int temp2 = *((unsigned int*)temp);
    #if AGDEBUGLEVEL > 5
    printf("read int: %i\n", temp2);
    #endif
    delete temp;
    return ntohl(temp2);
}

unsigned short AGMessage::read_short()
{
    char* temp = new char[2];
    read_string(temp, (unsigned short)2);
    unsigned short temp2 = *((unsigned short*)temp);
    #if AGDEBUGLEVEL > 5
    printf("read short: %i\n", temp2);
    #endif
    delete temp;
    return ntohs(temp2);
}

unsigned char AGMessage::read_byte()
{
    unsigned char* temp = new unsigned char[1];
    read_string((char*)temp, (unsigned short)1);
    unsigned char temp2 = *(temp);
    #if AGDEBUGLEVEL > 5
    printf("read byte: %i\n", temp2);
    #endif
    delete temp;
    return temp2;
}

int AGMessage::read_string(char* &MyBuff, bool terminate)
{
    /* We re-allocate for them, but they are responsible for deleting!!! */
    if (MyBuff != NULL && MyBuff != 0 ) delete(MyBuff);
    unsigned short size = read_short();
    /* FIX, problem if string is of length 0 and not termed... */
    if (terminate)
    {
        MyBuff = new char[size+1]; // for the \0
        memset(MyBuff, '\0', size + 1); // will terminate.
    }
    else
    {
        if (size == 0){ MyBuff = NULL; return 0;}
        MyBuff = new char[size];
        memset(MyBuff, '\0', size); // will terminate the rest whether it wants it or not.
    }
    
    #if AGDEBGULEVEL >4
    printf("auto-reading string of size: %i\n", size);
    #endif
    
    read_string(MyBuff, size);
    
    /* if it's already terminated... we are making a mistake here *** FIX */
    if (terminate)
    {
        MyBuff[size] = '\0'; // terminate the char*
        size++;
    }
    
    //unsigned char* newBuff = NULL;
    
    // Convert the string to our local format (for printing and file system operations)
    //size = string_serverToLocal(newBuff, (unsigned char*)MyBuff, size);
    //delete(MyBuff);
    //MyBuff = (char*)newBuff;  // FIX -- shoudl this be a char, or unsigned char?  does it matter?
    
    return size;
}


/* char* should be atleast char[l] big, or NULL*/
void AGMessage::read_string(char* &MyBuff, unsigned short l)
{
    /* FIX this seems to be a dumb way to do all this */
    
    if ( (MyBuff == NULL) || (MyBuff == 0) )
        MyBuff = new char[l];
    /* FIX, perhaps?? -- we're assuming they gave us a pointer of the right size!! */
    
    if( (BufferSize - (ReadPointer - DataBuffer)) < (unsigned short)l )
    {
        printf("ouch.  Can't read any more of the message.\n");
        printf("trying to read %i chars, only have %i left.\n\n",
                l , BufferSize - (ReadPointer - DataBuffer));
        err_exit("Sorry, ran off the end of the buffer when trying to read, can't continue.\nThis error means that when trying to read a \"message\" either from the server, or a file, I expected to be able to read more than was actually there.\n this error will be handled better in future revisions, but for now, just re-login.\n ");
    }
    #if AGDEBUGLEVEL > 5
    else {printf("begining read of length %i\n", l);}
    #endif
    //need to memset?
    memset(MyBuff,'\0', l); //why not. :-)  this will also terminate any short copies
    
    #if AGDEBUGLEVEL > 5
    printf("copying data\n");
    #endif
    memcpy(MyBuff, ReadPointer, l); //do the copy
    ReadPointer += l;
    /* BUFFER WILL BE DEALLOCATED BY THE CALLER! */
}

int AGMessage::getMessageStatus()
{
    if (SendPacket)
        printf("called getMessageStatus for a send packet... makes no sense!\n");
    
    #if AGDEBUGLEVEL > 4
    if (MessageStatus == AG_MESSAGE_COMPLETE)
    {
        //printf("message is complete\n");
    }
    else
    {
        printf("****message is NOT complete*****\n");
    }
    #endif
    return MessageStatus;
}

int AGMessage::getType()
{
    //not for really for sending packets...  except for debug info.
    return MessageType;
}


/****** FOR OUTGOING PACKETS **********/

void AGMessage::write_int(unsigned int data)
{
    int temp_int = htonl(data);
    write_string((char*)(&temp_int) , (unsigned short)4, false);
}
void AGMessage::write_short(unsigned short data)
{
    short int temp_short = htons(data);
    write_string((char*)(&temp_short) , (unsigned short)2, false);
}

void AGMessage::write_byte(unsigned char data)
{
    write_string((char*)(&data) , (unsigned short)1, false); //plus one needed???
}

void AGMessage::write_string(char* data, unsigned short l, bool includelength)
{
    /* FIX -- Don't allow ourselves to get here!!! handle this before here */
    if (l > 255)
        err_exit("I was asked to write a string of more than 255 characters.  Unfortunatly, this is a bug, and that action would cause bad things to happen. I have instead decided to stop here.  I appologize for the inconvience, this bug will be fixed soon.\n");
    
    #if AGDEBUGLEVEL > 5
    printf("writing string %s to message (with specified length: %i)\n", data, l);
    #endif
    if (includelength) write_short(l);
    //is the buffer big enough??
    if( (BufferSize - (WritePointer - DataBuffer)) < (unsigned short)l ) adjustBuffer();
    
    memcpy(WritePointer, data, l);
    WritePointer += l;
    MessageSize += l;
}

void AGMessage::write_string(char* q)
{

    if ((q == NULL) || (q == 0))
    {
        #if AGDEBUGLEVEL > 4
        printf("writting NULL string, intended?\n");
        #endif
        write_string("", 0, true);
    }
    else
    {
	#if AGDEBUGLEVEL > 5
	printf("writing string %s to message\n", q);
	#endif
        
        short unsigned int size = strlen(q);

        //unsigned char* newBuff = NULL;
    
        // Convert the string to the server format before sending
        //size = string_localToServer(newBuff, (unsigned char*)q, size);
        
        if (size > 255)
            err_print("I was asked to write a string of more than 255 characters.  Somehow the truncating function failed, or was not implemented properly for this platform.  PLEASE REPORT THIS BUG.  Unfortunatly, continuing will most likely get you kicked from the server.\n");
            
        write_string((char*)q, size, true);
        //delete (newBuff);
    }
}

void AGMessage::write_zeros(unsigned short howMany)
{
    char * temp = new char[howMany];
    bzero(temp, howMany);
    write_string(temp, howMany, false);
    delete(temp);
}

void AGMessage::adjustBuffer()
{
    #if AGDEBUGLEVEL > 4
    printf("adjusting buffer size\n");
    #endif
    //lock?
    char * NewBuffer = new char[BufferSize*2]; //allocate new space
    memcpy(NewBuffer, DataBuffer, BufferSize); // copy in data
    //fix pointers.
    ReadPointer = NewBuffer + (ReadPointer - DataBuffer);
    WritePointer = NewBuffer + (WritePointer - DataBuffer);
    //clear old space.
    delete (DataBuffer);
    //set pointer
    DataBuffer = NewBuffer;
    BufferSize *=2;
    //unlock
}

void AGMessage::setType(int type)
{
    if ( (type == -1) && (MessageType != -1))
    {
        printf("Unsetting the type.  BAD. DON'T use this call");
        MessageSize -=2; // unset the type.  DON'T ever use.
    }
    if (MessageType == -1) MessageSize +=2; // set the type.
    MessageType = type;
}

int AGMessage::sendMessage()
{
    #if AGDEBUGLEVEL > 2
    printf("\nSENDING PACKET:\n");
    #endif
    int packetSize = getCompleteMessageSize();
    #if AGDEBUGLEVEL > 3
    printf("Packetsize: %i\n", packetSize);
    #endif
    #if AGDEBUGLEVEL > 2
    printf("MessageType: %i\n", getType());
    #endif
    char* temp = new char[packetSize];
    
    int weReallyWrote = getCompleteMessageData(temp);
    
    if (packetSize < weReallyWrote)
    {
        #if AGDEBUGLEVEL > 4
        err_print("Oops, we just thrashed memory.  We wrote %i bytes to a pointer only allocated for %i.\n",
                weReallyWrote, packetSize);
        #endif
        err_exit("Oops, we just overwrote a section of memory, I'm exiting before I cause more damage.\n");
    }

    #if AGDEBUGLEVEL > 4
    dumpHex();
    #endif
    
    int error;
    socklen_t errorsize = sizeof(error);
    if ((getsockopt(TheSocket, SOL_SOCKET, SO_ERROR, &error, &errorsize) < 0) || (error != 0))
    {
        errno = error; //set errno
        err_print("got error before writing to socket, returning...\n");
        return (-1);
    }
    
    // HACK TO FIX SIGNAL HANDLING
    
    struct sigaction ignoreit;
    struct sigaction theold;
    
    ignoreit.sa_handler		= SIG_IGN;
    sigemptyset(&ignoreit.sa_mask);
    ignoreit.sa_flags		= 0;
    
    if (sigaction(SIGPIPE, &ignoreit, &theold) != 0)
        err_print("error ignoring signal\n"); // needed to ignore the sigpipes I can get here...
    
    int theerr = 0;
    
    if ((theerr = write_data(TheSocket, temp , getCompleteMessageSize())) < 0)
    {
        if (sigaction(SIGPIPE, &theold, NULL) != 0)
            err_print("error restoring signal handler\n");
            
        err_print("got error when attempting to write to socket, returning\n");
        
        return(-1);
    }
    
    if (sigaction(SIGPIPE, &theold, NULL) != 0)
        err_print("error restoring signal handler\n");
    
    #if AGDEBUGLEVEL > 3
    else
        printf("MESSAGE SENT.\n\n");
    #endif
    
    delete(temp);
    return 0;
}

void AGMessage::queueMessage(int i, server_status &STATS)
{
    #if AGDEBUGLEVEL > 2
    printf("\nQUEUING PACKET:\n");
    #endif
    int packetSize = getCompleteMessageSize();
    #if AGDEBUGLEVEL > 3
    printf("packetsize: %i\n", packetSize);
    #endif
    #if AGDEBUGLEVEL > 2
    printf("MessageType: %i\n", getType());
    #endif
    char* temp = new char[packetSize+20]; //FIX, just for safety

    #if AGDEBUGLEVEL > 3
    printf("got temp buffer of size: %i\n", packetSize);
    #endif
    
    int weReallyWrote = getCompleteMessageData(temp);

    #if AGDEBUGLEVEL > 3
    printf("filled temp buffer of size: %i with %i bytes\n", packetSize, weReallyWrote);
    #endif    
    
    if (packetSize < weReallyWrote)
    {
        #if AGDEBUGLEVEL > 4
        printf("Oops, we just thrashed memory.  We wrote %i bytes to a pointer only allocated for %i.\n",
                weReallyWrote, packetSize);
        #endif
        err_exit("Oops, we just overwrote a section of memory, I'm exiting before I cause more damage.\n");
    }

    #if AGDEBUGLEVEL > 4
    printf("this packet: \n");
    dumpHex();
    #endif
    
    
    /* does the buffer have space?*/
    if ( ( STATS.connections[i].Obuf + BUFFER_SIZE) > (STATS.connections[i].OwPtr + packetSize) )
    {
        memcpy(STATS.connections[i].OwPtr, temp, packetSize);
        STATS.connections[i].OwPtr += packetSize;
        #if AGDEBUGLEVEL > 3
        printf("MESSAGE QUEUED on connection: %i.\n\n", i);
        #endif
        FD_SET(STATS.connections[i].socket, &(STATS.wset));
    }
    else
    {
        #if AGDEBUGLEVEL > 1
        printf(" BUFFER TOO FULL on %i , FORCED TO SEND MESSAGE DIRECTLY, THIS COULD BE BAD!\n", i);
        #endif
        if (sendMessage() < 0)
            err_exit("There was an error while attempting to send.\nDying here is inappropriate, but I currently have no other choice.\nThis is a known bug.");
    }
    
    delete(temp);
}


void AGMessage::dumpHex()
{
    printf("data buffer: \n");
    char* start;
    
    #ifdef DUMPWHOLEBUFFER
        start = (char*)DataBuffer
	int wholeMessageSize = BufferSize;
    #else
        #if AGDEBUGLEVEL > 4
        printf("grabing size.\n");
        #endif
        
        int wholeMessageSize = getCompleteMessageSize();
        
        #if AGDEBUGLEVEL > 4
        printf("got size:%i\n", wholeMessageSize);
        #endif
	char* temp = new char[wholeMessageSize];
        #if AGDEBUGLEVEL > 4
        printf("sucessfully allocated char array of size:%i\n", wholeMessageSize);
        #endif
        int weReallyWrote = getCompleteMessageData(temp);
        
        #if AGDEBUGLEVEL > 4
        printf("got complete message data size it returned was:%i\n", weReallyWrote);
        #endif
        
	if (wholeMessageSize < weReallyWrote)
        {
            #if AGDEBUGLEVEL > 4
            printf("Oops, we just thrashed memory.  We wrote %i bytes to a pointer only allocated for %i.\n",
                    weReallyWrote, wholeMessageSize);
            #endif
            err_exit("Oops, we just overwrote a section of memory, I'm exiting before I cause more damage.\n");
        }
        start = temp;
    #endif
    
    printf("start before: %p\n", start);
    
    char* ptr = start;

    #if AGDEBUGLEVEL >4
	printf("wholeMessageSize: %i\n", wholeMessageSize);
    #endif

    char* ptr2 = ptr;

    /* Added support for little endian machines -- D.C. */

    while (ptr2 < (start + wholeMessageSize))
    {
        ptr = ptr2;
        while (ptr < (start + wholeMessageSize))
        {
            printf("%02hhX%02hhX ", *ptr, *(ptr+1)); // DC
            ptr +=2;
            if ((ptr - start) % 16 == 0) break;
        }
        ptr = ptr2;
        printf("    ");
        while (ptr < (start + wholeMessageSize))
        {
            // removed printing of control characters.  -- D.C.

            if(*ptr > 31)
                printf("%c", *ptr);
            else
                printf("_"); // don't print control characters
            if(*(ptr+1) > 31)
                printf("%c ", *(ptr+1));
            else
                printf("_ "); // don't print control characters
            
            ptr +=2;
            if ((ptr - start) % 16 == 0) break;
        }
        printf("\n");
        ptr2 = ptr;
    }

    printf("start after: %p ptr1: %p ptr2: %p temp: %p \n", start, ptr, ptr2, temp);

    #ifndef DUMPWHOLEBUFFER
    delete(temp);
    #endif
    
    #if AGDEBUGLEVEL > 4
    printf("leaving dumphex\n");
    #endif
}
