 /*
    AGMain.cpp
    OpenAG, libOpenAG, OpenAG X
 
    Created by Eric Seidel on Fri Nov 02 2001.
 
    Copyright (c) 2001-2002 Eric Seidel. All rights reserved.
 
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "common.h"

#include <errno.h>
#include <signal.h>
#include <sys/wait.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <assert.h>

#include "ParseMessage.h"

#include "AGMain.h"
#include "AGInit.h"
#include "AGMessage.h"
#include "socket.h"
#include "error.h"


int testcount = 0;

struct timezone dontcare = {0,0}; // FIX -- is this allowed?


bool doMainLoop(server_status &STATS)
{
    int		ready_count = 0;
    
    fd_set	rs = STATS.rset; // initialize.  Will actually be handled in handleBandwidthLimits...
    fd_set	ws = STATS.wset;
    
    /* Bandwidth limit stuff */
    handleBandwidthLimits(STATS, ws, rs);
    
    
    #if AGDEBUGLEVEL > 5
    printf("about to select with timeout: %i.%i\n", STATS.select_timer.tv_sec, STATS.select_timer.tv_usec);
    #endif
    
    /* dying here in Malloc DEBUG */
    
    
    /*  Handle selecting */
    if (STATS.connections[INFOSERVER].Ibuf == STATS.connections[INFOSERVER].IwPtr) // if it's not empty... kind of a hack.
    {
        //printf("ready to select...\n");
        //printf("var check: maxfd: %li  rs: %p  ws: %p  select_timer: %p\n",STATS.maxfd, &rs, &ws, STATS.select_timer);
        
        /* For those systems who destroy the timer pointer sent to them */
        timeval temp_timer = STATS.select_timer; // FIX -- is this really necessary, or is that just an ugly rumor?
        
        ready_count = select(STATS.maxfd + 1, &rs, &ws, NULL, &temp_timer);
        //printf("through select\n");
    }
    else
    {
        printf("MainServerBuffer is not empty, still: %i, not waiting.\n",
            STATS.connections[INFOSERVER].IwPtr - STATS.connections[INFOSERVER].Ibuf);
            // This will cause infinite loops if bad data is in buffer.
        // FIX - Setting ready_count to 1 here is a HACK to get by the timeout code...
        ready_count = 1;
    }
        
    
    #if AGDEBUGLEVEL > 3
    printf("returned from select: %i\n", ready_count);
    #endif

    /*  HANDLE TIME AND KEEP ALIVE */
    handleTimeAndKeepAlive(STATS);
    
    #if AGDEBUGLEVEL > 2
    if (ready_count == 0) // we arrived here because of a timeout.
    {
        testcount++;
        printf("Timeout, should see time check %i\n\n", testcount);
        return true;
    }
    else
        testcount = 0;
    #endif
    
    
    
    #if AGDEBUGLEVEL > 6
    printf("nConnections: %i\n", STATS.nConnections);
    #endif
    
    /* HANDLE ACTIVE CONNECTIONS */
    for (int i = 0; i < STATS.nConnections; i++)
    {
        STATS.current = i;
        #if AGDEBUGLEVEL > 5
        printf("entering for: %i\n", i);
        #endif
        
        /* READ BETWEEN STATS.connections[i].rPtr AND STATS.connections[i].wPtr -- WRITE FROM HERE.*/
        /* WRITE BETWEEN STATS.connections[i].wPtr AND STATS.connections[i].buf + BUFFER_SIZE -- READ INTO HERE */
        
        if (STATS.connections[i].flags == 0) /* connection not in use, ignore */
        {
            #if AGDEBUGLEVEL > 3
            printf("********ignoring connection record %i, not in use**********\n",i);
            #endif
        }
        else
        {
            /* HANDLE INDIVIDUAL CONNECTION TIMEOUTS */
            /* FIX - should I really make this test? */
            if (! (
                (  (STATS.connections[i].socket >= 0)
                && ( FD_ISSET(STATS.connections[i].socket, &rs) // true if readable
                  || FD_ISSET(STATS.connections[i].socket, &ws) ) )// true if writeable
                || (STATS.connections[i].hasOpenFile
                && ( FD_ISSET(STATS.connections[i].file, &rs)
                  || FD_ISSET(STATS.connections[i].file, &ws) ) ) 
                ) )
                // if it gets here, then NOTHING is ready on this connection.
            {
                #if AGDEBUGLEVEL > 4
                printf("none set\n");
                #endif
                time_t new_time = time(NULL);
                /* shoule execute for everyone first time through */
                if ( (new_time - STATS.connections[i].lastTime) > STATS.connections[i].TimeOut)
                {
                    /* Handle Connection Reset/Timeout */
                    #if AGDEBUGLEVEL > 3
                    printf("connection %i, TIMEOUT\n", i);
                    #endif
                    
                    /* send proper timeout message */
                    /* or simply delete the connection record */
                    
                    /* should work... */
                    if (i != 0)
                    {
                        if (!(STATS.connections[i].flags & TIMEOUT))
                        {
                            printf("Connection: %i has timed out.  with flags: %i the Info Server will be informed.\n", 
                                    i, STATS.connections[i].flags);
                            sendFileTransferState(i, STATS, FTS_TIMEOUT);
                            STATS.connections[i].flags &= TIMEOUT;
                            //deleteConnectionRecord(i, STATS);
                        }
                        else
                        {
                            printf("REPEATED TIMOUT on #%i!!\n", i);
                            deleteConnectionRecord(i, STATS);
                        }
                    }
                    
                    /* Mark the connection's last "touch" time */
                    STATS.connections[i].lastTime = time(NULL);
                }
            }
            else /*it will be handled below, so mark it as "touched" -- really should be done below, but this works. */
            {
                STATS.connections[i].lastTime = time(NULL);
                
                /* We could calculate the lowest timeout value and set that as the select timeout value... but no need */
                
                /*
                printf("about to enter big if.\n");
                
                if (FD_ISSET(STATS.connections[i].socket, &ws))
                    printf("connection: %i is STILL RETURNED socket writeable and flags: %i\n",
                            i, STATS.connections[i].flags);
                */
                
                /* HANDLE NEW CONNECTIONS */
                if ( (STATS.connections[i].flags & CONNECTING || STATS.connections[i].flags & ACCEPTING) &&
                    (STATS.connections[i].socket >= 0) &&
                    (FD_ISSET(STATS.connections[i].socket, &rs) || FD_ISSET(STATS.connections[i].socket, &ws)) )
                {                
                    #if AGDEBUGLEVEL > 3
                    printf("NEW CONNECTION READY\n");
                    #endif
                    
                    if (STATS.connections[i].flags & CONNECTING)
                    {
                        if(handleConnectionConnecting(STATS) == -1)
                            continue;
                        
                        #if AGDEBUGLEVEL > 3
                        printf("Delayed Connection Completed, telling server -- We're ready\n");
                        #endif
                        sendFileTransferState(i, STATS, FTS_SETTING_UP);
                    }
                        
                    else if (STATS.connections[i].flags & ACCEPTING)
                    {
                        if(handleConnectionAccepting(STATS) == -1)
                            continue;
                    }
                    
                    handleNewConnection(i, STATS);
                }
                
                /* HANDLE SOCKET READY TO READ FROM */
                else if( (STATS.connections[i].socket >= 0) && FD_ISSET(STATS.connections[i].socket, &rs) )
                {
                    #if AGDEBUGLEVEL > 2
                    printf("handling SOCKET READ on socket: %i of record: %i\n",STATS.connections[i].socket,i);
                    #endif
        
                    if (handleReadFromSocket(STATS) == 0)
                    {
                        int result = 0;
                        
                        /* Handle the DATA read */
                        if (STATS.connections[i].flags & WAIT_REQUEST)
                            result = handleIncomingFileRequest(STATS);
                            
                        else if (STATS.connections[i].flags & WAIT_RESPONSE)
                            result = handleIncomingFileRequestResponse(STATS);
                            
                        else if (STATS.connections[i].flags & TRANSFER_RECEIVING)
                            result = handleIncomingDataFromPeer(STATS);
                            
                        else if (STATS.connections[i].flags & INFOSERVER_FLAG)
                            result = handleIncomingDataFromInfoServer(STATS);
                            
                        else
                            printf("connection: %i ready to read, but didn't fall into a catagory!\n", i);
                            
                        if (result < 0)
                        {
                            //#if AGDEBUGLEVEL > 4
                            err_print("*************ERROR WHEN PROCESSING INCOMMING DATA ON CONNECTION: %i.***********\n", i);
                            //#endif
                            if (i == INFOSERVER)
                                clientInit(STATS);
                            else
                                deleteConnectionRecord(i,STATS);
                           continue;
                        }
                        
                    }
                    else
                        continue; // should get rid of the deleted but didn't skip the rest stuff.
                }
                
                /* HANDLE FILE READY TO READ FROM */
                else if( (STATS.connections[i].flags & TRANSFER_SENDING) &&
                            (STATS.connections[i].hasOpenFile)
                            && FD_ISSET(STATS.connections[i].file, &rs) )
                {
                    #if AGDEBUGLEVEL > 4
                    printf("File ready to read from");
                    #endif
                    if(handleIncomingDataFromFile(STATS) == -1) continue;
                }
                
                /* HANDLE SOCKET READY TO WRITE TO */
                else if( (STATS.connections[i].socket >= 0) && FD_ISSET(STATS.connections[i].socket, &ws) )
                {
                    #if AGDEBUGLEVEL > 2
                    printf("handling SOCKET WRITE on record: %i\n",i);
                    #endif
                    if (STATS.connections[i].flags & SENDING_REQUEST)
                        handleOutgoingFileRequest(STATS); //did it go through?
                    else if (STATS.connections[i].flags & SENDING_RESPONSE)
                        handleOutgoingFileRequestResponse(STATS);
                    
                    /* NOTICE: IF, not ELSE IF */
                    if ( (STATS.connections[i].OwPtr - STATS.connections[i].OrPtr)  > 0)
                    {
                        if (handleOutgoingDataToSocket(STATS) < 0)
                        {
                            //#if AGDEBUGLEVEL > 4
                            err_print("********************ERROR WRITING OUTGOING DATA on connection: %i.*********************\n", i);
                            //#endif
                            if (i == INFOSERVER)
                                clientInit(STATS);
                            else
                                deleteConnectionRecord(i,STATS);
                           continue;
                        }
                    }
                    else 
                    {
                        /* FIX -- What is this for? */
                        printf("We had to manually set it empty.  Somethign's fishy\n");
                        if (STATS.connections[i].hasOpenFile)
                            FD_CLR(STATS.connections[i].file, &(STATS.wset));
                        else
                            err_exit("Doesn't have an open file.  Isn't that bad?\n");
                    }
                }
                
                /* HANDLE FILE READY TO WRITE TO */
                else if( (STATS.connections[i].hasOpenFile)
                            && FD_ISSET(STATS.connections[i].file, &ws) )
                {
                    #if AGDEBUGLEVEL > 4
                    printf("handling file ready to write to\n");
                    #endif
                    if (handleOutgoingDataToFile(STATS) < 0) continue;
                }
                else
                {
                    printf("*********SOMETHING WENT WRONG, SHOULD HAVE FALLEN INTO A CATAGORY\n");
                    if (STATS.connections[i].socket >= 0)
                    {
                        if (FD_ISSET(STATS.connections[i].socket, &STATS.rset))
                            printf("connection: %i is SET socket readable\n",i);
                        if (FD_ISSET(STATS.connections[i].socket, &rs))
                            printf("connection: %i is RETURNED socket readable\n",i);
                
                        if (FD_ISSET(STATS.connections[i].socket, &STATS.wset))
                            printf("connection: %i is SET socket writeable\n",i);
                        if (FD_ISSET(STATS.connections[i].socket, &ws))
                            printf("connection: %i is RETURNED socket writeable\n",i);
                    }
                    else
                        printf("socket not initialized.\n");
                    printf("flags: %i\n", STATS.connections[i].flags);
                    printf("probably have a buffer that isn't emptying out size: %i in size: %i \n",
                        STATS.connections[i].OrPtr - STATS.connections[i].Obuf, 
                        STATS.connections[i].IrPtr - STATS.connections[i].Ibuf);
                }
                
                //printf("should have done catagory specific, about to do buffer test\n");
                
                if (STATS.connections[i].flags == 0)
                    printf("******DELETED connection %i, but didn't skip the rest ********\n", i);
                else
                    adjustBuffers(i, STATS);  //remove the already read stuff.
            }
        }
    }//close for
    
    return true;
}

void sendKeepAlive(server_status &STATS)
{
    if (STATS.setDisplayedServerStatus != NULL)
        STATS.setDisplayedServerStatus("Connected");
    
    AGMessage* KeepAlive = new AGMessage(STATS.connections[INFOSERVER].socket);
    KeepAlive->setType(READY_FOR_TRANSFER);
    
    time_t ThisKeepAlive = time(NULL);
    
    for (int i = 1; i < STATS.nConnections; i++)
    {
        if (STATS.connections[i].flags & TRANSFER_RECEIVING)
        {
            #if AGDEBUGLEVEL >3
		printf("ADDING connection #%i to keepalive\n",i);
	    #endif

	    KeepAlive->write_string((char*)STATS.connections[i].FileID,STATS.connections[i].FileIDLength, true);
            KeepAlive->write_int(getKPS(i, STATS));
            KeepAlive->write_zeros(4); /* UNKNOWN  -- may change.  FIX? */
            
            #if AGDEBUGLEVEL > 5
            printf("FileResumePosition: %i\n",STATS.connections[i].FileResumePosition);
            #endif
            KeepAlive->write_int(STATS.connections[i].FileResumePosition);
        }
    }
    #if AGDEBUGLEVEL > 3
    printf("SENDING KEEP ALIVE MESSAGE\n");
    #endif
    KeepAlive->queueMessage(INFOSERVER, STATS);

    
    STATS.LastKeepAlive = ThisKeepAlive; // could take the this excecution time into account, but why?
    
    delete(KeepAlive);
}

void sendFileTransferState(int i, server_status &STATS, int FileTransferState)
{
    #if AGDEBUGLEVEL > 2
    printf("SENDING FILE TRANSFER STATUS: %i\n",FileTransferState);
    #endif
    
    /* Send FileTransferState Message */
    AGMessage* FTSMessage = new AGMessage(STATS.connections[INFOSERVER].socket);
    FTSMessage->setType(FILE_TRANSFER_STATE);
    FTSMessage->write_string((char *)STATS.connections[i].FileID,
        STATS.connections[i].FileIDLength, true);
    FTSMessage->write_int(FileTransferState);
    FTSMessage->queueMessage(INFOSERVER, STATS);//send it.
    delete(FTSMessage);
    
    #if AGDEBUGLEVEL > 4
    printf("done sendingFileTransferState\n");
    #endif
}

int openCorrespondingFile(server_status &STATS)
{
    int i = STATS.current;
    int resultfd;
    int O_flags;
    
    /* determain what we're doign and set flags */
    
    if (STATS.connections[i].DirectionFlag == DIRECTION_READ)
    {
        O_flags = (O_WRONLY | O_NONBLOCK | O_APPEND);
        
        if (STATS.connections[i].FileResumePosition == 0)
        {
            #if AGDEBUGLEVEL > 3
            printf("OPENING NEW FILE FOR RECEIVING\n");
            #endif
            O_flags |= (O_CREAT | O_EXCL);
        }
        else
        {
            #if AGDEBUGLEVEL > 3
            printf("OPENING FILE FOR RESUMING AT: %i\n", STATS.connections[i].FileResumePosition);
            #endif            
            printf("resuming, should be ok\n");
        }
    }
    else
    {
        #if AGDEBUGLEVEL > 3
        printf("OPENING FILE FOR SENDING\n");
        #endif
        O_flags = (O_RDONLY | O_NONBLOCK);
    }
    
    
    
    /* Change to proper directory */

    if (STATS.connections[i].DirectionFlag == DIRECTION_READ)
    {
        if (chdir((char*)STATS.connections[i].FileDirectory) != 0)
        {
            if (errno == ENOENT)
            {
                err_print("Incomplete Downloads directory: %s doesn't exist, creating...\n",
                        STATS.Prefs->IncompleteDownloadsDirectory);
                int size = strlen(STATS.Prefs->IncompleteDownloadsDirectory) + 25;
                char* temp3 = new char[size];
                memset(temp3, '\0',size);
                snprintf(temp3, size, "mkdir -p \"%s\"", STATS.Prefs->IncompleteDownloadsDirectory);
                printf("about to execute: %s\n",temp3);
                if (system(temp3) != 0)
                   err_exit("Error creating incomplete downloads directory: %s\n  Please remedy this sitution by creating the directory by hand before attempting to continue.\n", STATS.Prefs->IncompleteDownloadsDirectory);
                delete(temp3);
                
                if (chdir((char*)STATS.connections[i].FileDirectory) != 0)
                    err_exit("Created incomplete downloads directory successful, but was unable to switch there... that's bad.  Please report.\n  Path: %s\n", STATS.Prefs->IncompleteDownloadsDirectory);
            }
            else
                err_exit("Unrecoverable error accessing incomplete downloads directory: %s\n",
                    STATS.Prefs->IncompleteDownloadsDirectory);
        }
    }
    else
    {
        if (chdir((char*)STATS.connections[i].FileDirectory) != 0)
        {
            err_print("Error accessing shares directory: %s\n", STATS.connections[i].FileDirectory);
            return (-1);
        }
    }
    
    
    /* actually open the file */
    
    if ( ( resultfd = open(STATS.connections[i].FileName, O_flags) ) < 0 )
    {
        if (STATS.connections[i].DirectionFlag == DIRECTION_SEND)
            return(-1);
        else
        {
            STATS.connections[i].FileResumePosition = 0;
            /* do I need to reset the directory? -- back to Temporary downloads directory?
                would it have changed?  been saved wierd? */
            if (getUnusedFileName(STATS.connections[i].FileName, STATS.connections[i].FileDirectory) < 0)
                return(-1);
            if (openCorrespondingFile(STATS) < 0)
                return(-1);
            else
                return 0;  // skip the rest, it was done at a deeper level
        }
    }
    
    /* update the FileSystem if needed. */
    if (STATS.setDisplayedFileSystemChange != NULL)
        if (STATS.connections[i].DirectionFlag == DIRECTION_READ)
            STATS.setDisplayedFileSystemChange(STATS.connections[i].FileDirectory);
    

    #if AGDEBUGLEVEL > 3
    printf("GOT FILE FD# %i\n",resultfd);
    #endif
             
    /* Handle all the extra stuff */
    
    /* Set file to be the newly opened file */
    STATS.connections[i].file = resultfd;
    /* Update that the there exists an open file for this conneciton */
    STATS.connections[i].hasOpenFile = true;
    
    if (STATS.connections[i].DirectionFlag == DIRECTION_SEND)
    {
        printf("Sending file: %s\n", STATS.connections[i].FileName);
        
        if ( (STATS.connections[i].FileResumePosition > 0) &&
            (lseek(STATS.connections[i].file, STATS.connections[i].FileResumePosition, SEEK_SET) < 0) )
        {
            err_print("Error seeking to position: %i  in fileID: %i\n", 
                STATS.connections[i].file, STATS.connections[i].FileResumePosition);
            err_print("That was of file: %s of directory %s\n",
                STATS.connections[i].FileName, STATS.connections[i].FileDirectory);
            err_print("We can't do anything here if we can't read the share file.\n");
            close(STATS.connections[i].file);
            STATS.connections[i].file = -1;
            STATS.connections[i].hasOpenFile = false;
            return (-1);
        }
        
        #if AGDEBUGLEVEL > 3
        printf("ADDING FILE TO READ LIST\n");
        #endif
        FD_SET(STATS.connections[i].file, &(STATS.rset));
    }
    else
    {
        printf("receiving file: %s\n", STATS.connections[i].FileName);
        
        /* update the DiskFileName in the resumes database */
        // we're assuming that Resumes exists...
        FileRecord* veryTemp;
        if ( (veryTemp = STATS.Resumes->getFileRecordByLocalID(STATS.connections[i].LocalID)) != NULL)
        {
            if (veryTemp->DiskFileName != NULL)
                delete(veryTemp->DiskFileName);
            veryTemp->DiskFileName = strdup(STATS.connections[i].FileName);
        }
        else
            err_print("couldn't access Resume FileRecord #%i\nThat may be trouble.\n", STATS.connections[i].LocalID);
        
        
        #if AGDEBUGLEVEL > 3
        printf("ADDING FILE TO WRITE LIST\n");
        #endif
        FD_SET(STATS.connections[i].file, &(STATS.wset));
        
        /* changing permissions */
        if (chmod((char*)STATS.connections[i].FileName, S_IWUSR | S_IRUSR ) < 0)
            err_print("Error changing permissions on file: %s won't be able to resume\n",
                        STATS.connections[i].FileName);
    }
    
    /* update STATS.maxfd */
    if (STATS.connections[i].file > STATS.maxfd)
        STATS.maxfd = STATS.connections[i].file;
    
    #if AGDEBUGLEVEL > 5
    printf("maxfd: %li\n", STATS.maxfd);
    #endif
    return 0;


    /* FIX LOOK UP WHERE THE FILE SHOULD BE.  GET FROM SHARES IF SHARE, WRITE TO DOWNLOAD IF DOWNOADING */

    /* lookup in ResumeDB or SharesDB  -- handled outside.*/
    
    /* check if it's the correct file -- handled outside */
    /* if not, send correct error message  -- outside */
    
    /* check validity of path */
    
    /* if not valid, set resume position to 0, try next one or send correct error message -- handled outside.*/

}

/* READING FUNCTIONS */

void handleNewConnection(int i, server_status &STATS)
{    
    /* We dont's send FTS_SETTING_UP here, because we would send it in case of
    a incoming connection twice (after listen() and here) 
    D.C. */
    
    //test first if this is a file transfer connection?
    if (STATS.connections[i].DirectionFlag == DIRECTION_SEND) //we're sending
    {
        STATS.connections[i].flags = WAIT_REQUEST;
        FD_CLR(STATS.connections[i].socket, &(STATS.wset));
        FD_SET(STATS.connections[i].socket, &(STATS.rset)); //should already be set
    }
    else //we're reading
    {
        STATS.connections[i].flags = SENDING_REQUEST;
        FD_CLR(STATS.connections[i].socket, &(STATS.rset));
        FD_SET(STATS.connections[i].socket, &(STATS.wset)); //should already be set
    }
}

int handleConnectionConnecting(server_status &STATS)
{
    int i = STATS.current;
    #if AGDEBUGLEVEL > 2
    printf("handling CONNECT on record: %i\n",i);
    #endif
    
    int error;
    socklen_t errorsize = sizeof(error);
    
    if (getsockopt(STATS.connections[i].socket, SOL_SOCKET, SO_ERROR, &error, &errorsize) < 0 || error != 0)
    {
        errno = error; //set errno
        handlePeerConnectingError(i,STATS);
        return(-1);
    }
    /* connection was successful */
    #if AGDEBUGLEVEL > 3
    printf("got outgoing connection %i\n",STATS.connections[i].socket);
    printf("CONNECTION COMPLETED\n");
    #endif
    return 0;
}

int handleConnectionAccepting(server_status &STATS)
{
    int i = STATS.current;
    #if AGDEBUGLEVEL > 2
    printf("handling ACCEPT on record: %i\n",i);
    #endif
    struct sockaddr_in PeerAddress;
    socklen_t PeerAddressLength = sizeof(PeerAddress);
    int tempsocket;
    char* ConnectedAddress;
    
    if ( (tempsocket =
            accept(STATS.connections[i].socket, (sockaddr *)&PeerAddress, &PeerAddressLength) ) < 0)
    {
        if (errno == EINPROGRESS)
            return (-1); //tell outside to continue.
        else if (errno == EWOULDBLOCK)
            return (-1);
        //handle errors here. /****FIX ***/
        err_exit("Unhandled Accept error.  Not wise to continue.  Sorry.  PLEASE REPORT THIS ERROR.\n");
    }
    
    //get the address of who connected.
    ConnectedAddress = inet_ntoa(PeerAddress.sin_addr);
    //should test mask against the address we received?
    //should chuck new connection and wait if it doesn't match?
    
    /*CLOSE THE LISTENING SOCKET, AND ADD THE NEW SOCKET */
    close(STATS.connections[i].socket);
    /* remove it from both read and write */
    FD_CLR(STATS.connections[i].socket, &(STATS.rset));
    FD_CLR(STATS.connections[i].socket, &(STATS.wset));
    // don't need to set it to -1 since we're about to set it below.
    
    /* Do I need to tell the server we've stopped listening? */
    STATS.connections[i].socket = tempsocket;
    
    /* HANDLE UPDATING MAXFD to include the new socket.*/
        if (STATS.connections[i].socket > STATS.maxfd)
             STATS.maxfd = STATS.connections[i].socket;
    
    #if AGDEBUGLEVEL > 5
    printf("maxfd: %li\n", STATS.maxfd);
    #endif
    
    #if AGDEBUGLEVEL > 2
    printf("GOT INCOMING CONNECTION ON SOCKET: %i, PeerAddress: %s\n", STATS.connections[i].socket, ConnectedAddress);
    #endif
    
    return 0;
}


void handleBandwidthLimits(server_status &STATS, fd_set &ws, fd_set &rs)
{
    timeval	new_timeval;

    timeval	outgoingDiff_timeval;
    timeval	incomingDiff_timeval;

    timeval	one_second;
    one_second.tv_sec = 1;
    one_second.tv_usec = 0;
    
    // test once, and assume for the next few calls
    if (STATS.Prefs != NULL)
    {
        gettimeofday(&new_timeval, &dontcare);
        timersub(&new_timeval, &STATS.lastOutgoingCheck, &outgoingDiff_timeval);
         
        if (STATS.Prefs->OutgoingLimit > 0) // if true, we're limiting outgoing speed.
        {
            if (outgoingDiff_timeval.tv_sec > 0) // if it has been one second.
            {
                // so now we reset count to 0, and set the last reset time to this time.
                STATS.outgoingByteCount = 0;
                STATS.lastOutgoingCheck.tv_sec = new_timeval.tv_sec;
                STATS.lastOutgoingCheck.tv_usec = new_timeval.tv_usec;
                #if AGDEBUGLEVEL > 6
                printf("it's been over a second, we're resetting outgoing\n");
                #endif
            }
            else // hasn't been a full second, so we are going to wait for the rest of that second before reseting.
                if (timercmp(&STATS.select_timer, &outgoingDiff_timeval, >))
                {
                    STATS.select_timer = outgoingDiff_timeval;
                }
            
            // both are gaurenteed to be positve, by above check.
            if (STATS.outgoingByteCount > STATS.Prefs->OutgoingLimit)
            {
                // oops, we've exceded our limit... only reading from infoserver.
                FD_ZERO(&ws);
                FD_SET(INFOSERVER, &ws);
                
                // we've cleared, now don't wait more than one second max!
                if (timercmp(&STATS.select_timer, &one_second, >))
                {
                    STATS.select_timer = one_second;
                }
            }
            else // we didn't excede our limit, so we'll just select as normal until we do.
            {
                ws = STATS.wset;
            }
        }
        else
            ws = STATS.wset;
        
        // calculate difference
        timersub(&new_timeval, &STATS.lastIncomingCheck, &incomingDiff_timeval);
        
        if (STATS.Prefs->IncomingLimit > 0) // if true, we're limiting incoming speed.
        {
            if (incomingDiff_timeval.tv_sec > 0) // if it has been one second.
            {
                // so now we reset count to 0, and set the last reset time to this time.
                STATS.incomingByteCount = 0;
                STATS.lastIncomingCheck.tv_sec = new_timeval.tv_sec;
                STATS.lastIncomingCheck.tv_usec = new_timeval.tv_usec;
                #if AGDEBUGLEVEL > 6
                printf("it's been over a second, we're resetting incoming\n");
                #endif
            }
            else // hasn't been a full second, so we are going to wait for the rest of that second before reseting.
                if (timercmp(&STATS.select_timer, &incomingDiff_timeval, >)) // compare them
                {
                    STATS.select_timer= incomingDiff_timeval;
                }
            
            // both are gaurenteed to be positve, by above check.
            if (STATS.incomingByteCount > STATS.Prefs->IncomingLimit)
            {
                // oops, we've exceded our limit... only writing to infoserver.
                FD_ZERO(&rs);
                FD_SET(INFOSERVER, &rs);
                
                // we've cleared, now don't wait more than one second max!
                if (timercmp(&STATS.select_timer, &one_second, >))
                {
                    STATS.select_timer= one_second;
                }
                #if AGDEBUGLEVEL > 6
                printf("oops we transferd: %li that exceded our limit of: %i\n", 
                    STATS.incomingByteCount, STATS.Prefs->IncomingLimit);
                #endif
            }
            else // we didn't excede our limit, so we'll just select as normal until we do.
            {
                rs = STATS.rset;
            }
        }
        else
            rs = STATS.rset;
    }
    else
        err_exit("Prefs Object suddenly disappeared from memory!  PLEASE REPORT THIS ERROR\n");

}

void handleTimeAndKeepAlive(server_status &STATS)
{
    // Check the time.
    time_t	new_time = time(NULL);
    
    //If 22 seconds has expired, send keep_alive message;
    
    time_t diff = new_time - STATS.old_time;
    if ( diff > 21 )
    {
        #if AGDEBUGLEVEL > 3
        printf("old time: %s\n", ctime(&STATS.old_time) );
        printf("new time: %s\n", ctime(&new_time) );
        #endif
        #if AGDEBUGLEVEL > 2
        printf("sanity check, TIME DIFFERENCE: %li seconds\n", diff );
        #endif
        // FIX -- this may change to use the new ClientStatus variable...
        if (STATS.sendKeepAlive) // set once the shares are sent.
            sendKeepAlive(STATS);
        STATS.old_time = new_time;
        // reset the timer after sending keepalive;
        STATS.select_timer.tv_sec = 22;
    }
    else
    {
        int left_til_keepAlive = (22 - diff);
        if (left_til_keepAlive > 0)
            STATS.select_timer.tv_sec = left_til_keepAlive;
        else
        {
            #if AGDEBUGLEVEL > 4
            printf("RESETING TIMEOUT TO 22\n");
            #endif
            STATS.select_timer.tv_sec = 22;
        }
    }
}


int handleReadFromSocket(server_status &STATS)
{
    int i = STATS.current;
    int n;
    
    #if AGDEBUGLEVEL > 3
    printf("asking to read %i, ",  BUFFER_SIZE - (STATS.connections[i].IwPtr - STATS.connections[i].Ibuf) );
    #endif
    
    assert( (BUFFER_SIZE - (STATS.connections[i].IwPtr - STATS.connections[i].Ibuf)) >= 0);
    
    assert( (BUFFER_SIZE - (STATS.connections[i].IwPtr - STATS.connections[i].Ibuf)) <= BUFFER_SIZE);
    
    /* Bandwidth restriction stuff */
    int readAmount = BUFFER_SIZE - (STATS.connections[i].IwPtr - STATS.connections[i].Ibuf);
    if ((i != INFOSERVER) && (STATS.Prefs != NULL) && (STATS.Prefs->IncomingLimit > 0))
    {
        // should be ok.
        int bandwidthleft = (STATS.Prefs->IncomingLimit - STATS.incomingByteCount);
        if ((bandwidthleft > 0) && (readAmount > bandwidthleft))
        {
            printf("restricted connection: #%i, wanted to read: %i only allowed %i\n", i, readAmount, bandwidthleft);
            readAmount = bandwidthleft;
        }
    }
    
    //read the data to the buffer, this is from the socket, which will be moved to a file.
    /* is this ready for non-blocking? */
    if ( ( n = read(STATS.connections[i].socket,
                    STATS.connections[i].IwPtr,
                    readAmount ) ) == 0 )
    {
        if (!(STATS.connections[i].flags & INFOSERVER_FLAG))
        {            
            #if AGDEBUGLEVEL > 2
            printf("  EOF on %i\n", STATS.connections[i].socket);
            #endif
            /*CLOSE and REMOVE IT FROM READ AND WRITE*/
            close(STATS.connections[i].socket);
            FD_CLR(STATS.connections[i].socket, &(STATS.rset));
            FD_CLR(STATS.connections[i].socket, &(STATS.wset));
            STATS.connections[i].socket = -1;
            

            /* FIX CAN THIS LOGIC BE CLEANED?
               Everything gets handled on handleOutgoingDataToFile()...
               There ist not much to do here.... D.C.
               Don't know wether it's working like this */
            
            #if AGDEBUGLEVEL > 3
            printf("CONNECTION CLOSED\n");
            printf("connection[i].flags=%i\n", STATS.connections[i].flags);
            #endif
            
            /* FIX THIS LOGIC TO TERMINATE WHEN THE SONG IS DONE, not when they stop sending. */
            
            if (STATS.connections[i].flags & TRANSFER_RECEIVING)
            {
                FD_SET(STATS.connections[i].file, &(STATS.wset)); // just make sure.
                //fprintf(stderr, "test 1");
            }
            else if(STATS.connections[i].flags & TRANSFER_SENDING) // do we ever get here?
            {
                printf("I don't think we should be in this code (readFromSocket)!  -- %s, %i\n", __FILE__, __LINE__);
                FD_SET(STATS.connections[i].file, &(STATS.rset)); // just make sure.
                //fprintf(stderr, "test 2"); 
            }
            else if(STATS.connections[i].flags & (WAIT_RESPONSE | SENDING_RESPONSE))
            {
                sendFileTransferState(i,STATS, FTS_CONNECTION_CLOSED);
                deleteConnectionRecord(i,STATS); // already closed above.
                return (-1);
            }
            STATS.connections[i].flags = CLOSING;  // FIX -- SHOULD I USE SHUTDOWN HERE?
            /* will send connection closed message after clearing buffer */
            return(-1);
        }
        else //info server connection closed
        {
            /* do we ever want to get here inentionally? */
            err_print("EOF received from unexpectedly infoserver.... restarting client\n");
            clientInit(STATS);
            return (-1);
        }
    }
    else if (n < 0) //error while reading.
    {
        if (!(STATS.connections[i].flags & INFOSERVER_FLAG))
        {
            if ( (errno == ECONNRESET)) // connection reset by peer
            {
                #if AGDEBUGLEVEL > 2
                printf("Lost connection to peer.  Notifying server.\n");
                #endif
                sendFileTransferState(i, STATS, FTS_CONNECTION_CLOSED); //reset by peer
                deleteConnectionRecord(i,STATS);
                return(-1); // skip the rest.
            }
            else if (errno == EWOULDBLOCK)
                return(-1); // think that's ok...
            else if (errno == ETIMEDOUT)
            {
                err_print("Operation timed out!  We shouldn't be blocking!!!\n");
                // FIX -- I don't need to delete the record her, do I?
                return(-1);
            }
            else
                err_exit("Unhandled Read error when reading from socket.  I can't continue.  Sorry.\n");
        }
        else
        {
            err_print("Information server closed the connection unexpectedly.  Restarting...\n");
            clientInit(STATS);
            return(-1); //will continue.
        }
        /* FIX *** HANDLE ERRORS */
    }
    else /* SUCCESS ADJUST POINTERS */
    {
        #if AGDEBUGLEVEL > 3
        printf("read: %i\n", n);
        #endif
        STATS.connections[i].IwPtr += n;
        
        if ((i != INFOSERVER) && (STATS.connections[i].flags == TRANSFER_RECEIVING))
            STATS.incomingByteCount += n; // for bandwidth restraint.
        
        if (STATS.connections[i].IwPtr-STATS.connections[i].IrPtr < 11)
        {
            #if AGDEBUGLEVEL > 3
            printf("ONLY %i BYTES IN BUFFER, WAITING FOR MORE\n",
                    STATS.connections[i].IwPtr-STATS.connections[i].IrPtr);
            #endif
            return(-1);
        }
    }
    return 0; /* do we ever hit this?  -- sure, if it doesn't have errors! */
}


int handleIncomingFileRequest(server_status &STATS)
{
    int i = STATS.current;
    
    #if AGDEBUGLEVEL > 2
    printf("handling INCOMING FILE REQUEST: %i\n",i);
    
    printf("**************WOW. WE ARE BEING ASKED TO SHARE!!!************\n");
    #endif
    
    //parse request;
    AGMessage* FileRequest =
            new AGMessage(STATS.connections[i].IrPtr,
                STATS.connections[i].IwPtr-STATS.connections[i].IrPtr , PEER_MESSAGE);
    int completeSize = FileRequest->getCompleteMessageSize();
    if (FileRequest->getMessageStatus() == AG_MESSAGE_COMPLETE) //crude... but works.
        STATS.connections[i].IrPtr += completeSize;
    else if (FileRequest->getMessageStatus() == AG_MESSAGE_INCOMPLETE)
    {
        printf("Incomming File Request Message Incomplete, waiting...\n");
        delete(FileRequest);
        return 0; // just wait for the buffer to fill more...
            /* Avoids this infinite loop... but what if we have BAD data.
            TCP should guarentee that we do not.  Still, the client will infinite loop if we have
            a short packet in the buffer.... not good.*/
    }
    else
    {        
        err_print("Bad File Request packet received from peer.  Must close the connection.\n");
        
        sendFileTransferState(i, STATS, FTS_CONNECTION_CLOSED);
        //deleteConnectionRecord(i, STATS); // handled outside.
        delete(FileRequest);
        return (-1);
    }
    
    unsigned short FileIDLength	= FileRequest->read_short();
    
    char* FileID = NULL;
    FileRequest->read_string(FileID,FileIDLength);
    int FileSize		= FileRequest->read_int();
    STATS.connections[i].FileResumePosition	= FileRequest->read_int();
    #if AGDEBUGLEVEL > 3
    printf("REQUESTED FILE SIZE %i\n", FileSize);
    printf("REQUESTED FILE RESUME POSITION %i\n", STATS.connections[i].FileResumePosition);
    #endif
    
    delete(FileRequest);
    
    /*******FIX******* VALIDATE REQUEST */
    
    if ( (FileIDLength != STATS.connections[i].FileIDLength) ||
         ( memcmp(FileID, STATS.connections[i].FileID, FileIDLength) != 0 ) ||
         (FileSize != STATS.connections[i].FileSize) )
    {
        err_print("Received FileRequest, but it was different than expected, ignoring and waiting for proper request.\n");
        delete (FileID);
        // already read the message, now we're just returning, ignoring, and waiting to listen for more.
        return 0;
    }
    delete(FileID); // done using that.
    
    printf("*********WE'RE being asked to send *******\n");
    
    FileRecord* theShare = 
        //STATS.Shares->getFileRecordByFileID(STATS.connections[i].FileID, STATS.connections[i].FileIDLength);
        STATS.Shares->getFileRecordByLocalID(STATS.connections[i].LocalID);
    if (theShare == NULL)
    {
        err_print("Didn't find share %s size: %i in shares database.  Informing server and closing peer connection.\n",
            STATS.connections[i].FileName, STATS.connections[i].FileSize);
        sendFileTransferState(i, STATS, FTS_INTERNAL_ERROR);
        //deleteConnectionRecord(i, STATS); // didn't find it,  Abort.
        return (-1);
    }
    
    if (theShare->FileSize != FileSize)
    {
        err_print("Asked to share and found share in database, but it has a different file size than expected\n");
        err_print("Found size: %i, Expected size: %i\n", theShare->FileSize, FileSize);
        err_print("Responding with proper size.\n");
    }
    
    if (STATS.connections[i].FileDirectory != NULL)
        delete (STATS.connections[i].FileDirectory);
    STATS.connections[i].FileDirectory = strdup(theShare->DirectoryName);
    
    // FIX THIS -- shouldn't overwrite that filename..
    
    if (STATS.connections[i].FileName != NULL)
        delete (STATS.connections[i].FileName);
    STATS.connections[i].FileName = strdup(theShare->DiskFileName);
    
    STATS.connections[i].FileSize = theShare->FileSize;
    
    /* THEN OPEN FILE */
    if (openCorrespondingFile(STATS) != 0)
    {
        err_print("Couldn't open File: %s to Send\n", STATS.connections[i].FileName);
        err_print("Deleting share from database\n");
        STATS.Shares->deleteFileRecordWithLocalID(theShare->LocalID);
        err_print("Not informing server that share was deleted...\n But aborting connection!\n");
        sendFileTransferState(i, STATS, FTS_INTERNAL_ERROR);
        return (-1);
    }
    
    printf("Ready to send share, now sending FileRequestResponse.\n");
    
    FD_CLR(STATS.connections[i].socket,&(STATS.rset));
    FD_SET(STATS.connections[i].socket,&(STATS.wset));
    STATS.connections[i].flags = SENDING_RESPONSE;
    return 0;
}

int handleIncomingFileRequestResponse(server_status &STATS)
{
    int i = STATS.current;
    
    #if AGDEBUGLEVEL > 2
    printf("handling INCOMING FILE REQUEST RESPONSE: %i\n",i);
    #endif
    
    /*have died of a segmentation fault while initializing below.
    handling INCOMING FILE REQUEST RESPONSE: 4
    
    7814 - INITIALIZING INCOMING PACKET
    7814 - begining read of length 4
    7814 - copying data
    Segmentation fault
    
    also:
    
    handling SOCKET READ on socket: 9 of record: 1
    asking to read 4000, read: 16
    handling INCOMING FILE REQUEST RESPONSE: 1
    Segmentation fault
    */
    AGMessage* FileRequestResponse =
        new AGMessage(STATS.connections[i].IrPtr,
            STATS.connections[i].IwPtr-STATS.connections[i].IrPtr,
            PEER_MESSAGE);
    int completeSize = FileRequestResponse->getCompleteMessageSize();
    if (FileRequestResponse->getMessageStatus() == AG_MESSAGE_COMPLETE) //crude... but works.
        STATS.connections[i].IrPtr += completeSize;
    else if (FileRequestResponse->getMessageStatus() == AG_MESSAGE_INCOMPLETE)
    {
        printf("Incomming File Request Response message incomplete, waiting...\n");
        delete(FileRequestResponse);
            /* Avoids this infinite loop... but what if we have BAD data.
            TCP should guarentee that we do not.  Still, the client will infinite loop if we have
            a short packet in the buffer.... not good.*/
        return(0);
    }
    else
    {
        delete(FileRequestResponse);
        err_print("Bad File Request Response packet received from peer. Killing peer connection and informing server.\n");
        sendFileTransferState(i, STATS, FTS_CONNECTION_CLOSED);
        return(-1);
    }
    /*
    STATS.connections[i].FileIDLength	= FileRequestResponse->read_short();
    STATS.connections[i].FileID	=
        FileRequestResponse->read_string(STATS.connections[i].FileIDLength);
    STATS.connections[i].FileSize	= FileRequestResponse->read_int();
    */
    
    #if AGDEBUGLEVEL > 5
    // just to maek sure that we're sending the right ones.
    FileRequestResponse->dumpHex();
    #endif
    
    char* OKMESSAGE = NULL;
    int OKMESSAGElength = FileRequestResponse->read_string(OKMESSAGE, DO_TERMINATE);
    delete(FileRequestResponse);
    
    #if AGDEBUGLEVEL > 2
    printf("RECEIVED REQUEST RESPONSE, WITH MESSAGE: %s\n\n", OKMESSAGE);
    #endif
    
    if (strcmp((char*)OKMESSAGE, "OK") != 0)
    {
        printf("PEER SAID REQUEST WAS NOT OK, rather: %s", OKMESSAGE);
        sendFileTransferState(i, STATS, FTS_CONNECTION_CLOSED);
        //deleteConnectionRecord(i,STATS); // handled outside.
        delete(OKMESSAGE);
        return (-1); //won't "continue" when outside, but that's ok, flags == 0 already.
    }
    else
    {
        delete(OKMESSAGE);
        
        #if AGDEBUGLEVEL > 2
        printf("*****OPENING FILE****\n");
        #endif
        if(STATS.connections[i].FileDirectory != NULL && STATS.connections[i].FileDirectory != 0)
            delete (STATS.connections[i].FileDirectory);
        STATS.connections[i].FileDirectory =  strdup(STATS.Prefs->IncompleteDownloadsDirectory);
        
        if (openCorrespondingFile(STATS) != 0)
        {
            err_print("Could not open a temporary file to start transfer of: %s on connection %i.\n",
                    STATS.connections[i].FileName, i);
            sendFileTransferState(i,STATS,FTS_INTERNAL_ERROR); /* what shoul we send to the server? */
            //deleteConnectionRecord(i,STATS); // handled outside.
            return (-1); // we're ok now, but this could potentially cause problems later.
        }
        #if AGDEBUGLEVEL > 3
        printf("GOT file descriptor for temp file: %i", STATS.connections[i].file);
        #endif
        //FD_SET(STATS.connections[i].file, &(STATS.wset));
        
        STATS.connections[i].flags = TRANSFER_RECEIVING;
    }
    return 0;
}

int handleIncomingDataFromPeer(server_status &STATS)
{
    /* All this does is check to see if we should keep listening to the peer
    or if we have to wait for the data to be written to disk */
    
    int i = STATS.current;
    #if AGDEBUGLEVEL > 2
    printf("handling INCOMING DATA FROM PEER: %i\n",i);
    #endif
    /* Buffer is full now, don't try to read from socket more. */
    if ( ( STATS.connections[i].Ibuf + BUFFER_SIZE ) == STATS.connections[i].IwPtr )
    {
        #if AGDEBUGLEVEL > 3
        printf("read buffer full on: %i\n",i);
        #endif
        FD_CLR(STATS.connections[i].socket,&(STATS.rset));
    }
    #if AGDEBUGLEVEL > 4
    /*
    printf("The file %i has current write status: %i and is set to be listened to: %i\n",
        STATS.connections[i].file,
        FD_ISSET(STATS.connections[i].file, &(ws)),
        FD_ISSET(STATS.connections[i].file, &(STATS.wset))); */
    printf("MaxFD is currently set to: %li\n", STATS.maxfd);
    #endif  
    
    /* Listen for file to empty Buffer */
    FD_SET(STATS.connections[i].file, &(STATS.wset));
    return 0;
}

int handleIncomingDataFromInfoServer(server_status &STATS)
{
    int i = STATS.current; // I shoudl be equal to INFORMATION_SERVER
    #if AGDEBUGLEVEL > 2
    printf("handling INCOMING DATA FROM INFOSERVER: %i\n",i);
    #endif
    //construct messages
    while ( (STATS.connections[i].IwPtr - STATS.connections[i].IrPtr) > 0)  // there is still stuff to read.
    {
        #if AGDEBUGLEVEL > 3
        printf("TOP OF MESSAGE PARSE LOOP\n");
        #endif
        int completeSize;
        AGMessage* IncomingMessage =
            new AGMessage(STATS.connections[i].IrPtr,
                STATS.connections[i].IwPtr - STATS.connections[i].IrPtr,
                    SERVER_MESSAGE); //get new message.
            
        completeSize = IncomingMessage->getCompleteMessageSize();
        if (IncomingMessage->getMessageStatus() == AG_MESSAGE_COMPLETE) //crude... but works.
            STATS.connections[i].IrPtr += completeSize;
        else if (IncomingMessage->getMessageStatus() == AG_MESSAGE_INCOMPLETE)
        {
            delete(IncomingMessage);
            break; /* Avoids this infinite loop... but what if we have BAD data.
                       TCP should guarentee that we do not.  Still, the client will infinite loop if we have
                        a short packet in the buffer.... not good.*/
        }
        else
        {
            delete(IncomingMessage);
            err_print("Recieved bad data from infoserver, must reset the client.\n");
            return(-1);
        }	
        #if AGDEBUGLEVEL > 4
        printf("about to parse\n");
        #endif
        parseMessage(STATS, IncomingMessage);
        #if AGDEBUGLEVEL > 4
        printf("returned from parse, about to delete message\n");
        #endif
        delete(IncomingMessage);  //why does it complain?
    }
    
    #if AGDEBUGLEVEL > 3
    if (STATS.connections[i].IrPtr != STATS.connections[i].IwPtr)
        printf("*****STUFF STILL IN BUFFER ****\n");
    #endif
    return 0;
}

int handleIncomingDataFromFile(server_status &STATS)
{
    int i = STATS.current;
    int n;
    #if AGDEBUGLEVEL > 2
    printf("handling FILE READ on record: %i\n",i);
    #endif
    
    /* saftey */
    assert( (BUFFER_SIZE - (STATS.connections[i].OwPtr - STATS.connections[i].Obuf)) >= 0);
    assert( (BUFFER_SIZE - (STATS.connections[i].OwPtr - STATS.connections[i].Obuf)) <= BUFFER_SIZE);
    
    //read the data to buffer.  This is from a file to a buffer which will be moved out to socket.
    if ( ( n = read(STATS.connections[i].file, STATS.connections[i].OwPtr,
                    BUFFER_SIZE - (STATS.connections[i].OwPtr - STATS.connections[i].Obuf) ) ) == 0 )
    {
        printf("EOF on %i\n", STATS.connections[i].file);
        close(STATS.connections[i].file);
        FD_CLR(STATS.connections[i].file, &(STATS.rset));
        FD_CLR(STATS.connections[i].file, &(STATS.wset));
        STATS.connections[i].file = -1;
        STATS.connections[i].hasOpenFile = false;
        
        STATS.connections[i].flags = CLOSING;
    }
    else if (n < 0)
    {
        if (errno == EWOULDBLOCK)
            return(-1); //should be fine.
        err_exit("Unhandled error encountered while reading from file.  This is currently unhandled.  Sorry.\n");
        //handle errors.
    }
    else /* SUCCESS ADJUST POINTERS */
        STATS.connections[i].OwPtr += n;
    
    /* Buffer is full now, don't try to read from file more. */
    if ( ( STATS.connections[i].Obuf + BUFFER_SIZE ) <= STATS.connections[i].OwPtr )
    {
        #if AGDEBUGLEVEL > 3
        printf("Buffer full, not reading more from file\n");
        #endif
        FD_CLR(STATS.connections[i].file,&(STATS.rset));
    }
    /* Listen for socket to empty Buffer */
    if (STATS.connections[i].socket >= 0)
        FD_SET(STATS.connections[i].socket, &(STATS.wset));
    else
        err_print("Read data from file, but no socket to write to... that's BAD.\n");
    return 0;
}


void handleOutgoingFileRequest(server_status &STATS)
{
    int i = STATS.current;
    
    /*RESUME POSITION SHOULD BE A DATABASE LOOKUP */
    if (STATS.Resumes != NULL)
    {
        FileRecord *theResume = 
            STATS.Resumes->getFileRecordBySuggestedNameAndSize( STATS.connections[i].FileName, STATS.connections[i].FileSize);
        if (theResume != NULL)
        {
            printf("**** FOUND RESUME IN DATABASE, RESUMING #%li  AT: %li\n", 
                theResume->LocalID, theResume->FileResumePosition);
            STATS.connections[i].FileResumePosition = theResume->FileResumePosition;
            STATS.connections[i].LocalID = theResume->LocalID;
            theResume->ConnectionPtr = &(STATS.connections[i]);
        }
        else
        {
            printf("File not found in resume database.  Adding a slot for it and setting it's name.\n");
            
            theResume = STATS.Resumes->getNewFileRecord();
            
            theResume->SuggestedFileName = strdup(STATS.connections[i].FileName);
            if (STATS.Prefs->IncompleteDownloadsDirectory != NULL)
           	theResume->DirectoryName	 = strdup(STATS.Prefs->IncompleteDownloadsDirectory);
            else
                printf("STATS.Prefs->IncompleteDownloadsDirectory was equal to NULL when creating resume record.  Please report this.\n");
            
            theResume->FileSize  = STATS.connections[i].FileSize;
            theResume->flags	  = UNVALIDATED_SHARE; // needs to be set to something!  FIX
            
            STATS.connections[i].LocalID = theResume->LocalID;
            theResume->ConnectionPtr = &(STATS.connections[i]);
            
            printf("adding new resume: %s with %i bytes of %i and LocalID: %i\n",
                STATS.connections[i].FileName, STATS.connections[i].FileResumePosition,
                STATS.connections[i].FileSize, STATS.connections[i].LocalID);
        }
    }
    
    AGMessage* FileRequest = new AGMessage(STATS.connections[i].socket);
    //no type
    FileRequest->write_string((char*)STATS.connections[i].FileID, STATS.connections[i].FileIDLength, true);
    FileRequest->write_int(STATS.connections[i].FileSize);
    FileRequest->write_int(STATS.connections[i].FileResumePosition);
    FileRequest->queueMessage(i, STATS);
    delete(FileRequest);
    
    /*Adjust select lists */
    FD_SET(STATS.connections[i].socket, &(STATS.rset));
    
    /* SET FLAGS */
    STATS.connections[i].flags = WAIT_RESPONSE;
}


void handleOutgoingFileRequestResponse(server_status &STATS)
{
    int i = STATS.current;
    //send OK message.
    AGMessage* FileRequestResponse = new AGMessage(STATS.connections[i].socket);
    //no type
    //FileRequestResponse->write_string((char*)FileID, FileIDLength, true); //not needed in this version.
    //FileRequestResponse->write_int(FileSize); //not needed in this version
    FileRequestResponse->write_string("OK", 2, true);
    printf("about to send share response\n");
    
    #if AGDEBUGLEVEL > 5
    // just to make sure we're sendign the righth ones.
    FileRequestResponse->dumpHex();
    #endif
    
    FileRequestResponse->queueMessage(i, STATS);
    delete(FileRequestResponse);
    
    /*Adjust select lists */
    FD_SET(STATS.connections[i].socket, &(STATS.wset));
    FD_SET(STATS.connections[i].file, &(STATS.rset));
    
    /* SET FLAGS */
    STATS.connections[i].flags = TRANSFER_SENDING;
    // should we have it try to write already?
    
    /* IF WE COULDN'T WRITE THIS, THE FILE NEEDS TO GET CLOSED... I GUESS THAT GETS HANDLED ABOVE? */
}

int handleOutgoingDataToSocket(server_status &STATS)
{
    int i = STATS.current;
    int n;
    
    int error;
    socklen_t errorsize = sizeof(error);
    if (getsockopt(STATS.connections[i].socket, SOL_SOCKET, SO_ERROR, &error, &errorsize) < 0 || error != 0)
    {
        errno = error; //set errno
        err_print("error trying to write data on connection: %i\n", i);
        return (-1);
    }
    
    /* Bandwidth restriction stuff */
    int writeAmount = STATS.connections[i].OwPtr-STATS.connections[i].OrPtr;
    if ((i != INFOSERVER) && (STATS.Prefs != NULL) && (STATS.Prefs->OutgoingLimit > 0))
    {
        // should be ok.
        int bandwidthleft = (STATS.Prefs->OutgoingLimit - STATS.outgoingByteCount);
        if ((bandwidthleft > 0) && (writeAmount > bandwidthleft))
        {
            printf("restricted connection: #%i, wanted to write: %i only allowed %i\n", i, writeAmount, bandwidthleft);
            writeAmount = bandwidthleft;
        }
    }
    
    //send data.
    
    // HACK TO FIX SIGNAL PROBLEMS.
    
    struct sigaction ignoreit;
    struct sigaction theold;
    
    ignoreit.sa_handler		= SIG_IGN;
    sigemptyset(&ignoreit.sa_mask);
    ignoreit.sa_flags		= 0;
    
    if (sigaction(SIGPIPE, &ignoreit, &theold) != 0)
        err_print("error ignoring signal\n"); // needed to ignore the sigpipes I can get here...
    
    n = write(STATS.connections[i].socket, STATS.connections[i].OrPtr,writeAmount);
    
    if (sigaction(SIGPIPE, &theold, NULL) != 0)
        err_print("error restoring signal handling\n");
    
    if (n < 0)
    {
        /* FIX HANDLE ERRORS */
        if (errno == EWOULDBLOCK)
            return(-1);
        
        return (-1);
    }
    else /* SUCCESS ADJUST POINTERS */
    {
        STATS.connections[i].OrPtr += n;
        if ((i != INFOSERVER) && (STATS.connections[i].flags == TRANSFER_SENDING) && STATS.connections[i].hasOpenFile)
        {
            #if AGDEBUGLEVEL > 4
            printf("sent %i bytes to peer!\n", n); // this will be off by the initial message.
            #endif
            
            //Above if is there to protect this value from being written to at bad times, for the wrong connection.
            // do I need these values when I am sending?  probably not.
            STATS.connections[i].FileResumePosition += n;  //here this refers to how much we've SENT
            // will be off a little from messages and not data, but who cares.
            STATS.connections[i].DataSinceLastKPS += n;
                // don't konw if the server even cares for outgoings...
            
            STATS.outgoingByteCount += n; // for bandwidth restraint.
        }
        
    #if AGDEBUGLEVEL > 4
        printf("wrote %i bytes to socket %i on connection %i\n", n, STATS.connections[i].socket, i);
    #endif
    }
    
    /* Buffer is completely empty now, don't try to write to socket more. */
    if ( (STATS.connections[i].OwPtr - STATS.connections[i].OrPtr) == 0 )
    {
        FD_CLR(STATS.connections[i].socket,&(STATS.wset));
        
        /* IF WE ARE SUPPOSED TO CLOSE DO SO. */
        if ( (i != INFOSERVER) && (STATS.connections[i].flags & CLOSING) )
        {
            //printf("closing connection, no more to send\n");
            
            sendFileTransferState(i,STATS,FTS_DOWNLOAD_COMPLETE);
            STATS.connections[i].flags = TRANSFER_COMPLETE;
            #if AGDEBUGLEVEL > 0
            printf("Completed Sending: %s\n", STATS.connections[i].FileName);
            #endif
            if (STATS.setDisplayedTransferComplete != NULL)
                STATS.setDisplayedTransferComplete(STATS.connections[i]);
            #if AGDEBUGLEVEL > 4
            else
                printf("not handling display stuff\n");
            #endif
            deleteConnectionRecord(i,STATS);
            return (0);
        }
    }
    #if AGDEBUGLEVEL > 3
    else printf("BUFFER NOT QUITE EMPTY, STILL %i TO WRITE\n",
                STATS.connections[i].OwPtr-STATS.connections[i].OrPtr);
    #endif
    
    /* Listen for file to fill Buffer */
    if ( (i != INFOSERVER) && (STATS.connections[i].hasOpenFile) )
    {
        //printf("set to listen for data from file\n");
        FD_SET(STATS.connections[i].file, &(STATS.rset));
    }
    
    return 0;
}

int handleOutgoingDataToFile(server_status &STATS)
{
    int i = STATS.current;
    int FileTransferState = FTS_CONNECTION_CLOSED; // is this right??? FIX
    int n;
    
    #if AGDEBUGLEVEL > 2
    printf("handling FILE WRITE on record: %i\n",i);
    #endif    
    n = write(STATS.connections[i].file, STATS.connections[i].IrPtr,
                STATS.connections[i].IwPtr-STATS.connections[i].IrPtr);
    if (n < 0)
    {
        /* FIX HANDLE ERRORS HERE */
        if (errno == EWOULDBLOCK)
            return(-1);
        err_exit("Unhandled error while writing to file: %s  Sorry.\n", STATS.connections[i].FileName);
    }
    else /* SUCCESS ADJUST POINTERS and RESUME POSITION*/
    {
        if (n > 0) // for efficency
        {
            STATS.connections[i].IrPtr += n;
            STATS.connections[i].FileResumePosition += n;
            STATS.connections[i].DataSinceLastKPS += n;
            
            /* update resumePosition resumes database */
            // assuming resumes exists!!!
            FileRecord* veryTemp;
            if ( (veryTemp = STATS.Resumes->getFileRecordByLocalID(STATS.connections[i].LocalID)) != NULL)
                veryTemp->FileResumePosition += n;
            else
            {
                printf("LocalID was: %i\n", STATS.connections[i].LocalID);
                err_print("couldn't access Resume FileRecord #%i to update FileResumePosition\nThat may be trouble.\n",
                        STATS.connections[i].LocalID);
            }
        }
        
    #if AGDEBUGLEVEL > 4
        printf("wrote %i bytes to file %i, name: %s on connection %i\n", n,
            STATS.connections[i].file,STATS.connections[i].FileName, i);
    #endif
    
        
        /* FOR TESTING ONLY!!!!!! */
        #ifdef AGQUICKDOWNLOADTEST
        STATS.connections[i].IrPtr = STATS.connections[i].IwPtr;
        STATS.connections[i].FileResumePosition = STATS.connections[i].FileSize;
        STATS.connections[i].flags = CLOSING;
        #endif
        
    }
    
    /* Listen for socket to fill Buffer */
    if (STATS.connections[i].socket >= 0)
        FD_SET(STATS.connections[i].socket, &(STATS.rset));
    
    /* Buffer is completely empty now, don't try to write to file more. */
    if ( (STATS.connections[i].IwPtr - STATS.connections[i].IrPtr) == 0 )
    {
        #if AGDEBUGLEVEL > 3
        printf("buffer empty, no longer writing to file\n");
        #endif
        FD_CLR(STATS.connections[i].file,&(STATS.wset));
        if (STATS.connections[i].flags & CLOSING)
        {
            if (STATS.connections[i].FileResumePosition == STATS.connections[i].FileSize)
            {
                FileTransferState = FTS_DOWNLOAD_COMPLETE; // the file was complete
                STATS.connections[i].flags = TRANSFER_COMPLETE;
                handleFileCompletelyreceived(STATS);
            }
            else
            {
                // FIX -- this should have been handled before here, could be removed, right?
                
                FileTransferState = FTS_CONNECTION_CLOSED; // the peer aborted.
                /* make sure it gets recorded in the partials database -- handled in deleteConnectionRecord*/
            }
            sendFileTransferState(i,STATS,FileTransferState);
            deleteConnectionRecord(i,STATS); // handles all closing of file descriptors
            return (-1);
        }
    }
    
    return 0;
}

void handleFileCompletelyreceived(server_status &STATS)
{
    int i = STATS.current;

    // are the paths the same?  Only move the file if they are different.  /they should already be "realed"
    if (strcmp(STATS.Prefs->FinishedDownloadsDirectory, STATS.connections[i].FileDirectory) != 0)
    {
        char* temp = new char[PATH_MAX];
        errno = 0;
        if ( (realpath(STATS.Prefs->FinishedDownloadsDirectory, temp) == NULL) 
            || (errno != 0) || (chdir(temp) != 0)) // FIX -- this might not work!
        {
            delete(temp);
            
            if (errno == ENOENT)
            {
                err_print("Completed Downloads directory: %s doesn't exist, creating...\n",
                        STATS.Prefs->FinishedDownloadsDirectory);
                
                int size = strlen(STATS.Prefs->FinishedDownloadsDirectory) + 25;
                char* temp3 = new char[size];
                memset(temp3, '\0', size);
                snprintf(temp3, size, "mkdir -p \"%s\"", STATS.Prefs->FinishedDownloadsDirectory);
                printf("about to execute: %s\n",temp3);
                if (system(temp3) != 0)
                {
                    err_print("Completed Downloads directory: %s was not found, attempted to create, but failed.  
                                Completed file: %s will be left in its current temporary directory: %s.\n",
                                STATS.Prefs->FinishedDownloadsDirectory, STATS.connections[i].FileName,
                                STATS.connections[i].FileDirectory);
                }
                else if (chdir((char*)STATS.Prefs->FinishedDownloadsDirectory) != 0)
                {
                        err_print("Completed Downloads directory: %s was not found, successfully created, 
                            but failed to access.\n  This error will repeat, and files will be left in the 
                            Temporary Downloads Directory", STATS.Prefs->FinishedDownloadsDirectory);
                }
                delete(temp3);
            }
            else if (errno == EACCES)
            {
                err_print("You do not have the correct permissions to access your specified download directory.  
                        Please correct this and relogin (to audiogalaxy).\n  
                        As such, all downloaded files will be left in your temporary directory.");
            }
            else
            {
                err_print("Unhandled error encountered while trying to verify your Download Path.  
                    Please check the error below and fix your download path.  
                    All downloaded files will be left in your temporary directory.  Please Report.\n");
            }
        }
        else
        {
            printf("the finished directory: %s already exists\n",STATS.Prefs->FinishedDownloadsDirectory);
            // could print it out if we wanted to...
            delete(temp); // finished directory exists.
        }
        
        // then switch back to the base directory.  We don't care if it succeeds.
        chdir((char*)STATS.BaseDirectory);
        
        //may have to gererate complete paths first.
        char* PathFrom; 
        char* PathTo;
        
        //Begin with just the file names.
        PathFrom = strdup(STATS.connections[i].FileName);
        PathTo = strdup(STATS.connections[i].FileName);
        
        // make the from path absolute for the current file's directory.
        makePathAbsolute(PathFrom, STATS.connections[i].FileDirectory);
        //#if AGDEBUGLEVEL > 3
        printf("between check, from: %s \nto:%s\n",PathFrom, PathTo);
        //#endif
        // get an unused name in the downloads directory.
        printf("test: %s\n", PathTo);
        if (getUnusedFileName(PathTo, STATS.Prefs->FinishedDownloadsDirectory) < 0)
        {
            printf("test1: %s\n", PathTo);
            printf("test2: %s\n", PathFrom);
            err_print("Failed to find a valid name for the file in the Finished downloads director, not moving from: %s to: %s.\n", PathFrom, PathTo);
            delete(PathTo);
            delete(PathFrom);
            return;
        }
        // make the pathTo absolute.
        makePathAbsolute(PathTo, STATS.Prefs->FinishedDownloadsDirectory);
        
        #if AGDEBUGLEVEL > 3
        printf("aftercheck, from: %s \nto:%s\n",PathFrom, PathTo);
        #endif
        
        /* move it and chmod it */
        if (chmod((char*)PathFrom, S_IWUSR | S_IRUSR | S_IWGRP | S_IRGRP) < 0)
            err_print("Error changing permissions on file: %s  You won't be able to read the file, please report.\n",
                STATS.connections[i].FileName);
            
        /* Move the file */
        if(AGMove(PathFrom, PathTo) > -1)
        {
            printf("moved OK\n");
            // moved fine, now update path information:
            if ((STATS.connections[i].FileDirectory != 0) && (STATS.connections[i].FileDirectory != NULL))
                delete(STATS.connections[i].FileDirectory); // should be ok.
            STATS.connections[i].FileDirectory = strdup(STATS.Prefs->FinishedDownloadsDirectory);
            
            /* add the share to the database  -- should also handle for ones left in temp too? */
            if (STATS.FinishedDownloadsAreShared)
                STATS.Shares->addNewFileRecordFromPath(STATS.connections[i].FileName,
                    STATS.connections[i].FileDirectory);
            // FIX -- NEED To notify the server of the share and update the total displayed!
        }
        else
        {
            printf("move failed\n");
            if (errno == EACCES)
                err_print("An access error occoured while moving a file from the temporary directory. Please report this error.  The file: %s, will be left in the temporary directory: %s\n", STATS.connections[i].FileName, STATS.connections[i].FileDirectory);
            else
                err_print("Download complete, but there was an unhandled error moving it to the completed downloads directory.  PLEASE REPORT THIS ERROR.\n");
        }
        
        #if AGDEBUGLEVEL > 4
        printf("attempted move from: %s \nto:%s\n",PathFrom, PathTo);
        #endif    
        
        if (STATS.setDisplayedFileSystemChange != NULL)
        {
            STATS.setDisplayedFileSystemChange(STATS.Prefs->IncompleteDownloadsDirectory);
            STATS.setDisplayedFileSystemChange(STATS.Prefs->FinishedDownloadsDirectory);
        }
        
        delete(PathFrom);
        delete(PathTo);
    }
    else
        printf("the finished directory exists, and the file is already there!\n");
    
    if (STATS.setDisplayedTransferComplete != NULL)
        STATS.setDisplayedTransferComplete(STATS.connections[i]);
    #if AGDEBUGLEVEL > 4
    else
        printf("not handling display stuff\n");
    #endif
}


int getUnusedFileName(char* &FileName, const char* FileDirectory)
{
    // FIX -- There is probably a better way to do this.
    int name_max = PATH_MAX - strlen(FileDirectory);
    if (name_max < 5)
    {
        err_print("Not enough space (%i) left for filename (%s) in %s!\n", name_max, FileName, FileDirectory);
        return (-1);
    }
        
    char* temp = new char[name_max];
    strcpy(temp, FileName);
    
    #if AGDEBUGLEVEL > 3
    printf("temp: %s\n", temp);
    #endif
    
    int bad_attempts = 0;
    errno = 0;
    
    char* OriginalDirectory = new char[PATH_MAX];
    if (getcwd(OriginalDirectory, PATH_MAX) == NULL)
    {
        delete(OriginalDirectory);
        delete(temp);
        err_print("Couldn't get original directory, bad.\n");
        return (-1); // is checked outside.
    }
    
    if (chdir(FileDirectory) != 0)
    {
        delete(OriginalDirectory);
        delete(temp);
        err_print("Couldn't change to the File Directory: %s to initiate search.\n", FileDirectory);
        return (-1); // is checked outside.
    }
    
    while((errno != ENOENT) && bad_attempts < 99) // errno != doesn't do anythign because of break below.
    {
        //char* temp2 = new char[PATH_MAX];
        
        if ( (access(temp, F_OK) == 0) || (errno != ENOENT) )
        { 
            printf("Couldn't use the filename: %s\n", temp);
            printf("errno = %i\n, meaning: %s", errno, strerror(errno));
            
            delete(temp);
            bad_attempts++;
                   
            // make the filename bigger.
            int len = strlen(FileName);
            // deleted above.
            temp = new char[name_max];
            char* otherTemp = new char[name_max];
            memset(otherTemp, '\0', name_max);
            strncpy(otherTemp, (const char *)FileName, len - 4 );
            snprintf(temp, name_max, "%s %i%s",otherTemp, bad_attempts, FileName + len - 4);
            delete(otherTemp);
            
            #if AGDEBUGLEVEL > 2
            printf("new name: %s\n", temp);
            #endif
        }
        else
        {
            //delete(temp2); // temp deleted below.
            #if AGDEBUGLEVEL > 3
            printf("breaking...\n");
            #endif
            break;
        }
    }
    if (chdir(OriginalDirectory) != 0)
        err_print("error changing back to original directory following getUsedFileName\n This is most likely because you removed it... this may cause funny things to happen.. who knows....\n");
        // don't care if it succededs.
    delete(OriginalDirectory);
    
    if (bad_attempts >= 99)
    {
        delete(temp);
        err_print("Ran out of attempts, This is A VERY RARE error.  Please clean out your temp directory and report.\n");
        return (-1);
    }
    delete(FileName); // unless it was changed outside, this should be fine.
    FileName = strdup(temp);
    delete(temp);
    return 0;
}

