/*
    AGConnection.cpp
    OpenAG, libOpenAG, OpenAG X
 
    Created by Eric Seidel on Sun Dec 02 2001.
 
    Copyright (c) 2001-2002 Eric Seidel. All rights reserved.
 
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "common.h"
#include "AGConnection.h"
#include "ParseMessage.h"

struct timezone tzdontcare = {0,0}; // FIX -- is this allowed?


/* CONNECTION FUNCTIONS */

int getNewConnectionRecord(server_status &STATS)
{
    for (int x = 1; x < MAXCONNECTIONS ; x++) // 0 should be reserved for INFOSOCKET
    {
        if (STATS.connections[x].flags == 0)
        {
            if ( (x+1) > STATS.nConnections ) STATS.nConnections = (x+1);
            
            //bzero(STATS.connections[x],sizeof(connection)); // just for safty...should be ok
            // we could... but this might cause memory leaks... if delete is bad...
            
            STATS.connections[x].IwPtr = STATS.connections[x].IrPtr =  STATS.connections[x].Ibuf;
            STATS.connections[x].OwPtr = STATS.connections[x].OrPtr =  STATS.connections[x].Obuf;
            #if AGDEBUGLEVEL > 4
            printf("IPTRs initialized to: w: %p r: %p b: %p\n",STATS.connections[x].IwPtr,
                    STATS.connections[x].IrPtr,STATS.connections[x].Ibuf);
            printf("OPTRs initialized to: w: %p r: %p b: %p\n",STATS.connections[x].OwPtr,
                    STATS.connections[x].OrPtr,STATS.connections[x].Obuf);
            #endif
            STATS.connections[x].hasOpenFile = false;
	    // hopefully these next few will prevent SEGFAULTS and not have any memoryleaks.
            
	    STATS.connections[x].FileID  		= NULL;
	    STATS.connections[x].ServerIP  		= NULL;
	    STATS.connections[x].FileName   		= NULL;
	    STATS.connections[x].FileDirectory	 	= NULL;
            
            STATS.connections[x].socket 		= (-1);
            STATS.connections[x].file 			= (-1);
            STATS.connections[x].hasOpenFile 		= false;
            STATS.connections[x].LocalID 		= (-1);

            STATS.connections[x].FileResumePosition 	= 0;
            STATS.connections[x].TimeOut 		= DEFAULT_TIMEOUT;
            STATS.connections[x].lastTime 		= time(NULL);
            STATS.connections[x].flags			= NOT_IN_USE;
            
            return x;
        }
    }
    return -1;
}

void deleteAllConnectionRecords(server_status &STATS)
{
    while (STATS.nConnections > 0)
    {
        if (STATS.connections[STATS.nConnections - 1].flags != NOT_IN_USE)
            deleteConnectionRecord(STATS.nConnections - 1, STATS);
        else
            STATS.nConnections--;
    }
}


void deleteConnectionRecord(int i, server_status &STATS)
{
    //#if AGDEBUGLEVEL > 2
    printf("REMOVING CONNECTION RECORD #%i with flags: %i\n", i, STATS.connections[i].flags);
    //#endif
    int which = i;
    #if AGDEBUGLEVEL > 4
    printf("updating maxfd\n");
    #endif
    /* degcrease max STATS.connections[i].socket, twice needed to check both */
    if ( (STATS.maxfd == STATS.connections[which].socket) || (STATS.maxfd == STATS.connections[which].file))
        STATS.maxfd--;
    if ( (STATS.maxfd == STATS.connections[which].socket) || (STATS.maxfd == STATS.connections[which].file))
        STATS.maxfd--;
    /*	this still could allow this number to get much higher than needed, if maxfd-1 is not in use... */
    
    #if AGDEBUGLEVEL > 4
    printf("unsetting socket: %i from r/w sets\n", STATS.connections[which].socket);
    #endif    
    /* Close socket if still open*/
    if(STATS.connections[which].socket >= 0)
    {
        if (close(STATS.connections[which].socket) != 0)
            printf("there was an error tryign to close socket %i for connection %i\n", 
                STATS.connections[which].socket, which);
        
        /* MAKE SURE THAT BOTH ARE REMOVED FROM READ/WRITE*/
        FD_CLR(STATS.connections[which].socket, &(STATS.rset));
        FD_CLR(STATS.connections[which].socket, &(STATS.wset));
        
        STATS.connections[which].socket = (-1);
    }
    
    FileRecord* incomplete = NULL;
    
    if (STATS.connections[which].LocalID > -1)
         incomplete = STATS.Resumes->getFileRecordByLocalID(STATS.connections[which].LocalID);
            // FIX -- should I lookup by Filename and size instead?
    
    if (incomplete != NULL)
        incomplete->ConnectionPtr = NULL;
    
    if (STATS.connections[which].hasOpenFile)
    {
        if (STATS.connections[which].DirectionFlag == DIRECTION_READ)
        {
            if (!(STATS.connections[which].flags & TRANSFER_COMPLETE))
            {
                #if AGDEBUGLEVEL > 4
                printf("attempting to delete temporary file: %s\n", STATS.connections[which].FileName);
                #endif
                
                if (incomplete == NULL)
                {
                    err_print("SHOULDN'T BE HERE!!!!\n");
                    
                    incomplete = STATS.Resumes->getNewFileRecord();
                    
                    incomplete->DiskFileName = strdup(STATS.connections[i].FileName);
                    //don't want to, but have to:
                    incomplete->SuggestedFileName = strdup(STATS.connections[i].FileName);
                    
                    incomplete->DirectoryName 	= strdup(STATS.connections[i].FileDirectory);
                    incomplete->SongLength 	= STATS.connections[i].SongLength;
                    incomplete->FileSize  	= STATS.connections[i].FileSize;
                    incomplete->flags	  	= UNVALIDATED_SHARE; // needs to be set to something!  FIX
                    
                    printf("adding new partial file: %s with %i bytes of %i\n",
                        STATS.connections[which].FileName, STATS.connections[which].FileResumePosition,
                        STATS.connections[which].FileSize);
                }
                else
                {
                    if (incomplete->DiskFileName == NULL) // hasn't been created yet.
                        STATS.Resumes->deleteFileRecordWithLocalID(incomplete->LocalID);
                        // is this ok?  we used the other id before
                    else
                        printf("updating partial file: %s with %i bytes of %i\n",
                                STATS.connections[which].FileName, STATS.connections[which].FileResumePosition,
                                STATS.connections[which].FileSize);
                }
                
                if (incomplete->DiskFileName != NULL)
                    // no real need to check... but this prevents it from touching somethign not in use.
                    incomplete->FileResumePosition = STATS.connections[which].FileResumePosition;
                
                // don't delete file.
            }
            else
                if (STATS.connections[which].LocalID > -1)
                    STATS.Resumes->deleteFileRecordWithLocalID(STATS.connections[which].LocalID);
        }
        /* Close file if still open */
        if(STATS.connections[which].file >= 0)
        {
            close(STATS.connections[which].file);
            #if AGDEBUGLEVEL > 4
            printf("unsetting file: %i from r/w sets\n", STATS.connections[which].file);
            #endif
            FD_CLR(STATS.connections[which].file, &(STATS.rset));
            FD_CLR(STATS.connections[which].file, &(STATS.wset));
        }
    }
    
    #if AGDEBUGLEVEL > 4
    printf("doing misc memory cleanup\n");
    #endif
    /* clean up memory */
    if (STATS.connections[which].FileID != NULL && STATS.connections[which].FileID != 0)
        delete(STATS.connections[which].FileID);
    if (STATS.connections[which].ServerIP != NULL && STATS.connections[which].ServerIP != 0)
        delete(STATS.connections[which].ServerIP);
    if (STATS.connections[which].FileName != NULL && STATS.connections[which].FileName != 0)
        delete(STATS.connections[which].FileName);
    if (STATS.connections[which].FileDirectory != NULL && STATS.connections[which].FileDirectory != 0)
        delete(STATS.connections[which].FileDirectory);
    /* if we were doing dynamic buffers, we would clean here */
    /* otherwise, the buffers should clean themselves with bzero */
    
    #if AGDEBUGLEVEL > 4
    printf("zeroing connection record, size: %li\n", sizeof(connection));
    #endif
    /* zero out the connection */
    bzero(&(STATS.connections[which]), sizeof(connection) );
    /* FIX - is this really necessary? */
    STATS.connections[which].FileID = NULL;
    STATS.connections[which].ServerIP = NULL;
    STATS.connections[which].FileName = NULL;
    STATS.connections[which].FileDirectory = NULL;
    STATS.connections[which].socket = (-1);
    STATS.connections[which].file = (-1);
    STATS.connections[which].hasOpenFile = false;
    STATS.connections[which].LocalID = (-1);
    
    /* reset pointers as well -- done on delete and initialize.*/
    STATS.connections[which].IwPtr = STATS.connections[which].IrPtr =  STATS.connections[which].Ibuf;
    STATS.connections[which].OwPtr = STATS.connections[which].OrPtr =  STATS.connections[which].Obuf;
    
    #if AGDEBUGLEVEL > 4
    printf("updating nConnections\n");
    #endif
    /* update max number of STATS.connections */
    while (STATS.connections[STATS.nConnections-1].flags == NOT_IN_USE)
        {STATS.nConnections--; if (STATS.nConnections == NOT_IN_USE) break;}
}


int getKPS(int i, server_status &STATS)
{
    #if AGDEBUGLEVEL > 3
    printf("in getKPS function\n");
    #endif
    
    if( (&STATS == NULL) || (&STATS == 0) || (STATS.connections[i].flags == NOT_IN_USE) )
        return 0;
    /* FIX -- ONLY SLIGHTLY LESS CRUDE THAN BEFORE */
    timeval TimeOfThisKPS;
    timeval theDiff;
    
    gettimeofday(&TimeOfThisKPS, &tzdontcare);
    
    timersub(&TimeOfThisKPS,  &STATS.connections[i].TimeOfLastKPS, &theDiff);
    
    #if AGDEBUGLEVEL > 3
    printf("entering KPS: %li DataSince: %lu TimeOfLast: %i %f TimeThis: %i %f, diff: %i %f\n",
            STATS.connections[i].BPS, STATS.connections[i].DataSinceLastKPS,
            STATS.connections[i].TimeOfLastKPS.tv_sec, STATS.connections[i].TimeOfLastKPS.tv_usec/1000000.0,
            TimeOfThisKPS.tv_sec, TimeOfThisKPS.tv_usec/1000000.0,
            theDiff.tv_sec, theDiff.tv_usec/1000000.0);
    #endif
    
    /* Calculate new KPS */
    if (timerisset(&STATS.connections[i].TimeOfLastKPS))
    {
        if ( theDiff.tv_sec > 0 ) // wait until it's been atleast a second before allowing recalculation
        {
            unsigned long temp = 0;
            
            #if AGDEBUGLEVEL > 5
            printf("BPS, before: %li\n",STATS.connections[i].BPS);
            
            temp = (STATS.connections[i].DataSinceLastKPS * 1000);
            unsigned long temp1 = ((theDiff.tv_sec * 1000) + (theDiff.tv_usec/1000));
            unsigned long temp2 = temp/temp1;
            printf("top: %lu bottom: %lu sec: %i usec: %i calc: %lu\n",temp,temp1,theDiff.tv_sec, theDiff.tv_usec, temp2);
            temp = temp2;
            #else
            temp = (STATS.connections[i].DataSinceLastKPS * 1000)/
                                        ((theDiff.tv_sec * 1000) + (theDiff.tv_usec/1000));
            #endif
            
            /* handle the averaging of the last several KPSes */
            
            unsigned long oldKPS = STATS.connections[i].BPSARRAY[STATS.connections[i].currentBPSRecord];
            STATS.connections[i].BPSARRAY[STATS.connections[i].currentBPSRecord] = temp;
            
            if (oldKPS > 0)
                STATS.connections[i].BPSTotal -= oldKPS;
            STATS.connections[i].BPSTotal += temp;
            
            if (STATS.connections[i].BPSRecordCount < NUMBER_BPS_RECORDS)
                STATS.connections[i].BPSRecordCount++;
            
            STATS.connections[i].BPS = STATS.connections[i].BPSTotal / STATS.connections[i].BPSRecordCount;
            
            STATS.connections[i].currentBPSRecord = (++STATS.connections[i].currentBPSRecord) % 5;
            
            /* Reset KPS count */
            STATS.connections[i].DataSinceLastKPS = 0;
            STATS.connections[i].TimeOfLastKPS.tv_sec = TimeOfThisKPS.tv_sec;
            STATS.connections[i].TimeOfLastKPS.tv_usec = TimeOfThisKPS.tv_usec;
        }
    }
    else
    {
        STATS.connections[i].TimeOfLastKPS.tv_sec = TimeOfThisKPS.tv_sec;
        STATS.connections[i].TimeOfLastKPS.tv_usec = TimeOfThisKPS.tv_usec;
        
        STATS.connections[i].BPS = STATS.connections[i].DataSinceLastKPS;
    }
    #if AGDEBUGLEVEL > 5
    printf("BPS: %li\n",STATS.connections[i].BPS);
    #endif
    
    return STATS.connections[i].BPS;
}


/* UPKEEP FUNCTIONS */

void adjustBuffers(int i, server_status &STATS)
{
    doAdjustBuffer(STATS.connections[i].Ibuf,
                    STATS.connections[i].IrPtr,
                    STATS.connections[i].IwPtr,
                    AGGRESSIVE_BUFFER_CLEAR);
    doAdjustBuffer(STATS.connections[i].Obuf,
                    STATS.connections[i].OrPtr,
                    STATS.connections[i].OwPtr,
                    AGGRESSIVE_BUFFER_CLEAR);
    /* could do it direct, but this is convient, and for legacy purposes... */
}

void doAdjustBuffer(char* buf, char* &rPtr, char* &wPtr, bool aggressiveClear)
{
    #if AGDEBUGLEVEL > 4
    printf("adjusting buffers.\n");
    #endif
    if (rPtr == wPtr)
        rPtr = wPtr = buf;
    else if (aggressiveClear) /* aggressive buffer clearing. */
    {
        #if AGDEBUGLEVEL > 3
        printf("DOING AGGRESSIVE BUFFER CLEAR\n");
        #endif
        int size = wPtr - rPtr;
        
        if (size < 0)
            err_exit("Connection Buffer is corrupt!!!  Data Size returned: %i\n", size);
        
    #if AGDEBUGLEVEL > 4
	printf("%i bytes still in buffer.\n", size);
    #endif
        if(size < BUFFER_SIZE)
        {
            #if AGDEBUGLEVEL > 4
            printf("about to create temp char array of size: %i\n", size);
            #endif
            char* temp;
            #if AGDEBUGLEVEL > 4
            printf("got a pointer, now filling with size: %i\n", size);
            #endif
            temp = new char[BUFFER_SIZE];
            #if AGDEBUGLEVEL > 4
            printf("allocated, this next one might be trouble\n");
            printf("about to copy data into temp T:%p rP:%p s:%i b+s:%p\n", temp, rPtr, size, buf+size);
            #endif
            memcpy(temp, rPtr, size); //or should it be move?
            wPtr = buf + size;
            rPtr = buf;
        #if AGDEBUGLEVEL > 4
            printf("buffer check, r: %p w: %p b:%p\n", rPtr,wPtr, buf);
        #endif
            memcpy(rPtr, temp , size); //could be optimized into one move if they don't overlap?? right?
            delete(temp);
        }
        #if AGDEBUGLEVEL > 4
        else
            printf("BUFFER FULL, can't clear any space.\n");
        #endif
    }
}
