;;###########################################################################
;; dataobj4.lsp
;; Copyright (c) 1991-98 by Forrest W. Young
;; This file contains methods for importing data. 
;; Together, dataobj1.lsp, dataobj2.lsp, dataobj3.lsp, and dataobj4.lsp
;; contain all the code to implement multivariate data objects.
;;###########################################################################

;;###########################################################################
;;Define methods for importing data
;;###########################################################################

;fwy4.25 following function redefined to include optional file
;fwy4.27 reworked to allow importing data with NIL elements.

(defun import-data (&optional file)
"Args: (&optional FILE)
Imports flat, rectangular ascii files as multivariate data. Data are read from file FILE, if specified. If FILE not specified a dialog box is presented. Data elements must be separated by white space. Number of variables is set equal to the number of data elements in the first record. Each record must have the same number of elements. Data elements may be numbers, symbols (i.e., unquoted alphanumerics) or strings (quoted alphanumerics). Strings may contain white space, symbols may not. Symbols are converted to upper case, strings retain original case. Variables are Numeric if they have no string or symbol values, otherwise they are Category. NIL data-elements can be entered as the symbol nil or NIL. The string "nil" (regardless of case) is interpreted as a string, not as a nil data-element."
  (let* ((data nil)
         (unw (unwind-protect
               (setf data 
                     (if file (read-data-columns file) (read-data-columns)))
               (when (not data)
                     (help 'import-data) 
                     (fatal-message "This file does not contain data that can be imported. See the listener for more complete information.") )
               ))
         (d nil)
         (datum nil)
         (nvar (length data))
         (nobs (length (first data)))
         (types (repeat "Numeric" nvar))
         )
    (when data
          (dotimes (i nvar)
           (dotimes (j nobs)
            (setf datum (select (select data i) j))
            (when (or (symbolp datum) (stringp datum))
                  (when (not (equal datum nil))
                        (setf (select (select data i) j)
                              (string (select (select data i) j)))
                        (setf (select types i) "Category"))))))
    (when data
          (setf data 
                (data "Imported-Data"
                      :title "Imported Data"
                      :types types
                      :variables (mapcar #'(lambda (x) 
                                    (format nil "Var~a" x)) (iseq nvar))
                      :data (combine (transpose (matrix (list nvar nobs) 
                                                       (combine data))))))
          (setf d (send data :data))
          (dotimes (i (length d))
                   (when (equal "NIL" (select d i)) 
                         (setf (select d i) NIL)))
          (browse-data data))
    data))

;;###########################################################################
;;Define methods for browsing, editing, summarizing and reporting data.
;;###########################################################################

(defmeth mv-data-object-proto :browse-data ()
  (if (not (eq *current-object* self)) (setcd self))
  (browse-data self))

(defmeth mv-data-object-proto :edit-data ()
  (if (not (eq *current-object* self)) (setcd self))
  (edit-data))

(defmeth mv-data-object-proto :report-data ()
  (if (not (eq *current-object* self)) (setcd self))
  (report-data))

(defmeth mv-data-object-proto :summarize-data (&rest args)
  (if (not (eq *current-object* self)) (setcd self))
  (apply #'summarize-data args))


(defmeth mv-data-object-proto :var-namelist (&key (show t))
"Method Args: none
Presents a namelist window containing a list of variable names."
  (let* ((var-window (name-list (send self :variables) 
                                :show nil
                              ; :go-away nil
                                :location location13
                                :title "Variables"))
         )
    (send var-window :size 
          (- (select namelist-size 0) msdos-fiddle)
          (- (select namelist-size 1) msdos-fiddle))
    (when show (send var-window :show-window))
    (send var-window :fix-name-list)
   ; (send var-window :has-h-scroll (max (screen-size)))
   ; (send var-window :has-v-scroll (max (screen-size)))
    (send var-window :redraw)
    var-window))

(defmeth mv-data-object-proto :obs-namelist (&key (show t))
"Method Args: none
Presents a window containing a list of observation labels."
  (let ((obs-window (name-list (send self :active-labels)
                               :show nil
                               :location location13
                               :title "Labels"))
        )
       
    (send obs-window :size 
          (- (select namelist-size 0) msdos-fiddle) long-namelist-length)
    (when show (send obs-window :show-window))
    (send obs-window :fix-name-list)
    ;(send obs-window :has-h-scroll (max (screen-size)))
    ;(send obs-window :has-v-scroll (max (screen-size)))
    (send obs-window :redraw)
    obs-window))
