;;#########################################################################
;; dataobj2.lsp
;; Copyright (c) 1991-98 by Forrest W. Young
;; This file contains the data object's menu methods and conversion methods. 
;; Together, dataobj1.lsp, dataobj2.lsp, dataobj3.lsp and dataobj4 contain
;; all the code to implement multivariate data objects.
;;#########################################################################

;;-------------------------------------------------------------------------
;; slot accessor methods (continued)
;;-------------------------------------------------------------------------

(defmeth mv-data-object-proto :dob-parents 
                                    (&optional (object-id-list nil set))
"Message args: (&optional object-id-list)
 Sets or retrieves the list of parent data and model objects." 
  (if set (setf (slot-value 'dob-parents) object-id-list))
  (slot-value 'dob-parents))

(defmeth mv-data-object-proto :dob-children 
                                  (&optional (object-id-list nil set))
"Message args: (&optional object-id-list)
 Sets or retrieves the list of children data and model objects." 
  (if set (setf (slot-value 'dob-children) object-id-list))
  (slot-value 'dob-children))

(defmeth mv-data-object-proto :guidemap-number (&optional (number nil set))
"Message args: (&optional number)
 Sets or retrieves the guidemap number (just for identification)." 
  (if set (setf (slot-value 'guidemap-number) number))
  (slot-value 'guidemap-number))

(defmeth mv-data-object-proto :guidemap-ancestors 
  (&optional (number-list nil set))
"Message args: (&optional number-list)
Sets or retrieves the guidemap ancestors list. This is a list of guidemap numbers created by link and used by return to traverse the guidemap hypertext." 
  (if set (setf (slot-value 'guidemap-ancestors) number-list))
  (slot-value 'guidemap-ancestors))

(defmeth mv-data-object-proto :add-parent (parent-object)
  (send self :dob-parents 
      (add-element-to-list (send self :dob-parents) parent-object)))

(defmeth mv-data-object-proto :add-child (child-object)
  (send self :dob-children 
        (add-element-to-list (send self :dob-children) child-object)))

(defmeth mv-data-object-proto :needs-computing (&optional (logical nil set))
"Message args: (&optional logical)
 Sets or retrieves whether the object needs computing (t) or not (nil)." 
  (if set (setf (slot-value 'needs-computing) logical))
  (slot-value 'needs-computing))

(defmeth mv-data-object-proto :auto-compute (&optional (logical nil set))
"Message args: (&optional logical)
 Sets or retrieves whether the object automatically re-computes (t) or not (nil) when needs-computing is set to t." 
  (if set (setf (slot-value 'auto-compute) logical))
  (slot-value 'auto-compute))

(defmeth mv-data-object-proto :datasheet-object 
  (&optional (object-id nil set))
"Message args: (&optional object-id)
 Sets or retrieves the object id of the data objects datasheet." 
  (if set (setf (slot-value 'datasheet-object) object-id))
  (slot-value 'datasheet-object))

(defmeth mv-data-object-proto :datasheet-arguments
  (&optional (list nil set))
"Message args: (&optional list)
 Sets or retrieves the list of datasheet arguments." 
  (if set (setf (slot-value 'datasheet-arguments) list))
  (slot-value 'datasheet-arguments))

(defmeth mv-data-object-proto :datasheet-open 
  (&optional (logical nil set))
"Message args: (&optional logical)
 Sets or retrieves whether the datasheet is open." 
  (if set (setf (slot-value 'datasheet-open) logical))
  (slot-value 'datasheet-open))

;;-------------------------------------------------------------------------
;; methods to convert mv to table data
;;-------------------------------------------------------------------------

(defmeth mv-data-object-proto :make-table-data 
                         (response-variable-name &optional menu-name)
;1 make new labels from category variables
  (let* ((cat-matrix (send self :active-data-matrix '(category)))
         (cat-variables  (send self :active-variables '(category)))
         (cell-labels (make-cell-labels cat-matrix cat-variables))
         (ungrouped-resp-var (send self :variable response-variable-name))
         (nobs (length ungrouped-resp-var))
;2 sort new labels into order and create data cells
         (urv-mat (send self :active-data-matrix '(all)))
         (sorted-labels-and-grouped-data 
          (sort-labels-and-group-data cell-labels urv-mat))
;3 create new table data object 
         (table nil))
    (when (not menu-name) 
          (setf menu-name (strcat "Table-" (send self :title))))
    (setf table (data menu-name
                      :created (send *desktop* :selected-icon)
                      :data (second sorted-labels-and-grouped-data)
                      :variables (list response-variable-name)
                      :ways cat-variables
                      :classes (third sorted-labels-and-grouped-data)))
    table))

(defun make-cell-labels (cat-matrix cat-variables)
  (let ((row nil)
        (labels nil)
        (string "")
        (nobs (first (size cat-matrix)))
        (nvar (second (size cat-matrix)))
        (value nil)
        )
    (dotimes 
     (i nobs)
     (setf row (row (send current-data :active-data-matrix '(category)) i))
     (setf string "")
     (dotimes 
      (j nvar)
      (setf value (select cat-matrix i j))
      (when (numberp value) 
            (setf value (format nil "~s" (select cat-matrix i j))))
      (setf string (strcat string 
           (select cat-variables j) "[" value "] ")))
     (setf labels (add-element-to-list labels string)))
    labels))  

(defun sort-labels-and-group-data (cell-labels ungrouped-data)
"Args: cell-labels ungrouped-data
Cells-labels is a list of cell-labels and ungrouped-data is a data-matrix. Sorts table cell labels into order and uses the sorted cell labels to make a new table data cell list and classes list . Returns a list with four element. The elements are the sorted labels, the data cell list, the sorted classes list, and a matrix of the sorted ungrouped data."
  (let* ((sorted-table (sort-and-permute-dob 
                        ungrouped-data cell-labels cell-labels nil))
         (sorted-data (first sorted-table))
         (sorted-resp-var (combine (col sorted-data 0)))
         (sorted-labels (second sorted-table))
         (nobs (length sorted-labels))
         (nways (1- (second (size ungrouped-data)))) 
         (data-cell-list nil)
         (classes-list nil)
         (start 0)
         (finish nil)) 
    (dotimes (i (1- nobs))
       (when (not (equal (select sorted-labels i) 
                         (select sorted-labels (1+ i))))
             (setf finish i)
             (setf data-cell-list (add-element-to-list data-cell-list 
                          (select sorted-resp-var (iseq start finish))))
             (setf start (1+ i))))
    (setf data-cell-list (add-element-to-list data-cell-list 
                         (select sorted-resp-var (iseq start (1- nobs)))))
    (dotimes (i nways)
             (setf classes-list (add-element-to-list classes-list
                   (remove-duplicates (combine (col sorted-data (1+ i)))
                                  :test 'equal))))
    (list sorted-labels data-cell-list classes-list (first sorted-table))))

;fwy 4.32 11/13/97 added for converting data for visualization and report
;fwy 4.32 11/23/97 modified for use by create-data
(defmeth mv-data-object-proto :mv-to-table 
  (&key can-convert? make-only-intermediate-mv)
"Method Args: can-convert? make-only-intermediate-mv
Converts multivariate data to table data. MV data must have only one active numeric or ordinal variable, but may have any number of active category variables (at least one). If necessary, an intermediate MV classification data object is created with the numeric or ordinal variable first, and the category variables following. When CAN-CONVERT? is t, the data are only checked to see if they can be converted. When MAKE-ONLY-INTERMEDIATE-MV is t, only the intermediate mv is made (if needed)."

  (let ((nnumord (send self :active-nvar '(numeric ordinal)))
        (ncat (send self :active-nvar '(category)))
        (numpos (position "Numeric" (send self :active-types '(all)) 
                           :test #'equal))
        (ordpos (position "Ordinal" (send self :active-types '(all)) 
                          :test #'equal))
        (catpos (position "Category" (send self :active-types '(all)) 
                          :test #'equal))
        (result nil)
        )
    
    (cond
      (can-convert? 
       ;do this when only want to know if can be converted
       (if (or (> nnumord 1) (= ncat 0))
           (setf result nil)
           (setf result t)))
      ((or (> nnumord 1) (= ncat 0))
       (error-message "These data cannot be converted to table data."))
      (t
       ;If data convertable, create intermediate mv data 
       ;with numeric variable first, if necessary
       
       (when (not (< (min (adjoin numpos ordpos)) catpos))
             (data (strcat "Cls-" (send self :name))
                   :variables 
                   (combine 
                    (send self :active-variables '(numeric ordinal))
                    (send self :active-variables '(category)))
                   :labels (send self :labels)
                   :created (send *workmap* :selected-icon)
                   :types (combine 
                           (send self :active-types '(numeric ordinal))
                           (send self :active-types '(category)))
                   :data (combine 
                          (bind-columns (send self :active-data-matrix 
                                              '(numeric ordinal))
                                        (send self :active-data-matrix 
                                              '(category))))))
       ;now create the table data
       (when (not make-only-intermediate-mv)
             (when *guidemap*
                   (when (send *guidemap* :gui)
                         (error-message "Guidemaps do not work for classification data in this release. They are being turned off.")
                         (send *guidemap* :close)))
           (setf result 
                 (create-data (strcat "Tab-" (send self :name)) :table t)))
       ))
    result))

;;---------------------------------------------------------------------------
;;data menu methods 
;;---------------------------------------------------------------------------
     
(defmeth mv-data-object-proto :set-menu&tool-states (data-mode)
"Method Args: data-mode
Sets states of tools to gray or normal depending on data-mode. Mode may be a string which is Table, Matrix, MV, Enabled or Disabled." 
  (let ((tools (send *toolbox* :icon-list)))
    (cond 
      ((equal data-mode "Enabled")
       (send *data-menu*  :enabled t) 
       (send *trans-menu* :enabled t)
       (send *tools-menu* :enabled t)
       (send *model-menu* :enabled t)
       (send (select tools 0) :icon-state "normal") ;help
       (send (select tools 1) :icon-state "gray")   ;anova  
       (send (select tools 2) :icon-state "gray")   ;coresp
       (send (select tools 3) :icon-state "gray")   ;mdscal
       (send (select tools 4) :icon-state "gray")   ;mulreg
       (send (select tools 5) :icon-state "gray")   ;nonpar
       (send (select tools 6) :icon-state "gray")   ;prncmp
       (send (select tools 7) :icon-state "gray")   ;regres
       (send (select tools 8) :icon-state "gray")   ;univar
       (mapcar #'(lambda (i)
              (send (select (send *data-menu* :items) i) :enabled t))
          (iseq   (length  (send  *data-menu* :items))))
       (mapcar #'(lambda (i)
              (send (select (send *trans-menu* :items) i) :enabled t))
          (iseq  (length  (send *trans-menu* :items)) ))
       (mapcar #'(lambda (i)
              (send (select (send *tools-menu* :items) i) :enabled t))
          (iseq (length  (send *tools-menu* :items)) ))
       (mapcar #'(lambda (i)
              (send (select (send *model-menu* :items) i) :enabled t))
          (iseq (length   (send *model-menu* :items)) ))
       )
      ((equal data-mode "MV")
       (send *vista* :menu-states data-mode)
       (send *data-menu*  :enabled t) 
       (send *trans-menu* :enabled t)
       (send *tools-menu* :enabled t)
       (send (select tools 0) :icon-state "normal")  ;help
       (send (select tools 1) :icon-state "normal")  ;anova  
       (send (select tools 2) :icon-state "normal")  ;coresp
       (send (select tools 3) :icon-state "gray")    ;mdscal
       (send (select tools 4) :icon-state "normal")  ;mulreg
       (send (select tools 5) :icon-state "gray")    ;nonpar
       (send (select tools 6) :icon-state "normal")  ;prncmp
       (send (select tools 7) :icon-state "normal")  ;regres
       (send (select tools 8) :icon-state "normal")  ;univar
       )
      ((equal data-mode "Table")
       (send *vista* :menu-states data-mode)
       (send *data-menu*  :enabled t) 
       (send *trans-menu* :enabled nil)
       (send *tools-menu* :enabled t)
       (send (select tools 0) :icon-state "normal") ;help
       (send (select tools 1) :icon-state "normal") ;anova
       (send (select tools 2) :icon-state "gray")   ;coresp
       (send (select tools 3) :icon-state "gray")   ;mdscal
       (send (select tools 4) :icon-state "gray")   ;mulreg
       (send (select tools 5) :icon-state "gray")   ;nonpar
       (send (select tools 6) :icon-state "gray")   ;prncmp
       (send (select tools 7) :icon-state "gray")   ;regres
       (send (select tools 8) :icon-state "gray") ;univar
       )
      ((equal data-mode "Matrix")
       (send *vista* :menu-states data-mode)
       (send *data-menu*  :enabled t) 
       (send *trans-menu* :enabled nil)
       (send *tools-menu* :enabled t)
       (send (select tools 0) :icon-state "normal") ;help
       (send (select tools 1) :icon-state "gray")   ;anova
       (send (select tools 2) :icon-state "gray")   ;coresp
       (send (select tools 3) :icon-state "normal") ;mdscal
       (send (select tools 4) :icon-state "gray")   ;mulreg
       (send (select tools 5) :icon-state "gray")   ;nonpar
       (send (select tools 6) :icon-state "gray")   ;prncmp
       (send (select tools 7) :icon-state "gray")   ;regres
       (send (select tools 8) :icon-state "gray")   ;univar
       )
      ((equal data-mode "Disabled")
       (send *data-menu*  :enabled nil) 
       (send *trans-menu* :enabled nil)
       (send *tools-menu* :enabled nil)
       (send *model-menu* :enabled nil)
       (send (select tools 0) :icon-state "normal")
       (send (select tools 1) :icon-state "gray")
       (send (select tools 2) :icon-state "gray")
       (send (select tools 3) :icon-state "gray")
       (send (select tools 4) :icon-state "gray")
       (send (select tools 5) :icon-state "gray")
       (send (select tools 6) :icon-state "gray")
       (send (select tools 7) :icon-state "gray")
       (send (select tools 8) :icon-state "gray")
       ))
    t))
        
(defmeth mv-data-object-proto :save-data (&optional file dont-save closing)
"Args: (&optional file dont-save closing)
FILE is a string. If datasheet is open, editable, and edited, the datasheet is saved so that the dataobject is updated (if the dataobject has family, a new dataobject is created). Then, unless DONT-SAVE is T, the updated or new data-object is written to FILE.lsp in a form suitable for use with the open-data command. CLOSING is t when datasheet being closed as well." 
  (let ((closed t)
        (dsob (send *current-data* :datasheet-object))
        )
    (when dsob (send dsob :error-check))
    (when (not dont-save)
          (set-working-directory *user-dir-name*)
          (when 
           (not file) 
           (setf file
#+macintosh(set-file-dialog "Save Data as File:" "" t)
#+msdos    (set-file-dialog "Save Data as File:")
#+X11    (if file (file-save-dialog "Save Data as File..." "*.lsp" "." file )
#+X11        (file-save-dialog "Save Data in File..." "*.lsp" "."))
             ))
          (when file (setf *user-dir-name* (get-working-directory))))
    (when (and (send self :datasheet-open)
               (send (send current-data :datasheet-object) :editable))
          (setf closed (send (send self :datasheet-object) 
                             :save-datasheet t closing)))
    (when (and file (not dont-save))
          (when closed
                (when dsob (send dsob :save-datasheet-arguments))
                (when (and *datasheet* (not (equal dsob *datasheet*)))
                      (send *datasheet* :save-datasheet-arguments))
                (setf file (string-downcase-if-not-X11 file))
                (when (and (> (length file) 3)
                           (string= ".lsp" file
                                    :start2 (- (length file) 4)))
                      (setf file (string-right-trim "lsp" file))
                      (setf file (string-right-trim "." file)))
                (format t "; saving ~s~%" file)
                (let ((f (open (strcat (string file) ".lsp") 
                               :direction :output))
                      (oldbreak *breakenable*))
                  (setq *breakenable* nil)
                  (send *current-data* :save-data-template f)
                  (setq *breakenable* oldbreak)
                  (close f)
                  (format t "; finished saving ~s~%" file)
                  f)))))

(defmeth mv-data-object-proto :save-data-template (f)
  (unwind-protect
   (print 
    `(data       ,(send self :name)
                 :title      ,(send self :title)
                 :about      ,(send self :about)
                 :variables ',(send self :active-variables '(all))
                 :types     ',(send self :active-types '(all))
                 :labels    ',(send self :active-labels)
                 :data      ',(send self :active-data '(all))
                 :datasheet-arguments ',(send self :datasheet-arguments)
                 )
    f)))

(defmeth mv-data-object-proto :delete-data ()
"Message args: none
Deletes a data object and all its child objects."
  (error-message "Not yet implemented."))

(defmeth mv-data-object-proto :create-data (&optional name &key (table nil)) 
"Message args: (&optional name &key (table nil))
Creates a new data object from the current active data.  Can optionally convert mv-table data to table data. If NAME is specified, the data object is named NAME (a string) and, if table is t, is converted to be a table object. If NAME not specified a dialog is presented for name and table. Optionally removes rows of multivariate data that have missing values (those coded as symbol NIL). Returns object identification of the new data object."
  (if (not (eq current-object self)) (setcd self))
  (setf merge-dob nil)
  (let* ((result nil)
         (menu-name name)
         (types (send self :active-types '(all)))
         (rest-types (remove-duplicates (rest types) :test 'equal))
         (numeric-var (first (send current-data :active-variables '(all))))
         (dataout (send self :active-data '(all)))
         (labelsout (send self :active-labels))
         (returned nil)
         )
    (when table
          (if (not (send self :mv-to-table :can-convert? t))
              (fatal-message "Cannot convert these data to table data.")))
    (when (not menu-name)
          (setf table (send self :mv-to-table :can-convert? t))
          (setf result (send self :close-dialog table))
          (setf menu-name (first result))
          (setf table (second result)))
    (when (and menu-name (= (length menu-name) 0)) 
          (error-message "You must specify a name."))
    (when (and menu-name (> (length menu-name) 0))
          ;check for missing values in multivariate data next
          (when (and (not table) 
                     (find 't (map-elements #'equal nil
                               (send current-data :active-data '(all)))))
                (when (two-button-dialog
        (format nil "Remove Rows with Missing~%Numeric or Ordinal Data?") 
                       :first-button "Remove" :second-button "Keep")
                      (setf returned (remove-missing-data-rows
                                      (send self :active-data-matrix '(all))
                                      :labels labelsout))
                      (setf dataout (combine (first returned)))
                      (setf labelsout (second returned))))
          (when (not (first dataout)) 
                (fatal-message "All rows contain missing values. New data not created."))

          (if table (send self :mv-to-table :make-only-intermediate-mv t))
          (if table
              (send *current-data* :make-table-data numeric-var menu-name)
              (data menu-name
                    :created (send *desktop* :selected-icon)
                    :title (strcat "Created from: " (send self :title))
                    :about (format nil "Created from data which said:~2%~a" (send self :about))
                    :data dataout
                    :variables (send self :active-variables '(all))
                    :labels labelsout
                    :types (send self :active-types '(all)))
              ))))

(defmeth mv-data-object-proto :can-make-table ()
"Message args: none
Returns t if data are convertable to table data, nil otherwise"
  (let* ((types (send self :active-types '(all)))
         (rest-types (remove-duplicates (rest types) :test 'equal))
         (numeric-var (first (send current-data :active-variables '(all)))))
    (and (equal (first   types) "Numeric")
         (= (length rest-types) 1)
         (equal (first rest-types) "Category"))))

(defmeth mv-data-object-proto :close-dialog (table)
"Gets name of new data object and whether it is to be a table or not."
  (let* ((title (send text-item-proto :new "Name of the New Data Object:"))
         (name (send edit-text-item-proto :new 
                     (strcat "Cr-" (send self :name))))
         (type-title (send text-item-proto :new "Make It:"))
         (type (send choice-item-proto :new (list
                     "Table Data"
                     "Multivariate Data")
                   ;  :location (list 84 72)
                     :value 0))
         (ok (send modal-button-proto :new "OK"
                   :action #'(lambda ()
                               (cond 
                                 (table (list (send name :text) 
                                              (= 0 (send type :value))))
                                 (t (list (send name :text) nil))))))
         (cancel (send modal-button-proto :new "Cancel"))
         (dialog (send modal-dialog-proto :new
                       (cond 
                         (table
                          (list title name type-title type (list ok cancel)))
                         (t (list title name (list ok cancel))))
                        :default-button ok))
         )
    (send dialog :modal-dialog)
    ))

(defun report-data ()
  (send current-data :report)
  t)

(defmeth mv-data-object-proto :report (&optional ok-types)
"Method Args:  (&optional ok-types)
Presents a listing of the data that includes the data object name, 
variable names and types and observation labels.  Reports only the active
ok-types data when the optional argument is used. Ok-types must be one of 
the following strings: all, numeric, ordinal, category, label. 
On Macintoshes the listing is presented in a separate window whose object 
identification is returned.  On other machines the listing is in the listener
window, T is returned."
  (if (not (eq current-object self)) (setcd self))
  (if (not ok-types) (setf ok-types '(all)))
  (let* ((data (send self :active-data ok-types))
         (n (length (send self :active-variables ok-types)))
         (m (/ (length data) n))
         (w nil)
         (dat-mat (matrix (list m n) data))
         (dat-lab (send self :active-labels))
         (both (bind-columns dat-lab dat-mat)))
    (setf w (report-header (strcat (send self :name) " Data Listing")))
    (display-string (format nil "Title: ~a" (send self :title)) w)
    (display-string
        (format nil "~%Data:  ~a~2%" (send self :name)) w)
    (display-string
        (format nil "Variable Names: ~a~%" 
                (send self :active-variables ok-types)) w)
    (display-string
        (format nil "Variable Types: ~a~2%" 
                (send self :active-types ok-types)) w)
    (display-string
          (format nil "Labeled Data Matrix: ~%") w)
    (print-matrix-to-window dat-mat w :labels dat-lab)
    w
    ))

(defun report ()
"Args: none
Generic function to ask an object for a report of itself"
  (send current-object :report))

(defmeth mv-data-object-proto :select-variables (var-name-list)
"Args: VAR-NAME-LIST
Selects the variables in VAR-NAME-LIST from the list of variables.  Displays
the selection in the variable window when it is open."
  (if (not (eq current-data self)) (setcd self));fwy4.25
  (let* ((w (send *vista* :var-window-object))
         (var-num-list ($position var-name-list (send self :variables)))
         (states (repeat 'NORMAL (send self :nvar)))
         )
    (cond 
      (w (send w :selection var-num-list)
         (send self :var-states 
               (send w :point-state (iseq (send w :num-points)))))
      (t (setf (select states var-num-list)
                     (repeat 'SELECTED (length var-num-list)))
         (send self :var-states states)))))

(defmeth mv-data-object-proto :select-observations (obs-label-list)
"Args: OBS-LABEL-LIST
Selects the observations in OBS-LABEL-LIST from the list of observation labels.  Displays the selection in the observation window when it is open."
  (if (not (eq current-data self)) (setcd self));fwy4.25
  (let* ((w (send *vista* :obs-window-object))
         (obs-num-list ($position obs-label-list (send self :labels)))
         (states (repeat 'NORMAL (send self :nobs)))
         )
    (cond 
      (w (send w :selection obs-num-list)
         (send self :obs-states 
               (send w :point-state (iseq (send w :num-points)))))
      (t (setf (select states obs-num-list)
                     (repeat 'SELECTED (length obs-num-list)))
         (send self :obs-states states)))))

(provide "dataobj2")