/*
 * marshall_tester.c
 */
/*
 * Copyright (c) of Carnegie Mellon University, 1997,1998,1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */


#include <nasd/nasd_options.h>
#include <nasd/nasd_types.h>
#include <nasd/nasd_error.h>
#include <nasd/nasd_drive_types.h>
#include <nasd/nasd_marshall.h>
#include <nasd/nasd_common.h>
#include <nasd/nasd_site.h>
#include <nasd/nasd_general.h>
#include <nasd/nasd_mq.h>
#include <nasd/nasd_marshall_tester.h>

int check_sizes(void);
int check_identifier(void);
int check_capability(void);
int check_security_param(void);
int check_timespec(void);
int check_drive_info(void);
int check_part_info(void);
int check_trace_basic(void);
int check_part_creat(void);
int check_part_creat_res(void);
int check_smpl_op(void);
int check_thrtl_op(void);
int check_getattr_args(void);
int check_getattr_res(void);
int check_setattr_args(void);
int check_setattr_res(void);
int check_create_args(void);
int check_create_res(void);
int check_flush_obj_args(void);
int check_flush_obj_res(void);
int check_eject_obj_args(void);
int check_eject_obj_res(void);
int check_initialize_args(void);
int check_initialize_res(void);
int check_change_key_args(void);
int check_change_key_res(void);
int check_trace_header(void);
int check_strt_iread_args(void);
int check_strt_iread_res(void);
int check_stop_iread_args(void);
int check_stop_iread_res(void);
#if NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_SRPC
int check_srpc_header(void);
#endif /* NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_SRPC */

int
main(
  int     argc,
  char  **argv)
{
  int fail = 0;
  
  if (check_sizes()) fail = 1;
  if (check_identifier()) fail = 1;
  if (check_security_param()) fail = 1;
  if (check_capability()) fail = 1;
  if (check_timespec()) fail = 1;
  if (check_drive_info()) fail = 1;
  if (check_part_info()) fail = 1;
/*
 * XXX check:
 * drive_opstats
 * cache_stat
 * layout_stat
 * ioqueue_stat
 * io_stat
 */
  if (check_trace_basic()) fail = 1;
  if (check_part_creat()) fail = 1;
  if (check_part_creat_res()) fail = 1;
  if (check_smpl_op()) fail = 1;
  if (check_thrtl_op()) fail = 1;
  if (check_getattr_args()) fail = 1;
  if (check_getattr_res()) fail = 1;
  if (check_setattr_args()) fail = 1;
  if (check_setattr_res()) fail = 1;
  if (check_create_args()) fail = 1;
  if (check_create_res()) fail = 1;
  if (check_flush_obj_args()) fail = 1;
  if (check_flush_obj_res()) fail = 1;
  if (check_eject_obj_args()) fail = 1;
  if (check_eject_obj_res()) fail = 1;
  if (check_initialize_args()) fail = 1;
  if (check_initialize_res()) fail = 1;
  if (check_change_key_args()) fail = 1;
  if (check_change_key_res()) fail = 1;
  if (check_trace_header()) fail = 1;
  if (check_strt_iread_args()) fail = 1;
  if (check_strt_iread_res()) fail = 1;
  if (check_stop_iread_args()) fail = 1;
  if (check_stop_iread_res()) fail = 1;
#if NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_SRPC
  if (check_srpc_header()) fail = 1;
#endif /* NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_SRPC */

  if (fail) {
    printf("Marshalling tests failed.\n");
    return 1;
  } else {
    printf("Marshalling tests succeeded.\n");
    return 0;
  }
}

int check_sizes(void) {
  int fail = 0;

  printf("Struct sizes:\n");
  printf("  sizeof(nasd_attribute_t) = %d (%d)\n", NASD_MT_SizeOf(nasd_attribute_t),
    NASD_ATTRIBUTE_SIZE);
  if (sizeof(nasd_attribute_t) != NASD_ATTRIBUTE_SIZE) {
    fail = 1;
    printf("problem: (size - nasd_attribute_t)\t %3d\n",
      NASD_MT_SizeOf(nasd_attribute_t));
  }
  printf("  sizeof(nasd_cookie_t) = %d\n", NASD_MT_SizeOf(nasd_cookie_t));
  printf("  sizeof(nasd_capability_t) = %d\n", NASD_MT_SizeOf(nasd_capability_t));
  printf("  sizeof(nasd_security_param_t) = %d\n", NASD_MT_SizeOf(nasd_security_param_t));
  printf("  sizeof(nasd_digest_nonce_t) = %d\n", NASD_MT_SizeOf(nasd_digest_nonce_t));
  printf("  sizeof(nasd_trace_header_t) = %d\n", NASD_MT_SizeOf(nasd_trace_header_t));
  printf("\n");

  /* control sizes */
  if (sizeof(nasd_ctrl_drive_info_t) != NASD_INFO_PAGESIZE) {
    fail = 1;
    printf("problem: (size - ctrl_drive_info)\t %3d\n",
	   NASD_MT_SizeOf(nasd_ctrl_drive_info_t));
  }

  if (sizeof(nasd_ctrl_part_info_t) != NASD_INFO_PAGESIZE) {
    fail = 1;
    printf("problem: (size - ctrl_part_info)\t %3d\n",
	   NASD_MT_SizeOf(nasd_ctrl_part_info_t));
  }

  if (sizeof(nasd_ctrl_drive_opstats_t) != NASD_INFO_PAGESIZE) {
    fail = 1;
    printf("problem: (size - ctrl_drive_opstats)\t %3d\n",
	   NASD_MT_SizeOf(nasd_ctrl_drive_opstats_t));
  }

  if (sizeof(nasd_ctrl_cache_stat_t) != NASD_INFO_PAGESIZE) {
    fail = 1;
    printf("problem: (size - ctrl_cache_stat)\t %3d\n",
	   NASD_MT_SizeOf(nasd_ctrl_cache_stat_t));
  }

  if (sizeof(nasd_ctrl_layout_stat_t) != NASD_INFO_PAGESIZE) {
    fail = 1;
    printf("problem: (size - ctrl_layout_stat)\t %3d\n",
	   NASD_MT_SizeOf(nasd_ctrl_layout_stat_t));
  }

  if (sizeof(nasd_ctrl_ioqueue_stat_t) != NASD_INFO_PAGESIZE) {
    fail = 1;
    printf("problem: (size - ctrl_ioqueue_stat)\t %3d\n",
	   NASD_MT_SizeOf(nasd_ctrl_ioqueue_stat_t));
  }

  if (sizeof(nasd_ctrl_io_stat_t) != NASD_INFO_PAGESIZE) {
    fail = 1;
    printf("problem: (size - ctrl_io_stat)\t %3d\n",
	   NASD_MT_SizeOf(nasd_ctrl_io_stat_t));
  }

  if (sizeof(nasd_ctrl_io_size_stat_t) != NASD_INFO_PAGESIZE) {
    fail = 1;
    printf("problem: (size - ctrl_io_size_stat)\t %3d\n",
	   NASD_MT_SizeOf(nasd_ctrl_io_size_stat_t));
  }

  if (sizeof(nasd_ctrl_trace_basic_t) != NASD_INFO_PAGESIZE) {
    fail = 1;
    printf("problem: (size - ctrl_trace_basic)\t %3d\n",
	   NASD_MT_SizeOf(nasd_ctrl_trace_basic_t));
  }

  /* sizes */
  
  if (sizeof(nasd_p_part_creat_dr_args_t) !=
      sizeof(nasd_p_part_creat_dr_args_otw_t)) {
    fail = 1;
    printf("problem: (size - p_part_creat_args)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_part_creat_dr_args_t),
	   NASD_MT_SizeOf(nasd_p_part_creat_dr_args_otw_t),
	   NASD_P_PART_CREAT_DR_ARGS_MAX);
  }
  
  if (sizeof(nasd_p_part_creat_dr_res_t) !=
      sizeof(nasd_p_part_creat_dr_res_otw_t)) {
    fail = 1;
    printf("problem: (size - p_part_creat_res)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_part_creat_dr_res_t),
	   NASD_MT_SizeOf(nasd_p_part_creat_dr_res_otw_t),
	   NASD_P_PART_CREAT_DR_RES_MAX);
  }
  
  if (sizeof(nasd_p_smpl_op_dr_args_t) !=
      sizeof(nasd_p_smpl_op_dr_args_otw_t)) {
    fail = 1;
    printf("problem: (size - p_smpl_op_args)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_smpl_op_dr_args_t),
	   NASD_MT_SizeOf(nasd_p_smpl_op_dr_args_otw_t),
	   NASD_P_SMPL_OP_DR_ARGS_MAX);
  }
  
  if (sizeof(nasd_p_thrtl_op_dr_args_t) !=
      sizeof(nasd_p_thrtl_op_dr_args_otw_t)) {
    fail = 1;
    printf("problem: (size - p_thrtl_op_args)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_thrtl_op_dr_args_t),
	   NASD_MT_SizeOf(nasd_p_thrtl_op_dr_args_otw_t),
	   NASD_P_THRTL_OP_DR_ARGS_MAX);
  }
  
  if (sizeof(nasd_p_setattr_dr_args_t) !=
      sizeof(nasd_p_setattr_dr_args_otw_t)) {
    fail = 1;
    printf("problem: (size - p_setattr_args)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_setattr_dr_args_t),
	   NASD_MT_SizeOf(nasd_p_setattr_dr_args_otw_t),
	   NASD_P_SETATTR_DR_ARGS_MAX);
  }
  
  if (sizeof(nasd_p_getattr_dr_args_t) !=
      sizeof(nasd_p_getattr_dr_args_otw_t)) {
    fail = 1;
    printf("problem: (size - p_getattr_args)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_getattr_dr_args_t),
	   NASD_MT_SizeOf(nasd_p_getattr_dr_args_otw_t),
	   NASD_P_GETATTR_DR_ARGS_MAX);
  }
  
  if (sizeof(nasd_p_getattr_dr_res_t) !=
      sizeof(nasd_p_getattr_dr_res_otw_t)) {
    fail = 1;
    printf("problem: (size - p_getattr_res)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_getattr_dr_res_t),
	   NASD_MT_SizeOf(nasd_p_getattr_dr_res_otw_t),
	   NASD_P_GETATTR_DR_RES_MAX);
  }
  
  if (sizeof(nasd_p_flush_obj_dr_args_t) !=
      sizeof(nasd_p_flush_obj_dr_args_otw_t)) {
    fail = 1;
    printf("problem: (size - p_flush_obj_args)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_flush_obj_dr_args_t),
	   NASD_MT_SizeOf(nasd_p_flush_obj_dr_args_otw_t),
	   NASD_P_FLUSH_OBJ_DR_ARGS_MAX);
  }
  
  if (sizeof(nasd_p_flush_obj_dr_res_t) !=
      sizeof(nasd_p_flush_obj_dr_res_otw_t)) {
    fail = 1;
    printf("problem: (size - p_flush_obj_res)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_flush_obj_dr_res_t),
	   NASD_MT_SizeOf(nasd_p_flush_obj_dr_res_otw_t),
	   NASD_P_FLUSH_OBJ_DR_RES_MAX);
  }
  
  if (sizeof(nasd_p_eject_obj_dr_args_t) !=
      sizeof(nasd_p_eject_obj_dr_args_otw_t)) {
    fail = 1;
    printf("problem: (size - p_eject_obj_args)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_eject_obj_dr_args_t),
	   NASD_MT_SizeOf(nasd_p_eject_obj_dr_args_otw_t),
	   NASD_P_EJECT_OBJ_DR_ARGS_MAX);
  }

  if (sizeof(nasd_p_eject_obj_dr_res_t) !=
      sizeof(nasd_p_eject_obj_dr_res_otw_t)) {
    fail = 1;
    printf("problem: (size - p_eject_obj_res)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_eject_obj_dr_res_t),
	   NASD_MT_SizeOf(nasd_p_eject_obj_dr_res_otw_t),
	   NASD_P_EJECT_OBJ_DR_RES_MAX);
  }

  if (sizeof(nasd_p_remove_dr_args_t) !=
      sizeof(nasd_p_remove_dr_args_otw_t)) {
    fail = 1;
    printf("problem: (size - p_remove_args)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_remove_dr_args_t),
	   NASD_MT_SizeOf(nasd_p_remove_dr_args_otw_t),
	   NASD_P_REMOVE_DR_ARGS_MAX);
  }
  
  if (sizeof(nasd_p_remove_dr_res_t) !=
      sizeof(nasd_p_remove_dr_res_otw_t)) {
    fail = 1;
    printf("problem: (size - p_remove_res)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_remove_dr_res_t),
	   NASD_MT_SizeOf(nasd_p_remove_dr_res_otw_t),
	   NASD_P_REMOVE_DR_ARGS_MAX);
  }
  
  if (sizeof(nasd_p_initialize_dr_args_t) != 
      sizeof(nasd_p_initialize_dr_args_otw_t)) {
    fail = 1;
    printf("problem: (size - p_initialize_args)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_initialize_dr_args_t),
	   NASD_MT_SizeOf(nasd_p_initialize_dr_args_otw_t),
	   NASD_P_INITIALIZE_DR_ARGS_MAX);
  }
  
  if (sizeof(nasd_p_initialize_dr_res_t) !=
      sizeof(nasd_p_initialize_dr_res_otw_t)) {
    fail = 1;
    printf("problem: (size - p_initialize_res)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_initialize_dr_res_t),
	   NASD_MT_SizeOf(nasd_p_initialize_dr_res_otw_t),
	   NASD_P_INITIALIZE_DR_RES_MAX);
  }
  
  if (sizeof(nasd_p_strt_iread_dr_args_t) != 
      sizeof(nasd_p_strt_iread_dr_args_otw_t)) {
    fail = 1;
    printf("problem: (size - p_strt_iread_args)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_strt_iread_dr_args_t),
	   NASD_MT_SizeOf(nasd_p_strt_iread_dr_args_otw_t),
	   NASD_P_STRT_IREAD_DR_ARGS_MAX);
  }
  
  if (sizeof(nasd_p_strt_iread_dr_res_t) !=
      sizeof(nasd_p_strt_iread_dr_res_otw_t)) {
    fail = 1;
    printf("problem: (size - p_strt_iread_res)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_strt_iread_dr_res_t),
	   NASD_MT_SizeOf(nasd_p_strt_iread_dr_res_otw_t),
	   NASD_P_STRT_IREAD_DR_RES_MAX);
  }
  
  if (sizeof(nasd_p_stop_iread_dr_args_t) != 
      sizeof(nasd_p_stop_iread_dr_args_otw_t)) {
    fail = 1;
    printf("problem: (size - p_stop_iread_args)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_stop_iread_dr_args_t),
	   NASD_MT_SizeOf(nasd_p_stop_iread_dr_args_otw_t),
	   NASD_P_STOP_IREAD_DR_ARGS_MAX);
  }
  
  if (sizeof(nasd_p_stop_iread_dr_res_t) !=
      sizeof(nasd_p_stop_iread_dr_res_otw_t)) {
    fail = 1;
    printf("problem: (size - p_stop_iread_res)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_stop_iread_dr_res_t),
	   NASD_MT_SizeOf(nasd_p_stop_iread_dr_res_otw_t),
	   NASD_P_STOP_IREAD_DR_RES_MAX);
  }
  
  if (sizeof(nasd_p_change_key_dr_args_t) !=
      sizeof(nasd_p_change_key_dr_args_otw_t)) {
    fail = 1;
    printf("problem: (size - p_change_key_args)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_change_key_dr_args_t),
	   NASD_MT_SizeOf(nasd_p_change_key_dr_args_otw_t),
	   NASD_P_CHANGE_KEY_DR_ARGS_MAX);
  }
  
  if (sizeof(nasd_p_change_key_dr_res_t) !=
      sizeof(nasd_p_change_key_dr_res_otw_t)) {
    fail = 1;
    printf("problem: (size - p_change_key_res)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_change_key_dr_res_t),
	   NASD_MT_SizeOf(nasd_p_change_key_dr_res_otw_t),
	   NASD_P_CHANGE_KEY_DR_RES_MAX);
  }
  
  if (sizeof(nasd_p_setattr_dr_args_t) !=
      sizeof(nasd_p_setattr_dr_args_otw_t)) {
    fail = 1;
    printf("problem: (size - p_setattr_args)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_setattr_dr_args_t),
	   NASD_MT_SizeOf(nasd_p_setattr_dr_args_otw_t),
	   NASD_P_SETATTR_DR_ARGS_MAX);
  }
  
  if (sizeof(nasd_p_setattr_dr_res_t) !=
      sizeof(nasd_p_setattr_dr_res_otw_t)) {
    fail = 1;
    printf("problem: (size - p_setattr_res)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_setattr_dr_res_t),
	   NASD_MT_SizeOf(nasd_p_setattr_dr_res_otw_t),
	   NASD_P_SETATTR_DR_RES_MAX);
  }
  
  if (sizeof(nasd_p_create_dr_res_t) !=
      sizeof(nasd_p_create_dr_res_otw_t)) {
    fail = 1;
    printf("problem: (size - p_create_res)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_create_dr_res_t),
	   NASD_MT_SizeOf(nasd_p_create_dr_res_otw_t),
	   NASD_P_CREATE_DR_RES_MAX);
  }
  
  if (sizeof(nasd_p_create_dr_args_t) !=
      sizeof(nasd_p_create_dr_args_otw_t)) {
    fail = 1;
    printf("problem: (size - p_create_args)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_p_create_dr_args_t),
	   NASD_MT_SizeOf(nasd_p_create_dr_args_otw_t),
	   NASD_P_CREATE_DR_ARGS_MAX);
  }

  if (sizeof(nasd_trace_header_t) !=
      sizeof(nasd_trace_header_otw_t)) {
    fail = 1;
    printf("(size - trace_header)\t %3d -> %3d (%3d)\n",
	   NASD_MT_SizeOf(nasd_trace_header_t),
	   NASD_MT_SizeOf(nasd_trace_header_otw_t),
	   NASD_TRACE_HEADER_SIZE);
  }

  if (fail)
    printf("Size check failed.\n\n");

  return fail;
}

int check_identifier(void) {
  nasd_identifier_t nid, nid_after;
  nasd_identifier_otw_t nid_otw;
  int fail = 0;

  /*
   * This code will twitch if the generated code
   * has alignment problems.
   */

{
  struct blah {  int foo; nasd_identifier_t nid; } foo1, foo2;
  struct blah_otw {  int foo; nasd_identifier_otw_t nid_otw; } foo_otw;

  foo1.foo = 0xb1a5;
  foo1.nid = nasd_uint64cast(0xfedcba9876543216);
  nasd_identifier_t_marshall(&foo1.nid, foo_otw.nid_otw);
  nasd_identifier_t_unmarshall(foo_otw.nid_otw, &foo2.nid);

  if (foo1.nid != foo2.nid) {
    fail = 1;
    printf("problem: nasd_identifier_t 0x%" NASD_ID_FMT
      " -> 0x%" NASD_ID_FMT " (1)\n",
      foo1.nid, foo2.nid);
  }
}

{
  struct blah {  char foo; nasd_identifier_t nid; } foo1, foo2;
  struct blah_otw {  char foo; nasd_identifier_otw_t nid_otw; } foo_otw;

  foo1.foo = 0xa5;
  foo1.nid = nasd_uint64cast(0xfedcba9876543216);
  nasd_identifier_t_marshall(&foo1.nid, foo_otw.nid_otw);
  nasd_identifier_t_unmarshall(foo_otw.nid_otw, &foo2.nid);

  if (foo1.nid != foo2.nid) {
    fail = 1;
    printf("problem: nasd_identifier_t 0x%" NASD_ID_FMT
      " -> 0x%" NASD_ID_FMT " (2)\n",
      foo1.nid, foo2.nid);
  }
}

  if (sizeof(nid) != 8) {
    fail = 1;
    printf("problem: sizeof nasd_identifier_t not \n");
  }

  if (sizeof(nasd_identifier_t) != sizeof(nasd_identifier_otw_t)) {
    fail = 1;
    printf("(size - nasd_identifier)\t %2d -> %2d (%2d)\n",
      NASD_MT_SizeOf(nasd_identifier_t),
      NASD_MT_SizeOf(nasd_identifier_otw_t),
      NASD_IDENTIFIER_T_SIZE);
  }

  nid = nasd_uint64cast(0xfedcba9876543216);

  nasd_identifier_t_marshall(&nid, nid_otw);
  nasd_identifier_t_unmarshall(nid_otw, &nid_after);

#if 0
{
  nasd_byte_t *t;
  int i;

  t = (nasd_byte_t *)nid_otw;
  printf("nasd_identifier_t 0x%" NASD_ID_FMT " -> 0x", nid);
  for(i=sizeof(nasd_identifier_otw_t);i>0;i--)
    printf("%02x", t[i-1]);
  printf(" -> 0x%" NASD_ID_FMT "\n", nid_after);
}
#endif

  if (nid != nid_after) {
    fail = 1;
    printf("problem: nasd_identifier_t 0x%" NASD_ID_FMT
      " -> 0x%" NASD_ID_FMT "\n",
      nid, nid_after);
  }

  nid = nasd_uint64cast(0xf1234567fedcba98);

  nasd_identifier_t_marshall(&nid, nid_otw);
  nasd_identifier_t_unmarshall(nid_otw, &nid_after);

  if (nid != nid_after) {
    fail = 1;
    printf("problem: nasd_identifier_t 0x%" NASD_ID_FMT
      " -> 0x%" NASD_ID_FMT "\n",
      nid, nid_after);
  }

  nid = nasd_uint64cast(0x8c20000000000080);

  nasd_identifier_t_marshall(&nid, nid_otw);
  nasd_identifier_t_unmarshall(nid_otw, &nid_after);

  if (nid != nid_after) {
    fail = 1;
    printf("problem: nasd_identifier_t 0x%" NASD_ID_FMT
      " -> 0x%" NASD_ID_FMT "\n",
      nid, nid_after);
  }

  if (fail) {
    printf("nasd_identifier_t check failed.\n\n");
  }

  return fail;
}

int check_timespec(void) {
  nasd_timespec_t ts, ts_after;
  nasd_timespec_otw_t ts_otw;
  int fail = 0;

  if (sizeof(nasd_timespec_t) != sizeof(nasd_timespec_otw_t)) {
    fail = 1;
    printf("(size - nasd_timespec_t)\t %2d -> %2d (%2d)\n",
      NASD_MT_SizeOf(nasd_timespec_t),
      NASD_MT_SizeOf(nasd_timespec_otw_t),
      NASD_TIMESPEC_T_SIZE);
  }

  ts.ts_sec = 1001;
  ts.ts_nsec = 5005;

  nasd_timespec_t_marshall(&ts, ts_otw);
  nasd_timespec_t_unmarshall(ts_otw, &ts_after);

  if (ts.ts_sec != ts_after.ts_sec) {
    fail = 1;
    printf("problem: ts_sec %d -> %d\n", ts.ts_sec, ts_after.ts_sec);
  }
  if (ts.ts_nsec != ts_after.ts_nsec) {
    fail = 1;
    printf("problem: ts_nsec %d -> %d\n", ts.ts_nsec, ts_after.ts_nsec);
  }

  if (fail) {
    printf("nasd_timespec_t check failed.\n\n");
  }

  return fail;
}

int check_capability(void) {
  nasd_capability_t capability, capability_after;
  nasd_capability_otw_t capability_otw;
  int fail = 0;

  capability.ni = 1234567;
  capability.expiration_seconds = 60*60;
  capability.rights = 100;
  capability.min_protection = 0x33;
  capability.AuditID = 60;
  capability.type = 6;
  capability.region_start = 1000;
  capability.region_end = 1100;
  capability.partnum = 3;

  nasd_capability_t_marshall(&capability, capability_otw);
  nasd_capability_t_unmarshall(capability_otw, &capability_after);

#define CHECKCAPABILITY_L(_x_,_fmt_) NASD_MT_CHECKBASIC(capability,_x_,_fmt_)

  CHECKCAPABILITY_L(ni,NASD_ID_FMT);
  CHECKCAPABILITY_L(expiration_seconds,"d");
  CHECKCAPABILITY_L(rights,"d");
  CHECKCAPABILITY_L(min_protection,"hu");
  CHECKCAPABILITY_L(AuditID,"hu");
  CHECKCAPABILITY_L(type,"hu");
  CHECKCAPABILITY_L(region_start,NASD_64s_FMT);
  CHECKCAPABILITY_L(region_end,NASD_64s_FMT);
  CHECKCAPABILITY_L(partnum,"hd");

#undef CHECKCAPABILITY_L

  if(fail)
    printf("Capability check failed.\n\n");

  return fail;
}

int check_security_param(void) {
  nasd_security_param_t security_param, security_param_after;
  nasd_security_param_otw_t security_param_otw;
  int fail = 0;

  security_param.partnum = 5;
  security_param.type = 6;
  security_param.actual_protection = 0x33;

  nasd_security_param_t_marshall(&security_param, security_param_otw);
  nasd_security_param_t_unmarshall(security_param_otw, &security_param_after);

#define CHECKSECURITY_PARAM(_x_,_fmt_) NASD_MT_CHECKBASIC(security_param,_x_,_fmt_)

  CHECKSECURITY_PARAM(partnum,"hd");
  CHECKSECURITY_PARAM(type,"hu");
  CHECKSECURITY_PARAM(actual_protection,"hu");

#undef CHECKSECURITY_PARAM

  if(fail)
    printf("Security param check failed.\n\n");

  return fail;
}

int check_drive_info(void) {
  nasd_ctrl_drive_info_t drive_info, drive_info_after;
  nasd_ctrl_drive_info_otw_t drive_info_otw;
  int fail = 0;

  /* ctrl_drive_info */
  drive_info.ctrl_id = 86;
  drive_info.max_parts = 87;
  drive_info.blocksize = 88;
  drive_info.num_parts = 89;
  drive_info.num_blocks = 90;
  drive_info.blocks_allocated = 90;
  
  nasd_ctrl_drive_info_t_marshall(&drive_info, drive_info_otw);
  nasd_ctrl_drive_info_t_unmarshall(drive_info_otw, &drive_info_after);

#define CHECKDRIVEINFO(_x_,_fmt_) NASD_MT_CHECKBASIC(drive_info,_x_,_fmt_)

  CHECKDRIVEINFO(ctrl_id,NASD_ID_FMT);
  CHECKDRIVEINFO(max_parts,NASD_64u_FMT);
  CHECKDRIVEINFO(blocksize,NASD_64u_FMT);
  CHECKDRIVEINFO(num_parts,NASD_64u_FMT);
  CHECKDRIVEINFO(num_blocks,NASD_64u_FMT);
  CHECKDRIVEINFO(blocks_allocated,NASD_64u_FMT);

#undef CHECKDRIVEINFO

  if (fail)
    printf("Drive info check failed.\n\n");

  return fail;
}

int check_part_info(void)  {
  nasd_ctrl_part_info_t part_info, part_info_after;
  nasd_ctrl_part_info_otw_t part_info_otw;
  int fail = 0;

  part_info.ctrl_id = 2;
  part_info.first_obj = nasd_int64cast(0x12345678abcdef0);
  part_info.num_obj = 3;
  part_info.part_size = 4;
  part_info.blocks_used = 5;
  part_info.blocks_allocated = 6;

  nasd_ctrl_part_info_t_marshall(&part_info, part_info_otw);
  nasd_ctrl_part_info_t_unmarshall(part_info_otw, &part_info_after);

#define CHECKPARTINFO(_x_,_fmt_) NASD_MT_CHECKBASIC(part_info,_x_,_fmt_)

  CHECKPARTINFO(ctrl_id,NASD_ID_FMT);
  CHECKPARTINFO(first_obj,NASD_ID_FMT);
  CHECKPARTINFO(num_obj,NASD_64u_FMT);
  CHECKPARTINFO(part_size,NASD_64u_FMT);
  CHECKPARTINFO(blocks_used,NASD_64u_FMT);
  CHECKPARTINFO(blocks_allocated,NASD_64u_FMT);
  CHECKPARTINFO(max_objs,NASD_64u_FMT);
  CHECKPARTINFO(blocksize,NASD_64u_FMT);

#undef CHECKPARTINFO

  if (fail)
    printf("Part info check failed.\n\n");

  return fail;
}


int check_trace_basic(void) {
  nasd_ctrl_trace_basic_t trace_basic, trace_basic_after;
  nasd_ctrl_trace_basic_otw_t trace_basic_otw;
  int fail = 0;

  trace_basic.ctrl_id = 4;
  trace_basic.enabled = 1;
  
  nasd_ctrl_trace_basic_t_marshall(&trace_basic, trace_basic_otw);
  nasd_ctrl_trace_basic_t_unmarshall(trace_basic_otw, &trace_basic_after);

#define CHECKTRB(_x_,_fmt_) NASD_MT_CHECKBASIC(trace_basic,_x_,_fmt_)

  CHECKTRB(ctrl_id,NASD_ID_FMT);
  CHECKTRB(enabled,NASD_64u_FMT);

#undef CHECKTRB

  if (fail) {
    printf("trace_basic check failed.\n\n");
  }

  return fail;
}


int check_part_creat(void) {
  nasd_p_part_creat_dr_args_t p_part_creat_args,p_part_creat_args_after;
  nasd_p_part_creat_dr_args_otw_t p_part_creat_args_otw;
  nasd_key_t akey,bkey,ckey;
  int i, fail = 0;
  
  for(i = 0; i < sizeof(nasd_key_t); i++) {
    akey[i] = 2 * i;
    bkey[i] = 2 * i + 1;
    ckey[i] = 2 * i - 1;
  }
        
  p_part_creat_args.in_partnum = 5;
  p_part_creat_args.in_blkcnt = 1234567;
  p_part_creat_args.in_min_protection = 666;
    
  bcopy(akey,p_part_creat_args.in_partition_key,sizeof(nasd_key_t));
  bcopy(bkey,p_part_creat_args.in_red_key,sizeof(nasd_key_t));
  bcopy(ckey,p_part_creat_args.in_black_key,sizeof(nasd_key_t));
  
  nasd_p_part_creat_dr_args_t_marshall(&p_part_creat_args,
				     p_part_creat_args_otw);
  nasd_p_part_creat_dr_args_t_unmarshall(p_part_creat_args_otw,
				       &p_part_creat_args_after);
    
#define CHECKPARTCREAT(_x_,_fmt_) NASD_MT_CHECKBASIC(p_part_creat_args,_x_,_fmt_)
  
  CHECKPARTCREAT(in_partnum,"d");
  CHECKPARTCREAT(in_min_protection,"hu")
  NASD_MT_CHECKKEY(p_part_creat_args,in_partition_key)
  NASD_MT_CHECKKEY(p_part_creat_args,in_black_key)
  NASD_MT_CHECKKEY(p_part_creat_args,in_red_key)
  CHECKPARTCREAT(in_blkcnt,NASD_64u_FMT);

#undef CHECKPARTCREAT

  if (fail)
    printf("Part creat check failed.\n\n");

  return fail;
}

int check_part_creat_res(void) {
  int fail = 0;
  nasd_p_part_creat_dr_res_t p_part_creat_res,p_part_creat_res_after;
  nasd_p_part_creat_dr_res_otw_t p_part_creat_res_otw;

  p_part_creat_res.nasd_status = 7;
    
  nasd_p_part_creat_dr_res_t_marshall(&p_part_creat_res,
				    p_part_creat_res_otw);
  nasd_p_part_creat_dr_res_t_unmarshall(p_part_creat_res_otw,
				      &p_part_creat_res_after);

  NASD_MT_CHECKSTATUS(p_part_creat_res,nasd_status)

  return fail;
}

int check_smpl_op(void) {
  int fail = 0;
  nasd_p_smpl_op_dr_args_t p_smpl_op_args,p_smpl_op_args_after;
  nasd_p_smpl_op_dr_args_otw_t p_smpl_op_args_otw;

  p_smpl_op_args.in_partnum = 5;
  p_smpl_op_args.in_identifier = 1234567;
  p_smpl_op_args.in_offset = 37;
  p_smpl_op_args.in_len = 362;
    
  nasd_p_smpl_op_dr_args_t_marshall(&p_smpl_op_args,p_smpl_op_args_otw);
  nasd_p_smpl_op_dr_args_t_unmarshall(p_smpl_op_args_otw,&p_smpl_op_args_after);
    
#define CHECKSMPLOPARGS(_x_,_fmt_) NASD_MT_CHECKBASIC(p_smpl_op_args,_x_,_fmt_)

  CHECKSMPLOPARGS(in_partnum,"d");
  CHECKSMPLOPARGS(in_identifier,NASD_ID_FMT);
  CHECKSMPLOPARGS(in_offset,NASD_64u_FMT);
  CHECKSMPLOPARGS(in_len,"d");

#undef CHECKSMPLOPARGS
  
  if (fail)
    printf("smpl_op check failed.\n\n");

  return fail;
}  

int check_thrtl_op(void) {
  int fail = 0;
  nasd_p_thrtl_op_dr_args_t p_thrtl_op_args,p_thrtl_op_args_after;
  nasd_p_thrtl_op_dr_args_otw_t p_thrtl_op_args_otw;

  p_thrtl_op_args.in_partnum = 5;
  p_thrtl_op_args.in_identifier = 1234567;
  p_thrtl_op_args.in_offset = 37;
  p_thrtl_op_args.in_bms_targ = nasd_int64cast(991234567890);
  p_thrtl_op_args.in_len = 362;

  nasd_p_thrtl_op_dr_args_t_marshall(&p_thrtl_op_args,p_thrtl_op_args_otw);
  nasd_p_thrtl_op_dr_args_t_unmarshall(p_thrtl_op_args_otw,&p_thrtl_op_args_after);

#define CHECKTHRTLOPARGS(_x_,_fmt_) NASD_MT_CHECKBASIC(p_thrtl_op_args,_x_,_fmt_)

  CHECKTHRTLOPARGS(in_partnum,"d");
  CHECKTHRTLOPARGS(in_identifier,NASD_ID_FMT);
  CHECKTHRTLOPARGS(in_offset,NASD_64u_FMT);
  CHECKTHRTLOPARGS(in_bms_targ,NASD_64u_FMT);
  CHECKTHRTLOPARGS(in_len,"d");

#undef CHECKTHRTLOPARGS

  if (fail)
    printf("thrtl_op check failed.\n\n");

  return fail;
}

int check_getattr_args(void) {
  int fail = 0;
  nasd_p_getattr_dr_args_t getattr_args,getattr_args_after;
  nasd_p_getattr_dr_args_otw_t getattr_args_otw;

  getattr_args.in_identifier = 1001;
  getattr_args.in_partnum = 3;

  nasd_p_getattr_dr_args_t_marshall(&getattr_args,getattr_args_otw);
  nasd_p_getattr_dr_args_t_unmarshall(getattr_args_otw,&getattr_args_after);
    
#define CHECKGETATTRARGS(_x_,_fmt_) NASD_MT_CHECKBASIC(getattr_args,_x_,_fmt_)

  CHECKGETATTRARGS(in_identifier,NASD_ID_FMT)
  CHECKGETATTRARGS(in_partnum,"hd")

#undef CHECKGETATTRARGS

  if (fail)
    printf("getattr_args check failed.\n\n");

  return fail;
}

int check_getattr_res(void) {
  int fail = 0;
  nasd_p_getattr_dr_res_t p_getattr_res,p_getattr_res_after;
  nasd_p_getattr_dr_res_otw_t p_getattr_res_otw;

  p_getattr_res.out_attribute.object_len = 123;
  p_getattr_res.nasd_status = 5;

  nasd_p_getattr_dr_res_t_marshall(&p_getattr_res,p_getattr_res_otw);
  nasd_p_getattr_dr_res_t_unmarshall(p_getattr_res_otw,&p_getattr_res_after);

  NASD_MT_CHECKSTATUS(p_getattr_res,nasd_status)
  NASD_MT_CHECKATTRIBUTE(p_getattr_res,out_attribute)

  if (fail)
    printf("getattr_res check failed.\n\n");

  return fail;
}

int check_setattr_args(void) {
  int fail = 0;
  nasd_p_setattr_dr_args_t setattr_args,setattr_args_after;
  nasd_p_setattr_dr_args_otw_t setattr_args_otw;

  setattr_args.in_identifier = 1001;
  setattr_args.in_partnum = 3;

  nasd_p_setattr_dr_args_t_marshall(&setattr_args,setattr_args_otw);
  nasd_p_setattr_dr_args_t_unmarshall(setattr_args_otw,&setattr_args_after);

#define CHECKSETATTRARGS(_x_,_fmt_) NASD_MT_CHECKBASIC(setattr_args,_x_,_fmt_)

  CHECKSETATTRARGS(in_identifier,NASD_ID_FMT)
  NASD_MT_CHECKATTRIBUTE(setattr_args,in_attribute)
  CHECKSETATTRARGS(in_fieldmask,"d")
  CHECKSETATTRARGS(in_partnum,"hd")
  CHECKSETATTRARGS(in_guard,"02x")

#undef CHECKSETATTRARGS

  if (fail)
    printf("setattr_args check failed.\n\n");

  return fail;
}
 
int check_setattr_res(void) {
  int fail = 0;
  nasd_p_setattr_dr_res_t p_setattr_res,p_setattr_res_after;
  nasd_p_setattr_dr_res_otw_t p_setattr_res_otw;

  p_setattr_res.out_attribute.object_len = 123;
  p_setattr_res.nasd_status = 5;
    
  nasd_p_setattr_dr_res_t_marshall(&p_setattr_res,p_setattr_res_otw);
  nasd_p_setattr_dr_res_t_unmarshall(p_setattr_res_otw,&p_setattr_res_after);

  NASD_MT_CHECKSTATUS(p_setattr_res,nasd_status)
  NASD_MT_CHECKATTRIBUTE(p_setattr_res,out_attribute)

  if (fail)
    printf("setattr_res check failed.\n\n");

  return fail;
}
 
int check_create_args(void) {
  int fail = 0;
  nasd_p_create_dr_args_t create_args,create_args_after;
  nasd_p_create_dr_args_otw_t create_args_otw;

  create_args.in_partnum = 3;

  nasd_p_create_dr_args_t_marshall(&create_args,create_args_otw);
  nasd_p_create_dr_args_t_unmarshall(create_args_otw,&create_args_after);
    
#define CHECKCREATEARGS(_x_,_fmt_) NASD_MT_CHECKBASIC(create_args,_x_,_fmt_)

  NASD_MT_CHECKATTRIBUTE(create_args,in_attribute)
  CHECKCREATEARGS(in_fieldmask,"d")
  CHECKCREATEARGS(in_partnum,"hd")

#undef CHECKCREATEARGS

    if (fail)
      printf("create_args check failed.\n\n");

    return fail;
}

int check_create_res(void) {
  int fail = 0;
  nasd_p_create_dr_res_t p_create_res,p_create_res_after;
  nasd_p_create_dr_res_otw_t p_create_res_otw;

  p_create_res.out_attribute.object_len = 123;
  p_create_res.out_identifier = 1001;
  p_create_res.nasd_status = 5;
    
  nasd_p_create_dr_res_t_marshall(&p_create_res,p_create_res_otw);
  nasd_p_create_dr_res_t_unmarshall(p_create_res_otw,&p_create_res_after);

#define CHECKCREATERES(_x_,_fmt_) NASD_MT_CHECKBASIC(p_create_res,_x_,_fmt_)

  NASD_MT_CHECKSTATUS(p_create_res,nasd_status)
  CHECKCREATERES(out_identifier,NASD_ID_FMT)
  NASD_MT_CHECKATTRIBUTE(p_create_res,out_attribute)

#undef CHECKCREATERES

  if (fail)
    printf("create_res check failed.\n\n");

  return fail;
}

int check_flush_obj_args(void) {
  int fail = 0;
  nasd_p_flush_obj_dr_args_t flush_obj_args,flush_obj_args_after;
  nasd_p_flush_obj_dr_args_otw_t flush_obj_args_otw;

  flush_obj_args.in_identifier = 1001;
  flush_obj_args.in_partnum = 3;
  nasd_p_flush_obj_dr_args_t_marshall(&flush_obj_args,flush_obj_args_otw);
  nasd_p_flush_obj_dr_args_t_unmarshall(flush_obj_args_otw,&flush_obj_args_after);
    
#define CHECKFLUSHOBJARGS(_x_,_fmt_) NASD_MT_CHECKBASIC(flush_obj_args,_x_,_fmt_)

  CHECKFLUSHOBJARGS(in_identifier,NASD_ID_FMT)
  CHECKFLUSHOBJARGS(in_partnum,"hd")

#undef CHECKFLUSHOBJARGS

  if (fail)
    printf("flush_obj_args check failed.\n\n");

  return fail;
}

int check_flush_obj_res(void) {
  int fail = 0;
  nasd_p_flush_obj_dr_res_t flush_obj_res,flush_obj_res_after;
  nasd_p_flush_obj_dr_res_otw_t flush_obj_res_otw;

  flush_obj_res.nasd_status = NASD_BAD_IDENTIFIER;

  nasd_p_flush_obj_dr_res_t_marshall(&flush_obj_res,flush_obj_res_otw);
  nasd_p_flush_obj_dr_res_t_unmarshall(flush_obj_res_otw,&flush_obj_res_after);

  NASD_MT_CHECKSTATUS(flush_obj_res,nasd_status)

  return fail;
}

int check_eject_obj_args(void) {
  nasd_p_eject_obj_dr_args_t eject_obj_args,eject_obj_args_after;
  nasd_p_eject_obj_dr_args_otw_t eject_obj_args_otw;
  int fail = 0;

  eject_obj_args.in_identifier = 1001;
  eject_obj_args.in_partnum = 3;

  nasd_p_eject_obj_dr_args_t_marshall(&eject_obj_args,eject_obj_args_otw);
  nasd_p_eject_obj_dr_args_t_unmarshall(eject_obj_args_otw,&eject_obj_args_after);

#define CHECKEJECTOBJARGS(_x_,_fmt_) NASD_MT_CHECKBASIC(eject_obj_args,_x_,_fmt_)

  CHECKEJECTOBJARGS(in_identifier,NASD_ID_FMT)
  CHECKEJECTOBJARGS(in_partnum,"hd")

#undef CHECKEJECTOBJARGS

  if (fail) {
    printf("eject_obj_args checks failed.\n\n");
  }

  return fail;
}

int check_eject_obj_res(void) {
  nasd_p_eject_obj_dr_res_t eject_obj_res,eject_obj_res_after;
  nasd_p_eject_obj_dr_res_otw_t eject_obj_res_otw;
  int fail = 0;

  eject_obj_res.nasd_status = NASD_BAD_IDENTIFIER;

  nasd_p_eject_obj_dr_res_t_marshall(&eject_obj_res,eject_obj_res_otw);
  nasd_p_eject_obj_dr_res_t_unmarshall(eject_obj_res_otw,&eject_obj_res_after);

  NASD_MT_CHECKSTATUS(eject_obj_res,nasd_status)

  if (fail) {
    printf("eject_obj_res checks failed.\n\n");
  }

  return fail;
}


int check_initialize_args(void) {
  int fail = 0;
  nasd_p_initialize_dr_args_t initialize_args,initialize_args_after;
  nasd_p_initialize_dr_args_otw_t initialize_args_otw;
  nasd_key_t mkey,dkey;
  int i;

  for(i=0;i<sizeof(nasd_key_t);i++) {
    mkey[i]=2*i;
    dkey[i]=2*i+1;
  }
        
  bcopy(mkey,initialize_args.in_master_key,sizeof(nasd_key_t));
  bcopy(dkey,initialize_args.in_drive_key,sizeof(nasd_key_t));

  nasd_p_initialize_dr_args_t_marshall(&initialize_args,initialize_args_otw);
  nasd_p_initialize_dr_args_t_unmarshall(initialize_args_otw,&initialize_args_after);

  NASD_MT_CHECKKEY(initialize_args,in_master_key)
  NASD_MT_CHECKKEY(initialize_args,in_drive_key)

  if (fail)
    printf("initialize_args check failed.\n\n");

  return fail;
}

int check_initialize_res(void) {
  int fail = 0;
  nasd_p_initialize_dr_res_t initialize_res,initialize_res_after;
  nasd_p_initialize_dr_res_otw_t initialize_res_otw;

  initialize_res.nasd_status = NASD_BAD_IDENTIFIER;

  nasd_p_initialize_dr_res_t_marshall(&initialize_res,initialize_res_otw);
  nasd_p_initialize_dr_res_t_unmarshall(initialize_res_otw,&initialize_res_after);
  
  NASD_MT_CHECKSTATUS(initialize_res,nasd_status)

  return fail;
}

int check_strt_iread_args(void) {
  int fail = 0;
  nasd_p_strt_iread_dr_args_t strt_iread_args,strt_iread_args_after;
  nasd_p_strt_iread_dr_args_otw_t strt_iread_args_otw;

  strt_iread_args.in_index_identifier = 1001;
  strt_iread_args.in_data_identifier = 2002;
  strt_iread_args.in_flownum = 5;
  strt_iread_args.in_interval = 60;
  strt_iread_args.in_client_addr.ip_addr = 0x8002bd4d;
  strt_iread_args.in_client_addr.port = 6969;
  strt_iread_args.in_partnum = 3;

  nasd_p_strt_iread_dr_args_t_marshall(&strt_iread_args,strt_iread_args_otw);
  nasd_p_strt_iread_dr_args_t_unmarshall(strt_iread_args_otw,&strt_iread_args_after);

#define CHECKSTARTIREADARGS(_x_,_fmt_) NASD_MT_CHECKBASIC(strt_iread_args,_x_,_fmt_)

  CHECKSTARTIREADARGS(in_index_identifier,NASD_ID_FMT)
  CHECKSTARTIREADARGS(in_data_identifier,NASD_ID_FMT)
  NASD_MT_CHECKTIMESPEC(strt_iread_args,in_earliest_start)
  NASD_MT_CHECKTIMESPEC(strt_iread_args,in_latest_start)
  NASD_MT_CHECKCLIENTADDR(strt_iread_args,in_client_addr)
  CHECKSTARTIREADARGS(in_interval,"d")
  CHECKSTARTIREADARGS(in_offset,"d")
  CHECKSTARTIREADARGS(in_flownum,"d")
  CHECKSTARTIREADARGS(in_partnum,"hd")

#undef CHECKSTARTIREADARGS

  if (fail)
    printf("strt_iread_args check failed.\n\n");

  return fail;
}

int check_strt_iread_res(void) {
  int fail = 0;
  nasd_p_strt_iread_dr_res_t strt_iread_res,strt_iread_res_after;
  nasd_p_strt_iread_dr_res_otw_t strt_iread_res_otw;

  strt_iread_res.nasd_status = NASD_BAD_IDENTIFIER;
  strt_iread_res.out_stream_id = 5;

  nasd_p_strt_iread_dr_res_t_marshall(&strt_iread_res,strt_iread_res_otw);
  nasd_p_strt_iread_dr_res_t_unmarshall(strt_iread_res_otw,&strt_iread_res_after);
  
#define CHECKSTARTIREADRES(_x_,_fmt_) NASD_MT_CHECKBASIC(strt_iread_res,_x_,_fmt_)

  NASD_MT_CHECKSTATUS(strt_iread_res,nasd_status)
  CHECKSTARTIREADRES(out_stream_id,"d")
  NASD_MT_CHECKTIMESPEC(strt_iread_res,out_start_time)

#undef CHECKSTARTIREADRES

  return fail;
}

int check_stop_iread_args(void) {
  int fail = 0;
  nasd_p_stop_iread_dr_args_t stop_iread_args,stop_iread_args_after;
  nasd_p_stop_iread_dr_args_otw_t stop_iread_args_otw;

  stop_iread_args.in_stream_id = 5;

  nasd_p_stop_iread_dr_args_t_marshall(&stop_iread_args,stop_iread_args_otw);
  nasd_p_stop_iread_dr_args_t_unmarshall(stop_iread_args_otw,&stop_iread_args_after);

#define CHECKSTOPIREADARGS(_x_,_fmt_) NASD_MT_CHECKBASIC(stop_iread_args,_x_,_fmt_)

  CHECKSTOPIREADARGS(in_stream_id,"d")

#undef CHECKSTOPIREADARGS

  if (fail)
    printf("stop_iread_args check failed.\n\n");

  return fail;
}

int check_stop_iread_res(void) {
  int fail = 0;
  nasd_p_stop_iread_dr_res_t stop_iread_res,stop_iread_res_after;
  nasd_p_stop_iread_dr_res_otw_t stop_iread_res_otw;

  stop_iread_res.nasd_status = NASD_BAD_IDENTIFIER;

  nasd_p_stop_iread_dr_res_t_marshall(&stop_iread_res,stop_iread_res_otw);
  nasd_p_stop_iread_dr_res_t_unmarshall(stop_iread_res_otw,&stop_iread_res_after);

  NASD_MT_CHECKSTATUS(stop_iread_res,nasd_status)
  
  return fail;
}

int check_change_key_args(void) {
  int fail = 0, i = 0;
  nasd_p_change_key_dr_args_t change_key_args,change_key_args_after;
  nasd_p_change_key_dr_args_otw_t change_key_args_otw;
  nasd_key_t tkey;

  change_key_args.in_partnum = 3;

  for(i=0;i<sizeof(nasd_key_t);i++)
    tkey[i]=2*i;
       
  bcopy(tkey,change_key_args.in_new_key,sizeof(nasd_key_t));
    
  nasd_p_change_key_dr_args_t_marshall(&change_key_args,
				     change_key_args_otw);
  nasd_p_change_key_dr_args_t_unmarshall(change_key_args_otw,
				       &change_key_args_after);

#define CHECKCHGKEYARGS(_x_,_fmt_) NASD_MT_CHECKBASIC(change_key_args,_x_,_fmt_)

  NASD_MT_CHECKKEY(change_key_args,in_new_key)
  CHECKCHGKEYARGS(in_partnum,"hd")
  CHECKCHGKEYARGS(in_type,"hu")

#undef CHECKCHGKEYARGS

  if (fail)
    printf("change_key_args check failed.\n\n");

  return fail;
}
 
int check_change_key_res(void) {
  int fail = 0;
  nasd_p_change_key_dr_res_t p_change_key_res,p_change_key_res_after;
  nasd_p_change_key_dr_res_otw_t p_change_key_res_otw;

  p_change_key_res.nasd_status = 5;
    
  nasd_p_change_key_dr_res_t_marshall(&p_change_key_res,
				    p_change_key_res_otw);
  nasd_p_change_key_dr_res_t_unmarshall(p_change_key_res_otw,
				      &p_change_key_res_after);

  NASD_MT_CHECKSTATUS(p_change_key_res,nasd_status)

  return fail;
}

int check_trace_header(void) {
  nasd_trace_header_t trace_header, trace_header_after;
  nasd_trace_header_otw_t hotw;
  int fail = 0;

  trace_header.tr_class = 11;
  trace_header.tr_type = 3;
  trace_header.tr_len = 529;
  trace_header.tr_seq = 0x1234;
  trace_header.tr_time.ts_sec = 37;
  trace_header.tr_time.ts_nsec = 3900;

  nasd_trace_header_t_marshall(&trace_header, hotw);
  nasd_trace_header_t_unmarshall(hotw, &trace_header_after);

#define CHECKTRACEHEADER(_x_,_fmt_) NASD_MT_CHECKBASIC(trace_header,_x_,_fmt_)

  CHECKTRACEHEADER(tr_class,"02x")
  CHECKTRACEHEADER(tr_type,"02x")
  CHECKTRACEHEADER(tr_len,"hu")
  CHECKTRACEHEADER(tr_seq,"u")
  NASD_MT_CHECKTIMESPEC(trace_header,tr_time)

#undef CHECKTRACEHEADER

  if (fail) {
    printf("Trace header checks failed.\n\n");
  }

  return fail;
}

#if NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_SRPC
int check_srpc_header(void) {
  int fail = 0;
  nasd_srpc_header_t header, header_after;
  nasd_srpc_header_otw_t header_otw;

  header.callid = 0x12345678;
  header.len    = 0x3456789a;
  header.op     = 0x56789abc;
  header.opstat = 0x789abcde;

  nasd_srpc_header_t_marshall(&header, header_otw);
  nasd_srpc_header_t_unmarshall(header_otw, &header_after);

#define CHECKHEADER(_val_,_fmt_) NASD_MT_CHECKBASIC(header,_val_,_fmt_)

  CHECKHEADER(callid,"u")
  CHECKHEADER(len,"u")
  CHECKHEADER(op,"u")
  CHECKHEADER(opstat,"u")

#undef CHECKHEADER

  if (fail) {
    printf("nasd_srpc_header_t checks failed.\n\n");
  }

  return fail;
}
#endif /* NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_SRPC */

/* Local Variables:  */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* End: */
