/*-
 * Copyright (c) 2015 Taylor R. Campbell
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

%{
#include <assert.h>
#include <inttypes.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>

#include "ast.h"
#include "attribute.h"
#include "eprintf.h"
#include "ereallocarr.h"
#include "syntax.h"

static struct string	string_edupzz(const char *);

static void	push(const char *, struct ast_srcloc, struct string *);
static void	pop(const char *);

static void	ast_option_destroy(const struct ast_option *);
static void	ast_type_destroy(const struct ast_type *);
static void	ast_enumerand_destroy(const struct ast_enumerand *);
static void	ast_enumeration_destroy(const struct ast_enumeration *);
static void	ast_rpc_destroy(const struct ast_rpc *);
static void	ast_service_destroy(const struct ast_service *);
static void	ast_package_destroy(const struct ast_package *);
static void	ast_import_destroy(const struct ast_import *);
static void	ast_extension_range_destroy(
		    const struct ast_extension_range *);
static void	ast_import_destroy(const struct ast_import *);
static void	ast_extensions_destroy(const struct ast_extensions *);
static void	ast_fieldval_destroy(const struct ast_fieldval *);
static void	ast_field_destroy(const struct ast_field *);
static void	ast_extend_destroy(const struct ast_extend *);
static void	ast_message_stmts_destroy(const struct ast_message_stmts *);
static void	ast_message_destroy(const struct ast_message *);
static void	ast_proto_destroy(const struct ast_proto *);

struct errctx {
	const char		*desc;
	struct ast_srcloc	loc;
	struct string		*id;
};

static struct errctx	*errctx = NULL;
static size_t		nerrctx = 0;

static int error = 0;
%}

%expect 0
%expect-rr 0

%token	<srcloc>	K_DEFAULT
%token	<srcloc>	K_ENUM
%token	<srcloc>	K_EXTEND
%token	<srcloc>	K_EXTENSIONS
%token	<srcloc>	K_FALSE
%token	<srcloc>	K_GROUP
%token	<srcloc>	K_IMPORT
%token	<srcloc>	K_MAX
%token	<srcloc>	K_MESSAGE
%token	<srcloc>	K_OPTION
%token	<srcloc>	K_OPTIONAL
%token	<srcloc>	K_PACKAGE
%token	<srcloc>	K_PUBLIC
%token	<srcloc>	K_REPEATED
%token	<srcloc>	K_REQUIRED
%token	<srcloc>	K_RETURNS
%token	<srcloc>	K_RPC
%token	<srcloc>	K_SERVICE
%token	<srcloc>	K_SYNTAX
%token	<srcloc>	K_TO
%token	<srcloc>	K_TRUE
%token	<srcloc>	K_WEAK

%token	<srcloc>	T_LROUND
%token	<srcloc>	T_RROUND
%token	<srcloc>	T_LSQUARE
%token	<srcloc>	T_RSQUARE
%token	<srcloc>	T_LCURLY
%token	<srcloc>	T_RCURLY
%token	<srcloc>	T_COMMA
%token	<srcloc>	T_SEMI
%token	<srcloc>	T_DOT
%token	<srcloc>	T_HYPHEN
%token	<srcloc>	T_EQUAL

%token	<id>			L_ID
%token	<integer>		L_INTEGER
%token	<real>			L_REAL
%token	<string>		L_STRING

%type	<proto>			proto
%type	<proto_stmts>		proto_stmts
%type	<proto_stmt>		proto_stmt
%type	<message>		stmt_message
%type	<message_stmts>		message_block
%type	<message_stmts>		message_stmts
%type	<message_stmt>		message_stmt
%type	<extensions>		extensions_head
%type	<extensions>		extensions
%type	<extension_ranges>	extension_ranges_head
%type	<extension_ranges>	extension_ranges
%type	<extension_range>	extension_range
%type	<field>			field
%type	<field_quant>		field_quant
%type	<type>			field_type
%type	<field_tag>		field_tag
%type	<field_options>		field_end
%type	<field_options>		field_optlist
%type	<field_options>		empty_field_opts
%type	<field_options>		field_opts_head
%type	<field_options>		field_opts
%type	<fieldval>		field_value
%type	<enumeration>		stmt_enum
%type	<enum_stmts>		enum_block
%type	<enum_stmts>		enum_stmts
%type	<enum_stmt>		enum_stmt
%type	<enumerand>		enumerand_def
%type	<enumerand_options>	enumerand_end
%type	<enumerand_value>	enumerand_value
%type	<enumerand_options>	enumerand_optlist
%type	<enumerand_options>	enumerand_opts_head
%type	<enumerand_options>	enumerand_opts
%type	<service>		stmt_service
%type	<service_stmts>		service_block
%type	<service_stmts>		service_stmts
%type	<service_stmt>		service_stmt
%type	<rpc>			stmt_rpc
%type	<rpc>			rpc_end
%type	<rpc_options>		rpc_optlist
%type	<rpc_options>		rpc_options
%type	<extend>		stmt_extend
%type	<extend_fields>		extend_block
%type	<extend_fields>		extend_stmts
%type	<import>		stmt_import
%type	<import_qual>		import_qual
%type	<package>		stmt_package
%type	<option>		stmt_option
%type	<option>		option
%type	<optval>		option_end
%type	<optname>		option_name
%type	<optpart>		option_name_part_nondefault
%type	<optpart>		option_name_part
%type	<optpart>		option_name_part_qualified
%type	<optval>		option_value
%type	<string>		dotted_name
%type	<string>		fq_dotted_name
%type	<string>		dotted_name_nongroup
%type	<type>			any_type
%type	<type>			userdef_type
%type	<string>		string
%type	<id>			id
%type	<id>			id_nonbool
%type	<id>			id_nondefault
%type	<id>			id_nongroup
%type	<id>			id_nonbool_nondefault_nongroup

%union {
	struct ast_srcloc		srcloc;
	struct ast_id			id;
	uintmax_t			integer;
	double				real;
	struct string			string;
	struct ast_proto		proto;
	struct ast_proto_stmts		proto_stmts;
	struct ast_proto_stmt		proto_stmt;
	struct ast_message		message;
	struct ast_message_stmts	message_stmts;
	struct ast_message_stmt		message_stmt;
	struct ast_extensions		extensions;
	struct ast_extension_ranges	extension_ranges;
	struct ast_extension_range	extension_range;
	struct ast_field		field;
	struct ast_field_quant_loc	field_quant;
	struct ast_type			type;
	field_tag_t			field_tag;
	struct ast_field_options	field_options;
	struct ast_fieldval		fieldval;
	struct ast_enumeration		enumeration;
	struct ast_enum_stmts		enum_stmts;
	struct ast_enum_stmt		enum_stmt;
	struct ast_enumerand		enumerand;
	struct ast_enumerand_options	enumerand_options;
	int32_t				enumerand_value; /* XXX typedef */
	struct ast_service		service;
	struct ast_service_stmts	service_stmts;
	struct ast_service_stmt		service_stmt;
	struct ast_rpc			rpc;
	struct ast_rpc_options		rpc_options;
	struct ast_extend		extend;
	struct ast_extend_fields	extend_fields;
	struct ast_import		import;
	enum ast_import_qual		import_qual;
	struct ast_package		package;
	struct ast_option		option;
	struct ast_optval		optval;
	struct ast_optname		optname;
	struct ast_optpart		optpart;
}

%%

proto
	: syntax_decl proto_stmts {
		$$.stmts = $2;
		picopbc_parsed(error, &$$);
		ast_proto_destroy(&$$);
		if (errctx) {
			free(errctx);
			errctx = NULL;
		}
	}
	;

syntax_decl
	: /* empty */
	| K_SYNTAX T_EQUAL string T_SEMI {
		if (string_len($3) != strlen("proto2") ||
		    memcmp("proto2", string_ptr($3), strlen("proto2")) != 0) {
			char *v = string_evisciiz($3);
			yyerrorf("unknown proto syntax: %s", v);
			free(v);
			/* XXX Abort instead?  */
			YYERROR;
		}
	}
	;

proto_stmts
	: /* empty */	{ $$.stmts = NULL; $$.nstmts = 0; }
	| proto_stmts proto_stmt {
		$$ = $1;
		if ($2.t != AST_PROTO_STMT_NULL) {
			assert($$.nstmts < SIZE_MAX);
			ereallocarr(&$$.stmts, $$.nstmts + 1,
			    sizeof($$.stmts[0]));
			$$.stmts[$$.nstmts] = $2;
			$$.nstmts++;
		}
	}
	;

proto_stmt
	: T_SEMI	{ $$.t = AST_PROTO_STMT_NULL; }
	| stmt_message	{ $$.t = AST_PROTO_STMT_MESSAGE; $$.u.message = $1; }
	| stmt_enum	{ $$.t = AST_PROTO_STMT_ENUMERATION;
			  $$.u.enumeration = $1; }
	| stmt_service	{ $$.t = AST_PROTO_STMT_SERVICE; $$.u.service = $1; }
	| stmt_extend	{ $$.t = AST_PROTO_STMT_EXTEND; $$.u.extend = $1; }
	| stmt_import	{ $$.t = AST_PROTO_STMT_IMPORT; $$.u.import = $1; }
	| stmt_package	{ $$.t = AST_PROTO_STMT_PACKAGE; $$.u.package = $1; }
	| stmt_option	{ $$.t = AST_PROTO_STMT_OPTION; $$.u.option = $1; }
	| error T_SEMI	{ yyclearin; yyerrok; $$.t = AST_PROTO_STMT_NULL; }
	;

/* Messages */

stmt_message
	: K_MESSAGE id { push("message", $1, &$2.id); } message_block {
		$$.loc = $1;
		$$.id = $2.id;
		$$.stmts = $4;
		pop("message");
	}
	;

message_block
	: T_LCURLY message_stmts T_RCURLY	{ $$ = $2; }
	| T_LCURLY message_stmts error T_RCURLY	{ yyclearin; yyerrok;
						  $$ = $2; }
	;

message_stmts
	: /* empty */	{ $$.stmts = NULL; $$.nstmts = 0; }
	| message_stmts message_stmt {
		$$ = $1;
		if ($2.t != AST_MESSAGE_STMT_NULL) {
			assert($$.nstmts < SIZE_MAX);
			ereallocarr(&$$.stmts, $$.nstmts + 1,
			    sizeof($$.stmts[0]));
			$$.stmts[$$.nstmts] = $2;
			$$.nstmts++;
		}
	}
	;

message_stmt
	: T_SEMI	{ $$.t = AST_MESSAGE_STMT_NULL; }
	| stmt_message	{ $$.t = AST_MESSAGE_STMT_MESSAGE; $$.u.message = $1; }
	| stmt_enum	{ $$.t = AST_MESSAGE_STMT_ENUMERATION;
			  $$.u.enumeration = $1; }
	| stmt_extend	{ $$.t = AST_MESSAGE_STMT_EXTEND; $$.u.extend = $1; }
	| stmt_option	{ $$.t = AST_MESSAGE_STMT_OPTION; $$.u.option = $1; }
	| extensions	{ $$.t = AST_MESSAGE_STMT_EXTENSIONS;
			  $$.u.extensions = $1; }
	| field		{ $$.t = AST_MESSAGE_STMT_FIELD; $$.u.field = $1; }
	| error T_SEMI	{ yyclearin; yyerrok; $$.t = AST_MESSAGE_STMT_NULL; }
	;

extensions_head
	: K_EXTENSIONS {
		push("extensions", $1, NULL);
		$$.loc = $1;
	}
	;

extensions
	: extensions_head extension_ranges T_SEMI {
		$$ = $1;
		$$.ranges = $2;
		pop("extensions");
	}
	| extensions_head extension_ranges error T_SEMI {
		yyclearin;
		yyerrok;
		$$ = $1;
		$$.ranges = $2;
		pop("extensions");
	}
	;

extension_ranges_head
	: /* empty */			{ $$.ranges = NULL; $$.nranges = 0; }
	| extension_ranges T_COMMA	{ $$ = $1; }
	| extension_ranges error T_COMMA
					{ yyclearin; yyerrok; $$ = $1; }
	;

extension_ranges
	: /* empty */			{ $$.ranges = NULL; $$.nranges = 0; }
	| extension_ranges_head extension_range {
		assert($$.nranges < SIZE_MAX);
		ereallocarr(&$$.ranges, $$.nranges + 1, sizeof($$.ranges[0]));
		$$.ranges[$$.nranges] = $2;
		$$.nranges++;
	}
	;

extension_range
	: field_tag		{ $$.min = $1; $$.max = $1; }
	| field_tag K_TO K_MAX	{ $$.min = $1; $$.max = FIELD_TAG_MAX; }
	| field_tag K_TO field_tag {
		if ($1 > $3) {
			yyerrorf("extension min/max out of order"
			    ": min %"PRI_field_tag" > max %"PRI_field_tag,
			    $1, $3);
			$$.min = $3;
			$$.max = $1;
		} else {
			$$.min = $1;
			$$.max = $3;
		}
	}
	;

field
	: field_quant K_GROUP { yyerror("field groups are not supported"); }
	    id T_EQUAL field_tag { push("field group", $1.loc, &$4.id); }
	    field_optlist message_block {
		$$.loc = $1.loc;
		$$.quant = $1.quant;
		$$.type.text = string_edupzz("(invalid type)");
		$$.id = $4.id;
		$$.tag = $6;
		$$.options = $8;
		ast_message_stmts_destroy(&$9);
		pop("field group");
	}
	| field_quant field_type id T_EQUAL field_tag
	    { push("field", $1.loc, &$3.id); } field_end {
		$$.loc = $1.loc;
		$$.quant = $1.quant;
		$$.type = $2;
		$$.id = $3.id;
		$$.tag = $5;
		$$.options = $7;
		pop("field");
	}
	;

field_quant
	: K_REQUIRED	{ $$.loc = $1; $$.quant = AST_FQ_REQUIRED; }
	| K_OPTIONAL	{ $$.loc = $1; $$.quant = AST_FQ_OPTIONAL; }
	| K_REPEATED	{ $$.loc = $1; $$.quant = AST_FQ_REPEATED; }
	;

field_type
	: any_type
	;

field_tag
	: L_INTEGER {
		if ($1 < FIELD_TAG_MIN) {
			yyerrorf("field tag too small: %"PRIuMAX
			    " (min %"PRIuMAX")", $1, (uintmax_t)FIELD_TAG_MIN);
			$$ = FIELD_TAG_MIN;
		} else if (FIELD_TAG_MAX < $1) {
			yyerrorf("field tag too large: %"PRIuMAX
			    " (max %"PRIuMAX")", $1, (uintmax_t)FIELD_TAG_MAX);
			$$ = FIELD_TAG_MAX;
		} else {
			$$ = $1;
		}
	}
	;

field_end
	: field_optlist T_SEMI		{ $$ = $1; }
	| error T_SEMI {
		yyclearin;
		yyerrok;
		$$.options = NULL;
		$$.noptions = 0;
	}
	;

field_optlist
	: /* empty */ {
		$$.default_present = false;
		$$.options = NULL;
		$$.noptions = 0;
	}
	| T_LSQUARE field_opts T_RSQUARE	{ $$ = $2; }
	| T_LSQUARE field_opts error T_RSQUARE	{ yyclearin; yyerrok;
						  $$ = $2; }
	;

empty_field_opts
	: /* empty */ {
		$$.default_present = false;
		$$.options = NULL;
		$$.noptions = 0;
	}

field_opts_head
	: empty_field_opts		{ $$ = $1; }
	| field_opts T_COMMA		{ $$ = $1; }
	| field_opts error T_COMMA	{ yyclearin; yyerrok; $$ = $1; }
	;

field_opts
	: empty_field_opts		{ $$ = $1; }
	| field_opts_head option {
		$$ = $1;
		assert($$.noptions < SIZE_MAX);
		ereallocarr(&$$.options, $$.noptions + 1,
		    sizeof($$.options[0]));
		$$.options[$$.noptions] = $2;
		$$.noptions++;
	}
	| field_opts_head K_DEFAULT T_EQUAL field_value {
		$$ = $1;
		if ($$.default_present) {
			yyerrorf("multiple default values");
		} else {
			$$.default_present = true;
			$$.default_value = $4;
		}
	}
	;

field_value
	: L_INTEGER	{ $$.t = AST_FIELDVAL_UINT; $$.u.uint = $1; }
	| T_HYPHEN L_INTEGER {
		$$.t = AST_FIELDVAL_SINT;
		if ($2 > (uint64_t)INT64_MAX + 1) {
			yyerrorf("negative too small: -%"PRIuMAX, $2);
			$$.u.sint = INT64_MIN;
		} else {
			$$.u.sint = -$2;
		}
	}
	| L_REAL	{ $$.t = AST_FIELDVAL_REAL; $$.u.real = $1; }
	| T_HYPHEN L_REAL
			{ $$.t = AST_FIELDVAL_REAL; $$.u.real = -$2; }
	| K_TRUE	{ $$.t = AST_FIELDVAL_BOOLEAN; $$.u.boolean = true; }
	| K_FALSE	{ $$.t = AST_FIELDVAL_BOOLEAN; $$.u.boolean = false; }
	| string	{ $$.t = AST_FIELDVAL_STRING; $$.u.string = $1; }
	| id_nonbool	{ $$.t = AST_FIELDVAL_ID; $$.u.id = $1.id; }
	;

/* Enums */

stmt_enum
	: K_ENUM id { push("enum", $1, &$2.id); } enum_block {
		$$.loc = $1;
		$$.id = $2.id;
		$$.stmts = $4;
		pop("enum");
	}
	;

enum_block
	: T_LCURLY enum_stmts T_RCURLY	{ $$ = $2; }
	| T_LCURLY enum_stmts error T_RCURLY
					{ yyclearin; yyerrok; $$ = $2; }
	;

enum_stmts
	: /* empty */	{ $$.stmts = NULL; $$.nstmts = 0; }
	| enum_stmts enum_stmt {
		$$ = $1;
		if ($2.t != AST_ENUM_STMT_NULL) {
			assert($$.nstmts < SIZE_MAX);
			ereallocarr(&$$.stmts, $$.nstmts + 1,
			    sizeof($$.stmts[0]));
			$$.stmts[$$.nstmts] = $2;
			$$.nstmts++;
		}
	}
	;

enum_stmt
	: T_SEMI	{ $$.t = AST_ENUM_STMT_NULL; }
	| stmt_option	{ $$.t = AST_ENUM_STMT_OPTION; $$.u.option = $1; }
	| enumerand_def	{ $$.t = AST_ENUM_STMT_ENUMERAND;
			  $$.u.enumerand = $1; }
	| error T_SEMI	{ yyclearin; yyerrok; $$.t = AST_ENUM_STMT_NULL; }
	;

enumerand_def
	: id T_EQUAL { push("enumerand", $1.loc, &$1.id); } enumerand_value
	    enumerand_end {
		$$.loc = $1.loc;
		$$.id = $1.id;
		$$.value = $4;
		$$.options = $5;
		pop("enumerand");
	}
	;

enumerand_end
	: enumerand_optlist T_SEMI	{ $$ = $1; }
	| error T_SEMI {
		yyclearin;
		yyerrok;
		$$.options = NULL;
		$$.noptions = 0;
 	}
	;

enumerand_value
	: L_INTEGER {
		if (INT32_MAX < $1) {
			yyerrorf("enumeration value too large: %"PRIuMAX, $1);
			$$ = INT32_MAX;
		} else {
			$$ = $1;
		}
	}
	| T_HYPHEN L_INTEGER {
		if ((uint32_t)INT32_MAX + 1 < $2) {
			yyerrorf("enumeration value too small: -%"PRIuMAX, $2);
			$$ = INT32_MIN;
		} else {
			$$ = -$2;
		}
	}
	;

enumerand_optlist
	: /* empty */			{ $$.options = NULL; $$.noptions = 0; }
	| T_LSQUARE enumerand_opts T_RSQUARE
					{ $$ = $2; }
	| T_LSQUARE enumerand_opts error T_RSQUARE
					{ yyclearin; yyerrok; $$ = $2; }
	;

enumerand_opts_head
	: /* empty */			{ $$.options = NULL; $$.noptions = 0; }
	| enumerand_opts T_COMMA	{ $$ = $1; }
	| enumerand_opts error T_COMMA	{ $$.options = NULL; $$.noptions = 0; }
	;

enumerand_opts
	: /* empty */			{ $$.options = NULL; $$.noptions = 0; }
	| enumerand_opts_head option	{
		$$ = $1;
		assert($$.noptions < SIZE_MAX);
		ereallocarr(&$$.options, $$.noptions + 1,
		    sizeof($$.options[0]));
		$$.options[$$.noptions] = $2;
		$$.noptions++;
	}
	;

/* Services */

stmt_service
	: K_SERVICE id { push("service", $1, &$2.id); } service_block {
		$$.loc = $1;
		$$.id = $2.id;
		$$.stmts = $4;
		pop("service");
	}
	;

service_block
	: T_LCURLY service_stmts T_RCURLY
			{ $$ = $2; }
	| T_LCURLY service_stmts error T_RCURLY
			{ yyclearin; yyerrok; $$ = $2; }
	;

service_stmts
	: /* empty */	{ $$.stmts = NULL; $$.nstmts = 0; }
	| service_stmts service_stmt {
		$$ = $1;
		if ($2.t != AST_SERVICE_STMT_NULL) {
			assert($$.nstmts < SIZE_MAX);
			ereallocarr(&$$.stmts, $$.nstmts + 1,
			    sizeof($$.stmts[0]));
			$$.stmts[$$.nstmts] = $2;
			$$.nstmts++;
		}
	}
	;

service_stmt
	: T_SEMI	{ $$.t = AST_SERVICE_STMT_NULL; }
	| stmt_option	{ $$.t = AST_SERVICE_STMT_OPTION; $$.u.option = $1; }
	| stmt_rpc	{ $$.t = AST_SERVICE_STMT_RPC; $$.u.rpc = $1; }
	| error T_SEMI	{ yyclearin; yyerrok; $$.t = AST_SERVICE_STMT_NULL; }
	;

stmt_rpc
	: K_RPC id { push("rpc", $1, &$2.id); } rpc_end {
		$$.loc = $1;
		$$.id = $2.id;
		$$.intype = $4.intype;
		$$.outtype = $4.outtype;
		$$.options = $4.options;
		pop("rpc");
	}
	;

rpc_end
	: T_LROUND userdef_type T_RROUND
	    K_RETURNS T_LROUND userdef_type T_RROUND
	    rpc_optlist {
		$$.intype = $2;
		$$.outtype = $6;
		$$.options = $8;
	}
	| error T_SEMI {
		yyclearin;
		yyerrok;
		$$.intype.text = string_edupzz("(invalid type)");
		$$.outtype.text = string_edupzz("(invalid type)");
		$$.options.options = NULL;
		$$.options.noptions = 0;
	}
	;

rpc_optlist
	: T_SEMI		{ $$.options = NULL; $$.noptions = 0; }
	| T_LCURLY rpc_options T_RCURLY
				{ $$ = $2; }
	| T_LCURLY rpc_options error T_RCURLY
				{ yyclearin; yyerrok; $$ = $2; }
	;

rpc_options
	: /* empty */		{ $$.options = NULL; $$.noptions = 0; }
	| rpc_options stmt_option {
		$$ = $1;
		assert($$.noptions < SIZE_MAX);
		ereallocarr(&$$.options, $$.noptions + 1,
		    sizeof($$.options[0]));
		$$.options[$$.noptions] = $2;
		$$.noptions++;
	}
	;

/* Extend */

stmt_extend
	: K_EXTEND userdef_type
	    { push("extend", $1, &$2.text); }
	    extend_block {
		$$.loc = $1;
		$$.type = $2;
		$$.fields = $4;
		pop("extend");
	}
	;

extend_block
	: T_LCURLY extend_stmts T_RCURLY	{ $$ = $2; }
	| T_LCURLY extend_stmts error T_RCURLY	{ yyclearin; yyerrok;
						  $$ = $2; }
	;

extend_stmts
	: /* empty */		{ $$.fields = NULL; $$.nfields = 0; }
	| extend_stmts T_SEMI	{ $$ = $1; }
	| extend_stmts field {
		$$ = $1;
		assert($$.nfields < SIZE_MAX);
		ereallocarr(&$$.fields, $$.nfields + 1, sizeof($$.fields[0]));
		$$.fields[$$.nfields] = $2;
		$$.nfields++;
	}
	| extend_stmts error T_SEMI
				{ yyclearin; yyerrok; $$ = $1; }
	;

/* Import */

stmt_import
	: K_IMPORT import_qual string T_SEMI {
		$$.loc = $1;
		$$.qual = $2;
		$$.pathname = $3;
	}
	;

import_qual
	: /* empty */		{ $$ = AST_IQ_DEFAULT; }
	| K_PUBLIC		{ $$ = AST_IQ_PUBLIC; }
	| K_WEAK		{ $$ = AST_IQ_WEAK; }
	;

/* Package */

stmt_package
	: K_PACKAGE dotted_name T_SEMI {
		$$.loc = $1;
		$$.name = $2;
	}
	;

/* Option */

stmt_option
	: K_OPTION option_name
	    { push("option", $1, NULL /* XXX mention name */); }
	    option_end {
		$$.loc = $1;
		$$.name = $2;
		$$.value = $4;
		pop("option");
	}
	;

option
	: option_name T_EQUAL option_value {
		$$.loc = $2;	/* XXX Use location of name.  */
		$$.name = $1;
		$$.value = $3;
	}
	;

option_end
	: T_EQUAL option_value T_SEMI	{ $$ = $2; }
	| error T_SEMI {
		yyclearin;
		yyerrok;
		$$.t = AST_OPTVAL_STRING;
		$$.u.string = string_edupzz("error");
	}
	;

option_name
	: option_name_part_nondefault {
		$$.parts = NULL;
		$$.nparts = 0;
		assert($$.nparts < SIZE_MAX);
		ereallocarr(&$$.parts, $$.nparts + 1, sizeof($$.parts[0]));
		$$.parts[$$.nparts] = $1;
		$$.nparts++;
	}
	| option_name T_DOT option_name_part {
		assert($$.nparts < SIZE_MAX);
		ereallocarr(&$$.parts, $$.nparts + 1, sizeof($$.parts[0]));
		$$.parts[$$.nparts] = $3;
		$$.nparts++;
	}
	;

/*
 * XXX This is a bit of a cop-out.  Only in the context of field
 * options is `default' not allowed as an initial name, because it
 * would be confused with a default value.  Propagating this to all the
 * other cases of option_name is too painful and probably nobody uses
 * option names like that.
 */
option_name_part_nondefault
	: id_nondefault	{ $$.t = AST_OPTPART_NORMAL; $$.u.normal = $1.id; }
	| option_name_part_qualified
			{ $$ = $1; }
	;

option_name_part
	: id		{ $$.t = AST_OPTPART_NORMAL; $$.u.normal = $1.id; }
	| option_name_part_qualified
			{ $$ = $1; }
	;

option_name_part_qualified
	: T_LROUND fq_dotted_name T_RROUND {
		$$.t = AST_OPTPART_EXTENSION;
		$$.u.extension = $2;
	}
	| T_LROUND error T_RROUND {
		yyclearin;
		yyerrok;
		$$.t = AST_OPTPART_EXTENSION;
		$$.u.extension = string_edupzz("error");
	}
	;

option_value
	: L_INTEGER	{ $$.t = AST_OPTVAL_UINT; $$.u.uint = $1; }
	| T_HYPHEN L_INTEGER {
		$$.t = AST_OPTVAL_SINT;
		if ($2 > (uint64_t)INT64_MAX + 1) {
			yyerrorf("negative too small: -%"PRIuMAX, $2);
			$$.u.sint = INT64_MIN;
		} else {
			$$.u.sint = -$2;
		}
	}
	| L_REAL	{ $$.t = AST_OPTVAL_REAL; $$.u.real = $1; }
	| T_HYPHEN L_REAL
			{ $$.t = AST_OPTVAL_REAL; $$.u.real = -$2; }
	| K_TRUE	{ $$.t = AST_OPTVAL_BOOLEAN; $$.u.boolean = true; }
	| K_FALSE	{ $$.t = AST_OPTVAL_BOOLEAN; $$.u.boolean = false; }
	| string	{ $$.t = AST_OPTVAL_STRING; $$.u.string = $1; }
	| id_nonbool	{ $$.t = AST_OPTVAL_ID; $$.u.id = $1.id; }
/*
#ifdef notyet
	| T_LCURLY option_value_block T_RCURLY
			{ $$.t = AST_OPTVAL_BLOCK; $$.u.block = $2; }
	| T_LCURLY error T_RCURLY {
		yyclearin;
		yyerrok;
		$$.t = AST_OPTVAL_STRING;
		$$.u.string = string_edupzz("error");
	}
#endif
*/
	;

/* Names and types */

dotted_name
	: id		{ $$ = $1.id; }
	| dotted_name T_DOT id {
		const struct string dot = STRING_CONST(".");
		$$ = string_econcatn_into(&$1, &dot, &$3.id, NULL);
		string_free($3.id);
	}
	;

fq_dotted_name
	: id		{ $$ = $1.id; }
	| T_DOT id {
		const struct string dot = STRING_CONST(".");
		$$ = string_econcat(dot, $2.id);
		string_free($2.id);
	}
	| fq_dotted_name T_DOT id {
		const struct string dot = STRING_CONST(".");
		$$ = string_econcatn_into(&$1, &dot, &$3.id, NULL);
		string_free($3.id);
	}
	;

dotted_name_nongroup
	: id_nongroup	{ $$ = $1.id; }
	| dotted_name T_DOT id {
		const struct string dot = STRING_CONST(".");
		$$ = string_econcatn_into(&$1, &dot, &$3.id, NULL);
		string_free($3.id);
	}
	;

/*
 * XXX any_type and userdef_type are supposed to be distinguished.
 * Presumably userdef_type should reject built-in types?
 */

any_type
	: T_DOT dotted_name {
		const struct string dot = STRING_CONST(".");
		$$.text = string_econcat(dot, $2);
		string_free($2);
	}
	| dotted_name_nongroup	{ $$.text = $1; }
	;

userdef_type
	: T_DOT dotted_name {
		const struct string dot = STRING_CONST(".");
		$$.text = string_econcat(dot, $2);
		string_free($2);
	}
	| dotted_name_nongroup	{ $$.text = $1; }
	;

/* Strings */

string
	: L_STRING		{ $$ = $1; }
	| string L_STRING	{ $$ = string_econcat_into($1, $2);
				  string_free($2); }
	;

/* Kludges */

/*
 * Could do this with a lexer hack, but one-token lookahead makes that
 * tricky, so this'll do for now.
 */

id
	: id_nonbool_nondefault_nongroup
			{ $$ = $1; }
	| K_DEFAULT	{ $$.loc = $1; $$.id = string_edupzz("default"); }
	| K_FALSE	{ $$.loc = $1; $$.id = string_edupzz("false"); }
	| K_TRUE	{ $$.loc = $1; $$.id = string_edupzz("true"); }
	;

id_nonbool
	: id_nonbool_nondefault_nongroup
			{ $$ = $1; }
	| K_DEFAULT	{ $$.loc = $1; $$.id = string_edupzz("default"); }
	| K_GROUP	{ $$.loc = $1; $$.id = string_edupzz("group"); }
	;

id_nondefault
	: id_nonbool_nondefault_nongroup
			{ $$ = $1; }
	| K_FALSE	{ $$.loc = $1; $$.id = string_edupzz("false"); }
	| K_GROUP	{ $$.loc = $1; $$.id = string_edupzz("group"); }
	| K_TRUE	{ $$.loc = $1; $$.id = string_edupzz("true"); }
	;

id_nongroup
	: id_nonbool_nondefault_nongroup
			{ $$ = $1; }
	| K_DEFAULT	{ $$.loc = $1; $$.id = string_edupzz("default"); }
	| K_FALSE	{ $$.loc = $1; $$.id = string_edupzz("false"); }
	| K_TRUE	{ $$.loc = $1; $$.id = string_edupzz("true"); }
	;

id_nonbool_nondefault_nongroup
	: L_ID		{ $$ = $1; }
	| K_ENUM	{ $$.loc = $1; $$.id = string_edupzz("enum"); }
	| K_EXTEND	{ $$.loc = $1; $$.id = string_edupzz("extend"); }
	| K_EXTENSIONS	{ $$.loc = $1; $$.id = string_edupzz("extensions"); }
	| K_IMPORT	{ $$.loc = $1; $$.id = string_edupzz("import"); }
	| K_MAX		{ $$.loc = $1; $$.id = string_edupzz("max"); }
	| K_MESSAGE	{ $$.loc = $1; $$.id = string_edupzz("message"); }
	| K_OPTION	{ $$.loc = $1; $$.id = string_edupzz("option"); }
	| K_OPTIONAL	{ $$.loc = $1; $$.id = string_edupzz("optional"); }
	| K_PACKAGE	{ $$.loc = $1; $$.id = string_edupzz("package"); }
	| K_PUBLIC	{ $$.loc = $1; $$.id = string_edupzz("public"); }
	| K_REPEATED	{ $$.loc = $1; $$.id = string_edupzz("repeated"); }
	| K_REQUIRED	{ $$.loc = $1; $$.id = string_edupzz("required"); }
	| K_RETURNS	{ $$.loc = $1; $$.id = string_edupzz("returns"); }
	| K_RPC		{ $$.loc = $1; $$.id = string_edupzz("rpc"); }
	| K_SERVICE	{ $$.loc = $1; $$.id = string_edupzz("service"); }
	| K_SYNTAX	{ $$.loc = $1; $$.id = string_edupzz("syntax"); }
	| K_TO		{ $$.loc = $1; $$.id = string_edupzz("to"); }
	| K_WEAK	{ $$.loc = $1; $$.id = string_edupzz("weak"); }
	;

%%

void
picopbc_parse(FILE *file, const char *filename,
    const struct syntaxopts *opts)
{

	picopbc_scan_init(file, filename, opts);
	yyparse();
}

static struct string
string_edupzz(const char *s)
{

	return string_edupz(s, strlen(s));
}

static void
push(const char *desc, struct ast_srcloc loc, struct string *id)
{

	assert(nerrctx < SIZE_MAX);
	ereallocarr(&errctx, nerrctx + 1, sizeof(errctx[0]));
	errctx[nerrctx].desc = desc;
	errctx[nerrctx].loc = loc;
	errctx[nerrctx].id = id;
	nerrctx++;
}

static void
pop(const char *desc)
{

	assert(0 < nerrctx);
	assert(desc == errctx[nerrctx - 1].desc);
	/* Lazily free or reallocate.  */
	if (--nerrctx == 0) {
		free(errctx);
		errctx = NULL;
	}
}

void
yyerror(const char *message)
{

	yyerrorf("%s", message);
}

void attr_printflike(1,2)
yyerrorf(const char *fmt, ...)
{
	extern char *yyfilename;
	extern int yylineno;
	extern int yycolumn;
	extern char *yytext;
	extern int yyleng;
	char *message;
	struct string context;
	bool eol;
	va_list va;
	size_t i;

	error = 1;

	context = string_edupz(yytext, yyleng);
	eol = (string_ptr(context)[0] == '\n');

	va_start(va, fmt);
	evasprintf(&message, fmt, va);
	va_end(va);

	(void)fprintf(stderr, "%s:%d:%d: error: %s", yyfilename,
	    yylineno - (eol? 1 : 0), yycolumn, message);
	if (!eol) {
		char *v = string_evisciiz(context);
		(void)fprintf(stderr, " near `%s'", v);
		free(v);
	}
	(void)fprintf(stderr, "\n");
	string_free(context);

	for (i = nerrctx; 0 < i--;) {
		char *id = errctx[i].id == NULL? NULL :
		    string_evisciiz(*errctx[i].id);

		(void)fprintf(stderr, "  in %s%s%s at %s:%zu:%zu\n",
		    errctx[i].desc, id? " " : "", id? id : "",
		    errctx[i].loc.filename,
		    errctx[i].loc.lineno,
		    errctx[i].loc.column);
	}
}

static void
ast_option_destroy(const struct ast_option *option)
{
	const struct ast_optpart *parts = option->name.parts;
	size_t i, nparts = option->name.nparts;

	switch (option->value.t) {
	case AST_OPTVAL_ID:
		string_free(option->value.u.id);
		break;
	case AST_OPTVAL_UINT:
	case AST_OPTVAL_SINT:
	case AST_OPTVAL_REAL:
	case AST_OPTVAL_BOOLEAN:
		break;
	case AST_OPTVAL_STRING:
		string_free(option->value.u.string);
		break;
	default:
		/* XXX Should not happen.  */
		break;
	}

	for (i = 0; i < nparts; i++) {
		switch (parts[i].t) {
		case AST_OPTPART_NORMAL:
			string_free(parts[i].u.normal);
			break;
		case AST_OPTPART_EXTENSION:
			string_free(parts[i].u.extension);
			break;
		default:
			/* XXX Should not happen.  */
			break;
		}
	}

	free(option->name.parts);
}

static void
ast_type_destroy(const struct ast_type *type)
{

	string_free(type->text);
}

static void
ast_enumerand_destroy(const struct ast_enumerand *enumerand)
{
	const struct ast_option *options = enumerand->options.options;
	size_t i, noptions = enumerand->options.noptions;

	for (i = 0; i < noptions; i++)
		ast_option_destroy(&options[i]);

	free(enumerand->options.options);
	string_free(enumerand->id);
}

static void
ast_enumeration_destroy(const struct ast_enumeration *enumeration)
{
	const struct ast_enum_stmt *stmts = enumeration->stmts.stmts;
	size_t i, nstmts = enumeration->stmts.nstmts;

	for (i = 0; i < nstmts; i++) {
		switch (stmts[i].t) {
		case AST_ENUM_STMT_NULL:
			/* XXX Should not happen.  */
			break;
		case AST_ENUM_STMT_OPTION:
			ast_option_destroy(&stmts[i].u.option);
			break;
		case AST_ENUM_STMT_ENUMERAND:
			ast_enumerand_destroy(&stmts[i].u.enumerand);
			break;
		default:
			/* XXX Should not happen.  */
			break;
		}
	}

	free(enumeration->stmts.stmts);
	string_free(enumeration->id);
}

static void
ast_rpc_destroy(const struct ast_rpc *rpc)
{
	const struct ast_option *options = rpc->options.options;
	size_t i, noptions = rpc->options.noptions;

	for (i = 0; i < noptions; i++)
		ast_option_destroy(&options[i]);

	free(rpc->options.options);
	ast_type_destroy(&rpc->outtype);
	ast_type_destroy(&rpc->intype);
	string_free(rpc->id);
}

static void
ast_service_destroy(const struct ast_service *service)
{
	const struct ast_service_stmt *stmts = service->stmts.stmts;
	size_t i, nstmts = service->stmts.nstmts;

	for (i = 0; i < nstmts; i++) {
		switch (stmts[i].t) {
		case AST_SERVICE_STMT_NULL:
			/* XXX Should not happen.  */
			break;
		case AST_SERVICE_STMT_OPTION:
			ast_option_destroy(&stmts[i].u.option);
			break;
		case AST_SERVICE_STMT_RPC:
			ast_rpc_destroy(&stmts[i].u.rpc);
			break;
		default:
			/* XXX Should not happen.  */
			break;
		}
	}

	free(service->stmts.stmts);
	string_free(service->id);
}

static void
ast_package_destroy(const struct ast_package *package)
{

	string_free(package->name);
}

static void
ast_import_destroy(const struct ast_import *import)
{

	string_free(import->pathname);
}

static void
ast_extension_range_destroy(const struct ast_extension_range *range)
{

	(void)range;		/* ignore */
}

static void
ast_extensions_destroy(const struct ast_extensions *extensions)
{
	const struct ast_extension_range *ranges = extensions->ranges.ranges;
	size_t i, nranges = extensions->ranges.nranges;

	for (i = 0; i < nranges; i++)
		ast_extension_range_destroy(&ranges[i]);

	free(extensions->ranges.ranges);
}

static void
ast_fieldval_destroy(const struct ast_fieldval *fieldval)
{

	switch (fieldval->t) {
	case AST_FIELDVAL_UINT:
	case AST_FIELDVAL_SINT:
	case AST_FIELDVAL_REAL:
	case AST_FIELDVAL_BOOLEAN:
		break;
	case AST_FIELDVAL_STRING:
		string_free(fieldval->u.string);
		break;
	case AST_FIELDVAL_ID:
		string_free(fieldval->u.id);
		break;
	}
}

static void
ast_field_destroy(const struct ast_field *field)
{
	const struct ast_option *options = field->options.options;
	size_t i, noptions = field->options.noptions;

	for (i = 0; i < noptions; i++)
		ast_option_destroy(&options[i]);

	free(field->options.options);
	if (field->options.default_present)
		ast_fieldval_destroy(&field->options.default_value);
	string_free(field->id);
	ast_type_destroy(&field->type);
}

static void
ast_extend_destroy(const struct ast_extend *extend)
{
	const struct ast_field *fields = extend->fields.fields;
	size_t i, nfields = extend->fields.nfields;

	for (i = 0; i < nfields; i++)
		ast_field_destroy(&fields[i]);

	free(extend->fields.fields);
	ast_type_destroy(&extend->type);
}

static void
ast_message_stmts_destroy(const struct ast_message_stmts *msg_stmts)
{
	const struct ast_message_stmt *stmts = msg_stmts->stmts;
	size_t i, nstmts = msg_stmts->nstmts;

	for (i = 0; i < nstmts; i++) {
		switch (stmts[i].t) {
		case AST_MESSAGE_STMT_NULL:
			/* XXX Should not happen.  */
			break;
		case AST_MESSAGE_STMT_MESSAGE:
			ast_message_destroy(&stmts[i].u.message);
			break;
		case AST_MESSAGE_STMT_ENUMERATION:
			ast_enumeration_destroy(&stmts[i].u.enumeration);
			break;
		case AST_MESSAGE_STMT_EXTEND:
			ast_extend_destroy(&stmts[i].u.extend);
			break;
		case AST_MESSAGE_STMT_OPTION:
			ast_option_destroy(&stmts[i].u.option);
			break;
		case AST_MESSAGE_STMT_EXTENSIONS:
			ast_extensions_destroy(&stmts[i].u.extensions);
			break;
		case AST_MESSAGE_STMT_FIELD:
			ast_field_destroy(&stmts[i].u.field);
			break;
		default:
			/* XXX Should not happen.  */
			break;
		}
	}

	free(msg_stmts->stmts);
}

static void
ast_message_destroy(const struct ast_message *message)
{

	ast_message_stmts_destroy(&message->stmts);
	string_free(message->id);
}

static void
ast_proto_destroy(const struct ast_proto *proto)
{
	const struct ast_proto_stmt *stmts = proto->stmts.stmts;
	size_t i, nstmts = proto->stmts.nstmts;

	for (i = 0; i < nstmts; i++) {
		switch (stmts[i].t) {
		case AST_PROTO_STMT_NULL:
			/* XXX Should not happen.  */
			break;
		case AST_PROTO_STMT_MESSAGE:
			ast_message_destroy(&stmts[i].u.message);
			break;
		case AST_PROTO_STMT_ENUMERATION:
			ast_enumeration_destroy(&stmts[i].u.enumeration);
			break;
		case AST_PROTO_STMT_SERVICE:
			ast_service_destroy(&stmts[i].u.service);
			break;
		case AST_PROTO_STMT_EXTEND:
			ast_extend_destroy(&stmts[i].u.extend);
			break;
		case AST_PROTO_STMT_IMPORT:
			ast_import_destroy(&stmts[i].u.import);
			break;
		case AST_PROTO_STMT_PACKAGE:
			ast_package_destroy(&stmts[i].u.package);
			break;
		case AST_PROTO_STMT_OPTION:
			ast_option_destroy(&stmts[i].u.option);
			break;
		default:
			/* XXX Should not happen.  */
			break;
		}
	}

	free(proto->stmts.stmts);
}
