/*-- portable graphics metacode reader

 * This program reads the portable graphics metacode and draws
 * graphics on hardware.  All hardware dependence, however, is
 * confined to a few low level routines starting with "pg".
 * This program shows its fortran origins in that array indices
 * typically start at 1 rather than 0.  This means that a bit of
 * space is wasted, but it is easier to keep this convention
 * than to change the entire program.
 */

#include <stdio.h>
#include <math.h>
#include <stdlib.h>

#define UNIQUEINFILE 1 /* set to 1 to copy input file to unique location --
                          this will probably only work for unix */

#define MAXAX 1.e20  /* keep axis values from getting too big */
#define SMALL 1.e-20 /* a small number limit */
#define BADDATA 1.e6 /* data exceeding this is assumed bad (default) */
#define FRACT 0.6328 /* constant for fill routines */
#define VLEN (-0.3)  /* vector arrow length */
#define VWID 0.1     /* vector arrow width */
#define MAG 50002    /* maximum size of line data buffers */
#define XPT 1001     /* maximum number of different x values */
#define YPT 1001     /* maximum number of different y values */
#define PPT 301      /* for perspective viewing routine ``surface'' */
#define NSEG 20      /* number of axis segments in perspective view */
#define THETAMAX 1.5 /* maximum elevation (radians) for perspective view */
#define WN  100      /* number of windows allowed */
#define NA 21        /* number of entries in window info block */

/* static (local) declarations */
static float a[NA];         /* current window information block */
static float dax[MAG],day[MAG]; /* line data buffers */
static float sav[11][NA];   /* saved window information blocks */
static float baddata;       /* actual bad data limit */

/* internal functions */
static float dott(),cinty(),cintx(),maglim();
static void panic(),rest(),draw(),mark(),conv(),norm(),mlabel();
static int survey(),cside(),insect(),conto(),vplot(),axes(),
     grid(),kfill(),nfill(),pfill(),xx(),pdraw();

/* global declarations -- for communication with the low level routines */
FILE *pgraf;         /* metacode input file */
int argcglobal;      /* location for command line arg stuff */
char **argvglobal;
float zoomfactor,deltheta,delphi,radfactor; /* for changes in perspective */
int viewinfo;        /* tells whether to display perspective info */
char tempfilename[40];  /* name of temporary pgraf3.out file */

main(argc,argv)
int argc;
char *argv[];
{
  char alfa[80];
  char tag;
  int id,nw,ntx,nty,ntz,ltype,mtype,num,nx,ny,jx,jy,i,dry,drx,shw;
  float xl,yl,xh,yh,sx,sy,sxl,sxh,syl,syh,szl,szh,dz,dvl,dvh;
  float xdat,ydat,msize,xp[3],f[3],zoom;
  float xstretch,ystretch,zstretch,fx,fy,fz;
  float xrel,yrel,zrel,rho,radius,theta,phi;

  /* globalize command line stuff */
  argcglobal = argc;
  argvglobal = argv;

  /* initialize bad data limit to default */
  baddata = BADDATA;

  /* construct a unique file name */
#if UNIQUEINFILE
  sprintf(tempfilename,"pgraf3.%d",getpid());

  /* copy the metacode file to the new name */
  sprintf(alfa,"cp pgraf3.out %s",tempfilename);
  if (system(alfa)) {
    fprintf(stderr,"error in executing `%s'\n",alfa);
    exit(1);
  }
#else
  strcpy(tempfilename,"pgraf3.out");
#endif

/* initialize variables for perspective viewing changes */
  zoomfactor = 1.;
  radfactor = 1.;
  deltheta = 0.;
  delphi = 0.;
  viewinfo = 0;

  /* open metacode file */
  if ((pgraf = fopen(tempfilename,"r")) == NULL) {
    fprintf(stderr,"can't open pgraf3.out\n");
    exit(1);
  }

  /* get a line */
  while (fscanf(pgraf,"%s",alfa) != EOF) {

    /* find first non-blank character */
    for (i=0; i<80; i++) {
      tag = alfa[i];
      if (tag != ' ')  break;
    }

    /* initialization */
    if (tag == 'i') {
      pginit();
    }

    /* set bad data value */
    else if (tag == 'x') {
      if (fscanf(pgraf,"%f",&baddata) == EOF) panic();
    }

    /* make a window */
    else if (tag == 'w') {
      if (fscanf(pgraf,"%d%f%f%f%f%d",&nw,&xl,&yl,&xh,&yh,&shw)
          == EOF) panic();
      nw = nw%WN;
      a[1] = xl;
      a[2] = xh;
      a[3] = yl;
      a[4] = yh;

      /* preventing the window from hitting the edge of the screen --
       * lines that hit the edge of such a window finish in odd ways
       */
      if(xh == 15.) xh = 14.99 ;
      if(yh == 10.) yh = 9.99;
      if(xl == 0.)  xl = .01;
      if(yl == 0.)  yl = .01;
      if(shw == 1){
        pgmove(xl,yl);
        pgdraw(xl,yh,1);
        pgdraw(xh,yh,1);
        pgdraw(xh,yl,1);
        pgdraw(xl,yl,1);
      }

      /* default parameter values */
      for (id = 5; id < NA; id++) a[id] = 0.;
      a[6] = 1.;
      a[8] = 1.;
      a[11] = 1.;
      a[12] = 1.;
      a[17] = 1.;
      a[18] = 1.;
      a[19] = 1.;
      a[20] = 1.;
      for (id = 1; id < NA; id++)
        sav[nw][id] = a[id];
    }

    /* draw axes */
    else if (tag == 'a') {
      if (fscanf(pgraf,"%d%f%f%f%f%d%d%d%d",
          &nw,&sxl,&sxh,&syl,&syh,&ntx,&nty,&drx,&dry) == EOF) panic();
      rest(nw);
      a[5] = maglim(MAXAX,sxl);
      a[6] = maglim(MAXAX,sxh);
      a[7] = maglim(MAXAX,syl);
      a[8] = maglim(MAXAX,syh);
      axes(ntx,nty,drx,dry);
      for (id = 1; id < NA; id++)
        sav[nw][id] = a[id];
    }

    /* draw a line */
    else if (tag == 'l') {
      if (fscanf(pgraf,"%d%d%d",&nw,&ltype,&num) == EOF)
        panic();
      rest(nw);
      if (num >= MAG) {
        pgflsh();
        fprintf(stderr,"pgraf: error caused by line > %d points\n",MAG - 1);
        cexit(1);
      }
      for (id=1; id <= num; id++) {
        if (fscanf(pgraf,"%f%f",&xdat,&ydat) == EOF) panic();
        dax[id] = xdat;
        day[id] = ydat;
      }
      draw(ltype,num,dax,day);
    }

    /* make a mark */
    else if (tag == 'm') {
      if (fscanf(pgraf,"%d%d%f%f%f",
          &nw,&mtype,&xdat,&ydat,&msize) == EOF) panic();
      rest(nw);
      mark(mtype,xdat,ydat,msize);
    }

    /* make contour plot */
    else if (tag == 'c') {
      if (fscanf(pgraf,"%d%d%d%f%d",&nw,&nx,&ny,&dz,&ltype)
          == EOF) panic();
      rest(nw);

      /* check for proper size ranges */
      if(nx > XPT){
        fprintf(stderr,"%s: number of points %d exceeds maximum of %d\n",
            argv[0],nx,XPT);
        cexit(1);
      }
      if(ny > YPT){
        fprintf(stderr,"%s: number of points %d exceeds maximum of %d\n",
            argv[0],ny,YPT);
        cexit(1);
      }
      conto(nx,ny,dz,ltype,dax,day);
    }

    /* make fill plot */
    else if (tag == 'd') {
      if (fscanf(pgraf,"%d%d%d%f%f%d",
          &nw,&nx,&ny,&dvl,&dvh,&ltype) == EOF) panic();
      if(nx > XPT){
        fprintf(stderr,"%s: number of points %d exceeds maximum of %d\n",
            argv[0],nx,XPT);
        cexit(1);
      }
      if(ny > YPT){
        fprintf(stderr,"%s: number of points %d exceeds maximum of %d\n",
            argv[0],ny,YPT);
        cexit(1);
      }
      rest(nw);
      kfill(nx,ny,dvl,dvh,ltype,dax,day);
    }

    /* make a vector plot */
    else if (tag == 'v') {
      if (fscanf(pgraf,"%d%d%d%f%f",&nw,&nx,&ny,&sx,&sy)
          == EOF) panic();
      if(nx > XPT){
        fprintf(stderr,"%s: number of points %d exceeds maximum of %d\n",
            argv[0],nx,XPT);
        cexit(1);
      }
      if(ny > YPT){
        fprintf(stderr,"%s: number of points %d exceeds maximum of %d\n",
            argv[0],ny,YPT);
        cexit(1);
      }
      rest(nw);
      vplot(nx,ny,sx,sy);
    }

    /* write a label */
    else if (tag == 'b') {
      if (fscanf(pgraf,"%d%f%f",&nw,&sx,&sy) == EOF) panic();
      if (fgets(alfa,80,pgraf) == NULL) break;
      if (fgets(alfa,80,pgraf) == NULL) break;
      rest(nw);
      mlabel(sx,sy,alfa);
    }

    /* make a grid of numbers in a window */
    else if (tag == 'g') {
      if (fscanf(pgraf,"%d%d%d%d%d",&nw,&nx,&ny,&jx,&jy)
          == EOF) panic();
      if(nx > XPT){
        fprintf(stderr,"%s: number of points %d exceeds maximum of %d\n",
            argv[0],nx,XPT);
        cexit(1);
      }
      if(ny > YPT){
        fprintf(stderr,"%s: number of points %d exceeds maximum of %d\n",
            argv[0],ny,YPT);
        cexit(1);
      }
      rest(nw);
      grid(nx,ny,jx,jy);
    }

    /* set up 2-D perspective view of 3-D object */
    else if (tag == 'p') {
      if (fscanf(pgraf,"%d%f%f%f%f%f%f%f",
          &nw,&xp[0],&xp[1],&xp[2],&f[0],&f[1],&f[2],&zoom)
          == EOF) panic();

      /* make viewer-induced modifications in viewing position */
      zoom *= zoomfactor*radfactor;
      xrel = xp[0] - f[0];
      yrel = xp[1] - f[1];
      zrel = xp[2] - f[2];
      rho = pow(xrel*xrel + yrel*yrel,0.5);
      radius = pow(rho*rho + zrel*zrel,0.5)*radfactor;
      if (rho < SMALL) {
	fprintf(stderr,"perspective view from top or bottom not allowed\n");
	cexit(1);
      }
      theta = atan2(zrel,rho) + deltheta;
      phi = atan2(yrel,xrel) + delphi;
      if (theta > THETAMAX) theta = THETAMAX;
      if (theta < -THETAMAX) theta = -THETAMAX;
      xp[0] = f[0] + radius*cos(theta)*cos(phi);
      xp[1] = f[1] + radius*cos(theta)*sin(phi);
      xp[2] = f[2] + radius*sin(theta);

/* restore window and print parameters to window */
      rest(nw);
      if (viewinfo) {
	sprintf(alfa,"viewpoint: %5g %5g %5g; zoom: %5g",
		xp[0],xp[1],xp[2],zoom);
	mlabel(0.,0.96,alfa);
      }

      /* compute focal direction */
      for (id = 0; id < 3; id++) {
        a[id + 9] = xp[id];
        f[id] -= xp[id];
      }

      /* compute normals to focal direction */
      a[12] = f[1];
      a[13] = -f[0];
      a[14] = 0.;
      a[15] = -f[0]*f[2];
      a[16] = -f[1]*f[2];
      a[17] = f[0]*f[0] + f[1]*f[1];
      norm(&a[12]);
      norm(&a[15]);

      /* get scaling right */
      if (zoom < SMALL) zoom = SMALL;
      a[6] = 1./zoom;
      a[5] = -a[6];
      if (a[2] - a[1] < SMALL) {
	fprintf(stderr,"window dimensions not properly defined\n");
	cexit(1);
      }
      else a[8] = a[6]*(a[4] - a[3])/(a[2] - a[1]);
      a[7] = -a[8];

      /*save it */
      for (id = 1; id < NA; id++) sav[nw][id] = a[id];
    }

    /* draw axes in perspective */
    else if (tag == 'h') {
      if (fscanf(pgraf,"%d%f%f%f%f%f%f%f%f%f",
          &nw,&sxl,&sxh,&xstretch,&syl,&syh,&ystretch,
          &szl,&szh,&zstretch) == EOF) panic();
      rest(nw);
      a[18] = xstretch;
      a[19] = ystretch;
      a[20] = zstretch;
      for (id = 1; id < NA; id++) sav[nw][id] = a[id];
      paxes(sxl,sxh,ntx,syl,syh,nty,szl,szh,ntz);
    }

    /* draw line in perspective */
    else if ( tag == 's') {
      if (pdraw() != 0) panic();
    }

    /* make a label in perspective */
    else if (tag == 'e') {
      if (fscanf(pgraf,"%d%f%f%f",&nw,&fx,&fy,&fz)
          == EOF) panic();
      if (fgets(alfa,80,pgraf) == NULL) panic();
      if (fgets(alfa,80,pgraf) == NULL) panic();
      rest(nw);
      project(fx,fy,fz,&sx,&sy);
      mlabel(sx,sy,alfa);
    }

    /* make a perspective view surface plot */
    else if (tag == 'j') {
      if (fscanf(pgraf,"%d%d%d%d",&nw,&nx,&ny,&ltype)
          == EOF) panic();
      rest(nw);

      /* check for proper size ranges */
      if(nx > PPT){
        fprintf(stderr,"%s: number of points %d exceeds maximum of %d\n",
            argv[0],ny,PPT);
        cexit(1);
      }
      if(ny > PPT){
        fprintf(stderr,"%s: number of points %d exceeds maximum of %d\n",
            argv[0],ny,PPT);
        cexit(1);
      }
      surface(nx,ny,ltype);
    }

    /* flush */
    else if (tag == 'f') {
      pgflsh();
    }

    /* debugging point out location*/
    else if(tag == 'u')
      fprintf(stderr,"Here \n");

      /* else there is an error */
    else {
      fprintf(stderr,"unrecognized command = %c\n",tag);
      break;
    }

    /* loop back to new command */
  }

  /* normal exit -- first remove temporary file */
  cexit(0);
}

/* cexit -- do a clean exit */
cexit(status)
int status;
{
#if UNIQUEINFILE
  char alfa[80];
  sprintf(alfa,"rm -f %s",tempfilename);
  system(alfa);
#endif
  exit(status);
}

static void panic()
{
  pgflsh();
  fprintf(stderr,"pgraf: unexpected end of file in pgraf3.out\n");
  cexit(1);
}

static void rest(nw)

/* restore active window */
int nw;
{
  int i;
  if (nw < 1) nw = 1;
  if (nw > 10) nw = 10;
  for (i = 1; i < NA; i++)
    a[i] = sav[nw][i];
}

static int conto(nx,ny,dz,ltyp,c,b)

/* contour plotting routine */
int nx,ny,ltyp;
float dz,c[],b[];
{
  float zv[6],xo,yo,z,zlmin,zlmax,vl[2][4],dx,dy,x1,x2,y1,y2,sxl,sxh,syl,syh;
  float lox[XPT + 1],loy[YPT + 1];
  int ix,iy,low,ihi,num,j,i,nvect;
  x1 = a[1];
  x2 = a[2];
  y1 = a[3];
  y2 = a[4];
  sxl= a[5];
  sxh = a[6];
  syl  = a[7];
  syh = a[8];

  /* calc dx and dy */
  dx = (x2 - x1)/(sxh - sxl);
  dy = (y2 - y1)/(syh - syl);
  xo = x1 -(sxl*dx);
  yo = y1 -(syl*dy);

  /* get x and y point locations*/
  for(ix = 0; ix <nx; ix++)
    if(fscanf(pgraf,"%f",&lox[ix+1]) == EOF) panic();
  for(iy = 0; iy <ny; iy++)
    if(fscanf(pgraf,"%f",&loy[iy+1]) == EOF) panic();

  /* get first row */
  for (ix=1; ix<=nx; ix++)
    if(fscanf(pgraf,"%f",&c[ix]) == EOF) panic();

  /* loop on rows */
  for (iy=2; iy<=ny; iy++) {

    /* get next row */
    for (ix=1; ix<=nx; ix++){
      b[ix] = c[ix];
    }
    for (ix=1; ix<=nx; ix++)
      if (fscanf(pgraf,"%f",&c[ix])== EOF) return(1);

    /* loop on columns */
    for (ix=2; ix<=nx;ix++) {

      /* get corner values */
      zv[1] = b[ix - 1];
      if (fabs(zv[1]) > baddata) continue;
      zv[2] = c[ix - 1];
      if (fabs(zv[2]) > baddata) continue;
      zv[3] = c[ix];
      if (fabs(zv[3]) > baddata) continue;
      zv[4] = b[ix];
      if (fabs(zv[4]) > baddata) continue;
      zv[5] = zv[1];

      /* find local mins and maxes */
      zlmin = zv[1];
      zlmax = zv[1];
      for (i=1; i<=4; i++) {
        if (zlmin > zv[i]) zlmin = zv[i];
        if (zlmax < zv[i]) zlmax = zv[i];
      }
      if (fabs(dz) > SMALL) {
        low = zlmin/dz;
        ihi = zlmax/dz;
        num = ihi - low + 1;

        /* loop on range */
        for (i=1; i<=num; i++) {
          z = (i - 1 + low)*dz;

          /* compute vectors and draw them */
          nvect = survey(dx,dy,xo,yo,z,zv,vl,lox,loy,ix,iy);
          for (j = 0; j < nvect; j++) {
            xx(vl[j][0],vl[j][1],vl[j][2],vl[j][3],ltyp);
          }
        }
      }
    }
  }
}

static int survey(dx,dy,xo,yo,z,zv,vlist,xloc,yloc,ix,iy)

/* make a survey of a box for possible contours -- returns number
   of contour vectors in box at value z and vector end points */
float dx,dy,xo,yo,z,zv[6],vlist[2][4],xloc[],yloc[];
int ix,iy ;
{
  int i,nints;
  float x[4],y[4],zdenom,xpos,ypos,sum;

  /* loop on sides, finding intersection points */
  nints = 0;
  for (i = 1; i < 5; i++) {
    if (((z >= zv[i]) && (z < zv[i + 1])) || 
        ((z < zv[i]) && (z >= zv[i + 1]))) {
      zdenom = zv[i + 1] - zv[i];
      if (fabs(zdenom) > SMALL){
        xpos = ((z - zv[i])/zdenom)*(xloc[ix] - xloc[ix-1]);
        ypos = ((z - zv[i])/zdenom)*(yloc[iy] - yloc[iy-1]);
      }
      else {
        xpos = (xloc[ix] - xloc[ix - 1])/2.;
        ypos = (yloc[iy] - xloc[iy - 1])/2.;
      }
      if (i == 1) {
        x[nints] = xloc[ix-1];
        y[nints] = yloc[iy - 1] + ypos;
      }
      else if (i == 2) {
        x[nints] = xloc[ix - 1] + xpos;
        y[nints] = yloc[iy];
      }
      else if (i == 3) {
        x[nints] = xloc[ix] ;
        y[nints] = yloc[iy] - ypos;
      }
      else {
        x[nints] = xloc[ix] - xpos;
        y[nints] = yloc[iy-1];
      }
      x[nints] = x[nints]*dx + xo;
      y[nints] = y[nints]*dy + yo;
      nints++;
    }
  }

  /* there should be 0, 2, or 4 intersections */
  if (nints == 0) return(0);
  else if (nints == 2) {
    vlist[0][0] = x[0];
    vlist[0][1] = y[0];
    vlist[0][2] = x[1];
    vlist[0][3] = y[1];
    vlist[1][0] = vlist[1][1] = vlist[1][2] = vlist[1][3] = 0.;
    return(1);
  }
  else if (nints == 4) {
    vlist[0][0] = x[0];
    vlist[0][1] = y[0];
    vlist[1][0] = x[2];
    vlist[1][1] = y[2];
    sum = (x[0] - x[1])*(x[0] - x[1]) + (y[0] - y[1])*(y[0] - y[1])
        + (x[2] - x[3])*(x[2] - x[3]) + (y[2] - y[3])*(y[2] - y[3])
        - (x[0] - x[3])*(x[0] - x[3]) - (y[0] - y[3])*(y[0] - y[3])
        - (x[1] - x[2])*(x[1] - x[2]) - (y[1] - y[2])*(y[1] - y[2]);
    if (sum > 0.) {
      vlist[0][2] = x[3];
      vlist[0][3] = y[3];
      vlist[1][2] = x[1];
      vlist[1][3] = y[1];
    }
    else {
      vlist[0][2] = x[1];
      vlist[0][3] = y[1];
      vlist[1][2] = x[3];
      vlist[1][3] = y[3];
    }
    return(2);
  }
  else {
    fprintf(stderr,"survey: nints = %d -- fatal error\n",nints);
    cexit(1);
  }
}

static int vplot(nx,ny,sx,sy)

/* make vector plot */
int nx,ny;
float sx,sy;
{
  int ix,iy,i;
  float x,y,px[6],py[6],rx,ry,sxl,sxh,syl,syh,xl,yl,xh,yh;
  float xo,yo,lox[XPT + 1],loy[YPT + 1],fx[2],fy[2],idx,idy,lx,ly,ng;
  sxl = a[5];
  sxh = a[6];
  syl = a[7];
  syh = a[8];
  xl  = a[1];
  xh  =a[2];
  yl  =a[3];
  yh =a[4];
  px[1] = 0.;
  py[1] = 0.;

  /*read point locations*/
  for(ix = 1;ix<=nx ; ix++)
    if(fscanf(pgraf,"%f",&lox[ix]) ==EOF) panic();
  for(iy = 1; iy<=ny;iy++)
    if(fscanf(pgraf,"%f",&loy[iy]) ==EOF) panic();

  /* loop on data points, drawing arrows */
  for (iy=1; iy <= ny; iy++) {
    for (ix=1; ix<=nx; ix++) {
      if (fscanf(pgraf,"%f%f",&x,&y) == EOF) return(1);

      /* skip bad data */
      if ((fabs(x) > baddata) || (fabs(y) > baddata)) continue;
      idx = (xh-xl)/(sxh-sxl);
      idy = (yh-yl)/(syh-syl);
      xo  = xl -(sxl*idx);
      yo  = yl -(syl*idy);
      rx = x*sx*idx ;
      ry = y*sy*idy ;
      lx = lox[ix]*idx +xo;
      ly = loy[iy]*idy +yo;

      /*draw the vector  */
      xx(lx,ly,(rx+lx),(ry+ly),1);
      ng = 1;

      /* draw the arrow */
      for(i=0;i<=1;i++){
        fx[i] = rx+lx + VLEN*rx +VWID*ry*ng ;
        fy[i] = ry+ly + VLEN*ry -VWID*rx*ng ;
        xx((lx+rx),(ly+ry),fx[i],fy[i],1);
        ng = -1;
      }
      xx(fx[0],fy[0],fx[1],fy[1],1);
    }
  }
}

static void draw(ltype,num,sx,sy)

/* draw a line of a given type */
int ltype,num;
float *sx, *sy;
{
  float x1,x2,y1,y2,sx1,sx2,sy1,sy2,x,y,oldy,oldx;
  int i;
  x1 = a[1];
  x2 = a[2];
  y1 = a[3];
  y2 = a[4];
  sx1 = a[5];
  sx2 = a[6];
  sy1 = a[7];
  sy2 = a[8];
  x = y = 0.;
  for (i = 1; i <= num; i++) {
    oldx = x;
    oldy = y;
    x = (sx[i] - sx1)/(sx2 - sx1);
    y = (sy[i] - sy1)/(sy2 - sy1);
    x = x1 + x*(x2 - x1);
    y = y1 + y*(y2 - y1);
    if (i > 1 && fabs(sx[i]) < baddata && fabs(sy[i]) < baddata &&
	fabs(sx[i - 1]) < baddata && fabs(sy[i - 1]) < baddata) {
      xx(oldx,oldy,x,y,ltype);
    }
  }
}

static void mark(mtype,sx,sy,size)

/* make a mark of specified type, position, and size */
int mtype;
float sx,sy,size;
{
  float x1,x2,y1,y2,sx1,sx2,sy1,sy2,x,y,cdx,cdy,bsize;
  int color;
  if ((fabs(sx) > baddata) || (fabs(sy) > baddata)) return;
  x1 = a[1];               /* get physical and symbolic window sizes */
  x2 = a[2];
  y1 = a[3];
  y2 = a[4];
  sx1 = a[5];
  sx2 = a[6];
  sy1 = a[7];
  sy2 = a[8];
  x = (sx - sx1)/(sx2 - sx1);   /* physical positions */
  y = (sy - sy1)/(sy2 - sy1);

  /* don't scribble outside of the canvas */
  if (x >= 0. && x <= 1. && y >= 0. && y <= 1.) {
    x = x1 + x*(x2 - x1);
    y = y1 + y*(y2 - y1);
    pginfo(&cdx,&cdy);       /* size of symbol */
    size *= cdx;
    bsize = 1.4*size;
    color = 0;
    while (mtype < 1) {
      mtype += 4;
      color--;
    }
    while (mtype > 4) {
      mtype -= 4;
      color++;
    }
    if (mtype == 1) {    /* plus sign */
      pgmove(x,y - bsize);
      pgdraw(x,y + bsize,1 + 4*color);
      pgmove(x - bsize,y);
      pgdraw(x + bsize,y,1 + 4*color);
    }
    else if (mtype == 2) {    /* times sign */
      pgmove(x - size,y - size);
      pgdraw(x + size,y + size,1 + 4*color);
      pgmove(x - size,y + size);
      pgdraw(x + size,y - size,1 + 4*color);
    }
    else if (mtype == 3) {    /* square */
      pgmove(x - size,y - size);
      pgdraw(x + size,y - size,1 + 4*color);
      pgdraw(x + size,y + size,1 + 4*color);
      pgdraw(x - size,y + size,1 + 4*color);
      pgdraw(x - size,y - size,1 + 4*color);
    }
    else {                    /* diamond */
      pgmove(x - bsize,y);
      pgdraw(x,y - bsize,1 + 4*color);
      pgdraw(x + bsize,y,1 + 4*color);
      pgdraw(x,y + bsize,1 + 4*color);
      pgdraw(x - bsize,y,1 + 4*color);
    }
  }
}

static int axes(ntx,nty,drx,dry)

/* make axes */
int ntx,nty,drx,dry;
{
  int i;
  float x1,x2,y1,y2,sx1,sx2,sy1,sy2,cdx,cdy,
  x,y,dx,dy,dsx,dsy,sx,sy,smax;
  char value[40];
  char gamma[80];
  char alfa[38],beta[38];
  x1 = a[1];
  x2 = a[2];
  y1 = a[3];
  y2 = a[4];
  sx1 = a[5];
  sx2 = a[6];
  sy1 = a[7];
  sy2 = a[8];

  /* find character dimensions */
  pginfo(&cdx,&cdy);

  /* redefine x1 and y1 */
  if(dry == 1) x1 = x1 + 7.0000*cdx;
  if(drx == 1) y1 = y1 + 4.0000*cdy;
  a[1] = x1;
  a[3] = y1;

  /* draw new axis lines */
  if(dry == 1){
    pgmove(x1,y2);
    pgdraw(x1,y1,1);
  }
  if(drx == 1){
    pgmove(x1,y1);
    pgdraw(x2,y1,1);
  }

  /* draw vertical tics and numbers */
  dsy = (sy2 - sy1)/(nty - 1);
  dy = (y2 - y1)/(nty - 1);
  smax = 0.;
  for (i = 2 - drx ; i < nty; i++) {
    sy = sy1 + (i - 1)*dsy;
    if (fabs(sy) > smax) smax = fabs(sy);
  }
  for (i = 2 - drx ; i < nty; i++) {
    y = y1 + (i - 1)*dy;
    pgmove(x1,y);
    pgdraw(x1 + 0.2,y,1);
    pgmove(x2,y);
    pgdraw(x2 - 0.2,y,1);
    sy = sy1 + (i - 1)*dsy;
    conv(sy,smax,value);
    if(dry == 1) pgtext(x1 - (strlen(value) + 1)*cdx,y,value,5);
  }

  /* draw horizontal tics and numbers */
  dx = (x2 - x1)/(ntx - 1);
  dsx = (sx2 - sx1)/(ntx - 1);
  smax = 0.;
  for ( i= 2 - dry; i < ntx; i++) {
    sx = sx1 + (i - 1)*dsx;
    if (fabs(sx) > smax) smax = fabs(sx);
  }
  for ( i= 2 - dry; i < ntx; i++) {
    x = x1 + (i - 1)*dx;
    pgmove(x,y1);
    pgdraw(x,y1 + 0.2,1);
    pgmove(x,y2);
    pgdraw(x,y2 - 0.2,1);
    sx = sx1 + (i - 1)*dsx;
    conv(sx,smax,value);
    if(drx == 1) pgtext(x - 0.5*(strlen(value) + 1)*cdx,y1 - 1.5*cdy,value,5);
  }

  /* make label */
  if (fgets(alfa,80,pgraf) == NULL) return(1);
  if (fgets(alfa,80,pgraf) == NULL) return(1);
  if (fgets(beta,80,pgraf) == NULL) return(1);

  /* remove cr from beta */
  for (i=0; i<38; i++) {
    if (beta[i] == '\0') {
      beta[i-1] = ' ';
      break;
    }
  }

  /* remove cr from alfa */
  for (i=0; i<38; i++) {
    if (alfa[i] == '\0') {
      alfa[i-1] = ' ';
      break;
    }
  }
  if(dry == 1 && drx == 1){
    sprintf(gamma,"%svs %s",beta,alfa);

    /* comments currently cut out when one of the axes is not drawn */
    /*else if(drx == 1) sprintf(gamma,"%s",alfa);
          else if(dry ==1) pgtext(x1-7.000*cdx,y1+cdy, beta)*/;
    pgtext(x1 - (7.0000 - 1.)*cdx,y1 - 3.*cdy,gamma,80);
  }
}

static void conv(x,xmax,value)
float x,xmax;
char *value;
{
  if (xmax < 10.) sprintf(value,"%.2f",x);
  else if (xmax < 100.) sprintf(value,"%.1f",x);
  else sprintf(value,"%.0f",x);
}

static int grid(nx,ny,jx,jy)

/* make a grid of numbers in a window */
int nx,ny,jx,jy;
{
  char value[51];
  char *vptr;
  float halflength;
  int ix,iy;
  float x1,x2,y1,y2,x,y,cdx,cdy,xo,yo,dx,dy,sxl,sxh,syl,syh,
    lox[XPT + 1],loy[YPT + 1];
  x1 = a[1];
  x2 = a[2];
  y1 = a[3];
  y2 = a[4];
  sxl = a[5];
  sxh = a[6];
  syl = a[7];
  syh = a[8];

  /* read marked point locations*/
  for(ix = 1 + jx/2; ix < nx - 1; ix += jx) {
    if(fscanf(pgraf,"%f",&lox[ix]) == EOF) return(0);
  }
  for(iy = 1 + jy/2; iy < ny - 1; iy += jy) {
    if(fscanf(pgraf,"%f",&loy[iy]) == EOF) return(0);
  }
  dx = (x2 - x1)/(sxh - sxl);
  dy = (y2 - y1)/(syh - syl);
  xo = x1 - (sxl*dx);
  yo = y1 - (syl*dy);

  /* read past end of line */
  if (fgets(value,8,pgraf) == NULL) return(1);
  pginfo(&cdx,&cdy);
  for (iy = 1 + jy/2; iy < ny - 1; iy += jy) {
    for (ix = 1 + jx/2; ix < nx - 1; ix += jx) {
      x = lox[ix]*dx + xo;
      y = loy[iy]*dy + yo;
      if (fgets(value,50,pgraf) == NULL) return(1);
      if (fabs(atof(value)) <= baddata) {
        vptr = value;
        while (*vptr == ' ') vptr++;
        halflength = (float)strlen(vptr)/2.;
        x -= halflength*cdx;
        y -= cdy/2.;
        if(x < x1 || (x + 2*halflength*cdx) > x2) continue;
        if(y < y1 || y + cdy > y2)  continue;
        pgtext(x,y,vptr);
      }
    }
  }
}

int mod(y,x)
int x,y;
{
  int m;
  int i;
  m = -1;
  if (x != 0) {
    if (y < 0) y = -y;
    i = y / x;
    m = y - x * i;
    if (m < 0) m = -m;
  }
  return(m);
}

static int kfill(nx,ny,dvl,dvh,ltyp,c,b)

/* fill routine */
int nx,ny,ltyp;
float dvl,dvh,c[],b[];
{
  float zv[6],x1,x2,y1,y2,xo,yo,xs,xt,ys,yt,rys,ryt,rxs,rxt;
  int nvect,j,iy,ix;
  float dy,dx,vl[2][4],sxl,sxh,syl,syh,lox[XPT + 1],loy[YPT + 1],tm;
  x1 = a[1];
  x2 = a[2];
  y1 = a[3];
  y2 = a[4];
  sxl = a[5];
  sxh = a[6];
  syl = a[7];
  syh = a[8];

  /* calc dx and dy */
  dx = (x2 - x1)/(sxh - sxl);
  dy = (y2 - y1)/(syh - syl);
  xo = x1 - (sxl*dx);
  yo = y1 - (syl*dy);

  /* get point x,y locations */
  for(ix = 1; ix <= nx;ix++)
    if(fscanf(pgraf,"%f", &lox[ix]) == EOF) panic();
  for(iy = 1; iy<= ny; iy++)
    if(fscanf(pgraf,"%f",&loy[iy]) == EOF) panic();

  /* get first row */
  for (ix = 1; ix <= nx; ix++)
    if (fscanf(pgraf,"%f",&c[ix])== EOF) return(1);

  /* loop on rows */
  for (iy = 2; iy <= ny; iy++) {

    /* get next row */
    for (ix = 1; ix <= nx; ix++)
      b[ix] = c[ix];
    for (ix = 1; ix <= nx; ix++)
      if (fscanf(pgraf,"%f",&c[ix])== EOF) return(1);

    /* loop on columns */
    for (ix = 2; ix <= nx; ix++) {

      /* get corner values */
      zv[1] = b[ix - 1];
      if (fabs(zv[1]) > baddata) continue;
      zv[2] = c[ix - 1];
      if (fabs(zv[2]) > baddata) continue;
      zv[3] = c[ix];
      if (fabs(zv[3]) > baddata) continue;
      zv[4] = b[ix];
      if (fabs(zv[4]) > baddata) continue;
      zv[5] = zv[1];

      /* do negative fill */
      xs = rxs = lox[ix - 1]*dx + xo;
      xt = rxt = lox[ix]*dx + xo;
      ys = rys = loy[iy - 1]*dy + yo;
      yt = ryt = loy[iy]*dy + yo;
      if (xt < xs) {
        tm = xs;
        xs = xt;
        xt = tm;
      }
      if (yt < ys) {
        tm = ys;
        ys = yt;
        yt = tm;
      }
      nvect = survey(dx,dy,xo,yo,dvl,zv,vl,lox,loy,ix,iy);
      for (j = 0; j < nvect; j++) {
        xx(vl[j][0],vl[j][1],vl[j][2],vl[j][3],ltyp);
      }
      nfill(xs,ys,xt,yt,zv,dvl,nvect,vl,ltyp,rxs,rxt,rys,ryt);

      /* do positive fill */
      nvect = survey(dx,dy,xo,yo,dvh,zv,vl,lox,loy,ix,iy);
      for (j = 0; j < nvect; j++) {
        xx(vl[j][0],vl[j][1],vl[j][2],vl[j][3],ltyp);
      }
      pfill(xs,ys,xt,yt,zv,dvh,nvect,vl,ltyp,rys,ryt,rxs,rxt);
    }
  }
}

/* compute intersection of contour and y fill line */
static float cinty(vlist,yval)
float vlist[4],yval;
{
  float denom;
  denom = vlist[3] - vlist[1];
  if ((denom >= 0.) && (denom < SMALL)) denom = SMALL;
  else if ((denom < 0.) && (denom > -SMALL)) denom = -SMALL;
  return(vlist[0] + (yval - vlist[1])*(vlist[2] - vlist[0])/denom);
}


/* compute intersection of contour and x fill line */
static float cintx(vlist,xval)
float vlist[4],xval;
{
  float denom;
  denom = vlist[2] - vlist[0];
  if ((denom >= 0.) && (denom < SMALL)) denom = SMALL;
  if ((denom < 0.) && (denom > -SMALL)) denom = -SMALL;
  return(vlist[1] + (xval - vlist[0])*(vlist[3] - vlist[1])/denom);
}

/* returns +1 if interpolated value > z, -1 otherwise */
static int cside(zv1,zv2,s1,s2,s,z)
float zv1,zv2,s1,s2,s,z;
{
  float zint;
  zint = zv1 + (zv2 - zv1)*(s - s1)/(s2 - s1);
  if (zint > z) return(1);
  else return(-1);
}

static int nfill(xs,ys,xt,yt,zv,z,nvect,vl,ltype,rxs,rxt,rys,ryt)
float xs,ys,xt,yt;
float zv[6];
float z;
int nvect;
float vl[2][4];
int ltype;
float rxs,rxt,rys,ryt;
{
  float cx,cy,del,yval,cinty(),xint1,xint2;
  int ilev,cside(),fside;

  /* return immediately if whole box empty */
  if ((nvect == 0) && (z <= zv[1])) return(0);

  /* get line spacing as a fraction of character height */
  pginfo(&cx,&cy);
  del = cy*FRACT;

  /* loop on y level */
  ilev = (ys)/del ;
  yval = del*(ilev+1) ;
  while(yval <= yt) {

    /* several possibilities */

    /* full box */
    if ((nvect == 0) && (z > zv[1])) {
      xx(rxs,yval,rxt,yval,ltype);
    }

    /* one line thru box */
    else if (nvect == 1) {
      xint1 = cinty(vl[0],yval);
      fside = cside(zv[1],zv[2],rys,ryt,yval,z);

      /* does contour intercept line in box? */
      if((xint1 > xs) && (xint1 < xt))  {

        /* starting on blank side */
        if (fside == 1) {
          xx(xint1,yval,rxt,yval,ltype);
        }

        /* starting on fill side */
        else {
          xx(rxs,yval,xint1,yval,ltype);
        }
      }

      /* nope -- its all or nothing */
      else if (fside == -1) {
        xx(rxs,yval,rxt,yval,ltype);
      }
    }

    /* two lines in box */
    else if (nvect == 2) {
      xint1 = cinty(vl[0],yval);
      xint2 = cinty(vl[1],yval);

      /* check slant of contours, and hence which corner is
         inside */
      if (vl[0][3] > vl[0][1]) {
        fside = cside(zv[1],zv[2],rys,ryt,rys,z);
      }
      else {
        fside = cside(zv[1],zv[2],rys,ryt,ryt,z);
      }

      /* outside fill */
      if (fside == 1) {
        if (xint1 > xs) {
          xx(rxs,yval,xint1,yval,ltype);
        }
        if (xint2 < xt) {
          xx(xint2,yval,rxt,yval,ltype);
        }
      }

      /* inside fill */
      else {
        if (xint1 < xs) xint1 = xs;
        if (xint2 > xt) xint2 = xt;
        xx(xint1,yval,xint2,yval,ltype);
      }
    }
    yval += del;
  }
}

/* do positive fill */
#define Y(a,b,c) xx(c,a,c,b,ltype)
static int pfill(xs,ys,xt, yt,zv,z,nvect,vl,ltype,rys,ryt,rxs,rxt)
int nvect,ltype;
float zv[6],z,vl[2][4],xs,xt,ys,yt,rys,ryt,rxs,rxt;
{
  float cx,cy,del,xval,cintx(),yint1,yint2,temp;
  int ilev,cside(),fside;

  /* return immediately if whole box empty */
  if ((nvect == 0) && (z >= zv[1])) return(0);

  /* line spacing again */
  pginfo(&cx,&cy);
  del = cy*FRACT;

  /* loop on x level */
  ilev = (xs)/del;
  xval = del*(ilev+1);
  while (xval <= xt) {

    /* several possibilities */

    /* full box */
    if ((nvect == 0) && (z < zv[1])) {
      Y(rys,ryt,xval);
    }

    /* one line thru box */
    else if (nvect == 1) {
      yint1 = cintx(vl[0],xval);
      fside = cside(zv[1],zv[4],rxs,rxt,xval,z);

      /* does contour intercept line in box? */
      if ((yint1 > ys) && (yint1 < yt)) {

        /* starting on blank side */
        if (fside == -1) {
          Y(yint1,ryt,xval);
        }

        /* starting on full side */
        else {
          Y(rys,yint1,xval);
        }
      }

      /* nope -- its all or nothing again */
      else if (fside == 1) {
        Y(rys,ryt,xval);
      }
    }

    /* two lines in box */
    else if (nvect == 2) {
      yint1 = cintx(vl[1],xval);
      yint2 = cintx(vl[0],xval);

      /* fix possible reverse order intersections */
      if (yint1 > yint2) {
        temp = yint1;
        yint1 = yint2;
        yint2 = temp;
      }

      /* check slant of contours, and hence which corner is
         inside */
      if (vl[0][3] > vl[0][1]) {
        fside = cside(zv[1],zv[4],rxs,rxt,rxs,z);
      }
      else {
        fside = cside(zv[1],zv[4],rxs,rxt,rxt,z);
      }

      /* outside fill */
      if (fside == -1) {
        if (yint1 > ys ) {
          Y(rys,yint1,xval);

        }
        if (yint2 < yt) {
          Y(yint2,ryt,xval);
        }
      }

      /* inside fill */
      else {
        if (yint1 < ys) yint1 = ys;
        if (yint2 > yt ) yint2 = yt;
        Y(yint1,yint2,xval);
      }
    }

    /* increment xval */
    xval += del;
  }
}

static int xx(ax,ay,bx,by,ltype)
int ltype;
float ax,ay, bx,by;
{
  float x1,x2,y1,y2,x,y,whr,outy,outx,cx,cy;
  x1 = a[1];
  x2 = a[2];
  y1 = a[3];
  y2 = a[4];
  if(ax>x2 || ax<x1 || ay>y2 || ay<y1){
    outy = ay;
    outx = ax;
    x    = ax;
    y    = ay;
  }
  else{
    outy = by;
    outx = bx;
    x    = bx;
    y    = by;
  }
  if((cx = bx-ax)>=0 && cx<SMALL) cx = SMALL;
  else if(cx<0 && cx>-SMALL) cx = -SMALL;
  if((cy = by-ay)>=0&& cy<SMALL) cy = SMALL;
  else if(cy<0 && cy>-SMALL) cy = -SMALL;
  if(ax>=x1 && ay>=y1 && ax<=x2 &&ay<=y2 && bx>=x1 &&
     by>=y1 &&bx<=x2 && by<=y2) {
    pgmove(ax,ay);
    pgdraw(bx,by,ltype);
    return(0);
  }
  whr = insect(ax,ay,bx,by);

  /* get intersection of window edge and line*/
  if(whr == 1){
    y = y1;
    x -= ((cx)/(cy))*(outy-y1);
  }
  else if(whr == 2){
    x = x2;
    y -= ((cy)/(cx))*(outx-x2);
  }
  else if(whr == 3){
    y = y2;
    x -= ((cx)/(cy))*(outy - y2);
  }
  else if(whr == 4){
    x= x1;
    y -= ((cy)/(cx))*(outx-x1);
  }
  if(x>x2|| y>y2 || x<x1 || y<y1) return;
  if (whr == 10) return(0);
  if(ax>x2 || ax<x1 || ay<y1 || ay>y2){
    pgmove(x,y);
    pgdraw(bx,by,ltype);
  }
  else{
    pgmove(ax,ay);
    pgdraw(x,y,ltype);
  }
}

/* insect finds if a line left the window and returns the side that it left*/
static int insect(ax,ay,bx,by)
float ax,ay,bx,by;
{
  float x,y,ox,oy,m,am[5],x1,x2,y1,y2,mden,mnum,adena,adenb,xl,yl;
  x1 = a[1];
  x2 = a[2];
  y1 = a[3];
  y2 = a[4];
  xl = a[6] - a[5];
  yl = a[8] - a[7];
  /*find the point inside the window*/
  if(fabs(ax*xl)>baddata||fabs(ay*yl)>baddata) return(10);
  if(fabs(bx*xl)>baddata||fabs(by*yl)>baddata) return(10);
  if(ax>x2 || ax<x1 || ay>y2 || ay<y1){

    /* set the inside point to (ox,oy)*/
    x = ax;
    y = ay;
    ox = bx;
    oy = by;
    if(bx>=x2 || bx<=x1 || by>=y2 || by<=y1) return(10); /* no pts in window*/

  }
  else if(bx>x2 || bx<x1 || by<y1 || by >y2){
    if(ax==x1 || ax == x2 || ay==y1 || ay==y2) return(10);
    x = bx;
    y = by;
    ox  = ax;
    oy  = ay;
  }
  else return(0);
  mnum = (y-oy);
  mden = (x-ox);
  adena = (x1-ox);
  adenb = (x2-ox);

  /* avoiding /0 problems */
  if(adena>-SMALL && adena <0) adena = -SMALL;
  else if (adena < SMALL && adena>= 0) adena = SMALL;
  if(adenb> -SMALL && adenb<0) adenb = -SMALL;
  else if(adenb<SMALL && adenb>=0) adenb = SMALL;
  if((mden<=0 && mden<=-SMALL) || ( mden>0 && mden>=SMALL)) m = mnum/mden;
  else if(mnum > 0)  return(3);
  else return(1);
  am[0] = am[4] = (y1-oy)/adena;
  am[1] = (y1-oy)/adenb;
  am[2] = (y2-oy)/adenb;
  am[3] = (y2 -oy)/adena;

  /* comparing slopes*/
  if(mnum>0)
    if(mden>0)
      if(m>am[2])
        return(3);
      else return(2);
    else if(m>am[3]) return(4);
    else return(3);
  else if(mden>0)
    if(m>am[1]) return(2);
    else return(1);
  else if(m>=am[0]) return(1);
  else return(4);
}

/* draw 3-D structure in perspective */
static int pdraw()
{
  int num,nw,ltype,loop;
  float bx,by,bz;

  /* get the data -- header line first -- */
  if (fscanf(pgraf,"%d%d%d",&nw,&ltype,&num) == EOF) return(1);

  /* restore window */
  rest(nw);

  /* -- and then the rest of it */
  if (num >= MAG) {
    pgflsh();
    fprintf(stderr,"pgraf: error caused by line > %d points\n",MAG - 1);
    cexit(1);
  }
  for (loop = 1; loop <= num; loop++) {
    if (loop < MAG) {
      if (fscanf(pgraf,"%f%f%f",&bx,&by,&bz) == EOF) return(1);
      project(bx,by,bz,dax + loop,day + loop);
    }
  }

  /* draw the sucker! */
  draw(ltype,num,dax,day);

  /* tell moma everything is ok */
  return(0);
}

/* project -- obtain the projections of the 3-D point on the 2-D plane */
project(bx,by,bz,x,y)
float bx,by,bz;
float *x,*y;
{
  float b[3];
  int id;
  b[0] = bx*a[18];
  b[1] = by*a[19];
  b[2] = bz*a[20];
  for (id = 0; id < 3; id++) b[id] -= a[id + 9];
  norm(b);
  *x = dott(b,&a[12]);
  *y = dott(b,&a[15]);
}

/* pseg -- draw a line segment in perspective */
pseg(xl,yl,zl,xh,yh,zh,ltype)
float xl,yl,zl,xh,yh,zh;
int ltype;
{
  project(xl,yl,zl,dax + 1,day + 1);
  project(xh,yh,zh,dax + 2,day + 2);
  draw(ltype,2,dax,day);
}

/* paxes -- draw axes in perspective */
paxes(sxl,sxh,ntx,syl,syh,nty,szl,szh,ntz)
float sxl,sxh;
int ntx;
float syl,syh;
int nty;
float szl,szh;
int ntz;
{
  char text[80];
  float x,y,del;
  int iseg,jseg;

/* segment axes so perspective works properly */
  for (iseg = 0; iseg < NSEG; iseg++) {
    jseg = iseg + 1;
    del = (sxh - sxl)/NSEG;
    pseg(sxl + del*iseg,syl,szl,sxl + del*jseg,syl,szl,1);
    del = (syh - syl)/NSEG;
    pseg(sxl,syl + del*iseg,szl,sxl,syl + del*jseg,szl,1);
    del = (szh - szl)/NSEG;
    pseg(sxl,syl,szl + del*iseg,sxl,syl,szl + del*jseg,1);
  }
  if( fscanf(pgraf,"%s",text) == EOF) return(1);
  project(sxh + .1*(sxh - sxl),syl,szl,&x,&y);
  mlabel(x,y,text);
  if( fscanf(pgraf,"%s",text) == EOF) return(1);
  project(sxl,syh + .1*(syh - syl),szl,&x,&y);
  mlabel(x,y,text);
  if( fscanf(pgraf,"%s",text) == EOF) return(1);
  project(sxl,syl,szh + .1*(szh - szl),&x,&y);
  mlabel(x,y,text);
}

/* surface -- draw a function of two variables in perspective */
surface(nx,ny,ltype)
int nx,ny,ltype;
{
  long i,ix,iy;
  float x[PPT + 1],y[PPT + 1],field[(PPT + 1)*(PPT + 1)];

/* get the data */
  for (ix = 0; ix < nx; ix++) {
    if (fscanf(pgraf,"%g",&x[ix]) == EOF) panic();
  }
  for (iy = 0; iy < ny; iy++) {
    if (fscanf(pgraf,"%g",&y[iy]) == EOF) panic();
  }
  for (i = 0; i < nx*ny; i++) {
    if (fscanf(pgraf,"%g",&field[i]) == EOF) panic();
  }

/* plot x lines */
  for (iy = 0; iy < ny; iy++) {
    for (ix = 0; ix < nx; ix++) {
      project(x[ix],y[iy],field[iy*nx + ix],
           &dax[ix + 1],&day[ix + 1]);
    }
    draw(ltype,nx,dax,day);
  }

/* plot y lines */
  for (ix = 0; ix < nx; ix++) {
    for (iy = 0; iy < ny; iy++) {
      project(x[ix],y[iy],field[iy*nx + ix],
           &dax[iy + 1],&day[iy + 1]);
    }
    draw(ltype,ny,dax,day);
  }
}

/* normalize a vector */
static void norm(x)
float x[3];
{
  int i;
  float sum;
  sum = 0.;
  for (i = 0; i < 3; i++) sum += x[i]*x[i];
  if (sum < SMALL) {
    for (i = 0; i < 3; i++) x[i] = 0.;
    return;
  }
  sum = sqrt(sum);
  for (i = 0; i < 3; i++) x[i] /= sum;
}

/* find a dot product */
static float dott(d,b)
float d[3],b[3];
{
  int i;
  float sum;
  sum = 0.;
  for (i = 0; i < 3; i++) sum += d[i]*b[i];
  return(sum);
}

/* make a label */
static void mlabel(sx,sy,alfa)
float sx,sy;
char alfa[];
{
  sx = (sx - a[5])/(a[6] - a[5]);
  if (sx < 0.0) sx = 0.0;
  if (sx > 1.0) sx = 1.0;
  sx = a[1] + sx*(a[2] - a[1]);
  sy = (sy - a[7])/(a[8] - a[7]);
  if (sy < 0.0) sy = 0.0;
  if (sy > 1.0) sy = 1.0;
  sy = a[3] + sy*(a[4] - a[3]);
  pgtext(sx,sy,alfa);
}

/* limit the magnitude of a quantity */
static float maglim(max,val)
float max,val;
{
  if (val > max) return(max);
  else if (val < -max) return(-max);
  else return(val);
}
