/* $Id$ */
/* Copyright (c) 1988 Carrick Sean Casey. All rights reserved. */
#include <sys/types.h>
#include <db.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "externs.h"
#include "userdb.h"

int	setsecure (int, int);
int	valuser (const char *, const char *);
void	nickdelete (int, const char *);
void	nickwritemsg (int, const char *, char *);
int	nickckmsg (int);
void	nickreadmsg (int);
void	nickwritetime (int, int);
void	nickchinfo (int, const char *, char *, int, const char *);
void	nickchpass (int, const char *, char *);
void	nickwrite (int, const char *);
int	nicklookup (int, const char *);

/* note:  we don't store trailing NUL in strings.  It would be easier if we did, but that would break compatibility with old userdbs. */

int
setsecure (int forWhom, int secure)
{
	DBT	key, data;
	char	kbuf[KEY_SIZE];
	int	retval = 0;

	if (strlen(u_tab[forWhom].realname) == 0) {
		senderror(forWhom, "You must be registered to change your security.");
		return -1;
		}

	if (!userdb)
		return -1;

	if (secure == 0) {
		strlcpy(kbuf, u_tab[forWhom].nickname, KEY_SIZE);
		strlcat(kbuf, ".secure", KEY_SIZE);
		lcaseit(kbuf);
		key.data = kbuf;
		key.size = strlen(key.data);

		if (userdb->del(userdb, &key, 0) >= 0)
			sends_cmdout (forWhom, "Security set to automatic.");
	} else if (secure == 1) {
		strlcpy(kbuf, u_tab[forWhom].nickname, KEY_SIZE);
		strlcat(kbuf, ".secure", KEY_SIZE);
		lcaseit(kbuf);
		key.data = kbuf;
		key.size = strlen(key.data);
		data.data = "SECURED";
		data.size = strlen(data.data);
		
		if (!userdb->put(userdb, &key, &data, 0))
			sends_cmdout (forWhom, "Security set to password required.");
	} else {
		error("Illegal setsecure value");
		retval = -2;
	}
	return retval;
}

int
valuser(const char *user, const char *password)
{
	DBT	key, data;
	char	kbuf[KEY_SIZE];
	char	line[LINE_SIZE];

	if (!userdb)
		return -1;

	if (strlen(password) == 0)
		return -1;

	strlcpy(kbuf, user, KEY_SIZE);
	strlcat(kbuf, ".password", KEY_SIZE);
	lcaseit(kbuf);
	key.data = kbuf;
	key.size = strlen(kbuf);

	if (userdb->get(userdb, &key, &data, 0))
		return -1;

	strncpy(line, data.data, data.size);
	line[data.size] = '\0';

	if (strcmp(line, password))
		return -1;
	else
		return 0;
}

void
nickdelete(int forWhom, const char *password)
{
	DBT	key, data;
	char	kbuf[KEY_SIZE];
	char	line[LINE_SIZE];
	int	i;

	if (strlen (password) <= 0)
	{
		senderror (forWhom, "Password cannot be null");
		return;
	}

	if (strlen(u_tab[forWhom].realname) == 0) {
		senderror(forWhom, "You must be registered to delete your entry.");
		return;
	}

	if (!userdb)
		return;

	strlcpy(kbuf, u_tab[forWhom].nickname, KEY_SIZE);
	strlcat(kbuf, ".password", KEY_SIZE);
	lcaseit(kbuf);
	key.data = kbuf;
	key.size = strlen(kbuf);

	if (userdb->get(userdb, &key, &data, 0)) {
	    senderror (forWhom, "You don't have a password.");
	    return;
	}
	
	strncpy(line, data.data, data.size);
	line[data.size] = '\0';

	if (!strcmp(line, password)) {
	    senderror (forWhom, "Password incorrect.");
	    return;
	}

	for (i=0; userdbfields[i]; i++) {
		strlcpy(kbuf, u_tab[forWhom].nickname, KEY_SIZE);
		strlcat(kbuf, userdbfields[i], KEY_SIZE);
		lcaseit(kbuf);
		key.data = kbuf;
		key.size = strlen(key.data);

		userdb->del(userdb, &key, 0);
	}

	for (i = 1; i <= MAX_WRITES; i++) {
		snprintf(kbuf, KEY_SIZE, "%s.header%d", u_tab[forWhom].nickname, i);
		lcaseit(kbuf);
		key.data = kbuf;
		key.size = strlen(kbuf);
		userdb->del(userdb, &key, 0);

		snprintf(kbuf, KEY_SIZE, "%s.message%d", u_tab[forWhom].nickname, i);
		lcaseit(kbuf);
		key.data = kbuf;
		key.size = strlen(kbuf);
		userdb->del(userdb, &key, 0);

		snprintf(kbuf, KEY_SIZE, "%s.from%d", u_tab[forWhom].nickname, i);
		lcaseit(kbuf);
		key.data = kbuf;
		key.size = strlen(kbuf);
		userdb->del(userdb, &key, 0);
	}

	sends_cmdout(forWhom, "Record Deleted");
}

void
nickwritemsg(int from, const char *to, char *message)
{
	char	kbuf[KEY_SIZE];
	DBT	key, data;
	char	line[LINE_SIZE], timebuf[LINE_SIZE];
	int	count, i;
	time_t	curtime;

	if (!userdb)
		return;

	if ((strlen(to) <= 0) || (strlen(message) <=0)) {
		sends_cmdout(from, "Usage: write nickname message text");
		return;
	}

	if (strlen(u_tab[from].realname) == 0) {
		senderror(from, "You must be registered to write a message.");
		return;
	}

	strlcpy(kbuf, to, KEY_SIZE);
	strlcat(kbuf, ".nick", KEY_SIZE);
	lcaseit(kbuf);
	key.data = kbuf;
	key.size = strlen(kbuf);
	if (userdb->get(userdb, &key, &data, 0)) {
		snprintf(line, LINE_SIZE, "%s is not registered", to);
		senderror(from, line);
		return;
	}

	strlcpy(kbuf, to, KEY_SIZE);
	strlcat(kbuf, ".nummsg", KEY_SIZE);
	lcaseit(kbuf);
	key.data = kbuf;
	key.size = strlen(kbuf);
	if (userdb->get(userdb, &key, &data, 0)) {
		strncpy(line, data.data, data.size);
		line[data.size] = '\0';
		count = atoi(line);
	} else
		count = 0;
	
	count++;
	if (count > MAX_WRITES) {
		senderror(from, "User mailbox full");
		return;
	}

	time(&curtime);
	strftime(timebuf, LINE_SIZE, "%d-%m-%Y %H:%M %Z", localtime(&curtime));
	snprintf(line, LINE_SIZE, "Message left at %s:", timebuf);
	snprintf(kbuf, KEY_SIZE, "%s.header%d", to, count);
	lcaseit(kbuf);
	key.data = kbuf;
	key.size = strlen(kbuf);
	data.data = line;
	data.size = strlen(line);
	userdb->put(userdb, &key, &data, 0);

	snprintf(kbuf, KEY_SIZE, "%s.from%d", to, count);
	lcaseit(kbuf);
	key.data = kbuf;
	key.size = strlen(kbuf);
	data.data = u_tab[from].nickname;
	data.size = strlen(data.data);
	userdb->put(userdb, &key, &data, 0);

	snprintf(kbuf, KEY_SIZE, "%s.message%d", to, count);
	lcaseit(kbuf);
	key.data = kbuf;
	key.size = strlen(kbuf);
	filtertext(message);
	data.data = message;
	data.size = strlen(data.data);
	userdb->put(userdb, &key, &data, 0);

	strlcpy(kbuf, to, KEY_SIZE);
	strlcat(kbuf, ".nummsg", KEY_SIZE);
	lcaseit(kbuf);
	key.data = kbuf;
	key.size = strlen(kbuf);
	snprintf(line, LINE_SIZE, "%d", count);
	data.data = line;
	data.size = strlen(line);
	userdb->put(userdb, &key, &data, 0);

	sendstatus(from, "Message", "Message text saved");

	if ((i = find_user(to)) > 0) {
		snprintf(line, LINE_SIZE, "%s is logged in now.", u_tab[i].nickname);
		sendstatus(from, "Warning", line);
		snprintf(line, LINE_SIZE, "You have %d message%s", count, (count > 1) ? "s": "");
		sendstatus(i, "Message", line);
	}
}

int
nickckmsg(int forWhom)
{
	char	kbuf[KEY_SIZE];
	char	line[LINE_SIZE];
	DBT	key, data;

	if (!userdb)
		return -1;

	if (strlen(u_tab[forWhom].realname) == 0)
		return -1;

	strlcpy(kbuf, u_tab[forWhom].nickname, KEY_SIZE);
	strlcat(kbuf, ".nummsg", KEY_SIZE);
	lcaseit(kbuf);
	key.data = kbuf;
	key.size = strlen(kbuf);

	if(!userdb->get(userdb, &key, &data, 0)) {
		strncpy(line, data.data, data.size);
		line[data.size] = '\0';
		return(atoi(line));
	} else {
		data.data = "0";
		data.size = strlen(data.data);
		userdb->put(userdb, &key, &data, 0);
	}
	
	return(0);
}

void
nickreadmsg(int forWhom)
{
	char	kbuf[KEY_SIZE];
	DBT	key, data;
	char	line[LINE_SIZE], from[MAX_NICKLEN+1];
	int	count, i;

	if (!userdb)
		return;

	if (strlen(u_tab[forWhom].realname) <= 0) {
		senderror(forWhom, "You must be registered to read messages.");
		return;
	}

	if ((count = nickckmsg(forWhom)) > 0) {
		for (i = 1; i <= count; i++) {
			snprintf(kbuf, KEY_SIZE, "%s.header%d", u_tab[forWhom].nickname, i);
			lcaseit(kbuf);
			key.data = kbuf;
			key.size = strlen(kbuf);

			if (!userdb->get(userdb, &key, &data, 0)) {
				strncpy(line, data.data, data.size);
				line[data.size] = '\0';
				sends_cmdout(forWhom, line);
			}
			userdb->del(userdb, &key, 0);

			snprintf(kbuf, KEY_SIZE, "%s.from%d", u_tab[forWhom].nickname, i);
			lcaseit(kbuf);
			key.data = kbuf;
			key.size = strlen(kbuf);
			if (!userdb->get(userdb, &key, &data, 0)) {
				strncpy(from, data.data, data.size);
				from[data.size] = '\0';
			} else
				strlcpy(from, "Server", LINE_SIZE);
			userdb->del(userdb, &key, 0);

			snprintf(kbuf, KEY_SIZE, "%s.message%d", u_tab[forWhom].nickname, i);
			lcaseit(kbuf);
			key.data = kbuf;
			key.size = strlen(kbuf);
			if (!userdb->get(userdb, &key, &data, 0)) {
				strncpy(line, data.data, data.size);
				line[data.size] = '\0';
				sendsavedmessage(from, forWhom, line);
			}
			userdb->del(userdb, &key, 0);
		}
	} else {
		senderror(forWhom, "No messages");
	}
}

void
nickwritetime(int forWhom, int class)
{
	char	kbuf[KEY_SIZE];
	DBT	key, data;
	char	line[LINE_SIZE];
	time_t	curtime;

	if (!userdb)
		return;
	
	if (strlen(u_tab[forWhom].realname) == 0)
		return;

	time(&curtime);
	strftime(line, LINE_SIZE, "%d-%m-%Y %H:%M %Z", localtime(&curtime));

	strlcpy(kbuf, u_tab[forWhom].nickname, KEY_SIZE);

	if (class == 0)
		strlcat(kbuf, ".signon", KEY_SIZE);
	else
		strlcat(kbuf, ".signoff", KEY_SIZE);

	lcaseit(kbuf);
	key.data = kbuf;
	key.size = strlen(kbuf);
	data.data = line;
	data.size = strlen(line);

	userdb->put(userdb, &key, &data, 0);
}

void
nickchinfo(int forWhom, const char *tag, char *newval, int max, const char *message)
{
	char	kbuf[KEY_SIZE];
	DBT	key, data;
	char	line[LINE_SIZE];
	char	newnewval[LINE_SIZE]; /* truncated newval, if needed */

	if (!userdb)
		return;
	
	if (strlen(u_tab[forWhom].realname) <= 0) {
		snprintf(line, LINE_SIZE, "You must be registered to set your %s.", message);
		senderror(forWhom, line);
		return;
	}

	if (strlen(newval) > max) {
		strncpy(newnewval, newval, max);
		newnewval[max] = '\0';
		newval = newnewval;
		snprintf (line, LINE_SIZE, "%s truncated to %d characters", message, max);
		senderror(forWhom, line);
	}

	strlcpy(kbuf, u_tab[forWhom].nickname, KEY_SIZE);
	strlcat(kbuf, ".", KEY_SIZE);
	strlcat(kbuf, tag, KEY_SIZE);
	lcaseit(kbuf);
	key.data = kbuf;
	key.size = strlen(kbuf);
	data.data = newval;
	data.size = strlen(newval);
	userdb->put(userdb, &key, &data, 0);

	snprintf(line, LINE_SIZE, "%s set to '%s'", message, newval);
	sends_cmdout(forWhom, line);
}

void
nickchpass(int forWhom, const char *oldpw, char *newpw)
{
	char	kbuf[KEY_SIZE];
	DBT	key, data;
	char	line[LINE_SIZE];
	int	l;

	if (!userdb)
		return;

	if ((l = strlen(newpw)) > MAX_PASSWDLEN)
		newpw[MAX_PASSWDLEN] = '\0';
	else if (l <= 0) {
		snprintf(line, LINE_SIZE, "Missing paramater");
		senderror(forWhom,line);
		return;
	}

	strlcpy(kbuf, u_tab[forWhom].nickname, KEY_SIZE);
	strlcat(kbuf, ".password", KEY_SIZE);
	lcaseit(kbuf);
	key.data = kbuf;
	key.size = strlen(kbuf);

	if (!userdb->get(userdb, &key, &data, 0)) {
		strncpy(line, data.data, data.size);
		line[data.size] = '\0';

		if (strncmp(line, oldpw, MAX_PASSWDLEN)) {
			snprintf(line, LINE_SIZE, "Authorization failure");
			senderror(forWhom, line);
		} else {
			data.data = newpw;
			data.size = strlen(newpw);
			userdb->put(userdb, &key, &data, 0);

			snprintf(line, LINE_SIZE, "Password changed");
			sendstatus(forWhom, "Pass", line);

			strlcpy(kbuf, u_tab[forWhom].nickname, KEY_SIZE);
			strlcat(kbuf, ".home", KEY_SIZE);
			lcaseit(kbuf);
			key.data = kbuf;
			key.size = strlen(kbuf);

			strlcpy(line, u_tab[forWhom].loginid, LINE_SIZE);
			strlcat(line, "@", LINE_SIZE);
			strlcat(line, u_tab[forWhom].nodeid, LINE_SIZE);
			data.data = line;
			data.size = strlen(line);

			userdb->put(userdb, &key, &data, 0);
		}
	} else {
		snprintf(line, LINE_SIZE, "Authorization failure");
		senderror(forWhom, line);
	}
}

void
nickwrite (int forWhom, const char *password)
{
	char	kbuf[KEY_SIZE];
	DBT	key, data;
	char	line[LINE_SIZE];
	int	i, j;

	if (!userdb)
	    return;

	if (strlen(password) <= 0) {
	    senderror (forWhom, "Password cannot be null");
	    return;
	}

	strlcpy(kbuf, u_tab[forWhom].nickname, KEY_SIZE);
	strlcat(kbuf, ".password", KEY_SIZE);
	lcaseit(kbuf);
	key.data = kbuf;
	key.size = strlen(kbuf);

	if(!userdb->get(userdb, &key, &data, 0)) {
		strncpy(line, data.data, data.size);
		line[data.size] = '\0';

		if (!strncmp(line, password, MAX_PASSWDLEN)) {
			snprintf(line, LINE_SIZE, "Nick registered");
			sends_cmdout(forWhom, line);

			strlcpy(u_tab[forWhom].realname, "registered", MAX_REALLEN);
			strlcpy(u_tab[forWhom].password, password, MAX_PASSWDLEN+1);
			nickwritetime(forWhom, 0);

			for (i = 1; i < MAX_GROUPS; i++)
				if ((g_tab[i].modtimeout > 0) && (strcmp(g_tab[i].missingmod, u_tab[forWhom].nickname)==0)) {
					g_tab[i].modtimeout = 0;
					g_tab[i].mod = forWhom;
					g_tab[i].missingmod[0] = '\0';
					snprintf(line, LINE_SIZE, "%s is the active moderator again.", u_tab[forWhom].nickname);
					for (j = 1; j < MAX_REAL_USERS; j++)
						if ((strcasecmp(u_tab[j].group, g_tab[i].name) == 0) && (j != forWhom))
							sendstatus(j, "Mod", line);
					snprintf(line, LINE_SIZE, "You are the moderator of group %s", g_tab[i].name);
					sendstatus(forWhom, "Mod", line);
			}
			if ((i = nickckmsg(forWhom)) > 0) {
				snprintf(line, LINE_SIZE, "You have %d message%s", i, (i > 1) ? "s" : "");
				sendstatus(forWhom, "Message", line);
			}
			for ( i = 1; i < MAX_GROUPS; i++ )
				talk_report (forWhom, i);
		} else {
			snprintf(line, LINE_SIZE, "Authorization failure");
			senderror(forWhom, line);
			u_tab[forWhom].realname[0] = '\0';
		}
	} else {
		strlcpy(line, password, LINE_SIZE);
		line[MAX_PASSWDLEN] = '\0';
		data.data = line;
		data.size = strlen(line);
		userdb->put(userdb, &key, &data, 0);

		strlcpy(kbuf, u_tab[forWhom].nickname, KEY_SIZE);
		strlcat(kbuf, ".nick", KEY_SIZE);
		lcaseit(kbuf);
		key.data = kbuf;
		key.size = strlen(kbuf);
		strlcpy(line, u_tab[forWhom].nickname, LINE_SIZE);
		data.data = line;
		data.size = strlen(line);
		userdb->put(userdb, &key, &data, 0);

		strlcpy(kbuf, u_tab[forWhom].nickname, KEY_SIZE);
		strlcat(kbuf, ".home", KEY_SIZE);
		lcaseit(kbuf);
		key.data = kbuf;
		key.size = strlen(kbuf);
		strlcpy(line, u_tab[forWhom].loginid, LINE_SIZE);
		strlcat(line, "@", LINE_SIZE);
		strlcat(line, u_tab[forWhom].nodeid, LINE_SIZE);
		data.data = line;
		data.size = strlen(line);
		userdb->put(userdb, &key, &data, 0);

		snprintf(line, LINE_SIZE, "Nick registered");
		sends_cmdout(forWhom, line);

		strlcpy(u_tab[forWhom].realname, "registered", LINE_SIZE);
		strlcpy(u_tab[forWhom].password, password, MAX_PASSWDLEN+1);
		nickwritetime(forWhom, 0);
	}
}

int
nicklookup(int forWhom, const char *theNick)
{
	char	kbuf[KEY_SIZE];
	DBT	key, data;
	char	line[LINE_SIZE];
	char	temp[LINE_SIZE];
	char	nickstr[LINE_SIZE];
	int	retval;
	int	l;
	int	count = 0;
	char	*s, *p, *lastw;
	char	tmp1, tmp2;
	char	s1[LINE_SIZE], s2[LINE_SIZE];

	if (!userdb)
		return -1;

	if (strlen(theNick) <= 0) {
		if ((forWhom > 0) && (forWhom < MAX_REAL_USERS))
			sends_cmdout(forWhom, "Usage: whois nickname");
		return -1;
	}
	
	if (strlen(theNick) > MAX_NICKLEN) {
		senderror(forWhom, "nickname too long");
		return -1;
	}

	strlcpy(kbuf, theNick, KEY_SIZE);
	strlcat(kbuf, ".nick", KEY_SIZE);
	lcaseit(kbuf);
	key.data = kbuf;
	key.size = strlen(kbuf);
	if (!userdb->get(userdb, &key, &data, 0)) {
		strncpy(nickstr, data.data, data.size);
		nickstr[data.size] = '\0';
	}

	strlcpy(kbuf, theNick, KEY_SIZE);
	strlcat(kbuf, ".home", KEY_SIZE);
	lcaseit(kbuf);
	key.data = kbuf;
	key.size = strlen(kbuf);

	if (!userdb->get(userdb, &key, &data, 0)) {
		retval = 0;

		if (forWhom >= 0) {
			strncpy(s1, data.data, data.size);
			s1[data.size] = '\0';

			snprintf(line, LINE_SIZE, "Nickname:     %-22s    Address:    %-24s", nickstr, s1);
			sends_cmdout(forWhom, line);

			strlcpy(kbuf, theNick, KEY_SIZE);
			strlcat (kbuf, ".phone", KEY_SIZE);
			lcaseit(kbuf);
			key.data = kbuf;
			key.size = strlen(kbuf);
			if (!userdb->get(userdb, &key, &data, 0)) {
				strncpy(s1, data.data, data.size);
				s1[data.size] = '\0';
			} else
				strlcpy(s1, "(None)", LINE_SIZE);

			strlcpy(kbuf, theNick, KEY_SIZE);
			strlcat(kbuf, ".realname", KEY_SIZE);
			lcaseit(kbuf);
			key.data = kbuf;
			key.size = strlen(kbuf);
			if (!userdb->get(userdb, &key, &data, 0)) {
				strncpy(s2, data.data, data.size);
				s2[data.size] = '\0';
			} else
				strlcpy(s2, "(None)", LINE_SIZE);

			snprintf(line, LINE_SIZE, "Phone Number: %-22s    Real Name:  %-24s", s1, s2);
			sends_cmdout(forWhom, line);

			strlcpy(kbuf, theNick, KEY_SIZE);
			strlcat(kbuf, ".signon", KEY_SIZE);
			lcaseit(kbuf);
			key.data = kbuf;
			key.size = strlen(kbuf);
			if (!userdb->get(userdb, &key, &data, 0)) {
				strncpy(s1, data.data, data.size);
				s1[data.size] = '\0';
			} else
				strlcpy(s1, "(unknown)", LINE_SIZE);

			strlcpy(kbuf, theNick, KEY_SIZE);
			strlcat(kbuf, ".signoff", KEY_SIZE);
			lcaseit(kbuf);
			key.data = kbuf;
			key.size = strlen(kbuf);
			if (!userdb->get(userdb, &key, &data, 0)) {
				strncpy(s2, data.data, data.size);
				s2[data.size] = '\0';
			} else
				strlcpy(s2, "(unknown)", LINE_SIZE);

			snprintf(line, LINE_SIZE, "Sign on:      %-22s    Sign off:   %-24s", s1, s2);
			sends_cmdout(forWhom, line);

			strlcpy(kbuf, theNick, KEY_SIZE);
			strlcat(kbuf, ".email", KEY_SIZE);
			lcaseit(kbuf);
			key.data = kbuf;
			key.size = strlen(kbuf);

			if (!userdb->get(userdb, &key, &data, 0)) {
				l = strlen(line);
				strlcpy(line, "E-mail addr:  ", LINE_SIZE);
				strncat(line, data.data, data.size);
				line[l + data.size] = '\0';
				sends_cmdout(forWhom, line);
			}

			strlcpy(kbuf, theNick, KEY_SIZE);
			strlcat(kbuf, ".www", KEY_SIZE);
			lcaseit(kbuf);
			key.data = kbuf;
			key.size = strlen(kbuf);
			if (!userdb->get(userdb, &key, &data, 0)) {
				l = strlen(line);
				strlcpy(line, "WWW:  ", LINE_SIZE);
				strncat(line, data.data, data.size);
				line[l + data.size] = '\0';
				sends_cmdout(forWhom, line);
			}

			strlcpy(kbuf, theNick, KEY_SIZE);
			strlcat(kbuf, ".addr", KEY_SIZE);
			lcaseit(kbuf);
			key.data = kbuf;
			key.size = strlen(kbuf);

			if (!userdb->get(userdb, &key, &data, 0)) {
				sends_cmdout(forWhom, "Street Address:");

				strncpy(line, data.data, data.size);
				line[data.size] = '\0';

				s = line;
				strlcpy(temp, "  ", LINE_SIZE);
				while (strlen(s) > 0) {
					if (*s == '|') {
						sends_cmdout(forWhom, temp);
						strlcpy(temp, "  ", LINE_SIZE);
					} else {
						l=strlen(temp);
						temp[l] = *s;
						temp[l+1] = '\0';
					}
					s++;
				}
				sends_cmdout(forWhom, temp);
			}

			strlcpy(kbuf, theNick, KEY_SIZE);
			strlcat(kbuf, ".text", KEY_SIZE);
			lcaseit(kbuf);
			key.data = kbuf;
			key.size = strlen(kbuf);
			if (!userdb->get(userdb, &key, &data, 0)) {
				strncpy(line, data.data, data.size);
				line[data.size] = '\0';
				s = line;
				/* traverse s and try to break on a word */
				p = s;
				lastw = p;
				while (*p != '\0') {
					if (*p == '\n') {
						*p++ = '\0';
						strlcpy(temp, "Text: ", LINE_SIZE);
						strlcat(temp, s, LINE_SIZE);
						sends_cmdout(forWhom, temp);
						count = 0;
						lastw = p;
						s = p;
						continue;
					}
					if (*p == ' ')
				/* remember location of last word break */
						lastw = p;

					/* break line if we are at max length */
					if (count == (MAX_TEXTLEN - 1)) {
						if ((p - lastw) > 40) {
				/* have to break in the middle of a word */
							tmp1 = *(p - 1);
							tmp2 = *p;
							*(p - 1) = '-';
							*p = '\0';
							strlcpy(temp, "Text: ", LINE_SIZE);
							strlcat(temp, s, LINE_SIZE);
							sends_cmdout(forWhom, temp);
							*(p - 1) = tmp1;
							*p = tmp2;
							p--;
							s = p;
						} else {
				/* break at last space char */
							tmp1 = *lastw;
							*lastw = '\0';
							strlcpy(temp, "Text: ", LINE_SIZE);
							strlcat(temp, s, LINE_SIZE);
							sends_cmdout(forWhom, temp);
							*lastw = tmp1;
							p = lastw + 1;
							s = p;
						}
						lastw = p;
						count = 0;
					} else {
						count++;
						p++;
					}
				}
				if (count > 0)
					strlcpy(temp, "Text: ", LINE_SIZE);
					strlcat(temp, s, LINE_SIZE);
					sends_cmdout(forWhom, temp);
				}
		} else {
			snprintf(temp, LINE_SIZE, "%s@%s", u_tab[-forWhom].loginid, u_tab[-forWhom].nodeid);
			strncpy(line, data.data, data.size);
			line[data.size] = '\0';
			if (strcasecmp(line, temp))
				retval = -2;
			else {
				strlcpy(kbuf, theNick, KEY_SIZE);
				strlcat(kbuf, ".secure", KEY_SIZE);
				lcaseit(kbuf);
				key.data = kbuf;
				key.size = strlen(kbuf);
				if (!userdb->get(userdb, &key, &data, 0))
					retval = -2;
			}
		}
	} else {
		if (forWhom >= 0) {
			snprintf(line, LINE_SIZE, "%s is not in the database.", theNick);
			senderror(forWhom, line);
		}
		retval = -1;
	}

	return retval;
}
