/*
 * Copyright (c) 2005 Jacob Meuser <jakemsr@jakemsr.com>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/*
 * $Id: bsdav_ratios.c,v 1.6 2006/03/24 10:22:44 jakemsr Exp $
 */

#include "includes.h"

#include <err.h>
#include <limits.h>
#include <stdlib.h>
#include <string.h>

#include "bsdav.h"


/* Parse ratio input string, for example "16:9",
 * into a bsdav_ratio structure.
 */

int
bsdav_parse_ratio(char *input, struct bsdav_ratio *ratio)
{
const char *errstr;
char	*q;
size_t	 span, slen;
int	 sret;

	slen = strlen(input);

 	span = strcspn(input, ":");
	if (span > slen) {
		warnx("ratio string would overflow");
		return(1);
	}

	if ((q = malloc(slen)) == NULL) {
		warn("slen, size = %lld", (long long)slen);
		return(1);
	}

	/* truncation is intentional */
	sret = snprintf(q, span + 1, input);
	if (sret < 0) {
		warnx("ratio string error");
		free(q);
		return(1);
	}
	ratio->n = (unsigned int)strtonum(q, 0, UINT_MAX, &errstr);
	if (errstr != NULL) {
		warnx("ratio numerator is %s: %s", errstr, q);
		free(q);
		return(1);
	}

	/* if no ':', assume denominator of 1 */
	if (span == slen)
		ratio->d = 1;
	else {
		slen -= span;
		if (slen <= span) {
			warnx("ratio denominator is NULL");
			free(q);
			return(1);
		}
		sret = snprintf(q, slen, input + span + 1);
		if (sret >= slen) {
			warnx("truncated ratio string %ld characters",
			    sret - span);
			free(q);
			return(1);
		} else if (sret < 0) {
			warnx("ratio string error");
			free(q);
			return(1);
		}
		/* denominator must be at least 1 */
		ratio->d = (unsigned int)strtonum(q, 1, UINT_MAX, &errstr);
		if (errstr != NULL) {
			warnx("ratio denominator is %s: %s", errstr, q);
			free(q);
			return(1);
		}
	}

	free(q);
	return(0);
}


/* Parse crop values input string, for example "0,8,0,8",
 * into a bsdav_crop structure.
 */

int
bsdav_parse_crop(char *input, struct bsdav_crop *crop)
{
char	**ap;
char	*crops[4];	/* t,l,b,r */
const char *errstr;


	for (ap = crops; ap < &crops[4] &&
	    (*ap = strsep(&input, ",")) != NULL;) {
		if (**ap != '\0')
			ap++;
	}
	*ap = NULL;

	/* top */
	if (crops[0] != '\0') {
		crop->t = (unsigned int)strtonum(crops[0], 0,
		    BKTR_HEIGHT_MAX, &errstr);
		if (errstr != NULL) {
			warnx("crop top is %s: %s", errstr, crops[0]);
			return(1);
		}
	} else
		crop->t = 0;

	/* bottom */
	if (crops[2] != '\0') {
		crop->b = (unsigned int)strtonum(crops[2], 0,
		    BKTR_HEIGHT_MAX - (long long)crop->t, &errstr);
		if (errstr != NULL) {
			warnx("crop bottom is %s: %s", errstr, crops[2]);
			return(1);
		}
	} else
		crop->b = 0;

	/* left */
	if (crops[1] != '\0') {
		crop->l = (unsigned int)strtonum(crops[1], 0,
		    BKTR_WIDTH_MAX, &errstr);
		if (errstr != NULL) {
			warnx("crop left is %s: %s", errstr, crops[1]);
			return(1);
		}
	} else
		crop->l = 0;

	/* right */
	if (crops[3] != '\0') {
		crop->r = (unsigned int)strtonum(crops[3], 0,
		    BKTR_WIDTH_MAX - (long long)crop->l, &errstr);
		if (errstr != NULL) {
			warnx("crop right is %s: %s", errstr, crops[3]);
			return(1);
		}
	} else
		crop->r = 0;

	return(0);
}
