/*
 * Copyright (c) 2005 Jacob Meuser <jakemsr@jakemsr.com>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/*
 * bsdav.h - audio/video API
 *
 * $Id: bsdav.h,v 1.19 2006/03/24 10:22:44 jakemsr Exp $
 *
 */

#ifndef _BSDAV_H_
#define _BSDAV_H_

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <stdio.h>

#define DEFAULT_VIDEO_DEVICE	"/dev/bktr0"
#if defined(__OpenBSD__) || defined(__NetBSD__)
# define DEFAULT_AUDIO_DEVICE	"/dev/audio"
# define HAVE_SUN_AUDIO
# include <sys/audioio.h>
# include <dev/ic/bt8xx.h>
#elif defined(__FreeBSD__) || defined(__DragonFly__)
# define DEFAULT_AUDIO_DEVICE	"/dev/dsp"
# define HAVE_OSS_AUDIO
# include <sys/soundcard.h>
# if defined(__FreeBSD__)
#  include <sys/param.h>
#  if __FreeBSD_version >= 502100
#   include <dev/bktr/ioctl_meteor.h>
#   include <dev/bktr/ioctl_bt848.h>
#  else
#   include <machine/ioctl_meteor.h>
#   include <machine/ioctl_bt848.h>
#  endif
# elif defined(__DragonFly__)
#  include <dev/video/meteor/ioctl_meteor.h>
#  include <dev/video/bktr/ioctl_bt848.h>
# endif
#endif

/* frame rates, aspect ratios, etc */

struct bsdav_ratio {
	unsigned int 	n;
	unsigned int 	d;
};


/* ring buffers */

struct bsdav_rb_buf {
	uint8_t		*buf;
	size_t		 size;
	struct timeval	 ts;
};

struct bsdav_ringbuf {
	struct bsdav_rb_buf *bufs;
	int		 num_bufs;
	int		 buf_in;
	int		 buf_out;
	int		 bufs_off;
};


/* video format info */

struct bsdav_vid_formats {
	char	*name;
	int	 bpp;
	int	 xv_id;
	int	 bktr_idx;
};

enum bsdav_vidfmts {
	BSDAV_VIDFMT_NONE	= 0,
	BSDAV_VIDFMT_YV12	= 1,
	BSDAV_VIDFMT_I420	= 2,
	BSDAV_VIDFMT_YV16	= 3,
	BSDAV_VIDFMT_YUY2	= 4,
	BSDAV_VIDFMT_UYVY	= 5,
	BSDAV_VIDFMT_RV15	= 6,
	BSDAV_VIDFMT_RV16	= 7,
	BSDAV_VIDFMT_LAST	= 8
};

/* libbsdav provides the following array.  The array index for a format
 * corresponds to it's bsdav_vidfmts value above.
 */
extern struct bsdav_vid_formats bsdav_vid_fmts[BSDAV_VIDFMT_LAST + 1];


/* audio format info */

struct bsdav_aud_formats {
	char	*name;
	int	 bps;
	int	 audio_enc;
	int	 audio_idx;
};

enum bsdav_audfmts {
	BSDAV_AUDFMT_NONE	= 0,
	BSDAV_AUDFMT_ULAW	= 1,
	BSDAV_AUDFMT_ALAW	= 2,
	BSDAV_AUDFMT_SLLE	= 3,
	BSDAV_AUDFMT_SLBE	= 4,
	BSDAV_AUDFMT_ULLE	= 5,
	BSDAV_AUDFMT_ULBE	= 6,
	BSDAV_AUDFMT_SLIN	= 7,
	BSDAV_AUDFMT_ULIN	= 8,
	BSDAV_AUDFMT_LAST	= 9
};

/* libbsdav provides the following array.  The array index for a format
 * corresponds to it's bsdav_audfmts value above.
 */
extern struct bsdav_aud_formats bsdav_aud_fmts[BSDAV_AUDFMT_LAST + 1];

#if defined(HAVE_SUN_AUDIO)
# define BSDAV_AUDMODE_PLAY	AUMODE_PLAY_ALL
# define BSDAV_AUDMODE_REC	AUMODE_RECORD
#elif defined(HAVE_OSS_AUDIO)
# define BSDAV_AUDMODE_PLAY	0
# define BSDAV_AUDMODE_REC	0
#endif

/* video norm info */

struct bsdav_vid_norms {
	char		*name;
	struct bsdav_ratio frame_rate;
	int		 bktr_id;
	int		 width;
	int		 height;
};

enum bsdav_vidnorms {
	BSDAV_VIDNORM_NONE	= 0,
	BSDAV_VIDNORM_NTSC	= 1,
	BSDAV_VIDNORM_PAL	= 2,
	BSDAV_VIDNORM_SECAM	= 3,
	BSDAV_VIDNORM_LAST	= 4
};

/* libbsdav provides the following array.  The array index for a format
 * corresponds to it's bsdav_vidnorms value above.
 */
extern struct bsdav_vid_norms bsdav_vid_nrms[BSDAV_VIDNORM_LAST + 1];


struct bsdav_vid_sources {
	char	*name;
	int	 bktr_id;
};

enum bsdav_vidsources {
	BSDAV_VIDSOURCE_NONE	= 0,
	BSDAV_VIDSOURCE_RCA	= 1,
	BSDAV_VIDSOURCE_TUNER	= 2,
	BSDAV_VIDSOURCE_SCOMP	= 3,
	BSDAV_VIDSOURCE_SVID	= 4,
	BSDAV_VIDSOURCE_INPT3	= 5,
	BSDAV_VIDSOURCE_LAST	= 6
};

/* libbsdav provides the following array.  The array index for a source
 * corresponds to it's bsdav_vidsources value above.
 */
extern struct bsdav_vid_sources bsdav_vid_srcs[BSDAV_VIDSOURCE_LAST + 1];


#define BKTR_WIDTH_MAX		768
#define BKTR_HEIGHT_MAX		576
#define BKTR_FPS_MAX		30


struct bsdav_crop {
	unsigned int	 t;	/* top */
	unsigned int	 l;	/* left */
	unsigned int	 b;	/* bottom */
	unsigned int	 r;	/* right */
};


/*
 * stream header
 */

struct bsdav_stream_header {
	long	tssec;	/* start time seconds */
	long	tsusec;	/* start time microseconds */
	int	vidfmt;	/* video format */
	int	vidwth;	/* video width */
	int	vidhgt;	/* video height */
	int	vidmfs;	/* video max. frame size */
	int	audfmt;	/* audio format */
	int	audchn;	/* audio channels */
	int	audsrt;	/* audio sample rate */
	int	audmfs;	/* audio max. frame size */
};

#define BSDAV_STRHDR_NMEM 10
#define BSDAV_STRHDR_SIZE 104  /* 8 + 16 * 2 + 8 * 8 */
#define BSDAV_STRHDR_SFMT "BSDAVSTR%016lx%016lx%08x%08x%08x%08x%08x%08x%08x%08x"

/*
 * frame header
 */

#define BSDAV_FRMTYP_NONE	0
#define BSDAV_FRMTYP_AUDIO	1
#define BSDAV_FRMTYP_VIDEO	2
#define BSDAV_FRMTYP_LAST	3

struct bsdav_frame_header {
	long	tssec;	/* timestamp seconds */
	long	tsusec;	/* timestamp micro seconds */
	int	type;	/* data type */
	int	size;	/* data size */
	int	num;	/* number */
};

#define BSDAV_FRMHDR_NMEM 5
#define BSDAV_FRMHDR_SIZE 64  /* 8 + 16 * 2 + 3 * 8 */
#define BSDAV_FRMHDR_SFMT "BSDAVFRM%016lx%016lx%08x%08x%08x"


/* Returns index of video format in bsdav_vid_fmts[], given a name.
 * Returns -1 on failure.
 */
int bsdav_find_vid_fmt(char *);

/* Returns index of audio format in bsdav_aud_fmts[], given a name.
 * Returns -1 on failure.
 */
int bsdav_find_aud_fmt(char *);

/* Returns index of video norm in bsdav_vid_nrms[], given a name.
 * Returns -1 on failure.
 */
int bsdav_find_vid_norm(char *);

/* Returns index of video source in bsdav_vid_srcs[], given a name.
 * Returns -1 on failure.
 */
int bsdav_find_vid_source(char *);


/* Parses ratio strings to struct bsdav_ratio, such as,
 *     "16:9" -> ratio.n = 16, ratio.d = 9 
 * Returns 0 on success, 1 on error.
 */
int bsdav_parse_ratio(char *, struct bsdav_ratio *);

/* Parses crop value strings such as "0,8,0,8" into a struct bsdav_crop.
 * Returns 0 on success, 1 on error.
 */
int bsdav_parse_crop(char *, struct bsdav_crop *);


/* Allocates memory and sets initial values for a ring buffer.
 * The num_bufs member must be set to the desired number of buffers.
 * Returns 0 on success, 1 on error.
 */
int bsdav_init_ringbuf(struct bsdav_ringbuf *, size_t);

/* Deallocates memory of a ring buffer and sets all pointers associated
 * with the ring buffer structures to NULL.
 */
void bsdav_free_ringbuf(struct bsdav_ringbuf *);

/* Add a new buffer to the ringbuffer.  Increments rb->buf_in and rb_bufs_off.
 * Returns 0 on success, 1 on error.
 */
int bsdav_rb_buf_in(struct bsdav_ringbuf *, uint8_t *, size_t, struct timeval);


/* Returns FILE position of stream header.
 */
long bsdav_find_stream_header(FILE *);

/* read and write stream headers */
int bsdav_write_stream_header(FILE *, struct bsdav_stream_header *);
int bsdav_read_stream_header(FILE *, struct bsdav_stream_header *);

/* pretty print the values of a stream header */
void bsdav_dump_stream_header(struct bsdav_stream_header *);


/* read and write frame headers */
int bsdav_write_frame_header(FILE *, struct bsdav_frame_header *);
int bsdav_read_frame_header(FILE *, struct bsdav_frame_header *);

/* read and write frame data */
int bsdav_read_frame_data(FILE *, uint8_t *, size_t, int);
int bsdav_write_frame_data(FILE *, uint8_t *, size_t, int);

/* pretty print the values of a frame header */
void bsdav_dump_frame_header(struct bsdav_frame_header *);

/* probe n MB of a stream to determine the frame rate */
double bsdav_probe_frame_rate(FILE *, long);


/* fills in aud_formats with current device info */
int bsdav_get_aud_formats(int);

/* pretty print audio device info */
int bsdav_list_audio_formats(char *, int);

/* fills in bsdav_vid_fmts[] with current device info */
int bsdav_get_vid_formats(int);

/* pretty print bktr device info */
int bsdav_list_bktr_formats(char *, int);


/* initialize audio device */
int bsdav_audio_init(int, int, int, int, int);

/* get total size of audio device's data buffer */
size_t bsdav_get_audhw_buf_size(int, int);

/* get position (number of available bytes) in the buffer */
size_t bsdav_get_audhw_buf_pos(int, int);


/* initialize bktr device */
int bsdav_bktr_init(int, int, int *, int *, int, int, int, int *,
    struct bsdav_crop *);

/* mmap(2) video device buffer.  Returns buffer size, or 0 if there
 * was an error.
 */
size_t bsdav_map_vid_buffer(uint8_t **, int, int, int, int);

/* munmap video device buffer */
void bsdav_unmap_vid_buffer(uint8_t **, size_t);


/* Write process ID to a file */
int bsdav_write_pid(char *);

#endif

