/*
   Copyright (C) 2008-2011 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, see <http://www.gnu.org/licenses/>.
*/
#include "settings.h"

#include "log.h"

#include <kconfig.h>
#include <ksharedconfig.h>
#include <kconfiggroup.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#define KOA_DEFAULT_CDDB_SERVER "http://freedb.freedb.org"
#define KOA_DEFAULT_CDDB_PORT 80
#define KOA_DEFAULT_CDDB_ACCESS Settings::CDDB_ANY
#define KOA_DEFAULT_COVER_IMAGE_SEARCH_URL "http://images.google.com/images?q=%1"

static const char* deviceNames[] = { "/dev/cdrom", "/dev/dvd", "/dev/sr", 0 };

namespace KoverArtist
{

Settings* Settings::sInstance = 0;


Settings::Settings()
:localImages(true)
,removeImageBorders(true)
,cdDevice()
,cddbServer(KOA_DEFAULT_CDDB_SERVER)
,cddbPort(KOA_DEFAULT_CDDB_PORT)
,cddbAutoSetTitle(true)
,cddbDirs()
,cddbCacheFiles(true)
,cddbAccess(KOA_DEFAULT_CDDB_ACCESS)
,coverImageSearchUrl(KOA_DEFAULT_COVER_IMAGE_SEARCH_URL)
{
}


void Settings::load()
{
   KConfig cfg("koverartistrc");

   const KConfigGroup& globalsGroup = cfg.group("Globals");
   localImages = globalsGroup.readEntry("local_images", true);
   removeImageBorders = globalsGroup.readEntry("remove_image_borders", true);
   coverImageSearchUrl = globalsGroup.readEntry("cover_image_search_url", KOA_DEFAULT_COVER_IMAGE_SEARCH_URL);

   const KConfigGroup& audioCdGroup = cfg.group("AudioCD");
   cdDevice = audioCdGroup.readEntry("cd_device", "");
   cddbServer = audioCdGroup.readEntry("cddb_server", KOA_DEFAULT_CDDB_SERVER);
   cddbPort = audioCdGroup.readEntry("cddb_port", KOA_DEFAULT_CDDB_PORT);
   cddbAutoSetTitle = audioCdGroup.readEntry("cddb_auto_set_title", true);
   cddbCacheFiles = audioCdGroup.readEntry("cddb_cache_files", true);
   cddbAccess = (CddbAccess) audioCdGroup.readEntry("cddb_access", (int) KOA_DEFAULT_CDDB_ACCESS);
   audioCdGroup.readXdgListEntry("cddb_dirs", cddbDirs);

   if (cdDevice.isEmpty())
      cdDevice = findCdDevice();

   if (cddbDirs.isEmpty())
   {
      cddbDirs.append(QString(getenv("HOME")) + "/.cddb");
#ifdef CDDB_PATH
#define QUOTE(x) #x
      cddbDirs.append(QUOTE(CDDB_PATH));
#endif
   }
}


void Settings::sync()
{
   KConfig cfg("koverartistrc");

   KConfigGroup globalsGroup(&cfg, "Globals");
   globalsGroup.writeEntry("local_images", localImages);
   globalsGroup.writeEntry("remove_image_borders", removeImageBorders);
   globalsGroup.writeEntry("cover_image_search_url", coverImageSearchUrl);

   KConfigGroup audioCdGroup(&cfg, "AudioCD");
   audioCdGroup.writeEntry("cd_device", cdDevice);
   audioCdGroup.writeEntry("cddb_server", cddbServer);
   audioCdGroup.writeEntry("cddb_port", cddbPort);
   audioCdGroup.writeEntry("cddb_auto_set_title", cddbAutoSetTitle);
   audioCdGroup.writeXdgListEntry("cddb_dirs", cddbDirs);
   audioCdGroup.writeEntry("cddb_cache_files", cddbCacheFiles);
   audioCdGroup.writeEntry("cddb_access", int(cddbAccess));

   cfg.sync();
}


QString Settings::findCdDevice()
{
   QString name;

   for (int i=0; deviceNames[i]; ++i)
   {
      for (int idx=-1; idx<4; ++idx)
      {
         if (idx<0) name = deviceNames[i];
         else name = QString("%1%2").arg(deviceNames[i]).arg(idx);

         if (access(name.toLocal8Bit(), R_OK)==0)
         {
            logDebug("Found cd device: %1").arg(name);
            return name;
         }
      }
   }

   logWarning("No accessable cd device found");
   return QString::null;
}


} //namespace
